#include "dat.h"
#include "fns.h"

enum {
	SSH_MSG_USERAUTH_PK_OK	= 60,
};

struct AuthContext {
	int		nkeys;
};

static	char	*msgnameauthpk(int);

static int
authpkinit(Conn *c, AuthContext **ctx)
{
	int rv;

	rv = 0;
	if(*ctx == nil){
		*ctx = emalloc(sizeof **ctx);
		if(c->authcerts == nil)
			c->authcerts = certprivload(c);
		if(c->authcerts == nil)
			rv = -1;
	}
	debug(DbgAuth, "authpkinit %d %d", (*ctx)->nkeys, rv);
	c->msgnameauth = msgnameauthpk;
	return rv;
}

static int
authpk(Conn *c, Channel *rc, Channel *wc, AuthContext *ctx, char *service)
{
	int n;
	int rv;
	char *fp;
	uchar buf[16];

	Msg *m;
	Sig *sig;
	Cert *cert;
	CertList *cl;
	Blob *b, *certblob, *sigblob;

	USED(rc);

	cert = nil;
	sigblob = nil;
	certblob = nil;
	cl=c->authcerts;
	for(n=0; cl != nil && n < ctx->nkeys; n++)
		cl=cl->link;
	if(cl == nil)
		goto Abort;

	ctx->nkeys++;
	cert = cl->cert;

	/* cert marshalled once, but used multiple times */
	certblob = mkblob(sizecert(cert));
	putcert(certblob, cert);

	fp = fingerprintcert(cert, FPHex, md5, MD5dlen);
	debug(DbgAuth, "authpk: key fingerprint: %s", fp);
	free(fp);

	debug(DbgAuth, "authpk: request %s with publickey auth.", service);
	m = ssh_msg_userauthreq(c, service, "publickey");
	putbool(&m->b, 0);		/* false -> initial request */
	putstring(&m->b, cert->name);
	putblobstring(&m->b, certblob);
	sendmsg(wc, m);
	debug(DbgAuth, "authpk: initial request sent");

	m = authrecvmsg(c, rc, MSGIGN);
	if(m->type != SSH_MSG_USERAUTH_PK_OK){
		debug(DbgAuth, "authpk: rejected by server");
		goto Error;
	}
	/* XXX: verify algorithm name and key blob returned are the same */
	freemsg(m);

	/* construct blob to be signed */
	n = sizeblobstring(c->sessionid)+1;
	n += sizestring(c->user)+sizestring(service)+sizestring("publickey");
	n += 1+sizestring(cert->name)+sizeblobstring(certblob);
	b = mkblob(n);

	/* XXX: use msgpayload() */
	putblobstring(b, c->sessionid);
	buf[0] = SSH_MSG_USERAUTH_REQUEST;
	putbytes(b, buf, 1);
	putstring(b, c->user);
	putstring(b, service);
	putstring(b, "publickey");
	putbool(b, 1);
	putstring(b, cert->name);
	putblobstring(b, certblob);
	sig = cert->sign(cert, b->bp, b->wp - b->bp);
	clearblob(b);

	sigblob = mkblob(sizesig(sig));
	putsig(sigblob, sig);

	debug(DbgAuth, "authpk: about to send signed session ID");

	m = ssh_msg_userauthreq(c, service, "publickey");
	putbool(&m->b, 1);		/* true -> second request */
	putstring(&m->b, cert->name);
	putblobstring(&m->b, certblob);
	putblobstring(&m->b, sigblob);
	sendmsg(wc, m);

	sig->free(sig);			/* sig is local; cert is not */
	clearblob(sigblob);
	clearblob(certblob);

	m = authrecvmsg(c, rc, MSGIGN);
	rv = authgeneric(c, rc, wc, m, 0);
	return rv;

 Error:
	rv = authgeneric(c, rc, wc, m, 0);
	/* fall through */

 Abort:
	clearblob(sigblob);
	clearblob(certblob);
	return UAfail;
}

static char*
msgnameauthpk(int t)
{
	if(t == SSH_MSG_USERAUTH_PK_OK)
		return "SSH_MSG_USERAUTH_PK_OK";
	return "<unknown>";
}

Auth authpublickey = {
	"publickey",
	authpkinit,
	authpk
};
