/*
 * Simple root file system
 */

#include "a.h"

typedef struct Xfile Xfile;
struct Xfile
{
	Xfile *parent;
	Xfile **child;
	int nchild;
	Dir d;
	char *data;
};

static Xfile *root;
static int qidgen;

int
rootcreate(char *path, ulong mode)
{
	char *p, *nextp;
	int i;
	Xfile *xf, *nf;
	
	if(root == nil){
		p = path;
		xf = nil;
		goto Create;
	}

	xf = root;
	for(p=path; p && *p; p=nextp){
		if(*p == '/'){
			nextp = p+1;
			continue;
		}
		nextp = strchr(p, '/');
		if(nextp == nil){
			/* last piece */
			for(i=0; i<xf->nchild; i++)
				if(strcmp(p, xf->child[i]->d.name) == 0){
					werrstr("%s already exists", path);
					return -1;
				}
			goto Create;
		}
		for(i=0; i<xf->nchild; i++){
			if(memcmp(p, xf->child[i]->d.name, nextp-p) == 0)
			if(xf->child[i]->d.name[nextp-p] == 0){
				xf = xf->child[i];
				goto Found;
			}
		}
		werrstr("%.*s does not exist", utfnlen(path, nextp-path), path);
		return -1;

	Found:
		nextp++;
	}
	werrstr("%s already exists", path);
	return -1;

Create:
	nf = emalloc(sizeof *nf);
	nf->d.name = estrdup(p);
	nf->d.mode = mode;
	nf->d.uid = user;
	nf->d.gid = user;
	nf->d.muid = "";
	nf->d.qid.type = mode>>24;
	nf->d.qid.path = ++qidgen;
	nf->d.qid.vers = 0;
	nf->d.mtime = time(0);
	nf->d.atime = time(0);
	if(xf){
		nf->parent = xf;
		xf->child = erealloc(xf->child, (xf->nchild+1)*sizeof(xf->child[0]));
		xf->child[xf->nchild++] = nf;
	}else{
		nf->parent = nf;
		root = nf;
	}
	return 0;
}

Xfile*
rootwalk(Xfile *xf, char *name)
{
	int i;
	
	for(i=0; i<xf->nchild; i++)
		if(strcmp(xf->child[i]->d.name, name) == 0)
			return xf->child[i];
	return nil;
}

void
rootreqthread(void *v)
{
	int i, tot, n, off, have;
	uchar *a;
	Fs *fs;
	Req *r;
	Xfile *xf, *nf;
	static int need[] = { 4, 2, 6, 1};

	fs = v;
	while((r = recvp(fs->reqchan)) != nil){
		xf = nil;
		if(r->fid){
			xf = r->fid->aux;
			if(xf == nil)
				r->fid->aux = xf = root;
		}
		switch(r->ifcall.type){
		case Tflush:
			respond(r, nil);
			break;
	
		case Topen:
			have = (xf->d.mode>>6)&7;
			if((r->ifcall.mode&~3) || (need[r->ifcall.mode&3] & ~have)){
				respond(r, "permission denied");
				break;
			}
			respond(r, nil);
			break;	

		case Tread:
			if(!(xf->d.mode&DMDIR)){
				/* files */
				tot = xf->d.length;
				if(r->ifcall.offset >= tot){
					r->ofcall.count = 0;
					respond(r, nil);
					break;
				}
				if(r->ifcall.count+r->ifcall.offset > tot)
					r->ofcall.count = tot - r->ifcall.offset;
				memmove(r->ofcall.data, xf->data, r->ofcall.count);
				respond(r, nil);
				break;
			}
			/* directories */
			off = r->ifcall.offset;
		fprint(2, "kids of %s\n", xf->d.name);
			for(i=0; i<xf->nchild && off > 0; i++)
				off -= sizeD2M(&xf->child[i]->d);
			tot = r->ifcall.count;
			a = r->ofcall.data;
			have = 0;
			for(; i<xf->nchild; i++){
				n = convD2M(&xf->child[i]->d, a+have, tot-have);
				if(n == BIT16SZ)
					break;
				have += n;
			}
			if(i < xf->nchild && have == 0){
				respond(r, "directory entry too large");
				break;
			}
			r->ofcall.count = have;
			respond(r, nil);
			break;

		case Tstat:
			r->d = xf->d;
			r->d.name = estrdup(r->d.name);
			r->d.uid = estrdup(r->d.uid);
			r->d.gid = estrdup(r->d.gid);
			r->d.muid = estrdup(r->d.muid);
			respond(r, nil);
			break;

		case Twalk:
			for(i=0; i<r->ifcall.nwname; i++){
				nf = rootwalk(xf, r->ifcall.wname[i]);
				if(nf == nil)
					break;
				/* not going to bother with permission checking */
				r->ofcall.wqid[i] = nf->d.qid;
				xf = nf;
			}
			r->ofcall.nwqid = i;
			if(r->ifcall.nwname && !r->ofcall.nwqid){
				wrespond(r, "path not found");
				break;
			}
			r->newfid->aux = xf;
			wrespond(r, nil);
			break;

		case Tcreate:
		case Tremove:
		case Twstat:
		case Twrite:
			respond(r, "permission denied");
			break;	
		}
	}
}

Fs*
mkrootfs(Qid *rootqid)
{
	Fs *fs;

	rootcreate("/", DMDIR|0555);

	fs = emalloc(sizeof *fs);
	fs->name = "root";
	fs->reqchan = chancreate(sizeof(void*), 4);
	threadcreate(rootreqthread, fs, STACK);

	*rootqid = root->d.qid;
	return fs;
}
