;;;; -*- Mode: lisp; indent-tabs-mode: nil -*-

(defpackage :eval
  (:use :common-lisp :csp)
  (:export
   #:start
   #:espawn))

;; Example: multithreaded debugging.
;;
;; Allow the user to choose an appropriate restart interactively when
;; a thread raises a condition.
;; 
;; Call start to start the REPL. Any new threads should be started
;; with espawn to give them access to the debugger.
;; 
;; This is just a proof-of-concept. The code has been deliberately kept simple.
;; 
;; The main difficulty the code faces is controlling access to the standard
;; input, making sure that two threads aren't competing for the same input.
;; 
;; I don't believe the problem can be solved in general (one *standard-input*, many threads)
;; but we try at least to solve the case for the REPL vs. erroring thread.
;; If a thread raises a condition half-way through a typed expression,
;; the expression reading is put on hold while the user chooses a restart.
;; any other threads raising a condition in the meantime will be blocked.
;; 
;; The main problem we can't solve is coping with the :interactive keyword
;; in restart-case correctly. The :interactive function needs to be called in the context
;; of the originating thread, but there's no way of calling it without committing
;; to the restart, and by doing that we lose the ability to know when the
;; interaction has completed. The upshot of this is that if there's a :interactive
;; function, it will compete for input with the REPL.
;; 
;; Exercises for the reader (none of these should be too hard :-])
;; 
;; - Interactive REPL when in debug mode. expressions should be
;; interpreted in the context of the thread that raised the error.
;; 
;; - Multi-window debug; we could create a new window/panel
;; for each condition raised, each with its own REPL, thus avoiding
;; the single-input problem.
;; 
;; - Remove the terminate-thread restart from the central
;; evaluation loop.
;; 
;; - Handle eof properly.
;; 
;; - Attach to currently running threads that have not raised a condition.
;; 
;; - All the other stuff usually provided by the debugger.
;; 
;; - Get it to play nicely with SLIME.

(in-package :eval)

;; when a thread intercepts a condition, it sends on *condc*
;; (cond restarts reply), where cond is the condition itself,
;; restarts is a list of (restart restart-text), the list of all possible
;; restarts + their associated text, and reply is a channel
;; on which will be sent the chosen restart, once the user
;; has made the choice.
(defvar *condc* (chan))

;; read lines upon request from the standard input;
;; to request a line, send a channel, say r, on rc;
;; readlinesproc will then read a line and send it on r.
(defun readlinesproc (rc)
  (loop
   (let ((reply (? rc)))
     (note "got readline request")
     (handler-case
	 (multiple-value-bind (s no-nl) (read-line)
	   (when (not no-nl)
	     (setf s (string+ s #(#\newline))))
              (note "got line ~s~%" s)
	   (! reply s))
       (end-of-file ()
	 (! reply nil))))))

;; get expressions from central process and evaluate them.
(defun evalproc (ec)
  (let ((reply (chan)))
    (loop
     (note "evalproc sending")
     (! ec reply)
     (note "evalproc sent")
     (let ((expr (? reply)))
       (note "evalproc got reply")
       (restart-case (progn
		       (format t "~s~%" (eval expr))
		       (force-output))
	 (abort () :report "Return to top level" nil))))))

;; moderate access to reader.
;; if we're asked to read an expression, then
;; accumulate lines into a string buffer until we don't get eof,
;; upon which we can send the complete expression back to be evaluated.
;; if we're told about a condition that needs handling, we
;; put the expression reading on hold and deal with it there and then.
(defun centralproc (linesc evalc condc)
  (let ((evalreplyc nil) (ebuf "") (linereplyc (chan)))
    (loop
     (alt
       ((? evalc reply)
        (note "centralproc got eval req")
        (setf evalreplyc reply)
        (format t "+ ")
        (force-output)
        (note "sending line request")
        (! linesc linereplyc)
        (note "sent line request"))
       ((? linereplyc line)
        (note "centralproc got line")
        (setf ebuf (concatenate 'string ebuf line))
        (multiple-value-bind (expr status) (parsestring ebuf)
          (case status
            (:ok
             (! evalreplyc expr)
             (setf evalreplyc nil)
             (setf ebuf ""))
            (:err
             (format t "some error~%")
             (setf ebuf ""))
            (:partial
             (! linesc linereplyc)))))
       ((? condc (cond restarts reply))
        (note "centralproc got condition")
        ;; if we've been asked for an expression to evaluate,
        ;; then we've already asked the line reader for a line,
        ;; so to get a line, get its reply and send another request,
        ;; otherwise send the request and receive the reply.
        (! reply
           (dialog cond restarts
                   (if evalreplyc
                       #'(lambda () (let ((k (? linereplyc))) (! linesc linereplyc) k))
                       #'(lambda () (! linesc linereplyc) (? linereplyc)))))
        (when evalreplyc
          (format t "+ ~a" ebuf)
          (force-output)))))))

(defun parsestring (s)
  (handler-case
      (let ((r (read (make-string-input-stream s))))
	(values r :ok))
    (end-of-file () (values nil :partial))
    (t () (values nil :error))))

;; interact with the user to find out which condition to use.
;; restarts is a list of (restart restart-name), as condition-report
;; needs to be evaluated in the context of the originating thread,
;; so we can't get it here.
(defun dialog (cond restarts getline)
  (format t "~a~%" cond)
  (setf restarts (reverse restarts))
  (loop
   (let ((n 0))
     (dolist (c restarts)
       (format t "~d: [~a] ~a~%" (incf n) (restart-name (car c)) (cadr c)))
     (format t "which one? ")
     (force-output)
     (let ((ans (parse-integer (funcall getline) :junk-allowed t)))
       (when (and (>= ans 1) (<= ans n))
	 (return-from dialog (car (nth (1- ans) restarts)))))
     (format t "choose a number between 1 and ~d~%" n))))

(defmacro espawn (&rest forms)
  "Same as csp:spawn, but wrap the forms in a handler to
allow the debugger to be notified correctly"
  `(spawn
    (handler-bind
        ((t #'(lambda (c)
                (let ((reply (chan)))
                  (! *condc*
                     (list
                      c
                      (mapcar
                       #'(lambda (x) (list x (condition-message x))) (compute-restarts))
                      reply))
                  (invoke-restart (? reply))))))
      ,@forms)))

(defun start ()
  "Start the REPL and enable threaded debugging"
  (let ((linesc (chan)) (evalc (chan)))
    (spawn (readlinesproc linesc))
    (espawn (evalproc evalc))
    (centralproc linesc evalc *condc*)))

(defun string+ (&rest s)
  (apply #'concatenate 'string s))

;; pinched from swank:

(defun unsafe-condition-message (condition)
  #+sbcl
    (let ((sb-int:*print-condition-references* nil))
      (princ-to-string condition))
  #-sbcl
    (princ-to-string condition))

(defun condition-message (condition)
  "Safely print condition to a string, handling any errors during
printing."
  (let ((*print-pretty* t))
    (handler-case
        (unsafe-condition-message condition)
      (error (cond)
        ;; Beware of recursive errors in printing, so only use the condition
        ;; if it is printable itself:
        (format nil "Unable to display error condition~@[: ~A~]"
                (ignore-errors (princ-to-string cond)))))))

