#include <u.h>
#include <libc.h>
#include <fcall.h>
#include "dat.h"
#include <auth.h>
#include "fns.h"


static char Ebadfid[] = "unknown fid";
static char Efidinuse[] = "fid already in use";
static char Efileexcl[] = "exclusive files are blocked by recover";
static char Enoauth[] = "authentication not required";

/*
 * send back to client
 */
static void
csend(Fcall *f)
{
	uchar *buf;
	long n;
	
	buf = emalloc(MAXPKT);
	n = convS2M(f, buf, MAXPKT);
	if(n < 0)
		sysfatal("csend serialization %r");

	if(chatty && f->type != Rread && f->type != Rwrite)
		syslog(0, logfile, "[%d] srv <- %F", getpid(), f);

	last[nlast++&(nelem(last)-1)] = *f;

	chat("srv -> %F\n", f);
	if(write(srvfd, buf, n) != n)
		sysfatal("write to posted service: %r");
	free(buf);
}

/*
 * send error to client
 */
static void
cerror(ushort tag, char *msg)
{
	Fcall f;

	f.type = Rerror;
	f.tag = tag;
	f.ename = msg;
	csend(&f);
}

/*
 * add elem to c->name
 */
static void
namewalk(Cname *c, char *elem)
{
	c->name = erealloc(c->name, c->len+strlen(elem)+1);
	strcat(c->name, "/");
	strcat(c->name, elem);
	cleanname(c->name);
	c->len = strlen(c->name)+1;
}

/*
 * set name to the next element needed to turn sname into cname.
 * if sname == "/usr/rob/lib/profile" and cname == "/usr", then
 * name gets set to "rob".
 */
static void
nextelem(char *name, char *sname, char *cname)
{
	char *p, *q;

	assert(strlen(sname) <= strlen(cname));
	assert(strncmp(sname, cname, strlen(sname)) == 0);

	p = cname+strlen(sname);
	if(*p == '/')
		p++;
	q = strchr(p, '/');
	if(q == nil)
		q = p+strlen(p);
	memmove(name, p, q-p);
	name[q-p] = 0;
}

/*
 * print a request
 */
static void
dumpreq(Req *r, void*p)
{
	syslog(0, logfile, "[%d] req %F\n", getpid(), &r->fcall);
	USED(p);
}

/*
 * connection died and has been redialed.  restart a previously-sent request.
 */
static void
restartreq(Req *r, void *p)
{
	uchar *buf;
	long n;
	Req *flushr;

	USED(p);
	buf = emalloc(MAXPKT);
	if(r->internal){
		/*
		 * internal messages are specific to the connection they
		 * are created for.  no need to repeat them on later connections.
		 * other code will do that if necessary.
		 */
		chat("	freeing request: internal\n");
		freereq(r);
	}
	else if(r->fcall.type == Tclunk){
		/*
		 * hanging up the connection is the uber-clunk.
		 */
		r->fcall.type++;
		r->fcall.tag = r->tag.local;
		r->fcall.fid = r->fid->fid.local;
		freefid(r->fid);
		csend(&r->fcall);
		freereq(r);
	}else if(r->fcall.type == Tflush){
		/*
		 * we were trying to cancel a request so just don't restart it.
		 * RSC 7/15/2005: I'm sure this is broken.  If flushr has
		 * already been restarted, freeing it isn't quite correct!
		 * Also it might break the forallreqs loop.
		 * Perhaps we need to discard the flushes in a separate pass.
		 */
		if(flushr = lookuprreq(r->fcall.oldtag)){
			switch(flushr->fcall.type){
			case Tattach:
				freefid(flushr->fid);
				break;
			case Twalk:
				if(flushr->newfid)
					freefid(flushr->newfid);
				break;
			}
			freereq(flushr);
		}
		r->fcall.type++;
		r->fcall.tag = r->tag.local;
		csend(&r->fcall);
		freereq(r);
	}else{
		/*
		 * if the fid is usable, retransmit the request.
		 * RSC 7/15/2005: otherwise do we need to queue it?
		 */
		if(fidready(r->fid)){
			n = convS2M(&r->fcall, buf, MAXPKT);
			chat("net <- (restarted) %F\n", &r->fcall);
			write(netfd, buf, n);
		}
		else
			chat("	fid not ready for request %F\n", &r->fcall);
	}
	free(buf);
}	

extern int ispipenet;

/*
 * redial the connection and get going again
 */
void
redial(void)
{
	Attach *a;
	int nreq;
	

	chat("[%d] dialing %s ", getpid(), dialstring);
Retry:
	close(netfd);
	
	if(!ispipenet)
		netfd = (*doconnect)(dialstring);
	else
		netfd = open(dialstring, ORDWR);
	if(netfd < 0){
Error:
		chat("error: %r\n");
		syslog(0, logfile, "[%d] error: %r", getpid());
		if(gen == 0)
			sysfatal("can't establish initial connection");
		sleep(10);
		goto Retry;
	}

	chat("connected!\n");
	if(xversion() < 0)
		goto Error;

	for(a=attachlist; a; a=a->link){
		if(authattach(a) < 0)
			goto Retry;
		a->rgen = gen+1;
		a->gen = gen+1;
		a->rootfid->gen = gen+1;
		a->rootfid->rgen = gen+1;
	}
	gen++;
	nreq = forallreqs(restartreq, nil);
	if(nreq)
		chat("# restarted %d requests\n", nreq);
}

static void
packwalk(Cname *cname, uchar *buf, Fcall *f)
{
	char *wname[MAXWELEM];
	int i;

	strncpy((char *)buf, cname->name, cname->len);
	f->nwname = getfields((char *)buf,wname,MAXWELEM,1,"/");
	for (i=0; i<f->nwname; i++){
		f->wname[i]=wname[i];
	}
}


/*
 * is the fid f usable on the current connection?
 */
int
fidready(Fid *f)
{
	Req *r;

	if(f == nil)
		return 1;

	if(f->gen == gen)	/* fid is known on this connection */
		return 1;

	if(f->rgen == gen){	 /*we will walk it later when we are attached*/
		return 0;
	}

	if(attachready(f->attach) && f != f->attach->rootfid){
		/*
		 * RSC 7/15/2005: attachready might have changed f->gen and f->rgen, no?
		 */
		r = allocreq(taggen++, nil);
		r->fid = f->attach->rootfid;
		r->newfid = f;
		r->internal = 1;
		r->fcall.type = Twalk;
		packwalk(&f->cname, r->buf, &r->fcall);
		f->rgen = gen;
		queuereq(r);
	}

	return 0;
}

/*
 * is the attach point a usable on this connection?
 */
int
attachready(Attach *a)
{
	Req *r;

	if(a->gen == gen)	
		return 1;

	if(a->rgen == gen)
		return 0;

	a->rgen = gen;
	r = allocreq(taggen++, nil);
	r->internal = 1;
	r->fid = a->rootfid;
	r->fcall.type = Tattach;
	r->fcall.tag = r->tag.remote;
	r->fcall.fid = r->fid->fid.remote;
	r->fcall.afid = NOFID;
	r->fcall.uname = eve;
	r->fcall.aname = a->aname;
	queuereq(r);

	return 0;
}

/*
 * find the attachment for the given specifier, new one if necessary
 */
Attach*
attachment(char *spec)
{
	Attach *a;
	Fid *f; 
	
	assert(spec);

	for(a=attachlist; a; a=a->link)
		if(strcmp(a->aname, spec) == 0)
			return a;

	a = emalloc(sizeof *a);
	a->aname = estrdup(spec); /* BUG, all should be freed when clunk */
	a->rootfid = f = allocfid(fidgen++);
	f->type = QTDIR;
	f->dirscount = 0;
	f->dirccount = 0;
	f->attach = a;
	f->cname.name = estrdup("/");
	f->cname.len = 2;
	f->sname.name = estrdup("/");
	f->sname.len = 2;
	a->link = attachlist;
	attachlist = a;

	return a;
}


void
listensrv(void *x)
{
	long n;
	ushort oldtag;
	Attach *a;
	Fcall f;
	Fid *fid, *nfid;
	Req *rspc, *r;

	USED(x);
	rspc = emalloc(sizeof(Req));
	while((n=read9pmsg(srvfd, rspc->buf, MAXPKT)) >= 0){
		if(convM2S(rspc->buf, n, &f) != n)
			sysfatal("listensrv: error convM2S: %r");
		qlock(&thelock);
		if(chatty && f.type != Tread && f.type != Twrite)
			syslog(0, logfile, "[%d] srv <- %F", getpid(), &f);
		last[nlast++&(nelem(last)-1)] = f;
		chat("srv <- %F\n", &f);
		switch(f.type){
		case Tauth:
			f.type=Rerror;
			cerror(f.tag, Enoauth);
			free(rspc);
			break;
		case Tflush:
			r = lookuplreq(f.oldtag);
			if(r == nil){
				f.type++;
				csend(&f);
				break;
			}
			oldtag = r->tag.remote;
			r->flushing = 1;

			r = allocreq(f.tag, rspc);
			r->fcall.type = Tflush;
			r->fcall.oldtag = oldtag;
			queuereq(r);
			break;
			
		case Tattach:
			if((nfid=allocfid(f.fid)) == nil){
				cerror(f.tag, Efidinuse);
				break;
			}
			a = attachment(f.aname); 
			r = allocreq(f.tag, rspc);
			r->fcall.type = Twalk;
			r->newfid = nfid;
			r->fid = a->rootfid;
			r->isattach = 1;
			chat("	queuing %F\n", &(r->fcall));
			queuereq(r);
			break;

		case Twalk:
			if((fid=lookuplfid(f.fid)) == nil){
				cerror(f.tag, Ebadfid);
				break;
			}
			if((nfid=allocfid(f.newfid)) == nil){
				goto Walknotclone;
				break;
			}
			r = allocreq(f.tag, rspc);
			r->fcall = f; 
			r->fid = fid;
			r->newfid = nfid;
			queuereq(r);
			break;
		case Tversion:
			f.type++;
			f.version = "9P2000";
			f.msize = currmsize;
			csend(&f);
			free(rspc);
			break;

/*		case Twalk:
		case Topen:
		case Tcreate:
		case Tread:
		case Twrite:
		case Tclunk:
		case Tremove:
		case Tstat:
		case Twstat:
*/
Walknotclone:
		default:
			if((fid = lookuplfid(f.fid)) == nil){
				cerror(f.tag, Ebadfid);
				free(rspc);
				break;
			}
			/*
			 * We can't recover ORCLOSE files.  Instead we drop the ORCLOSE
			 * flag and perform the remove at close ourselves.
			 */
			if(f.type == Topen || f.type == Tcreate){
				fid->orclose = (f.mode & ORCLOSE) == ORCLOSE;
				f.mode &= ~ORCLOSE;
			}
			if(f.type == Topen && fid->type&QTEXCL){
				cerror(f.tag, Efileexcl);	//Exclusive open use files not allowed.
				free(rspc);
				break;
			}
			
			r = allocreq(f.tag, rspc);
			if(f.type == Tclunk && fid->orclose){
				f.type = Tremove;
				r->isclunk = 1;
			}
			r->fcall = f;
			r->fid = fid;
			queuereq(r);
			break;
		}
		qunlock(&thelock);
		rspc = malloc(sizeof(Req));
	}
	chat("Exiting listensrv %r\n");
	postnote(PNGROUP, getpid(), "anybody want a peanut?");
	_exits("listensrv");
}

/*
 * copy parameters from old to new
 */
Fid*
clonefid(Fid *old, Fid *new)
{
	new->cname.name = estrdup(old->cname.name);
	new->cname.len = strlen(new->cname.name)+1;
	new->sname.name = estrdup(old->sname.name);
	new->sname.len = strlen(new->sname.name)+1;
	new->gen = old->gen;
	new->attach = old->attach;
	return new;
}

static void
externalresponse(Req *r, Fcall *f)
{
	int i;
	Fcall cf;
	Fid *fid;
	Req *flushr;
	int nwq;
	Qid *wq;
	
	cf = *f;
	cf.tag = r->tag.local;
	
	if(r->fid){
		if(f->type == Rread && r->fid->type&QTDIR)
			r->fid->dirscount += f->count;
		else{
			r->fid->dirscount=0;
			r->fid->dirccount = 0;
		}
		if(r->fid->dirscount > r->fid->dirccount)
			r->fid->dirccount = r->fid->dirscount;
	
		if(f->type != Rerror){
			//if(r->fid->fid.remote != cf.fid)
			//	syslog(0, logfile, "[%d] fid mismatch %F %F", getpid(), &r->fcall, f);
			cf.fid = r->fid->fid.local;
		}
	}

	if(cf.type == Rattach){
		syslog(0, logfile, "[%d] unexpected Rattach %F %F", getpid(), &r->fcall, f);
		dumplast();
	}

	if(r->isattach && cf.type != Rerror){
		assert(cf.type == Rwalk);
		cf.type = Rattach;
		cf.qid = r->fid->attach->rootqid; //BUG, dont change specs...
	}
	if(r->isclunk){
		assert(cf.type == Rremove || cf.type == Rerror);
		cf.type = Rclunk;
	}

	csend(&cf);

	switch(f->type){
	case Rclunk:
	case Rremove:
		if(!r->flushing)
			freefid(r->fid);
		break;
	case Rwalk:
		if(!r->flushing){
			if(r->fcall.nwname!= f->nwqid){ //Walk did not succeed, leave everything unaffected
				chat("	newname: %d!= nwqid: %d, not walking\n", r->fcall.nwname, f->nwqid);
				freefid(r->newfid);
				r->newfid = nil;
				break;
			}
			nwq = f->nwqid;
			wq = f->wqid;
			if(r->newfid){
				clonefid(r->fid, r->newfid);
				r->newfid->type = wq[nwq-1].type;
			}
			else
				r->fid->type = wq[nwq-1].type;
			for(i = 0; i<f->nwqid; i++){
				namewalk(&r->newfid->cname, r->fcall.wname[i]);
				namewalk(&r->newfid->sname, r->fcall.wname[i]);
			}
		}
		break;
	case Rcreate:
		fid = r->fid;
		if(!r->flushing){
			namewalk(&fid->cname, r->fcall.name);
			namewalk(&fid->sname, r->fcall.name);
			fid->isopen = 1;
			fid->omode = r->fcall.mode;
		}
		break;
	case Ropen:
		fid = r->fid;
		if(!r->flushing){
			fid->isopen = 1;
			fid->omode = r->fcall.mode;
		}
		break;
	case Rflush:
		/*
		  *	If we sent a flush for them, 
		  *	we have to free the frozen fids now
		  */
		if(flushr = lookuprreq(r->fcall.oldtag)){
			switch(flushr->fcall.type){
			case Tremove:
			case Tattach:
				if(flushr->fid)
					freefid(flushr->fid);
				break;
			case Twalk:
				if(flushr->newfid && flushr->newfid != flushr->fid)
					freefid(flushr->newfid);
				break;
			}
			freereq(flushr);
		}
		break;
	case Rerror:
		switch(r->fcall.type){
		case Twalk:
			if(r->newfid != nil){
				freefid(r->newfid);
				r->newfid = nil;
			}
			break;
		case Tclunk:	/* can't happen */
		case Tremove:
			freefid(r->fid);
			break;
		}
	}

	/*
	 * If we're waiting for an Rflush to come back for that message,
	 * we can't free the request until it comes back (else we'll reuse
	 * the tag, and get confused when the Rflush arrives).
	 */
	if(r->flushing == 0)
		freereq(r);
}


static void
clientrecover(Req *r, void *v)
{
	Fid *f;


	f = v;

	chat("recovering[%lud], remote  path is: %s, for fcall %F\n" ,f->fid.remote,f->cname.name,&r->fcall);
	if(r->fid->fid.remote == f->fid.remote){
		queuereq(r);
	}

}

static void
clienterror(Req *r, void *v)
{
	Fid *f;
	Req *nr;
	Fcall fc;

	f = v;

	if(r->fid == f){
		/*
		 * If we have an outstanding Tclunk,
		 * we need to send back an Rclunk rather than an Rerror,
		 * and we have to clunk the fid on the remote server.
		 *
		 * If we have an outstanding Tremove,
		 * we need to clunk the fid on the remote server.
		 */
		switch(r->fcall.type){
		case Tclunk:
			fc.type = Rclunk;
			fc.tag = r->tag.local;
			fc.fid = f->fid.local;
			csend(&fc);
			/* fall through */
		case Tremove:
			nr = allocreq(taggen++, nil);
			nr->fid = f;
			nr->fcall.type = Tclunk;
			nr->internal = 1;
			queuereq(nr);
			if(r->fcall.type == Tclunk)
				break;
			/* Tremove fall through */
		default:
			cerror(r->tag.local, "couldn't recover fid after lost connection");
		}
		freereq(r);
	}
}

static void
internalresponse(Req *r, Fcall *f)
{
	Fid *fid;
	Attach *a;
	int i, nwq, isdir;
	Qid *wq;

	switch(f->type){
	default:
		syslog(0, logfile, "[%d] unexpected internal response %F %F", getpid(), f, &r->fcall);
		dumplast();
		freereq(r);
		break;
	case Rread:
		freereq(r);	
		break;
	case Rclunk:
		freereq(r);
		break;

	case Rattach:
		fid = r->fid;
		a = fid->attach;
		
		a->gen = gen;
		a->rootqid = f->qid;
		freereq(r);
		fid->gen = gen;
		forallreqs(clientrecover, fid);
		break;

	case Rwalk:
		if(r->fcall.nwname != f->nwqid){ //Walk did not succeed, leave everything unaffected
			chat("	newname: %d!= nwqid: %d, not walking\n",r->fcall.nwname,f->nwqid);
			freefid(r->newfid);
			r->newfid = nil;
			break;
		}

		if(r->flushing)
			break;

		wq = f->wqid;
		nwq = f->nwqid;
		if(r->newfid){
			clonefid(r->fid, r->newfid);
			r->newfid->type = wq[nwq-1].type;
			isdir = r->newfid->type&QTDIR;
			fid = r->newfid;
		}
		else{
			r->fid->type = wq[nwq-1].type;
			isdir = r->fid->type&QTDIR;
			fid = r->fid;
		}
		for(i=0; i<f->nwqid; i++){
			namewalk(&fid->cname, r->fcall.wname[i]);
			namewalk(&fid->sname, r->fcall.wname[i]);
		}
		chat("fid local %llud server %s client %s\n", fid->fid.local, fid->sname.name, fid->cname.name);
		if(strcmp(fid->sname.name, fid->cname.name) != 0){
			r->fcall.type = Twalk;
			r->fid = fid;
			r->fcall.nwname = 1;
			r->fcall.wname[0] = (char *)r->buf;
			nextelem(r->fcall.wname[0], fid->sname.name, fid->cname.name);
			queuereq(r);
		}else if(fid->isopen){
			r->fcall.type = Topen;
			r->fcall.mode = fid->omode;
			r->fid = fid;
			if(isdir){
				chat("	recovering an open dir, special case\n");
				r->fid->dirccount = r->fid->dirscount; //paranoid
				r->fid->dirscount = 0;
			}
			queuereq(r);
		}else
			goto Recover;
		break;

	case Ropen:
		fid = r->fid;
		fid->type = r->fcall.qid.type;
		/* fall through */
	Recover:
		fid->gen = gen;
		chat("[%d] reestablished fid %llud %lud %s\n", getpid(), fid->fid.local, fid->fid.remote, fid->cname.name);
		freereq(r);
		forallreqs(clientrecover, fid);
		break;

	case Rerror:
		fid = r->fid;
		fid->gen = gen;
		freereq(r);
		chat("[%d] reestablished fid %llud %lud %s %ld\n", getpid(), fid->fid.local, fid->fid.remote, fid->cname.name,fid->cname.len);
		forallreqs(clienterror, fid);
		break;
	}
}

void dumpreqs(void);


void
listennet(void *x)
{
	uchar *buf;
	Fcall f;
	long n;
	Req *r;
	int nreq;

	USED(x);

	buf = emalloc(MAXPKT);
	for(;;){
		for(;;){
			if((n = netread9pmsg(netfd, buf, MAXPKT)) <= 0)
				break;

			if(convM2S(buf, n, &f) != n){
				fprint(2, "convM2S: %r\n");
				break;
			}
			
			qlock(&thelock);

			last[nlast++&(nelem(last)-1)] = f;
			if(chatty && f.type != Rread && f.type != Rwrite)
				syslog(0, logfile, "[%d] net -> %F", getpid(), &f);

			chat("net -> %F\n", &f);
			
			r = lookuprreq(f.tag);
			if(r == nil){
				syslog(0, logfile, "[%d] unexpected: %F", getpid(), &f);
				dumplast();
		
				forallreqs(dumpreq, nil);
				qunlock(&thelock);
				continue;
			}
			if(f.type != r->fcall.type+1 && f.type != Rerror){
				syslog(0, logfile, "[%d] mismatch %F got %F", getpid(), &r->fcall, &f);
				qunlock(&thelock);
				continue;
			}
			


			if(r->fcall.type == Ropen){
				r->fid->vers=r->fcall.qid.vers;
				r->fid->type=r->fcall.qid.type;
			}
			if(r->internal)
				internalresponse(r, &f);
			else
				externalresponse(r, &f);
				
			qunlock(&thelock);
		}

		/* wait for requests to come in */
		do{ 
			qlock(&thelock);
			nreq = forallreqs(nil, nil);
			qunlock(&thelock);
			sleep(1000);
		}while(nreq == 0);
		qlock(&thelock);
		redial();
		qunlock(&thelock);
	}
	free(buf);
	chat("Listennet is exiting\n");
}
