#include <u.h>
#include <libc.h>
#include <fcall.h>
#include "dat.h"
#include <auth.h>
#include "fns.h"

static uint
hashid(uvlong id)
{
	return id%NHASH;
}

static vlong
allocremote(Idmap *rmap)
{
	ulong i;
	uvlong bit;

	for(i=0; i<nelem(rmap->v); i++)
		if(rmap->v[i] != ~0UL)
			break;
	if(i >= nelem(rmap->v))
		sysfatal("out of rfids");

	bit = ~rmap->v[i];
	bit &= -bit;	/* grab lowest bit */
	rmap->v[i] ^= bit;
	assert(bit != 0);
	for(i=i*8*ULONGBITS; !(bit&1); i++)
		bit >>= 1;

	return i;
}

static void
freeremote(Idmap *map, ushort remote)
{
	int i;
	vlong bit;

	assert(remote < MAXFID);

	i = remote/(8*ULONGBITS);
	bit = 1UL<<(remote%(8*ULONGBITS));
	map->v[i] &= ~bit;
}

static Id**
llookuplocal(Idmap *map, uvlong local)
{
	Id **lf;

	for(lf=&map->lhash[hashid(local)]; *lf; lf=&(*lf)->llink)
		if((*lf)->local == local)
			break;
	return lf;
}

static Id**
llookupremote(Idmap *map, ulong remote)
{
	Id **lf;

	for(lf=&map->rhash[hashid(remote)]; *lf; lf=&(*lf)->rlink)
		if((*lf)->remote == remote)
			break;
	return lf;
}

Id*
lookuplocal(Idmap *map, uvlong local)
{
	Id *r;
	r = *llookuplocal(map, local);
	return r;
}

Id*
lookupremote(Idmap *map, ulong remote)
{
	Id *r;
	r = *llookupremote(map, remote);
	return r;
}

Id*
allocid(Idmap *map, uvlong local, int sz, void *idspc)
{
	uint h;
	Id *id;

	if(id=lookuplocal(map, local)){
		fprint(2, "duplicate id at %p\n", id);
		dumplast();
		return nil;	/* sure to break something */
	}

	if(!idspc){
		assert(sz >= sizeof(*id));
		id = emalloc(sz);
	}
	else
		id = (Id *)idspc;

	id->local = local;
	id->remote = allocremote(map);

	h = hashid(local);
	id->llink = map->lhash[h];
	map->lhash[h] = id;

	h = hashid(id->remote);
	id->rlink = map->rhash[h];
	map->rhash[h] = id;

	return id;
}

void
freeid(Idmap *map, Id *id)
{
	Id **ll, **lr;

	ll = llookuplocal(map, id->local);
	lr = llookupremote(map, id->remote);
	assert(*ll == id && *lr == id);

	*ll = id->llink;
	*lr = id->rlink;
	freeremote(map, id->remote);

	free(id);
}
