 /* 
 * srv.c
 * simple overlay  file server library
 *
 */

#include <olfs.h>
#include "node.h"


static Node *rootnode;
static char *realpath;
static Olfs *olfs;


int
dprint(char *fmt, ...)
{
	int n;
	va_list args;


	if( chatty9p < 2 )
		return 0;
	va_start(args, fmt);
	n = vfprint(2, fmt, args);
	va_end(args);
	return n;
}



/* initialize function */
void
init(void)
{	
	setpath(olfs->realpath);
	rootnode = newnode(nil, "");
	return;
}


static void
fsattach(Req *r)
{
	char *spec;
	
	if(olfs->attach){
		olfs->attach(r);
		return;
	}

	/* this fs provides single tree */
	spec = r->ifcall.aname;
	if(spec && spec[0]){
		respond(r, "invalid attach specifier");
		return;
	}

	r->fid->aux = rootnode;
	r->fid->qid = ((Node*)r->fid->aux)->qid;
	r->ofcall.qid = r->fid->qid;
	respond(r, nil);
}


static char*
fswalk1(Fid *fid, char *name, Qid *qid)
{
	int i;
	Node *node, *np;

	if(olfs->walk1){
		return olfs->walk1(fid, name, qid);
	}


	node  = fid->aux;

	if(strcmp(name, ".")==0){
		return nil;
	}

	if(strcmp(name, "..")==0){
		if(node->parent){
			fid->aux = node->parent;
			fid->qid = node->parent->d->qid;
			*qid = fid->qid;
		}else{
			*qid = fid->qid;
		}

		return nil;
	}


	/* read dir and create node */

	if(node->dirdirty)
		nodedirread(node);

	if(olfs->nameencode)
		name = olfs->nameencode(name);

	for(i=0;i<node->files;i++){
		if(strcmp((node->dir+i)->name, name) == 0){
			np = newnode(node, name);
			fid->aux = np;
			fid->qid = np->qid;
			*qid=fid->qid;
			return nil;
		}
	}

	return "not found";	
}

static char*
fsclone(Fid  *ofid, Fid *fid)
{
	Node *node, *onode;

	if(olfs->clone){
		return olfs->clone(ofid, fid);
	}

	onode = (Node*)ofid->aux;
	node = emalloc(sizeof(Node));
	*node = *onode;
	node->d = reallocdir(onode->d,0);
	if(node->qid.type&QTDIR)
		nodedirread(node);

	fid->aux = node;
	return nil;
}


/* fill stat information */
void
fillstat(Dir *dir, Dir *d, int isroot)
{
	if(olfs->fillstat){
		olfs->fillstat(dir, d, isroot);
		return;
	}
	if(olfs->namedecode && (! isroot))
		d->name = olfs->namedecode(d->name);

	*dir = *d;
	dir->name = estrdup(d->name);
	dir->uid = estrdup(d->uid);
	dir->gid = estrdup(d->gid);
	dir->muid = estrdup(d->muid);

	return;
}

void
wfillstat(Dir *dir, Node *np)
{
	char errbuf[ERRMAX];

	if(olfs->wfillstat){
		olfs->wfillstat(dir, np);
		return;
	}

	if(olfs->namedecode)
		dir->name = olfs->nameencode(dir->name);

	if(dirwstat(np->realname, dir) <0){
		errbuf[0] = '\0';
		errstr(errbuf, sizeof errbuf);
		fprint(2, "%s\n",errbuf);
	}

}

int
fsdirgen(int n, Dir *dir, void *p)
{
	Node *np;
	np = (Node*)p;
	
	if(np->dirdirty)
		nodedirread(np);

	if(n < np->files){
		fillstat(dir, np->dir+n, 0);
		return 0;
	}

	return -1;
}

static void
fsstat(Req *r)
{
	Node *np;
	int isroot;

	np = (Node*) r->fid->aux;
	if(np){
		if(np->dirty)
			nodedirstat(np);
	
		if(strcmp(np->realname, rootnode->realname)==0)
			isroot=1;
		else
			isroot=0;

		fillstat(&(r->d), np->d, isroot);
		respond(r, nil);
		return;
	}else{
		respond(r, "something error in fsstat()");
	}
}

static void
fswstat(Req *r)
{
	Node *np;
	Dir *dir;

	dir  = &(r->d);

	np = (Node*)r->fid->aux;

	if(np){
		wfillstat(dir, np);
		np->dirty=2;
		respond(r, nil);
		return;
	}
	respond(r, "somthing error in fswstat()");
	
}



/* open the file */
static void
fsopen(Req *r)
{
	int fd, mode;
	Node *node;
	char errbuf[ERRMAX];

	if(olfs->open){
		olfs->open(r);
		return;
	}


	node = ((Node*)r->fid->aux);

	if( node->open ){
		close( node->fd );
		node->open = 0;
	}

	if(olfs->modeconvert)
		mode = olfs->modeconvert(r, r->ifcall.mode);
	else
		mode = r->ifcall.mode;

	fd = open(node->realname, mode);
	if(fd == -1){
		errbuf[0] = '\0';
		errstr(errbuf, sizeof errbuf);
		respond(r, errbuf);
		return;
	}
	
	node->fd = fd;
	node->open = 1;

	respond(r, nil);
}
			
/* destroy fid */
static void
fsdestroyfid(Fid *fid)
{
	Node *node;

	if(olfs->destroyfid){
		olfs->destroyfid(fid);
		return;
	}

	node = (Node*)fid->aux;

	if(node==nil){
		return;
	}

	if(node->open){
		close(node->fd);
	}

	free(node->d);
	if(node->dir)
		free(node->dir);
	free(node);
}

/* read file */
static void
fsread(Req *r)
{
	Node *np;
	char buf[BUFSIZ], errbuf[ERRMAX];
	ulong len;
	int fd;


	if(olfs->read){
		olfs->read(r);
		return;
	}

	np = (Node*)r->fid->aux;

	if(np->qid.type & QTDIR){
		dirread9p(r, fsdirgen, (void *)np);
	}else{
		fd = np->fd;
		if((len = pread(fd, buf, sizeof(buf),r->ifcall.offset) ) < 0 ){
			errbuf[0] = '\0';
			errstr(errbuf, sizeof errbuf);
			respond(r, errbuf);
		}else{
			if(olfs->readfilter)
				olfs->readfilter(buf);

			memmove(r->ofcall.data, buf, len);
			r->ofcall.count = len;
		}
	}
	
	np->dirty = 1;	// access time

	respond(r,nil);
}




static void
fswrite(Req *r)
{

	int fd;
	long len;
	Node *np;
	char *buf, errbuf[ERRMAX];

	if(olfs->write){
		olfs->write(r);
		return;
	}

	np = (Node *)r->fid->aux;

	/* do nothing, just respond error */
	if(np->qid.type&QTDIR)
		respond(r, "directories are not writable");


	fd = np->fd;
	buf = r->ifcall.data;
	
	if(olfs->writefilter)
		olfs->writefilter(buf);

	if( (len = pwrite(fd, r->ifcall.data, r->ifcall.count, r->ifcall.offset)) >=0){
		r->ofcall.count = len;
		respond(r, nil);
	}else{
		errbuf[0] = '\0';
		errstr(errbuf, sizeof errbuf);
		respond(r, errbuf);
	}

	np->dirty = 2;
}

void
fsremove(Req *r)
{
	char errbuf[ERRMAX];
	Node *np;

	if(olfs->remove){
		olfs->remove(r);
		return;
	}

	np = (Node*)r->fid->aux;
	np->dirty = 2;
	np->parent->dirdirty = 1;

	if( remove(np->realname) < 0){
		errbuf[0] = '\0';
		errstr(errbuf, sizeof errbuf); 
		respond(r, errbuf);
	}
	else
		respond(r, nil);

	
	return;
}

void
fscreate(Req *r)
{
	char *name, path[FILENAME_MAX], errbuf[ERRMAX];
	ulong perm;
	int mode;
	Node *np;
	int fd;

	if(olfs->create){
		olfs->create(r);
		return;
	}

	np = (Node*)r->fid->aux;
	np->dirdirty = 1;

	if( np->qid.type&QTDIR == 0){
		respond(r, "not a directory");
		return;
	}

	if(olfs->nameencode)
		name = olfs->nameencode(r->ifcall.name);
	else
		name = r->ifcall.name;


	sprint(path, "%s/%s",np->realname, name);

	if(olfs->permconvert)
		perm = olfs->permconvert(r, r->ifcall.perm);
	else
		perm = r->ifcall.perm;

	if(olfs->modeconvert)
		mode = olfs->modeconvert(r, r->ifcall.mode);
	else
		mode = r->ifcall.mode;


	if((fd = create(path, mode, perm)) < 0){
		errbuf[0] = '\0';
		errstr(errbuf, sizeof errbuf);
		respond(r, errbuf);
		return;
	}

	np = newnode(r->fid->aux, name);
	np->fd = fd;
	np->open = 1;
	r->fid->aux = np;
	r->fid->qid = np->qid;
	r->ofcall.qid = r->fid->qid;

	respond(r, nil);
	
}


static Srv srvfs = {
	.attach	= fsattach,
	.walk1	= fswalk1,
	.stat		= fsstat,
	.wstat	= fswstat,
	.clone	= fsclone,
	.destroyfid = fsdestroyfid,
	.open	= fsopen,
	.read		= fsread,
	.write	= fswrite,
	.create	= fscreate,
	.remove	= fsremove,
};



void
postolfs(Olfs *srv, char *srvname,  char *mtpt, int flag)
{
	olfs = emalloc(sizeof(Olfs));
	memmove(olfs, srv, sizeof(Olfs)); 
	init();
	postmountsrv(&srvfs, srvname, mtpt, flag);
}