/*
 * cryptfs.c
 * crypt file server.
 *
 */

#include "cryptfs.h"

CryptKey ckey;


enum {
	BUFSIZE = 8*1024,
};


void 
usage(void)
{
	print("usage %s [-d] [-D] [-s srvname] [-m mtpt]  target\n",argv0);
	exits("usage");
}


char *
cnameencode(char *name)
{
	return fnencrypt(&ckey, name);
}

char *
cnamedecode(char *name)
{
	return fndecrypt(&ckey, name);
}

int
cmodeconvert(Req *r, int mode)
{
	USED(r);

	if(mode & OWRITE){
		mode &= ~OWRITE;
		mode |= ORDWR;	// write -> read+write
	}
	return mode;
}


void
cfsread(Req *r)
{
	Node *node;
	uchar buf[BUFSIZE];
	ulong len, count, offset, padding;
	ulong start, flength;
	int fd, bsize;


	node = (Node*)r->fid->aux;
	
	if(node->qid.type & QTDIR){
		dirread9p(r, fsdirgen, r->fid->aux);
	}else{
		if(node->dirty >1)
			nodedirstat(node);
		flength = node->d->length;

		fd = node->fd;
		bsize = ckey.bsize;

		// count = r->ifcall.count;
		offset = r->ifcall.offset;
		
		start = offset/bsize*bsize;
		padding = offset-start;

		if(len = pread(fd, buf, sizeof(buf), start)){
			if( (start+len) % bsize && (start+len) !=  flength) {
				len = len/bsize*bsize;
			}
			bufdecrypt(&ckey, buf, len, start);

			if(r->ifcall.count < len-padding)
				count = r->ifcall.count;
			else
				count = len-padding;

			memmove(r->ofcall.data, buf+padding, count);
			r->ofcall.count = count;
		}
	}

	node->dirty = 1;	// access

	respond(r,nil);
}

void
cfswrite(Req *r)
{

	int fd, i;
	long len;
	long offset, count, fboffset, wlen, lboffset, flength, lbtrim;

	Node * node;
	char *buf;
	uchar wbuf[BUFSIZE];
	int bsize;

	node = (Node *)r->fid->aux;

	/* do nothing, just respond error */
	if(node->qid.type&QTDIR){
		respond(r, "directories are not writable");
		return;
	}


	fd = node->fd;
	buf = r->ifcall.data;

	if(node->dirty>1)
		nodedirstat(node);

	flength = node->d->length;

	offset = r->ifcall.offset;
	count = r->ifcall.count;
	bsize = ckey.bsize;
	if(count > BUFSIZE-bsize)
		count = BUFSIZE-bsize;


	// first/last block offset
	fboffset = offset/bsize*bsize;
	lboffset = (offset+count)/bsize*bsize;
	lbtrim = 0;
	if(offset+count > flength)
		lbtrim = bsize - ( offset+count - lboffset);

	if(offset != fboffset){
		len = pread(fd, wbuf, bsize, fboffset);
		if(len != bsize && len+fboffset < flength){
			respond(r, "cannot decrypt old data");
			return;
		}

		bufdecrypt(&ckey, wbuf, len, fboffset);
	}

	if(offset+count != lboffset){
		len = pread(fd, wbuf+(lboffset-fboffset), bsize, lboffset);
		if(len != bsize && lboffset+len < flength){
			respond(r, "can not decrypt old data");
			return;
		}

		bufdecrypt(&ckey, wbuf+lboffset-fboffset, len, lboffset);
	}

	for(i=0; i<count; i++){
		wbuf[i+offset-fboffset] = buf[i];
	}

	wlen = lboffset-fboffset+bsize-lbtrim;
	bufencrypt(&ckey, wbuf, wlen, fboffset);

	count = 0;
	while(count<wlen ){
		if( (len = pwrite(fd, wbuf+count, wlen-count, fboffset+count)) < 0){
			respond(r, "write error (fswrite(): pwrite)");
			return;
		}
		count+=len;
	}

	r->ofcall.count = count;
	node->dirty = 2;

	respond(r, nil);
}


Olfs olfs = {
	.read = cfsread,
	.write = cfswrite,
	.nameencode = cnameencode,
	.namedecode = cnamedecode,
	.modeconvert = cmodeconvert,
};



void
main(int argc, char **argv)
{
	char *srvname, *mtpt, *path;
	UserPasswd *up;
	ulong mode;

	srvname = nil;
	mtpt = nil;
	// mtpt = "/n/cryptfs";
	mode = MREPL|MCREATE;

	/* parse argument macro */
	ARGBEGIN{
	case 'D':
		chatty9p++;	/* debug flag */
		break;
	case 's':
		srvname = EARGF(usage());
		break;
	case 'm':
		mtpt = EARGF(usage());
		break;
	default:
		usage();
	}ARGEND;

	if(argc != 1)
		usage();


	if(*(argv[0]+strlen(argv[0])-1)!='/'){
		path = emalloc(strlen(argv[0])+2);
		sprint(path, "%s/",argv[0]);
	}else{
		path = estrdup(argv[0]);
	}
	if(! mtpt){
		mtpt = path;
	}

	olfs.realpath = estrdup(path);

	if(chatty9p)
		fprint(2,"srvname %s mtpt %s realpath %s\n",
			srvname, mtpt, path);

	up = auth_getuserpasswd(auth_getkey, "proto=pass server=cryptfs service=%s", path);
	ckey.algorithm = DES3;
	genkey(up->passwd, &ckey);

	postolfs(&olfs, srvname, mtpt, mode);
	exits(nil);
}
