#include "lib.h"
#include "getargs.h"
#include "l2u.h"
#include "hplay.h"
#include "file.h"

#define SUN_MAGIC 	0x2e736e64		/* Really '.snd' */
#define SUN_HDRSIZE	24			/* Size of minimal header */
#define SUN_UNSPEC	((unsigned)(~0))	/* Unspecified data size */
#define SUN_ULAW	1			/* u-law encoding */
#define SUN_LIN_8	2			/* Linear 8 bits */
#define SUN_LIN_16	3			/* Linear 16 bits */

file_write_p file_write = nil;
file_term_p  file_term  = nil;

static char	*linear_file;
static char	*au_file;
static int	channels = 1;
static int	au_fd = -1;            /* file descriptor for .au ulaw file */
static int	linear_fd = -1;

static unsigned	au_encoding = SUN_ULAW;
static unsigned	au_size = 0;

static void 
wblong(int fd, unsigned long x)
{
	uchar bytes[4], *p;

	p = bytes;
	p[0] = x>>24;
	p[1] = x>>16;
	p[2] = x>>8;
	p[3] = x;
	write(fd, bytes, 4);
}

static void
swab(short *buf, int n, int endian)
{
	uchar *p;
	int v;

	p = (uchar*)buf;	/* do it in place */
	for(; --n >= 0; p += 2) {
		v = *buf++;
		if(endian == 0) {
			p[0] = v;	/* little endian */
			p[1] = v>>8;
		} else {
			p[0] = v>>8;
			p[1] = v;
		}
	}
}

void 
au_header(int fd, unsigned enc, unsigned rate, unsigned size, char *comment)
{
	if(!comment)
		comment = "";
	wblong(fd, SUN_MAGIC);
	wblong(fd, SUN_HDRSIZE + strlen(comment));
	wblong(fd, size);
	wblong(fd, enc);
	wblong(fd, rate);
	wblong(fd, channels);                   /* channels */
	write(fd, comment, strlen(comment));
}

static void
aufile_write(int n, short *data)
{
	if(channels > 1) {
		short *v;
		int i;
		v = malloc(n*sizeof(short)*2);
		if(v == nil) {
			fprint(2, "say: can't malloc: %r\n");
			exits("malloc");
		}
		for(i=0; i<n; i++)
			v[2*i] = v[2*i+1] = data[i];
		data = v;
		n *= 2;
	}
	if(n > 0) {
		if(linear_fd >= 0) {
			int size = n*sizeof(short);
			swab(data, n, 0);
			if(write(linear_fd, data, size) != size) {
				fprint(2, "say: error writing %s: %r\n", linear_file);
				exits("err");
			}
		}
		if(au_fd >= 0) {
			if(au_encoding == SUN_LIN_16) {
				int size = n*sizeof(short);
				swab(data, n, 1);
				if(write(au_fd, data, size) != size) {
					fprint(2, "%s: error writing %s: %r\n", program, au_file);
					exits("err");
				}
				au_size += size;
			} else if(au_encoding == SUN_ULAW) {
				uchar	*plabuf = (uchar *) malloc(n);
				if(plabuf) {
					uchar	*p = plabuf;
					uchar	*e = p + n;
					while(p < e) {
						*p++ = short2ulaw(*data++);
					}
					if(write(au_fd, plabuf, n) != n) {
						fprint(2, "%s: error writing %s: %r\n", program, au_file);
						exits("err");
					}
					au_size += n;
					free(plabuf);
				} else
					fprint(2, "%s : No memory for ulaw data\n", program);
			} else
				abort();
		}
	}
	if(channels > 1)
		free(data);
}

static void
aufile_term(void)
{
	/* Finish ulaw file */
	if(au_fd >= 0) {
		/* Now go back and overwite header with actual size */
		if(seek(au_fd, 8, 0) == 8)
			wblong(au_fd, au_size);
		if(au_fd != 1)
			close(au_fd);
		au_fd = -1;
	}
	/* Finish linear file */
	if(linear_fd >= 0) {
		if(linear_fd != 1)
			close(linear_fd);
		linear_fd = -1;
	}
}

int
file_init(int argc, char *argv[])
{
	argc = getargs("File output", argc, argv,
	    "l", "", &linear_file, "Raw 16-bit linear pathname",
	    "C", "%d", &channels, "Number of channels (default: 1)",
	    "o", "", &au_file,     "Sun/Next audio file name",
	    nil);
	if(help_only)
		return argc;

	if(au_file) {
		if(strcmp(au_file, "-") == 0) {
			au_fd = 1;                   /* stdout */
		} else {
			au_fd = create(au_file, OWRITE, 0666);
			if(au_fd < 0)
				fprint(2, "say: can't create %s: %r\n", au_file);
		}
		if(au_fd >= 0) {
			if(samp_rate > 8000)
				au_encoding = SUN_LIN_16;
			else
				au_encoding = SUN_ULAW;
			au_header(au_fd, au_encoding, samp_rate, SUN_UNSPEC, "");
			au_size = 0;
		}
	}

	if(linear_file) {
		if(strcmp(linear_file, "-") == 0) {
			linear_fd = 1 /* stdout */ ;
		} else {
			linear_fd = create(linear_file, OWRITE, 0666);
			if(linear_fd < 0)
				fprint(2, "say: can't create %s: %r\n", linear_file);
		}
	}
	if(au_fd >= 0 || linear_fd >= 0) {
		file_write = aufile_write;
		file_term  = aufile_term;
	}
	return argc;
}
