-- libdav ver. 1.2
-- lua ver. 5.2 compliant
-- you need 9lua
-- update: 2014/09/07
-- coded by Kenar (Kenji arisawa)

gen=require("gen")
dir=require("dir")
str=require("str")

dprint = gen.dprint
date = gen.date
basename = gen.basename
split = str.split

test = dir.test
lsdir = dir.lsdir

sub = string.sub
gsub = string.gsub
char = string.char
match = string.match
find = string.find
byte = string.byte
format = string.format


function truerandom()
  local fd = p9.open("/dev/random")
  local s = p9.read(fd,3)   -- 3 is max, 4 makes a problem
  local n = 0
  for i=1,#s do
    n = 256*n + byte(sub(s,i,i))
  end
  p9.close(fd)
  return n
end

function uuidgen()
  -- UUID version 4 (random bit except six bit)
  -- look RFC4122 for the document
  -- example: 79F49CF7-75DC-456D-A5DF-769AAC78E22C
  -- the value is obtained using uuidgen on unix; these bits are
  -- 0111 1001 1111 0100 1001 1100 1111 0111
  -- 0111 0101 1101 1100
  -- 0100 0101 0110 1101	# 0100 .... .... ....
  -- 1010 0101 1101 1111	# 10.. .... .... ....
  -- 0111 0110 1001 1010 1010 1100 0111 1000 1110 0010 0010 1100
  local r,s,s1,s2,s3,s4,s5
  math.randomseed(truerandom())
  s = ""
  for i=1,16 do
    r = math.random(0,255)  -- r is 0 .. 255
    s = s .. format("%02x",r)
  end
  s1 = sub(s,1,8)
  s2 = sub(s,9,12)
  s3 = sub(s,13,16)
  s4 = sub(s,17,20)
  s5 = sub(s,20,32)
  s3 = "4"..sub(s3,2,4) -- "4" is the version no.
  t = sub(s4,1,1)
  t = format("%x", 0x80 + p9.bit(byte(t), "&", 0x3f))
  s4 = t..sub(s4,2,4)
  return format("urn:uuid:%s-%s-%s-%s-%s",s1,s2,s3,s4,s5)
end

function urlencode(s)
  -- % is the escape char in Lua re
  -- applicable only to path part
  local unsafe = " ;?#:@!&=+$,<>%\"{}|\\^[]()*`'"
  return gsub(s,"(.)",function(a)
    if (byte(a) > 127) or find(unsafe,a,1,true) then
      return format("%%%02X",byte(a))
      -- %HH encoding; upper case is recommended by RFC3987
    end
  end)
end

function urldecode(s)
  return gsub(s,"(%%%x%x)",function(a)
    return char(tonumber(sub(a,2,3),16))
  end)
end

function mketag(dm)
  return dm.qid.."-"..dm.mtime
end

function prop_getlastmodified(dm)
  -- dm = p9.dirstat(file)
  local t
  t = dm.mtime
  return "<getlastmodified>"..date("http",t).."</getlastmodified>\n"
end

function prop_creationdate(dm)
  -- dm = p9.dirstat(file)
  local t
  t = dm.mtime  -- Plan9 does not have "creation date", so we used mtime
  return "<creationdate>"..date("iso",t).."</creationdate>\n"
end

function prop_getcontentlength(dm)
  -- dm = p9.dirstat(file)
  local t
  if match(dm.mode,"^02000") then
    return nil,"HTTP/1.1 404 Not Found"
  end
  t = format("%d",dm.length)
  return "<getcontentlength>"..t.."</getcontentlength>\n"
end

function prop_getcontenttype(dm)
  -- dm = p9.dirstat(file)
  local t,e,m
  -- Apache/2.2.9 (Unix) returns
  -- <getcontenttype>httpd/unix-directory</getcontenttype>
  -- for directory
  -- so we follow apache
  if match(dm.mode,"^02000") then
   return "<getcontenttype>httpd/unix-directory</getcontenttype>\n"
  end
  t = dm.name
  e = match(t,"([^.]*)$")
  -- "abc.def" -> "def"
  -- "abc." -> ""
  -- "abc" -> "abc"
  if #t == #e then
    e = ""
  end
  --
  -- dprint("mime:",mimetype,e)
  m = mimetype[e]
  if m == nil then
    m = "application/octet-stream"
  end
  return "<getcontenttype>"..m.."</getcontenttype>\n"
end

function prop_getetag(dm)
  -- rfc2518: The getetag property MUST be defined
  -- dm = p9.dirstat(file)
  return "<getetag>"..mketag(dm).."</getetag>\n"
end

function prop_resourcetype(dm)
  -- dm = p9.dirstat(file)
  -- luaconf.h:492: #define LUA_NUMBER	double
  if isdir(dm) then
    return "<resourcetype><collection/></resourcetype>\n"
  end
  return "<resourcetype/>\n"
end

function prop_supportedlock(dm)
  -- rfc2518: WebDAV compliant server is not required to support locking
  -- Mac OSX doesn't allow write if lock is not supported.
  -- dm = p9.dirstat(file)
  if locksupport then
    return [[
<supportedlock>
 <lockentry>
  <lockscope><exclusive/></lockscope>
  <locktype><write/></locktype>
 </lockentry>
 <lockentry>
  <lockscope><shared/></lockscope>
  <locktype><write/></locktype>
 </lockentry>
</supportedlock>
]]
  else
    return "<supportedlock/>"  -- should not come here
  end
end

function prop_displayname(dm)
  return "<displayname>"..dm.name.."</displayname>\n"
end

function prop_lockdiscovery(dm,path)
  local t,f,s,u,timeout
  local p = tokenpath(path)
  f = io.open(p)
  if f == nil then
    return "<lockdiscovery/>\n"  -- we follow Apache/2.2.9
  end
  s = f:read("*a")
  t = {}
  gsub(s,"([^:]+):([^%s]*)%s*",function(a,b)
    t[a] = b
  end)
  timeout = tonumber(t["expire"]) - os.time()
  u = [[
<lockdiscovery>
<activelock>
<locktype><%s/></locktype>
<lockscope><%s/></lockscope>
<depth>%s</depth>
<owner><href>%s</href></owner>
<timeout>Second-%s</timeout>
<locktoken><href>%s</href></locktoken>
<lockroot><href>%s</href></lockroot>
</activelock>
</lockdiscovery>
]]
  if timeout > 0 then
    return format(u,t["locktype"],t["lockscope"],t["depth"],
           t["owner"],timeout,t["locktoken"],t["lockroot"])
  else
    os.remove(p)
    return "<lockdiscovery/>"  -- we follow Apache/2.2.9
  end
end

prophash = {
["propfind prop displayname"] = prop_displayname,
["propfind prop resourcetype"] = prop_resourcetype,
["propfind prop creationdate"] = prop_creationdate,
["propfind prop getlastmodified"] = prop_getlastmodified,
["propfind prop getcontenttype"] = prop_getcontenttype,
["propfind prop getcontentlength"] = prop_getcontentlength,
["propfind prop getetag"] = prop_getetag,
["propfind prop supportedlock"] = prop_supportedlock,
["propfind prop lockdiscovery"] = prop_lockdiscovery,
}

function mkprop(proplist,path)
  -- assuming trailing "/" for dirname
  local s,i,f,d,v,r,err,t
  -- We follow Apache/2.2.9 (Unix)
  d = p9.dirstat(path)
  if d == nil then -- should not happen
    return nil
  end
  if proplist == nil then
    proplist = {["propfind allprop"] = ""}
  end
  t = ""
  if proplist["propfind allprop"] then
    for k,v in pairs(prophash) do
      s = v(d,target)
      if s then
        t = t..s
      end
    end
  elseif proplist["propfind propname"] then
    r = match(d.mode,"^02000")
    for k,v in pairs(prophash) do
      s = split(k)[3]
      if s ~= "getcontentlength" or r == nil then
        t = t .. "<" .. s .. "/>\n"
      end
    end
  else
    err = {}
    i = 0
    for k,v in pairs(proplist) do
      f = prophash[k]
      if f then
        s,r = f(d,target)
        if s then
          t = t..s
        else
          err[k] = r
          i = i + 1
        end
      else
        err[k] = "HTTP/1.1 404 Not Found"
      end
    end
  end
  if i == 0 then
    err = nil
  end
  return "<prop>\n"..t.."</prop>\n", err
end

function mkresponse(target,htarget,namelist,proplist)
  -- usage: namelist=lsdir(name); mkresponse(target,htarget,namelist,proplist)
  -- proplist is a set of namelist like:
  --   ["propfinf prop getcontentlength"]=""
  --   ["propfind propname"]=""
  local s,ps,href,errs
  s = '<?xml version="1.0" encoding="utf-8"?>\n'
  s = s .. '<multistatus xmlns="DAV:">\n'
  for i,v in ipairs(namelist) do
    s = s .. "<response>\n"
    if v == "" then
      href = htarget
    else
	  href = htarget.."/"..urlencode(v)
    end
    s = s .. "<href>"..href.."</href>\n"
    s = s .. "<propstat>\n"
    if v == "" then
      ps,errs = mkprop(proplist,target)
    else
      ps,errs = mkprop(proplist,target.."/"..v)
    end
    if ps then
      s = s .. ps
      s = s .. "<status>HTTP/1.1 200 OK</status>\n"
    else
      -- dprint(v)
      s = s .. "<status>HTTP/1.1 404 Not Found</status>\n"
    end
    s = s .. "</propstat>\n"
    if errs then
      -- errs["propstat prop getcontentlength"]="HTTP/1.1 404 Not Found"
      -- windows client has many unsupported properties.
      -- we gather properties with same error message
      local t
      local u = {}
      for k,e in pairs(errs) do
        if u[e] == nil then
          u[e] = {}
        end
        u[e][k] = true
      end
      for k,v in pairs(u) do
        s = s .. "<propstat>\n<prop>\n"
        for k1,v1 in pairs(v) do
          t = split(k1)
          s = s .. "<".. t[3] .. "/>\n"
        end
        s = s .. "</prop>\n<status>" .. k .. "</status>\n</propstat>\n"
      end
    end
    s = s .. "</response>\n"
  end
  s = s .. "</multistatus>\n"
  return s
end

function readmimetype(path)
  local hash={}
  local t
  for line in io.lines(path) do
	-- the line is something like:
	-- .aif		audio		x-aiff		-		y
	-- Lua doesn't have split()
    if match(line,"^#") == nil then
      t = split(line)
      if sub(t[1],1,1) ~= "#" and t[1] ~= "-" and t[2] ~= "-" and
        t[3] ~= "-"  then
	    hash[sub(t[1],2)] = t[2] .. "/" .. t[3]
	  end
    end
  end
  hash[""] = "application/octet-stream"
  return hash
end

                       -- some locking stuffs --

function unlock(path,tokens,tree)
  -- path must be a dir
  -- tokens must be a table
  -- there might be several locks in several "._.locktokens"
  -- return nil if no locks
  -- return false if some of locks can not be open by the tokens
  -- return true if all locks can be open by the token
  if test("-e",path) == false then
    error()
  end
  if tree ~= nil then
    local r,s,u,dm
    dm = p9.readdir(path)
    u = nil
    for k,v in pairs(dm) do
      if v then
        s,r = unlock(path.."/"..k,tokens,true)
        if s == false then
          return s,r
        elseif s == true then
          u = true
        end
      end
    end
    return u,nil
  end
  local p,f,s,token1,expire,time,d,m,kv
  d = path.."/._.locktokens"
  m = p9.readdir(d)
  if m == nil then
    return nil,nil
  end
  time = os.time()
  for k,v in pairs(m) do
    p = d.."/"..k
    f = io.open(p) -- should not be nil
    s = f:read("*a")
    f:close()
    kv = {}
    gsub(s, "([^:]+):([^%s]*)%s*",function(a,b)
      kv[a] = b
    end)
    if time > kv["expire"] then
      os.remove(p)
    else
      if tokens[kv["locktoken"]] == nil then
        return false,p
      end
    end
  end
  return true,nil
end

function copy(old,new)
  local base_o,base_n,n;
  -- trailig '/' must be strip out from old
  n = #old
  if sub(old,n) == '/' then
    old = sub(old,1,n-1)
  end
  base_old = basename(old)
  base_new = basename(new)
  if base_old == base_new then
    dir.cpdir(old,new)
    return
  end

  if test("-f", old) then
    dir.copy(old, new)
    return
  end
  -- Then old is a directory if OS is Plan9
  if test("-e", new) then
    if not dir.removable(new) then
      header['Status'] = '403 Permission Denied'
    else
      dir.remove(new)
    end
  end
  dir.cpdir(old, new)
end

-- old and new must be a direcory
function move(old,new)
  --[[ rfc4918
   If a resource exists at the destination, the destination resource
   will be deleted as a side-effect of the MOVE operation, subject to
   the restrictions of the Overwrite header.
   ]]

  local base_o,base_n,n
  -- trailig '/' must be strip out from $old
  n = #old
  if sub(old,n) == '/' then
    old = sub(old,1,n-1)
  end

  base_o = basename(old)
  base_n = basename(new)

  if base_o == base_n then
    os.rename(old,new)
    return
  end

  if test("-f", old) then
    dir.copy(old, new)
    os.remove(old)
    return
  end
  -- Then old is a directory if OS is Plan9
  -- new must not exist
  dir.cpdir(old, new)
  dir.remove(old)
end

remove = dir.remove

function tokenpath(path)
  -- let path is "/foo/bar" where bar is dir or file
  -- then tokenpath is "/foo/._.tokenpath/bar"
  local b,n,locktokens
  -- confirmation: trim trailing "/" in the path if present
  path = gsub(path,"/$","")
  b,n = basename(path)
  locktokens = b.."/._.locktokens"
  if not test("-e", locktokens) then
    p9.mkdir(locktokens)
  end
  return locktokens.."/"..n
end

function lock(path,tokens,lockinfo)
  -- assume path is "/foo/bar/name" and the last "name" is that of file
  -- then we create a lock file to "/foo/bar/._.locktokens/name"
  -- the content is a randum number
  --
  -- the format: "token:value expire:value"
  local a,b,f,t,p,d,s,token,expire,kv
  kv = {}
  p = tokenpath(path)
  f = io.open(p)
  if f then
    s = f:read("*a")
    f:close()
    gsub(s, "([^:]+):([^%s]*)%s*",function(a,b)
      kv[a] = b
    end)
    if os.time() < tonumber(kv["expire"]) and (tokens == nil or tokens[kv["locktoken"]] == nil) then
      return nil
    end
   end
  f = io.open(p,"w")
  if f == nil then -- should not happen if we configure correctly
    return nil
  end
  token = uuidgen()
  expire = os.time() + timeout
  t = "locktoken:"..token.." expire:"..expire.." "
  for k,v in pairs(lockinfo) do
    t = t .. k .. ":" .. v .. " "
  end
  f:write(t)
  f:close()
  return token
end

function unlock1(path,token)
  local f,t,p,d,s,token1,expire,tokens
  p = tokenpath(path)
  f = io.open(p)
  if f == nil then
    return nil
  end
  s = f:read("*a")
  f:close()
  kv = {}
  gsub(s, "([^:]+):([^%s]*)%s*",function(a,b)
    kv[a] = b
  end)
  if os.time() > tonumber(kv["expire"]) then
    os.remove(p)
    return nil
  end
  if type(token) == "string" then
    if kv["locktoken"] == token then
      os.remove(p)
      return true
    else
      return false
    end
  end
  -- then token is a table
  if token[kv["locktoken"]] == true then
    return true
  end
  return false
end

function rmlock(path)
  local p
  p = tokenpath(path)
  os.remove(p)
end



function init()
  mimetype = readmimetype("/sys/lib/mimetype")
  if mimetype == nil then error("no mime file") end
end

init()

return {
	truerandom=truerandom,
	uuidgen=uuidgen,
	urlencode=urlencode,
	urldecode=urldecode,
	mketag=mketag,
	prop_lockdiscovery=prop_lockdiscovery,
	prophash=prophash,
	mkprop=mkprop,
	mkresponse=mkresponse,
	copy=copy,
	move=move,
	remove=remove,
	tokenpath=tokenpath,
	lock=lock,
	unlock=unlock,
	unlock1=unlock1,
	rmlock=rmlock,
	init=init,
}
