#! /usr/bin/env python
import unittest
import moop
from StringIO import StringIO
import sys

Func = moop.Func
Prop = moop.Prop
Obj = moop.Obj

def assertEquals(got, expect, error):
    assert got == expect, '''%s: expected %s, got %s''' % (error, `expect`, `got`)


class DocTestCase(unittest.TestCase):
    def setUp(self):
        self.doctest = None
    def checkDoc(self):
        if self.doctest:
            errors, tests = self.doctest(0)
            assertEquals(errors, 0,
                         '%s Doctest failed, %s errors out of %s tests' % (self.__class__.__name__, errors, tests))

class FuncTestCase(unittest.TestCase):
    def setUp(self):
        moop.gSafeGlobals = moop.makeSafeGlobals()
        self.f = Func('test(arg1=None, arg2=0)')
        self.f.source = ['a = 1', 'print a, arg1, arg2', 'return arg1']
        self.f.compile()
        self.oldstdout = sys.stdout
        sys.stdout = StringIO()
        self.o = Obj(1, None)

    def tearDown(self):
        moop.gSafeGlobals = {}
        sys.stdout.close()
        sys.stdout = self.oldstdout
        moop.gObjlist = {}
        self.o = None
        moop.gOwner = None

    def checkFuncName(self):
        assertEquals(self.f.name, 'test(arg1=None,arg2=0)', 'incorrect func name')

    def checkSource(self):
        assertEquals(self.f.source, ['a = 1', 'print a, arg1, arg2', 'return arg1'], 'invalid source')

    def checkCompile(self):
        assert self.f.code, 'code missing'

    def checkList(self):
        ret = self.f.list()
        assertEquals(sys.stdout.getvalue(), "a = 1\nprint a, arg1, arg2\nreturn arg1\n", 'invalid listing')
        assert ret == None, 'invalid return from list'

    def checkListTo(self):
        ret = self.f.list(toLine=2)
        assertEquals(sys.stdout.getvalue(), "a = 1\nprint a, arg1, arg2\n", 'invalid toLine listing')
        assert ret == None, 'invalid return from list'

    def checkNoArgCall(self):
        ret = self.f()
        assertEquals(sys.stdout.getvalue(), "1 None 0\n", 'invalid call()')
        assert ret == None, 'invalid return from no arg call'
         
    def checkOneArgCall(self):
        ret = self.f(2)
        assertEquals(sys.stdout.getvalue(), "1 2 0\n", 'invalid call(2)')
        assertEquals(ret, 2, 'invalid return from one arg call')

    def checkTwoArgCall(self):
        ret = self.f(2, 3)
        assertEquals(sys.stdout.getvalue(), "1 2 3\n", 'invalid call(2, 3)')
        assertEquals(ret, 2, 'invalid return from two arg call')

    def checkPrint(self):
        assertEquals(str(self.f), "unbound test(arg1=None,arg2=0)", 'invalid string-ization of func')

    def checkBoundPrint(self):
        f = Func('test(arg1=None,arg2=0)')
        f.boundTo = self.o
        assertEquals(str(f), "test(arg1=None,arg2=0)", 'invalid string-ization of bound func')

    def checkHacking(self):
        try:
            moop.gOwner = self.o
            other = Obj(2, None)
        except 'Hacker', e:
            pass
        moop.gOwner = None
        assertEquals(e, "Hacking is grounds for termination of your account.", 'invalid hacker exception')
        
    def checkOwnerCantBeChangeDirectly(self):
        e = None
        try:
            self.o.__dict__['wizard'] = 1
            other = Obj(2, None)
            moop.gOwner = other
            self.f.newProp = "Test"
        except 'PermError', e:
            pass
        s = "%s can't modify %s of %s" % (moop.gOwner, "newProp", self.f)
        moop.gOwner = None
        assertEquals(e, s, 'should have gotten PermError exception for not being owner')

    def checkOwnerMustBeWizard(self):
        e = None
        try:
            self.o.__dict__['wizard'] = 0
            moop.gOwner = self.o
            self.f.newProp = "Test"
        except 'PermError', e:
            pass
        s = "%s can't modify %s of %s" % (moop.gOwner, "newProp", self.f)
        moop.gOwner = None
        assertEquals(e, s, 'should have gotten PermError exception for not being wizard')

    def checkOtherAttrsOnlyChangedByOwner(self):
        try:
            self.o.__dict__['wizard'] = 1
            other = Obj(2, None)
            moop.gOwner = self.o
            self.f.owner = other
        except 'PermError', e:
            pass
        assertEquals(e, "Can't directly change func owner", 'attrs changed by non-owner should cause PermError exception')

    def checkNoGlobalOwner(self):
        self.o.__dict__['wizard'] = 0  # ordinarily would cause assignment to fail
        moop.gOwner = None
        self.f.newProp = "Test"
        assertEquals(self.f.newProp, "Test", "assignment to newProp failed")

    def checkSourceResetsCode(self):
        self.f.source = ''
        self.f.compile()
        assertEquals(self.f.code, None, 'invalidating source failed to invalidate code object')

    def checkUnbind(self):
        self.f.unbind()
        assert self.f.boundTo is None, 'unbinding Func failed'

    def checkSuper(self):
        pass # need to understand better to write this...

class PropTestCase(unittest.TestCase):
    def setUp(self):
        moop.gSafeGlobals = moop.makeSafeGlobals()
        self.oldstdout = sys.stdout
        sys.stdout = StringIO()
        self.o = Obj(1, None)
        self.p = Prop(self.o, 'test')

    def tearDown(self):
        moop.gSafeGlobals = {}
        sys.stdout.close()
        sys.stdout = self.oldstdout
        moop.gObjlist = {}
        self.o = None
        moop.gOwner = None

    def checkInit(self):
        assertEquals(self.p.owner, self.o, 'owner not set correctly')
        assertEquals(self.p.val, 'test', 'val not set correctly')
        assertEquals(self.p.r, 1, 'r not set to default')
        assertEquals(self.p.w, 0, 'w not set to default')
        assertEquals(self.p.c, 1, 'c not set to default')

    def checkResetFuncOwner(self):
        f = Func('test()')
        assert f.owner is None, 'func should start unowned'
        self.p = Prop(self.o, f)
        assertEquals(f.owner, self.o, 'func owner not set to prop owner when passed as val')
        assertEquals(self.p.val, f, 'val not set to func')

    def checkPermStr(self):
        assertEquals(self.p.permstr(), 'rc', 'default perm string created incorrectly')
        self.p.w = 1
        assertEquals(self.p.permstr(), 'rwc', 'altered perm string created incorrectly')
        f = Func('test()')
        self.p = Prop(self.o, f)
        assertEquals(self.p.permstr(), 'rcx', 'perm string should show executable if function and x')
        self.p.val.x = 0
        assertEquals(self.p.permstr(), 'rc', 'perm string should not show executable if function and not x')

class ObjTestCase(unittest.TestCase):
    def setUp(self):
        moop.gSafeGlobals = moop.makeSafeGlobals()
        self.oldstdout = sys.stdout
        sys.stdout = StringIO()
        self.o = Obj(1, None)
        self.p = Prop(self.o, 'test')

    def tearDown(self):
        moop.gSafeGlobals = {}
        sys.stdout.close()
        sys.stdout = self.oldstdout
        moop.gObjlist = {}
        self.o = None
        moop.gOwner = None

    def checkAncestors(self):
        o2 = Obj(2, self.o)
        o3 = Obj(3, self.o)
        o4 = Obj(4, o2, o3)
        assertEquals(o2.ancestors(), [self.o], 'invalid list of ancestors')
        assertEquals(o3.ancestors(), [self.o], 'invalid list of obj %s ancestors' % o3.id)
        assertEquals(o4.ancestors(), [o2, self.o, o3, self.o], 'invalid list of multiple ancestors')
        
def suite():
    FuncSuite = unittest.makeSuite(FuncTestCase, 'check')
    PropSuite = unittest.makeSuite(PropTestCase, 'check')
    ObjSuite = unittest.makeSuite(ObjTestCase, 'check')
    suite = unittest.TestSuite((FuncSuite, PropSuite, ObjSuite))
    return suite

if __name__ == '__main__':
    testSuite = suite()
    runner = unittest.TextTestRunner()
    runner.run(testSuite)
