#! /usr/bin/env python
# (c) 1999 - AccesSoft - See license.txt for details
# $Id: marksub.py,v 1.1 2004/02/12 20:20:18 daveron Exp $ 
#----------------------------------------------------------------------
#    This module translates markup tags in a variety of ways.
#----------------------------------------------------------------------

"""
Marksub - substitutes various codes for markup tags in text

>>> msg = 'This is <b>bold, <i>italic</i></b>, <u>underlined</u> text.'
>>> print msg
This is <b>bold, <i>italic</i></b>, <u>underlined</u> text.
>>> print marksub( msg, plain )
['This is bold, italic, underlined text.']
>>> print marksub( msg, showcodes )
['This is <b>bold, <i>italic</i></b>, <u>underlined</u> text.']
>>> print marksub( msg, html )
['This is <b>bold, <i>italic</i></b>, underlined text.']
>>> print marksub( msg, vt100, 40 )
['This is \\033[1mbold, \\033[7mitalic\\033[0m\\033[0m,', '\\033[4munderlined\\033[0m text.']

"""
#"

import string
replace = string.replace

gMarkupCodes = "biul"

#----------------------------------------------------------------------
# FUNCTION: marksub
#    Given a string s and a translation dictionary trans, 
#   replace each markup tag with its translated version. 
#
#    This function also splits the lines if linelen > 0.
#    It returns a list of strings to output.
#----------------------------------------------------------------------

def marksub( s, transFunc, linelen=0 ):
    if linelen > 0:
        lines = splitLines(s,linelen)
    else:
        lines = [s]
    for i in range(0,len(lines)):
            lines[i] = transFunc(lines[i])
    return lines
            
#----------------------------------------------------------------------
# FUNCTION: splitLines
#    This function splits the given string into lines.
#    Ideally, it takes into account markup tags which are invisible.
#    (Currently, it just ignores them, which may cause lines to
#    come out a little shorter when lots of markups are present.)
#----------------------------------------------------------------------
def splitLines(s, linelen):
    lines = string.split(s, "\n")
    outlines = []
    for line in lines:
        while len(line) > linelen:
            pos = string.rfind(line[:linelen],' ')
            if pos < 0:
                break
            outlines.append(line[:pos])
            line = line[pos+1:]
        outlines.append(line)
    return outlines

#----------------------------------------------------------------------
# TRANSLATION SET: plain()
#    This set strips all markups.
#----------------------------------------------------------------------
def plain(line):
    for c in gMarkupCodes:
        line = replace(line, '<%s>' % c, '')
        line = replace(line, '</%s>' % c, '')
    return line

#----------------------------------------------------------------------
# TRANSLATION SET: showcodes()
#    This set leaves all codes visible and untranslated.
#----------------------------------------------------------------------
def showcodes(line):
    return line


#----------------------------------------------------------------------
# TRANSLATION SET: html()
#    This generates mostly standard HTML.
#----------------------------------------------------------------------
def html(line):
    invalid = 'lu'
    for c in invalid:
        line = replace(line, '<%s>' % c, '')
        line = replace(line, '</%s>' % c, '')
    return line


#----------------------------------------------------------------------
# TRANSLATION SET: vt100()
#    This set generates VT100 escape sequences.
#----------------------------------------------------------------------
def vt100(line):
    ATT = chr(27)+'['
    plain = ATT + '0m'
    line = replace(line, '<b>', ATT + '1m')
    line = replace(line, '</b>', plain)
    line = replace(line, '<l>', ATT + '5m')
    line = replace(line, '</l>', plain)
    line = replace(line, '<u>', ATT + '4m')
    line = replace(line, '</u>', plain)
    line = replace(line, '<i>', ATT + '7m')
    line = replace(line, '</i>', plain)
    return line

     

def _test(verbose=0):
    import doctest
    import marksub
    return doctest.testmod(marksub, verbose=verbose)

if __name__ == '__main__':
    _test(0)
