namespace eval blipt {
    variable default_title_complement "Blipt"
    variable current_title "Welcome"
    variable title_complement "Blipt"
    variable current_place [list]
    variable history [list]

    variable doc ".document.body"
    variable elements_index 0

    proc init {} {
        variable doc

        init_gui
        tags_config
        key_bindings

        # ttk::themes
        # > classic default clam alt
        ttk::style theme use clam
    }

    proc init_gui {} {
        variable current_title
        # Main window:
        wm title . $current_title
        wm geometry . 100x20

        # -------------------------
        # Document area:
        frame .document -background white -height 100

        # Background image:
        ttk::label .document.background -background orange
        place .document.background -x 0 -y 0 -relwidth 1 -relheight 1

        text .document.body -background LemonChiffon2 -foreground black \
            -yscrollcommand {.document.scroll set} \
            -setgrid true -wrap word \
            -padx 30 -pady 15
        ttk::scrollbar .document.scroll -command {.document.body yview}

        pack .document.scroll -side right -fill y
        pack .document.body -expand yes -fill both
        pack .document -expand yes -fill both

        focus .document.body
        .document.body configure -state disabled
    }

    proc geometry {width height} {
        wm geometry . "${width}x${height}"
    }

    # -----------------------------------
    # Key bindings:
    proc key_bindings {} {
        variable doc

        # Make mouse wheel work as expected over various elements:
        bind TLabel <Button-3> {
            event generate $blipt::doc <Button-3>
        }
        bind TLabel <Button-4> {
            event generate $blipt::doc <Button-4>
        }
        bind TLabel <Button-5> {
            event generate $blipt::doc <Button-5>
        }
        bind Frame <Button-3> {
            event generate $blipt::doc <Button-3>
        }
        bind Frame <Button-4> {
            event generate $blipt::doc <Button-4>
        }
        bind Frame <Button-5> {
            event generate $blipt::doc <Button-5>
        }

        # Normal keys:
        bind $doc q {
            exit 0
        }
        # h: go back
        bind $doc h {blipt::go_back}
        bind $doc <3> {blipt::go_back}
        # r: reload
        bind $doc r {blipt::reload}
        bind $doc p {
            set c [blipt::get_current_place]
            puts $c
            clipboard clear
            clipboard append -format STRING $c
        }
        bind $doc 0 {
            $blipt::doc configure -background LemonChiffon2 -foreground black
        }
        bind $doc 9 {
            $blipt::doc configure -background WhiteSmoke -foreground black
        }
        bind $doc 8 {
            $blipt::doc configure -background "dark orange" -foreground black
        }
        bind $doc 7 {
            $blipt::doc configure -background "dark khaki" -foreground black
        }
        bind $doc 6 {
            $blipt::doc configure -background "dark red" -foreground white
        }
        bind $doc 5 {
            $blipt::doc configure -background black -foreground white
        }
        bind $doc 4 {
            $blipt::doc configure -background "dark sea green" -foreground black
        }
        bind $doc 3 {
            $blipt::doc configure -background "dark gray" -foreground black
        }
    }

    proc tags_config {} {
        variable doc
        # -----------------------------------
        # text widget basic configuration.
        # See:
        # - http://pages.cpsc.ucalgary.ca/~saul/personal/archives/Tcl-Tk_stuff/tcl_examples/

        $doc tag configure h1 -font \
            "-family {$config::h1font} -size [expr $config::base_font_size + 20]"
        $doc tag configure h2 -font \
            "-family {$config::hfont} -size [expr $config::base_font_size + 12]"
        $doc tag configure h3 -font \
            "-family {$config::hfont} -size [expr $config::base_font_size + 6]"
        $doc tag configure h4 -font \
            "-family {$config::hfont} -size [expr $config::base_font_size + 4]"
        $doc tag configure h5 -font \
            "-family {$config::hfont} -size [expr $config::base_font_size + 2]"
        $doc tag configure p -font \
            "-family {$config::pfont} -size $config::base_font_size"
        $doc tag configure bold -font \
            "-family {$config::pfont} -size $config::base_font_size -weight bold"
        $doc tag configure italic -font \
            "-family {$config::pfont} -size $config::base_font_size -slant italic"
        $doc tag configure overstrike -font \
            "-family {$config::pfont} -size $config::base_font_size -overstrike true"
        $doc tag configure fixed -font \
            "-family {$config::fxfont} -size $config::base_font_size"
        $doc tag configure link -font \
            "-family {$config::pfont} -size $config::base_font_size" -foreground red
        $doc tag configure fixed_link -font \
            "-family {$config::fxfont} -size [expr $config::base_font_size + 2]" -foreground red

        # TODO: allow users to change the size relations.
    }

    # Protocol methods
    proc background_img {file_path} {
        if {$file_path == ""} {
            pack configure .document.body -padx 0
        } else {
            set image_object [image create photo -file [prepare_image $file_path]]
            .document.background configure -image $image_object

            pack configure .document.body -padx 20
        }
    }
    proc background_color {color} {
        if {$color == ""} {
            pack configure .document.body -padx 0
            update
        } else {
            .document.background configure -background $color
            pack configure .document.body -padx 20
        }
    }

    proc h1 {text} {
        variable doc

        $doc insert end "$text\n" h1
    }
    namespace export h1

    proc h2 {text} {
        variable doc

        $doc insert end "$text\n" h2
    }
    namespace export h2

    proc h3 {text} {
        variable doc

        $doc insert end "$text\n" h3
    }
    namespace export h3

    proc h4 {text} {
        variable doc

        $doc insert end "$text\n" h4
    }
    namespace export h4

    proc h5 {text} {
        variable doc

        $doc insert end "$text\n" h5
    }
    namespace export h5

    proc hr {} {
        variable doc

        update
        set cols [expr [winfo width .] / 9]
        for {set i 0} {$i < $cols} {incr i} {
            $doc insert end "_"
        }
        $doc insert end "\n\n"
    }
    namespace export hr

    proc newline {} {
        variable doc

        $doc insert end "\n" p
    }
    namespace export newline

    proc p {text} {
        variable doc
        $doc insert end "$text" p
    }
    namespace export p

    proc bold {text} {
        variable doc
        $doc insert end "$text" bold
    }
    proc italic {text} {
        variable doc
        $doc insert end "$text" italic
    }
    proc overstrike {text} {
        variable doc
        $doc insert end "$text" overstrike
    }

    proc fixed {text} {
        variable doc
        $doc insert end "$text" fixed
    }
    namespace export fixed

    proc create_image {source} {
        update
        set doc_width [expr [winfo width .document.body] - 100]

        set img_obj [image create photo -file $source]
        set image_width [image width $img_obj]

        # https://wiki.tcl-lang.org/page/Fast+image+resizing
        # There's no nice way to resize images...
        if {$image_width > $doc_width} {
            # TODO: create in multiples of 300:
            set multiplier [expr $doc_width / $config::images_step_size]
            set stepped_width [expr $config::images_step_size * $multiplier]

            set h [expr $stepped_width * 2]
            set target "${source}:${stepped_width}.png"
            set exists [file exists $target]
            if {!$exists} {
                exec -- convert $source -resize "${stepped_width}x$h" "$target"
            }
            return [image create photo -file $target]
        } else {
            return $img_obj
        }
    }
    proc prepare_image {filename} {
        set path "$filename"

        # If it's a JPG file, convert to PNG first:
        set is_jpg [string match "*.jpg" [string tolower $path]]
        set is_jpeg [string match "*.jpeg" [string tolower $path]]
        set is_webp [string match "*.webp" [string tolower $path]]
        if {$is_jpg || $is_jpeg || $is_webp} {
            set new_path "$path.png"
            set exists [file exists $new_path]
            if {!$exists} {
                if {[catch {exec -- convert $path $new_path}]} {
                    puts "convert: error ($path)"
                    return ""
                }
            }
            set path $new_path
        }
        return $path
    }
    proc open_image {filename} {
        set path [prepare_image $filename]
        if {$path == ""} {return ""}
        return [create_image $path]
    }
    namespace export open_image
    proc lazy_img {{txt ""}} {
        variable doc
        variable elements_index

        set name ".frame_[incr elements_index]"

        frame $name -background LemonChiffon2
        pack [ttk::label "$name.image"]
        pack [ttk::label "$name.legend" -text $txt -background LemonChiffon2]
        $doc window create end -window $name -align center
        return $name
    }
    namespace export lazy_img

    proc img {filename {txt ""}} {
        set new_image [open_image $filename]

        set name [lazy_img $txt]
        $name.image configure -image $new_image

        return $name
    }
    namespace export img

    proc _link {text command tag} {
        variable doc
        variable elements_index

        uplevel 2 $doc tag bind "link_$elements_index" <1> "{$command}"
        $doc insert end "$text" [list "link_$elements_index" $tag]

        incr elements_index
    }
    proc link {text command} {
        _link $text $command link
    }
    namespace export link
    proc fixed_link {text command} {
        _link $text $command fixed_link
    }
    namespace export fixed_link

    proc cls {} {
        variable doc
        $doc delete 0.0 end
        gc::clean
        update
    }
    namespace export cls

    # ------------------------
    proc title {title} {
        variable current_title
        variable title_complement
        set current_title $title
        wm title . "$title - $title_complement"
    }
    proc set_title_complement {complement} {
        variable current_title
        variable default_title_complement
        variable title_complement

        if {$complement == ""} {set complement $default_title_complement}
        set title_complement $complement
        wm title . "$current_title - $complement"
    }
    namespace export title

    proc change_doc {cmd} {
        variable doc

        $doc configure -state normal
        eval "$cmd"
        $doc configure -state disabled
        update
    }
    namespace export change_doc

    proc go_to {cmd} {
        uplevel 1 "eval \"blipt::history_push {$cmd}\""
        change_doc $cmd
    }
    namespace export go_to

    # ----------------------------------------------
    # History:
    proc history_push {cmd} {
        variable history
        set history [lappend history "$cmd"]
    }
    namespace export history_push
    proc history_pop {} {
        variable history
        set cmd [lindex $history end]
        set history [lrange $history 0 end-1]
        return $cmd
    }
    namespace export history_pop
    proc go_back {} {
        # The current page:
        set cmd1 [history_pop]

        # The previous page:
        set cmd2 [history_pop]
        if {$cmd2 == ""} {
            history_push "$cmd1"
            return
        }

        # Go!
        go_to "$cmd2"
    }
    proc reload {} {
        set current_entry [history_pop]
        puts "Reload: $current_entry"
        eval "blipt::go_to {$current_entry}"
    }
    proc set_current_place {place} {
        variable current_place
        set current_place $place
        on_set_current_place $place
    }
    proc on_set_current_place {place} {
        plugins::on_set_current_place $place
    }
    namespace export set_current_place
    proc get_current_place {} {
        variable current_place
        return $current_place
    }
    namespace export get_current_place

    proc load_driver {file_path} {
        global argv

        set fd [open $file_path r]
        set parts [file split [file rootname $file_path]]
        set driver_name [lindex $parts end]

        namespace eval driver "variable name {$driver_name}"

        namespace eval driver {
            variable config_dir [config::get_config_dir $name]
        }
        namespace eval driver [read $fd]
    }
    namespace export load_driver
}
