; MODSTUF1.ASM
;
; This is the first part of the assembler "half" of the player module.
; Here we handle time-critical things like actual resampling and mixing of
; mod samples.  This file is broken into four parts to get around the
; assembler's 64k limit on source files.
;
; This file must be assembled with Modstuf.inc, which defines the structures
; below.
;

; PUBLIC declarations.
;
PUBLIC	_c4spds		; playback rates at C4 for finetunes
PUBLIC	_samplerecs	; array of sample data structures
PUBLIC	_ninstruments	; number of samples in the file
PUBLIC	_channelrecs	; array of channel data structures
PUBLIC	_patterntable	; pattern table
PUBLIC	_pattablesize	; number of entries in the pattern table
PUBLIC	_nfilepatterns	; number of patterns in the file
PUBLIC	_patternsegs	; table of segment addresses of pattern buffers
PUBLIC	_patbufsize	; size of a pattern buffer in paragraphs
PUBLIC	_visiteds	; table of visited flags for pattern table entries
PUBLIC	_currentpat	; current entry in the pattern table
PUBLIC	_nextpat	; next entry in the pattern table (entry to play at
			;   after this division)
PUBLIC	_songendjump	; song end jump position
PUBLIC	_loopdisable	; loop disable flag
PUBLIC	_currentpatseg	; segment address of current pattern buffer
PUBLIC	_partmixseg	; segment address of the partial-mix buffer
PUBLIC	_inpartmix	; number of samples in the partial-mix buffer
PUBLIC	_samplespertick	; number of output samples per tick
PUBLIC	_setspeedparm	; set speed parameter
PUBLIC	_beatspermin	; number of beats per minute
PUBLIC	_bpmdefl	; default for number of beats per minute
PUBLIC	_ticksperrow	; number of ticks per division
PUBLIC	_ticksdefl	; default for number of ticks per division
PUBLIC	_ticksleft	; ticks remaining in this division, plus one
PUBLIC	_patdelayparm	; pattern delay parameter
PUBLIC	_currentrow	; current row in the pattern
PUBLIC	_nextrow	; next row in the pattern (row to play at after this
			; division)
PUBLIC	_patloopstart	; pattern loop start
PUBLIC	_patloopcount	; pattern loop count
PUBLIC	_nchannels	; number of channels
PUBLIC	_bytesperrow	; bytes per row in the pattern table
PUBLIC	_bufsegrecs	; array of 2k buffer segment data records
PUBLIC	_globalvol	; global volume, 0-64
PUBLIC	_globalvoldefl	; default for global volume
PUBLIC	_initmodstuf	; initialization routine for the player module
PUBLIC	BYTE2HEX	; routine to convert a byte to 2 hex digits
PUBLIC	SAMPSTRING	; routine to convert sample number to ASCII
PUBLIC	EFFSTRING	; routine to convert effect and parameter to ASCII
PUBLIC	PERIODSTRING	; routine to convert period to ASCII note
PUBLIC	VOLUMESTRING	; routine to convert volume to ASCII
PUBLIC	FIL2MEMNOTES	; routine to convert notes to internal format
PUBLIC	NOTE2PERIOD	; routine to convert .s3m note to period
PUBLIC	SETBEATS	; routine to set the number of beats per minute
PUBLIC	_getsamples	; routine to get 2k worth of samples

;
; Multiplication table for 8086.  Not needed for the 286 version.
;
#IF !M_I286
EXTRN	MULTBL:WORD	; multiplication table for 8086
#ENDIF

;
; Establish the segment ordering.  Code comes first.  This is the C
; standard segment ordering for Small model.
;
_TEXT	SEGMENT WORD PUBLIC 'CODE'
_TEXT	ENDS

_DATA	SEGMENT PARA PUBLIC 'DATA'
_DATA	ENDS

CONST	SEGMENT WORD PUBLIC 'CONST'
CONST	ENDS

_BSS	SEGMENT WORD PUBLIC 'BSS'
_BSS	ENDS

STACK	SEGMENT PARA STACK 'STACK'
STACK	ENDS

DGROUP	GROUP	_DATA,CONST,_BSS,STACK

_DATA	SEGMENT PARA PUBLIC 'DATA'
		;
		; Table of adjustment factors for arpeggio.
		;
		EVEN
ARPADJS		DW	0FFFFh,0F1A2h,0E412h,0D745h
		DW	0CB30h,0BFC9h,0B505h,0AADCh
		DW	0A145h,09838h,08FADh,0879Ch
		DW	08000h,078D1h,07209h,06BA2h
		;
		; Table of C4SPD values for finetunes -8 to 7.
		;
		EVEN
_c4spds		DW	7895, 7941, 7985, 8046
		DW	8107, 8169, 8232, 8280
		DW	8363, 8413, 8463, 8529
		DW	8581, 8651, 8723, 8757
		;
		; Sine wave table for vibrato and tremolo.
		;
		EVEN
SINETBL		DW	00000h,01918h,031F1h,04A50h
		DW	061F8h,078ADh,08E3Ah,0A268h
		DW	0B505h,0C5E4h,0D4DBh,0E1C6h
		DW	0EC83h,0F4FAh,0FB15h,0FEC4h
		DW	0FFFFh,0FEC4h,0FB15h,0F4FAh
		DW	0EC83h,0E1C6h,0D4DBh,0C5E4h
		DW	0B505h,0A268h,08E3Ah,078ADh
		DW	061F8h,04A50h,031F1h,01918h
		DW	00000h
		;
		; Ramp table for vibrato and tremolo.
		;
		EVEN
RAMPTBL		DW	0FFFFh,0F800h,0F000h,0E800h
		DW	0E000h,0D800h,0D000h,0C800h
		DW	0C000h,0B800h,0B000h,0A800h
		DW	0A000h,09800h,09000h,08800h
		DW	08000h,07800h,07000h,06800h
		DW	06000h,05800h,05000h,04800h
		DW	04000h,03800h,03000h,02800h
		DW	02000h,01800h,01000h,00800h
		DW	00000h
		;
		; Square wave table for vibrato and tremolo.
		;
		EVEN
SQUARETBL	DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh,0FFFFh,0FFFFh,0FFFFh
		DW	0FFFFh
		;
		; Half-step periods.
		;
		EVEN
HALFSTEPS	DW	27392, 25856, 24400, 23040, 21712, 20496
		DW	19344, 18256, 17232, 16272, 15376, 14512
		DW	13696, 12928, 12192, 11520, 10848, 10240
		DW	 9664,  9120,  8608,  8128,  7680,  7248
		DW	 6848,  6464,  6100,  5760,  5428,  5124
		DW	 4836,  4564,  4308,  4068,  3844,  3628
		DW	 3424,  3232,  3048,  2880,  2712,  2560
		DW	 2416,  2280,  2152,  2032,  1920,  1812
		DW	 1712,  1616,  1525,  1440,  1357,  1281
		DW	 1209,  1141,  1077,  1017,   961,   907
		DW	  856,   808,   762,   720,   678,   640
		DW	  604,   570,   538,   508,   480,   453
		DW	  428,   404,   381,   360,   339,   320
		DW	  302,   285,   269,   254,   240,   226
		DW	  214,   202,   190,   180,   170,   160
		DW	  151,   143,   135,   127,   120,   113
		DW	  107,   101,    95,    90,    85,    80
		DW	   76,    71,    67,    64,    60,    57
		DW	   53,    50,    48,    45,    42,    40
		DW	   38,    36,    34,    32,    30,    28
		DW	   27,    25,    24,    22,    21,    20
		DW	   19,    18,    17,    16,    15,    14
		DW	   13,    13,    12,    11,    11,    10
		DW	    9,     9,     8,     8,     7,     7
		DW	    7,     6,     6,     6,     5,     5
		DW	    5,     4,     4,     4,     4,     4
		DW	    3,     3,     3,     3,     3,     3
		DW	    2,     2,     2,     2,     2,     2
		;
		; Table of hex digits for converting to ASCII hex strings,
		; extended to allow displaying of values > 15 in one character.
		;
HEXDIGITS	DB	"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		;
		; Table of note names for displaying period values.
		;
		EVEN
NOTENAMES	LABEL	WORD
		DB	"C C#D D#E F F#G G#A A#B "
		;
		; Sample data structures for samples 0-99.
		;
		EVEN
_samplerecs	DB	100 * (TYPE SAMPREC) DUP (0)
		;
		; Number of instrument samples, extended to 16 bits.
		;
		EVEN
_ninstruments	DW	0
		;
		; Channel data structures for channels 0-31.
		;
		EVEN
_channelrecs	DB	32 * (TYPE CHANREC) DUP (0)
		;
		; Pattern table, numbers extended to 16 bits.
		;
		EVEN
_patterntable	DW	128 DUP (0)
		;
		; Number of entries in the pattern table, 1-128, extended
		; to 16 bits.
		;
		EVEN
_pattablesize	DW	0
		;
		; Number of patterns in the file, word 1-256.
		;
		EVEN
_nfilepatterns	DW	0
		;
		; Table of segment addresses of pattern buffers.
		;
		EVEN
_patternsegs	DW	256 DUP (0)
		;
		; Size of a pattern buffer in paragraphs.
		;
		EVEN
_patbufsize	DW	0
		;
		; Table of visited flags for each entry in the pattern
		; table.
		;
_visiteds	DB	128 DUP (0)
		;
		; Current entry in the pattern table, 0-127, and next pattern
		; (entry to play at after this division), 0-128, extended to
		; 16 bits.
		;
		EVEN
_currentpat	DW	0
_nextpat	DW	0
		;
		; Song end jump position, extended to 16 bits.
		;
		EVEN
_songendjump	DW	0
		;
		; Loop disable flag, 1 if pattern table looping disabled.
		;
_loopdisable	DB	0
		;
		; Segment address of current pattern buffer.
		;
		EVEN
_currentpatseg	DW	0
		;
		; Current offset in the partial-mix buffer for the final mix
		; routine.  This should precede _partmixseg so that the LDS
		; instruction can be used.
		;
		EVEN
PARTMIXOFFS	DW	0
		;
		; Segment address of the partial-mix buffer.
		;
		EVEN
_partmixseg	DW	0
		;
		; Number of samples in the partial-mix buffer, word 0-3341.
		;
		EVEN
_inpartmix	DW	0
		;
		; Number of output samples per tick, word 39-3341.
		;
		EVEN
_samplespertick	DW	0
		;
		; Set speed parameter, byte 0-255.
		;
_setspeedparm	DB	0
		;
		; Number of "beats" per minute.
		;
_beatspermin	DB	0
		;
		; Default for number of "beats" per minute at the start of
		; the song.
		;
_bpmdefl	DB	125
		;
		; Number of ticks per division, extended to 16 bits.
		;
		EVEN
_ticksperrow	DW	0
		;
		; Default for number of ticks per division at the start of
		; the song.
		;
		EVEN
_ticksdefl	DW	6
		;
		; Number of ticks remaining in this division, plus one,
		; word 0-3872; we will decrement this at the beginning of
		; a tick, then if 0, new division.
		;
		EVEN
_ticksleft	DW	0
		;
		; Pattern delay parameter, byte 0-120.
		;
_patdelayparm	DB	0
		;
		; Current row in the pattern and next row (row to play at
		; after this division), byte 0-63.
		;
_currentrow	DB	0
_nextrow	DB	0
		;
		; New note flag, 1 if there is a new note on the current
		; channel.
		;
NEWNOTE		DB	0
		;
		; Pattern loop start, byte 0-63.
		;
_patloopstart	DB	0
		;
		; Pattern loop count, byte 0-15.
		;
_patloopcount	DB	0
		;
		; Number of channels, 4, 6, 8, 16 or 32, extended to 16 bits.
		;
		EVEN
_nchannels	DW	0
		;
		; Bytes per row, byte (4, 6, 8, 16 or 32) * sizeof( struct
		; noterec ).
		;
_bytesperrow	DB	0
		;
		; 2k buffer segment data structures for segments 0-31.  This
		; data is kept on each buffer segment so that it can be
		; displayed to the user when the segment is played.
		;
		EVEN
_bufsegrecs	DB	32 * (TYPE BUFSEGREC) DUP (0)
		;
		; Global volume and default for global volume.  For mods,
		; the default for global volume is 64; for ScreamTracker
		; (.S3M) files, the default comes from the file header.
		;
_globalvol	DB	64
_globalvoldefl	DB	64
		;
		; Array of pointers to effect setup routines, indexed by
		; the major effect number.
		;
		EVEN
EFSETUPS	DW	OFFSET EF0SETUP
		DW	OFFSET EF1SETUP
		DW	OFFSET EF2SETUP
		DW	OFFSET EF3SETUP
		DW	OFFSET EF4SETUP
		DW	OFFSET EF5SETUP
		DW	OFFSET EF6SETUP
		DW	OFFSET EF7SETUP
		DW	OFFSET DONOTHING
		DW	OFFSET EF9SETUP
		DW	OFFSET EFASETUP
		DW	OFFSET EFBSETUP
		DW	OFFSET EFCSETUP
		DW	OFFSET EFDSETUP
		DW	OFFSET EFESETUP
		DW	OFFSET EFFSETUP
		DW	OFFSET EFGSETUP
		DW	OFFSET EFHSETUP
		DW	OFFSET EFISETUP
		DW	OFFSET EFJSETUP
		DW	OFFSET EFKSETUP
		DW	OFFSET EFLSETUP
		DW	OFFSET EFMSETUP
		DW	OFFSET EFNSETUP
		DW	OFFSET EFOSETUP
		DW	OFFSET EFPSETUP
		DW	OFFSET EFQSETUP
		DW	OFFSET EFRSETUP
		DW	OFFSET EFSSETUP
NEFFECTS	EQU	($-OFFSET EFSETUPS) / 2
		;
		; Array of pointers to extended effect setup routines,
		; indexed by the minor effect number.
		;
		EVEN
EFESETUPS	DW	OFFSET EFENONE
		DW	OFFSET EFE1SETUP
		DW	OFFSET EFE2SETUP
		DW	OFFSET EFE3SETUP
		DW	OFFSET EFE4SETUP
		DW	OFFSET EFE5SETUP
		DW	OFFSET EFE6SETUP
		DW	OFFSET EFE7SETUP
		DW	OFFSET EFENONE
		DW	OFFSET EFE9SETUP
		DW	OFFSET EFEASETUP
		DW	OFFSET EFEBSETUP
		DW	OFFSET EFECSETUP
		DW	OFFSET EFEDSETUP
		DW	OFFSET EFEESETUP
		DW	OFFSET EFENONE
		;
		; Array of pointers to ScreamTracker extended effect setup
		; routines, indexed by the minor effect number.
		;
		EVEN
EFSSETUPS	DW	OFFSET EFENONE
		DW	OFFSET EFE3SETUP
		DW	OFFSET EFE5SETUP
		DW	OFFSET EFE4SETUP
		DW	OFFSET EFE7SETUP
		DW	OFFSET EFENONE
		DW	OFFSET EFENONE
		DW	OFFSET EFENONE
		DW	OFFSET EFENONE
		DW	OFFSET EFENONE
		DW	OFFSET EFENONE
		DW	OFFSET EFE6SETUP
		DW	OFFSET EFECSETUP
		DW	OFFSET EFEDSETUP
		DW	OFFSET EFEESETUP
		DW	OFFSET EFENONE
;
; Static data for the _getsamples() routine begins. ******************
;
		;
		; State of the _getsamples() routine:  0 if returning the
		; first 2k buffer full; 2 if the last partial-mix buffer
		; has not been encountered yet; 4 if the last partial-mix
		; buffer has been done, and we are ramping to the baseline;
		; 6 if we are at the baseline, and one more buffer full of
		; silence should be returned; and 8 if we are all done
		; playing.
		;
		EVEN
GS_STATE	DW	8
		;
		; Subroutines of _getsamples() for song playback for each
		; of the 5 possible states.
		;
		EVEN
GS_SONGROUTINES	DW	OFFSET GSSONGSTART
		DW	OFFSET GSSONGPLAY
		DW	OFFSET GSRAMP
		DW	OFFSET GSONEMORE
		DW	OFFSET GSALLDONE
		;
		; Subroutines of _getsamples() for sample playback for each
		; of the 5 possible states.
		;
		EVEN
GS_SAMPROUTINES	DW	OFFSET GSSAMPSTART
		DW	OFFSET GSSAMPPLAY
		DW	OFFSET GSRAMP
		DW	OFFSET GSONEMORE
		DW	OFFSET GSALLDONE
		;
		; Subroutine table to use for current item.
		;
		EVEN
GS_ITEMTBL	DW	OFFSET GS_SONGROUTINES
		;
		; Pointer to sample record for sample currently being
		; played.
		;
		EVEN
GS_SAMPREC	DW	OFFSET _samplerecs
		;
		; Sample step for playing samples, in words.
		;
		EVEN
GS_STEPFRAC	DW	0
GS_STEPINT	DW	0
		;
		; Current position in the sample buffer when playing a sample.
		; GS_SAMPSEG is the current segment, but GS_SAMPINT and
		; GS_SAMPFRAC are kept in words relative to that segment.
		;
		EVEN
GS_SAMPFRAC	DW	0
GS_SAMPINT	DW	0
GS_SAMPSEG	DW	0
		;
		; Current 64k half of an EMS sample.
		;
GS_SAMPHALF	DB	0
		;
		; Last sample output, used while ramping to the baseline.
		;
GS_LASTSAMP	DB	80h
		;
		; Pointers to final-mix routines for various numbers of
		; channels in the mod.
		;
		EVEN
GS_MIXERS	DW	OFFSET GSMIX4
		DW	OFFSET GSMIX6
		DW	OFFSET GSMIX8
		DW	OFFSET GSMIX10
		DW	OFFSET GSMIX6
		DW	OFFSET GSMIX14
		DW	OFFSET GSMIX8
		DW	OFFSET GSMIX18
		DW	OFFSET GSMIX10
		DW	OFFSET GSMIX22
		DW	OFFSET GSMIX6
		DW	OFFSET GSMIX26
		DW	OFFSET GSMIX14
		DW	OFFSET GSMIX30
		DW	OFFSET GSMIX8
		;
		; Pointer to final-mix routine for song playback (mixes
		; stereo to mono).
		;
		EVEN
GS_MIXPROC	DW	OFFSET GSMIX4
_DATA	ENDS

_TEXT	SEGMENT WORD PUBLIC 'CODE'
;
; NEARHALF routine, to find the period value of the nearest full half-
; step to a given period.  Takes the period to find in AX.  In addition,
; the following must be set in the default data segment:
;
;    HALFSTEPS      168 words  period values for full half-steps
;
; Returns the period of the full half-step nearest to the given period
; in AX, and the offset of that period in the HALFSTEPS table in BX.
; Destroys DX, SI, DI.
;
	EVEN
NEARHALF:
	XOR	SI,SI		; SI is the low end of the search interval
	MOV	DI,167		; DI is the high end of the search interval
	;
	; Loop until (low - high) = 1, or until we find the value.
	;
NEARHALF_LOOP:
	MOV	BX,DI		; (low - high) = 1? then stop looking
	SUB	BX,SI
	CMP	BX,1
	JBE	NEARHALF_LOOPEND
	MOV	BX,SI		; BX is the middle of the search interval
	ADD	BX,DI
	AND	BL,0FEh		; (average of SI,DI) * 2 (word offset)
	CMP	HALFSTEPS[BX],AX
	JA	NEARHALF_FIXLOW
	JE	NEARHALF_DONE	; (we might get lucky and find the value)
	SHR	BX,1		; middle is too low, set high to middle
	MOV	DI,BX
	JMP	SHORT NEARHALF_LOOP
	EVEN
NEARHALF_FIXLOW:
	SHR	BX,1		; middle is too high, set low to middle
	MOV	SI,BX
	JMP	SHORT NEARHALF_LOOP
	;
	; The loop is done, and we didn't find the exact value.  The best
	; one is either SI (low) or the DI (high) entry in the table.
	;
	EVEN
NEARHALF_LOOPEND:
	MOV	BX,AX		; save the value we're looking for in BX
	SHL	SI,1		; convert SI to word offset
	SHL	DI,1		; convert DI to word offset
	SUB	AX,HALFSTEPS[SI]
	CWD
	XOR	AX,DX
	SUB	AX,DX
	XCHG	AX,BX		; BX is absolute difference of (sought - low)
	SUB	AX,HALFSTEPS[DI]
	CWD
	XOR	AX,DX
	SUB	AX,DX
	MOV	DX,AX		; DX is absolute difference of (sought - high)
	XCHG	BX,SI		; BX is offset of best; SI = |sought - low|
	MOV	AX,HALFSTEPS[BX] ; AX is best period so far
	CMP	SI,DX		; is low better than high? all done if so
	JBE	NEARHALF_DONE
	MOV	BX,DI		; high value is best
	MOV	AX,HALFSTEPS[DI]
NEARHALF_DONE:
	RET
;
; GETARPSTEP routine, to compute the sample step for arpeggio.  Takes
; the base period in CX, C4SPD for the sample in BX, and number of
; half-steps above the base period in DX.  In addition, the following
; must be set in the default data segment:
;
;    _clockspersam  word       number of 3.57MHz clocks per output sample
;    ARPADJS        16 words   array of adjustment factors
;
; Returns the sample step in DX.AX.  Destroys BX, CX, DI.
;
	EVEN
GETARPSTEP:
	OR	DX,DX			; 0 half-steps above?
	JZ	GETSTEP			; no adjustment to make then
	MOV	DI,DX			; adjust period for # of half-steps
	SHL	DI,1
	MOV	AX,ARPADJS[DI]
	MUL	CX
	SHL	AX,1			; (round to nearest)
	ADC	DX,0
	JNZ	GETARPSTEP_GOOD		; if result is 0, set it to 1
	MOV	DX,1
GETARPSTEP_GOOD:
	MOV	CX,DX
	; control falls through to next routine
;
; GETSTEP routine, to compute the sample step.  Takes the period in CX
; and C4SPD value in BX.  In addition, the following must be set in the
; default data segment:
;
;    _clockspersam  word       number of 3.57MHz clocks per output sample
;
; Returns the sample step in DX.AX.  Destroys BX, CX, DI.
;
GETSTEP:
	MOV	AX,_clockspersam	; multiply (r*4) * c
	SHL	AX,1
	SHL	AX,1
	MUL	BX
	MOV	DI,AX			; save low word of product in DI
	MOV	AX,DX			; divide high word by p
	XOR	DX,DX
	DIV	CX
	XCHG	AX,DI			; high word in DI, get low word
	DIV	CX
	MOV	BX,AX			; save low word in BX
	XOR	AX,AX			; divide again to get fraction
	DIV	CX
	SHR	CX,1			; round to nearest
	CMP	CX,DX
	ADC	AX,0
	ADC	BX,0
	ADC	DI,0
	XCHG	AX,BX			; save fraction in BX, low word in AX
	MOV	DX,DI			; high word in DX
	MOV	CX,8363			; divide by 8363
	DIV	CX	; (won't overflow because r <= 895, c <= 65535)
	XCHG	AX,BX			; save integer in BX, fraction in AX
	DIV	CX
	SHR	CX,1			; round to nearest
	CMP	CX,DX
	ADC	AX,0
	ADC	BX,0
	MOV	DX,BX			; sample step in DX.AX
	RET
;
; MAJOR EFFECT SETUP ROUTINES ****************************************
;
; These routines should modify no registers.  On entry, DS is the default
; data segment, BX addresses the channel data structure, and ES:DI addresses
; the note structure for the current note.
;
; DONOTHING routine, used as a placeholder when there is no effect on a
; channel, or the effect does not need a routine.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
DONOTHING:
	RET
;
; EF0SETUP routine, the effect setup routine for effect 0 (arpeggio).  The
; special case where the parameter is 0, which is a normal note with no
; effect, has already been taken care of by the caller.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF0SETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AX,[BX].CHANSTEPFRAC	; set sample step 0 to normal step
	MOV	[BX].CHANASTEP0FRAC,AX
	MOV	AX,[BX].CHANSTEPINT
	MOV	[BX].CHANASTEP0INT,AX
	MOV	DL,ES:[DI].NOTEEFFPARM	; get parameter in DL
	JMP	EFLSETUP_SAVEINFO	; the rest is same as effect L
;
; EF1SETUP routine, the effect setup routine for effect 1 (slide up).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF1SETUP:
	PUSH	AX
	PUSH	DX
	MOV	AX,[BX].CHANSLIDEINCR	; get current slide increment in AX
	MOV	DL,ES:[DI].NOTEEFFPARM	; get parameter in DL
	OR	DL,DL			; parameter 0?
	JZ	EF1SETUP_DOABS
	XOR	DH,DH			; not 0 - set slide increment
	MOV	AX,DX
	SHL	AX,1
	SHL	AX,1
	JMP	SHORT EF1SETUP_SETSIGN
	EVEN
EF1SETUP_DOABS:
	CWD		; keeping old slide increment, take absolute value
	XOR	AX,DX
	SUB	AX,DX
EF1SETUP_SETSIGN:
	NEG	AX			; make slide increment negative
	MOV	[BX].CHANSLIDEINCR,AX	; save it
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF1TICK ; set tick procedure
	POP	DX
	POP	AX
	RET
;
; EF2SETUP routine, the effect setup routine for effect 2 (slide down).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF2SETUP:
	PUSH	AX
	PUSH	DX
	MOV	AX,[BX].CHANSLIDEINCR	; get current slide increment in AX
	MOV	DL,ES:[DI].NOTEEFFPARM	; get parameter in DL
	OR	DL,DL			; parameter 0?
	JZ	EF2SETUP_DOABS
	XOR	DH,DH			; not 0 - set slide increment
	MOV	AX,DX
	SHL	AX,1
	SHL	AX,1
	JMP	SHORT EF2SETUP_SAVEINCR
	EVEN
EF2SETUP_DOABS:
	CWD		; keeping old slide increment, take absolute value
	XOR	AX,DX
	SUB	AX,DX
EF2SETUP_SAVEINCR:
	MOV	[BX].CHANSLIDEINCR,AX	; save it
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF1TICK ; set tick procedure
	POP	DX
	POP	AX
	RET
;
; EF3SETUP routine, the effect setup routine for effect 3 (slide to note).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF3SETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	MOV	CX,[BX].CHANPERIODGOAL	; get period goal in CX
	MOV	AX,ES:[DI].NOTEPERIOD	; get period in AX
	OR	AX,AX			; if period 0, do not adjust goal
	JZ	EF3SETUP_CHKPARM
	MOV	CX,AX			; set period goal equal to period
	MOV	[BX].CHANPERIODGOAL,AX
EF3SETUP_CHKPARM:
	MOV	AX,[BX].CHANSLIDEINCR	; get current slide increment in AX
	MOV	DL,ES:[DI].NOTEEFFPARM	; get parameter in DL
	OR	DL,DL			; parameter 0?
	JZ	EF3SETUP_DOABS
	XOR	DH,DH			; not 0 - set slide increment
	MOV	AX,DX
	SHL	AX,1
	SHL	AX,1
	JMP	SHORT EF3SETUP_SETSIGN
	EVEN
EF3SETUP_DOABS:
	CWD		; keeping old slide increment, take absolute value
	XOR	AX,DX
	SUB	AX,DX
EF3SETUP_SETSIGN:
	CMP	CX,[BX].CHANEFFPERIOD	; if goal < current, negate increment
	JAE	EF3SETUP_SETINCR
	NEG	AX
EF3SETUP_SETINCR:
	MOV	[BX].CHANSLIDEINCR,AX	; save new (?) slide increment
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF3TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF3END ; set end procedure
	POP	DX
	POP	CX
	POP	AX
	RET
;
; EF4SETUP routine, the effect setup routine for effect 4 (vibrato).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF4SETUP:
#IF M_I286
	PUSHA
#ELSE
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameters in AL
	MOV	DL,AL			; AL = first parameter, DL = second
#IF M_I286
        SHR     AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	JZ	EF4SETUP_CHKDEPTH	; first parameter 0?
	CBW				; 1st parm <> 0 - set vibrato increment
	MOV	[BX].CHANVIBINCR,AX
EF4SETUP_CHKDEPTH:
	AND	DL,0Fh			; second parameter 0?
	JZ	EF4SETUP_CHKNOTE
	XOR	DH,DH			; 2nd parm <> 0 - set vibrato depth
	SHL	DX,1
	SHL	DX,1
	MOV	[BX].CHANVIBDEPTH,DX
EF4SETUP_CHKNOTE:
	CMP	ES:[DI].NOTEPERIOD,0	; is it a new note?
	JZ	EF4SETUP_SAMENOTE	; if new note:
	MOV	SI,[BX].CHANVIBPOS	; SI is position in vibrato wave table
	CMP	[BX].CHANVIBNORETRG,1	; is retrigger disabled?
	JE	EF4SETUP_NORETRG
	XOR	SI,SI			; retrigger vibrato (set position = 0)
	MOV	[BX].CHANVIBPOS,SI
	JMP	SHORT EF4SETUP_LOOKUP
	EVEN
EF4SETUP_NORETRG:
	CMP	SI,33			; position < 33? use it directly then
	JB	EF4SETUP_LOOKUP
	NEG	SI			; position >= 33, use 64-position
	ADD	SI,64
EF4SETUP_LOOKUP:
	SHL	SI,1			; get waveform table index
	ADD	SI,[BX].CHANVIBWAVE	; SI addresses period adjustment
	MOV	AX,[SI]			; AX is fixed-point adjustment
	MUL	[BX].CHANVIBDEPTH	; multiply by vibrato depth
	SHL	AX,1			; round to nearest
	ADC	DX,0			; DX is (positive) adjustment
	CMP	[BX].CHANVIBPOS,32	; position < 32? adjust up
	JB	EF4SETUP_SETPERIOD
	NEG	DX			; position >= 32, adjust down
EF4SETUP_SETPERIOD:
	ADD	DX,[BX].CHANEFFPERIOD	; DX is period for first tick
	JG	EF4SETUP_HIGHENUF	; if < 1, set to 1
	MOV	DX,1
	JMP	SHORT EF4SETUP_SETSTEP
	EVEN
EF4SETUP_HIGHENUF:
	CMP	DX,27392		; if > 27392, set to 27392
	JLE	EF4SETUP_SETSTEP
	MOV	DX,27392
EF4SETUP_SETSTEP:
	PUSH	BX			; save pointer to channel data
	MOV	CX,DX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
EF4SETUP_SAMENOTE:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF4TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF4END ; set end procedure
#IF M_I286
	POPA
#ELSE
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	AX
#ENDIF
	RET
;
; EF5SETUP routine, the effect setup routine for effect 5 (slide to note
; plus volume slide).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF5SETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter in AL
	JMP	EFNSETUP_SAVEINFO	; the rest is same as effect N
;
; EF6SETUP routine, the effect setup routine for effect 6 (vibrato plus
; volume slide).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF6SETUP:
#IF M_I286
	PUSHA
#ELSE
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameters in AL
	JMP	EFMSETUP_SAVEINFO	; the rest is same as effect M
;
; EF7SETUP routine, the effect setup routine for effect 7 (tremolo).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF7SETUP:
	PUSH	AX
	PUSH	DX
	PUSH	SI
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameters in AL
	MOV	DL,AL			; AL = first parameter, DL = second
#IF M_I286
        SHR     AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	JZ	EF7SETUP_CHKDEPTH	; first parameter 0?
	CBW				; 1st parm <> 0 - set tremolo increment
	MOV	[BX].CHANTREMINCR,AX
EF7SETUP_CHKDEPTH:
	AND	DL,0Fh			; second parameter 0?
	JZ	EF7SETUP_CHKNOTE
	XOR	DH,DH			; 2nd parm <> 0 - set tremolo depth
	MOV	[BX].CHANTREMDEPTH,DX
EF7SETUP_CHKNOTE:
	CMP	ES:[DI].NOTEPERIOD,0	; is it a new note?
	JZ	EF7SETUP_SAMENOTE	; if new note:
	MOV	SI,[BX].CHANTREMPOS	; SI is position in tremolo wave table
	CMP	[BX].CHANTREMNORETRG,1	; is retrigger disabled?
	JE	EF7SETUP_NORETRG
	XOR	SI,SI			; retrigger tremolo (set position = 0)
	MOV	[BX].CHANTREMPOS,SI
	JMP	SHORT EF7SETUP_LOOKUP
	EVEN
EF7SETUP_NORETRG:
	CMP	SI,33			; position < 33? use it directly then
	JB	EF7SETUP_LOOKUP
	NEG	SI			; position >= 33, use 64-position
	ADD	SI,64
EF7SETUP_LOOKUP:
	SHL	SI,1			; get waveform table index
	ADD	SI,[BX].CHANTREMWAVE	; SI addresses period adjustment
	MOV	AX,[SI]			; AX is fixed-point adjustment
	MUL	[BX].CHANTREMDEPTH	; multiply by tremolo depth
	SHL	AX,1			; round to nearest
	ADC	DX,0			; DX is (positive) adjustment
	CMP	[BX].CHANTREMPOS,32	; position < 32? adjust up
	JB	EF7SETUP_SETVOL
	NEG	DX			; position >= 32, adjust down
EF7SETUP_SETVOL:
	ADD	DL,[BX].CHANEFFVOL	; add effective volume
	JGE	EF7SETUP_HIGHENUF	; if < 0, set to 0
	XOR	DL,DL
	JMP	SHORT EF7SETUP_SETMIX
	EVEN
EF7SETUP_HIGHENUF:
	CMP	DL,64			; if > 64, set to 64
	JLE	EF7SETUP_SETMIX
	MOV	DL,64
EF7SETUP_SETMIX:
	MOV	[BX].CHANMIXVOL,DL	; set mixing volume
EF7SETUP_SAMENOTE:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF7TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF7END ; set end procedure
	POP	SI
	POP	DX
	POP	AX
	RET
;
; EF9SETUP routine, the effect setup routine for effect 9 (set sample offset).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EF9SETUP:
	PUSH	AX
	PUSH	DX
	PUSH	SI
	MOV	AH,ES:[DI].NOTEEFFPARM	; get parameter in AH
	XOR	AL,AL			; (multiply by 256)
	MOV	SI,[BX].CHANSAMPPTR	; past end of sample?
	CMP	AX,[SI].SAMPLEN
	JAE	EF9SETUP_DONE		; invalid if so
	XOR	DX,DX			; set fraction of sample position to 0
	MOV	[BX].CHANSAMPFRAC,DX
	CMP	[SI].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	EF9SETUP_INEMS
	;
	; Case 1:  Sample in conventional RAM.
	;
	SHL	AX,1			; convert offset to bytes
#IF M_I286
	RCR	DX,4			; get segment adjustment
#ELSE
	RCR	DX,1
	SHR	DX,1
	SHR	DX,1
	SHR	DX,1
#ENDIF
	MOV	[BX].CHANSAMPOFFS,AX	; set integer of sample position
	ADD	DX,[SI].SAMPSEG		; set sample segment
	MOV	[BX].CHANSAMPSEG,DX
	JMP	SHORT EF9SETUP_DONE
	;
	; Case 2:  Sample in expanded RAM.
	;
	EVEN
EF9SETUP_INEMS:
	SHL	AX,1			; convert offset to bytes
	RCL	DL,1			; set sample half
	MOV	[BX].CHANSAMPOFFS,AX	; set integer of sample position
	MOV	[BX].CHANSAMPHALF,DL	; save sample half
EF9SETUP_DONE:
	POP	SI
	POP	DX
	POP	AX
	RET
;
; EFASETUP routine, the effect setup routine for effect A (volume slide).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFASETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	OR	AL,AL			; nothing to do if parameters are 0
	JZ	EFASETUP_DONE
	TEST	AL,0F0h			; is first parameter 0?
	JZ	EFASETUP_DOWNSLIDE	; if so, use second (slide down)
#IF M_I286
        SHR     AL,4			; get first parameter
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	MOV	[BX].CHANVOLINCR,AL	; set increment
	JMP	SHORT EFASETUP_SETPROCS
	EVEN
EFASETUP_DOWNSLIDE:
	AND	AL,0Fh			; sliding down - use second parameter
	NEG	AL
	MOV	[BX].CHANVOLINCR,AL	; set increment
EFASETUP_SETPROCS:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFATICK ; set tick procedure
EFASETUP_DONE:
	POP	AX
	RET
;
; EFBSETUP routine, the effect setup routine for effect B (pattern jump).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFBSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	XOR	AH,AH
	MOV	_nextpat,AX
	MOV	_nextrow,AH
EFBSETUP_DONE:
	POP	AX
	RET
;
; EFCSETUP routine, the effect setup routine for effect C (set volume).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFCSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	CMP	AL,64
	JA	EFCSETUP_DONE
	MOV	[BX].CHANEFFVOL,AL
	MOV	[BX].CHANMIXVOL,AL
EFCSETUP_DONE:
	POP	AX
	RET
;
; EFDSETUP routine, the effect setup routine for effect D (pattern break).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFDSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
        MOV     AH,AL			; it's BCD, convert it to binary
#IF M_I286
        SHR     AH,4
#ELSE
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
#ENDIF
        AND     AL,0Fh
        AAD
	CMP	AL,64			; invalid if >= 64
	JAE	EFDSETUP_DONE
	MOV	_nextrow,AL
	MOV	AX,_currentpat
	INC	AX
	MOV	_nextpat,AX
EFDSETUP_DONE:
	POP	AX
	RET
;
; EFESETUP routine, the effect setup routine for effect E (extended effect).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFESETUP:
	PUSH	AX
	PUSH	SI
	MOV	AL,ES:[DI].NOTEEFFPARM
#IF M_I286
        SHR     AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	CWD
	SHL	AX,1
	MOV	SI,AX
	MOV	AL,ES:[DI].NOTEEFFPARM
	AND	AL,0Fh
	JMP	EFESETUPS[SI]
;
; EFFSETUP routine, the effect setup routine for effect F (set speed).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFFSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	MOV	_setspeedparm,AL	; save it - DOTICK will take care of it
	POP	AX
	RET
;
; EFGSETUP routine, the effect setup routine for effect G (set ticks/division),
; ScreamTracker effect A.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFGSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	OR	AL,AL			; 0 is invalid
	JZ	EFGSETUP_DONE
	XOR	AH,AH			; set ticks/division
	MOV	_ticksperrow,AX
EFGSETUP_DONE:
	POP	AX
	RET
;
; EFHSETUP routine, the effect setup routine for effect H (volume
; slide/fineslide), ScreamTracker effect D.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFHSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	OR	AL,AL			; if 0, use previous
	JNZ	EFHSETUP_SAVEINFO
	MOV	AL,[BX].CHANINFOBYTE
EFHSETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,AL	; save infobyte for next time
	MOV	AH,AL			; AH = first parameter, AL = second
#IF M_I286
	SHR	AH,4
#ELSE
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
#ENDIF
	AND	AL,0Fh			; if second = 0, slide up
	JNZ	EFHSETUP_CHKSLIDEDOWN
	MOV	[BX].CHANVOLINCR,AH	; set increment
	JMP	SHORT EFHSETUP_SETPROCS
	EVEN
EFHSETUP_CHKSLIDEDOWN:
	OR	AH,AH			; if first = 0, slide down
	JNZ	EFHSETUP_CHKFINEUP
	NEG	AL			; set (negative) increment
	MOV	[BX].CHANVOLINCR,AL
EFHSETUP_SETPROCS:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFATICK ; set tick procedure
	JMP	SHORT EFHSETUP_DONE
	EVEN
EFHSETUP_CHKFINEUP:
	CMP	AL,0Fh			; if second = 0Fh, fine slide up
	JNE	EFHSETUP_CHKFINEDOWN
	MOV	AL,AH
	ADD	AL,[BX].CHANEFFVOL	; add effective volume to parameter
	CMP	AL,64			; if result > 64, set to 64
	JBE	EFHSETUP_SETVOL
	MOV	AL,64
	JMP	SHORT EFHSETUP_SETVOL
	EVEN
EFHSETUP_CHKFINEDOWN:
	CMP	AH,0Fh			; if first = 0Fh, fine slide down
	JNE	EFHSETUP_DONE
	NEG	AL		; subtract parameter from effective volume
	ADD	AL,[BX].CHANEFFVOL
	JGE	EFHSETUP_SETVOL		; if result < 0, set to 0
	XOR	AL,AL
EFHSETUP_SETVOL:
	MOV	[BX].CHANEFFVOL,AL	; set effective and mixing volume
	MOV	[BX].CHANMIXVOL,AL
EFHSETUP_DONE:
	POP	AX
	RET
;
; EFISETUP routine, the effect setup routine for effect I (slide/fineslide
; down), ScreamTracker effect E.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFISETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	OR	AL,AL			; if 0, use previous
	JNZ	EFISETUP_SAVEINFO
	MOV	AL,[BX].CHANINFOBYTE
EFISETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,AL	; save infobyte for next time
	CMP	AL,0F0h			; if >= 0F0h, fine slide down
	JB	EFISETUP_CHKEXTRAFINE
	AND	AL,0Fh			; AX = slide increment
	JZ	EFISETUP_DONE		; if 0, nothing to do
	CBW
	SHL	AX,1
	SHL	AX,1
	JMP	SHORT EFISETUP_ADDPERIOD
	EVEN
EFISETUP_CHKEXTRAFINE:
	CMP	AL,0E0h			; if >= 0E0h, extra fine slide down
	JB	EFISETUP_SLIDE
	AND	AL,0Fh			; AX = slide increment
	JZ	EFISETUP_DONE		; if 0, nothing to do
	CBW
EFISETUP_ADDPERIOD:
	ADD	AX,[BX].CHANEFFPERIOD	; add to period
	CMP	AX,27392		; if greater than 27392, set to 27392
	JBE	EFISETUP_SETPERIOD
	MOV	AX,27392
EFISETUP_SETPERIOD:
	MOV	[BX].CHANEFFPERIOD,AX	; save new effective period
	PUSH	BX			; save pointer to channel data
	MOV	CX,AX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; get sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
	JMP	SHORT EFISETUP_DONE
	EVEN
EFISETUP_SLIDE:
	XOR	AH,AH			; normal slide - set slide increment
	SHL	AX,1
	SHL	AX,1
	MOV	[BX].CHANSLIDEINCR,AX	; save it
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF1TICK ; set tick procedure
EFISETUP_DONE:
	POP	DI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; EFJSETUP routine, the effect setup routine for effect J (slide/fineslide
; up), ScreamTracker effect F.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFJSETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	OR	AL,AL			; if 0, use previous
	JNZ	EFJSETUP_SAVEINFO
	MOV	AL,[BX].CHANINFOBYTE
EFJSETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,AL	; save infobyte for next time
	CMP	AL,0F0h			; if >= 0F0h, fine slide up
	JB	EFJSETUP_CHKEXTRAFINE
	AND	AL,0Fh			; AX = slide increment
	JZ	EFJSETUP_DONE		; if 0, nothing to do
	CBW
	SHL	AX,1
	SHL	AX,1
	JMP	SHORT EFJSETUP_ADDPERIOD
	EVEN
EFJSETUP_CHKEXTRAFINE:
	CMP	AL,0E0h			; if >= 0E0h, extra fine slide up
	JB	EFJSETUP_SLIDE
	AND	AL,0Fh			; AX = slide increment
	JZ	EFJSETUP_DONE		; if 0, nothing to do
	CBW
EFJSETUP_ADDPERIOD:
	NEG	AX			; make increment negative
	ADD	AX,[BX].CHANEFFPERIOD	; add to period
	JG	EFJSETUP_SETPERIOD	; if less than 1, set to 1
	MOV	AX,1
EFJSETUP_SETPERIOD:
	MOV	[BX].CHANEFFPERIOD,AX	; save new effective period
	PUSH	BX			; save pointer to channel data
	MOV	CX,AX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; get sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
	JMP	SHORT EFJSETUP_DONE
	EVEN
EFJSETUP_SLIDE:
	XOR	AH,AH			; normal slide - set slide increment
	SHL	AX,1
	SHL	AX,1
	NEG	AX			; make increment negative
	MOV	[BX].CHANSLIDEINCR,AX	; save it
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF1TICK ; set tick procedure
EFJSETUP_DONE:
	POP	DI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; EFKSETUP routine, the effect setup routine for effect K (tremor),
; ScreamTracker effect I.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFKSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	OR	AL,AL			; if 0, continue tremor
	JZ	EFKSETUP_SETPROCS
	MOV	AH,AL			; AH = first parameter, AL = second
#IF M_I286
	SHR	AH,4
#ELSE
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
#ENDIF
	AND	AL,0Fh
	JZ	EFKSETUP_DONE		; if offtime = 0, nothing to do
	OR	AH,AH			; if ontime = 0, turn the note off
	JNZ	EFKSETUP_SETONEND
	MOV	[BX].CHANMIXVOL,AH
	JMP	SHORT EFKSETUP_DONE
	EVEN
EFKSETUP_SETONEND:
	MOV	[BX].CHANTREMOROFF,AH	; set tremor off and back on times
	ADD	AL,AH
	MOV	[BX].CHANTREMOREND,AL
	MOV	[BX].CHANTREMORCOUNT,0	; set tremor count to 0
EFKSETUP_SETPROCS:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFKTICK ; set tick procedure
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF7END ; set end procedure
EFKSETUP_DONE:
	POP	AX
	RET
;
; EFLSETUP routine, the effect setup routine for effect L (arpeggio),
; ScreamTracker effect J.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFLSETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	AX,[BX].CHANSTEPFRAC	; set sample step 0 to normal step
	MOV	[BX].CHANASTEP0FRAC,AX
	MOV	AX,[BX].CHANSTEPINT
	MOV	[BX].CHANASTEP0INT,AX
	MOV	DL,ES:[DI].NOTEEFFPARM	; get parameter in DL
	OR	DL,DL			; if 0, use previous
	JNZ	EFLSETUP_SAVEINFO
	MOV	DL,[BX].CHANINFOBYTE
EFLSETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,DL	; save infobyte for next time
	PUSH	BX			; save BX
#IF M_I286
        SHR     DL,4			; get first parameter in DX
#ELSE
	SHR	DL,1
	SHR	DL,1
	SHR	DL,1
	SHR	DL,1
#ENDIF
	XOR	DH,DH
	MOV	CX,[BX].CHANEFFPERIOD	; get base period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETARPSTEP		; get sample step 1 (BX, CX, DI gone)
	POP	BX			; restore BX
	MOV	[BX].CHANASTEP1FRAC,AX	; set sample step 1
	MOV	[BX].CHANASTEP1INT,DX
	PUSH	BX			; save BX again
	MOV	DL,[BX].CHANINFOBYTE	; get second parameter in DX
	AND	DL,0Fh
	XOR	DH,DH
	MOV	CX,[BX].CHANEFFPERIOD	; get base period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETARPSTEP		; get sample step 1 (BX, CX, DI gone)
	POP	BX			; restore BX
	MOV	[BX].CHANASTEP2FRAC,AX	; set sample step 2
	MOV	[BX].CHANASTEP2INT,DX
	MOV	[BX].CHANARPCOUNT,0	; set arpeggio counter to 0
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF0TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF0END ; set end procedure
	POP	DI
	POP	DX
	POP	CX
	POP	AX
	RET
;
; EFMSETUP routine, the effect setup routine for effect M (vibrato plus
; volume slide), ScreamTracker effect K.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFMSETUP:
#IF M_I286
	PUSHA
#ELSE
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameters in AL
	OR	AL,AL			; if 0, use previous
	JNZ	EFMSETUP_SAVEINFO
	MOV	AL,[BX].CHANINFOBYTE
EFMSETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,AL	; save infobyte for next time
	MOV	DL,AL			; AL = second parameter, DL = first
	AND	AL,0Fh			; is the second parameter 0?
	JZ	EFMSETUP_SLIDEUP
	NEG	AL			; not 0 - sliding volume down
	MOV	[BX].CHANVOLINCR,AL	; save (negative) increment
	JMP	SHORT EFMSETUP_CHKNOTE
	EVEN
EFMSETUP_SLIDEUP:
#IF M_I286
	SHR	DL,4			; sliding volume up
#ELSE
	SHR	DL,1
	SHR	DL,1
	SHR	DL,1
	SHR	DL,1
#ENDIF
	MOV	[BX].CHANVOLINCR,DL	; save (positive) increment
EFMSETUP_CHKNOTE:
	CMP	ES:[DI].NOTEPERIOD,0	; is it a new note?
	JZ	EFMSETUP_SAMENOTE	; if new note:
	MOV	SI,[BX].CHANVIBPOS	; SI is position in vibrato wave table
	CMP	[BX].CHANVIBNORETRG,1	; is retrigger disabled?
	JE	EFMSETUP_NORETRG
	XOR	SI,SI			; retrigger vibrato (set position = 0)
	MOV	[BX].CHANVIBPOS,SI
	JMP	SHORT EFMSETUP_LOOKUP
	EVEN
EFMSETUP_NORETRG:
	CMP	SI,33			; position < 33? use it directly then
	JB	EFMSETUP_LOOKUP
	NEG	SI			; position >= 33, use 64-position
	ADD	SI,64
EFMSETUP_LOOKUP:
	SHL	SI,1			; get waveform table index
	ADD	SI,[BX].CHANVIBWAVE	; SI addresses period adjustment
	MOV	AX,[SI]			; AX is fixed-point adjustment
	MUL	[BX].CHANVIBDEPTH	; multiply by vibrato depth
	SHL	AX,1			; round to nearest
	ADC	DX,0			; DX is (positive) adjustment
	CMP	[BX].CHANVIBPOS,32	; position < 32? adjust up
	JB	EFMSETUP_SETPERIOD
	NEG	DX			; position >= 32, adjust down
EFMSETUP_SETPERIOD:
	ADD	DX,[BX].CHANEFFPERIOD	; DX is period for first tick
	JG	EFMSETUP_HIGHENUF	; if < 1, set to 1
	MOV	DX,1
	JMP	SHORT EFMSETUP_SETSTEP
	EVEN
EFMSETUP_HIGHENUF:
	CMP	DX,27392		; if > 27392, set to 27392
	JLE	EFMSETUP_SETSTEP
	MOV	DX,27392
EFMSETUP_SETSTEP:
	PUSH	BX			; save pointer to channel data
	MOV	CX,DX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
EFMSETUP_SAMENOTE:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF6TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF4END ; set end procedure
#IF M_I286
	POPA
#ELSE
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	AX
#ENDIF
	RET
;
; EFNSETUP routine, the effect setup routine for effect N (slide to note
; plus volume slide), ScreamTracker effect L.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFNSETUP:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter in AL
	OR	AL,AL			; if 0, use previous
	JNZ	EFNSETUP_SAVEINFO
	MOV	AL,[BX].CHANINFOBYTE
EFNSETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,AL	; save infobyte for next time
	MOV	DL,AL			; AL = second parameter, DL = first
	AND	AL,0Fh			; is the second parameter 0?
	JZ	EFNSETUP_SLIDEUP
	NEG	AL			; not 0 - sliding volume down
	MOV	[BX].CHANVOLINCR,AL	; save (negative) increment
	JMP	SHORT EFNSETUP_GETGOAL
 	EVEN
EFNSETUP_SLIDEUP:
#IF M_I286
	SHR	DL,4			; sliding volume up
#ELSE
	SHR	DL,1
	SHR	DL,1
	SHR	DL,1
	SHR	DL,1
#ENDIF
	MOV	[BX].CHANVOLINCR,DL	; save (positive) increment
EFNSETUP_GETGOAL:
	MOV	CX,[BX].CHANPERIODGOAL	; get period goal in CX
	MOV	AX,ES:[DI].NOTEPERIOD	; get period in AX
	OR	AX,AX			; if period 0, do not adjust goal
	JZ	EFNSETUP_GETINCR
	MOV	CX,AX			; set period goal equal to period
	MOV	[BX].CHANPERIODGOAL,AX
EFNSETUP_GETINCR:
	MOV	AX,[BX].CHANSLIDEINCR	; get current slide increment in AX
	CWD				; take absolute value
	XOR	AX,DX
	SUB	AX,DX
EFNSETUP_SETSIGN:
	CMP	CX,[BX].CHANEFFPERIOD	; if goal < current, negate increment
	JAE	EFNSETUP_SETINCR
	NEG	AX
EFNSETUP_SETINCR:
	MOV	[BX].CHANSLIDEINCR,AX	; save new (?) slide increment
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF5TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF3END ; set end procedure
	POP	DX
	POP	CX
	POP	AX
	RET
;
; EFOSETUP routine, the effect setup routine for effect O (retrigger plus
; volume slide), ScreamTracker effect Q.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFOSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter in AL
	OR	AL,AL			; if 0, use previous
	JNZ	EFOSETUP_SAVEINFO
	MOV	AL,[BX].CHANINFOBYTE
EFOSETUP_SAVEINFO:
	MOV	[BX].CHANINFOBYTE,AL	; save infobyte for next time
	AND	AL,0Fh			; get retrigger speed
	JZ	EFOSETUP_DONE		; retrigger speed 0 is invalid
	MOV	[BX].CHANRETRIGSPEED,AL	; set retrigger speed and count
	MOV	[BX].CHANRETRIGCOUNT,AL
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFOTICK ; set tick procedure
EFOSETUP_DONE:
	POP	AX
	RET
;
; EFPSETUP routine, the effect setup routine for effect P (set bpm),
; ScreamTracker effect T.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFPSETUP:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	MOV	BL,ES:[DI].NOTEEFFPARM	; get parameter
	CMP	BL,20h			; < 20h is invalid
	JB	EFPSETUP_DONE
	MOV	_beatspermin,BL
	XOR	BH,BH
	SHL	BX,1
	MOV	AX,_samprate		; samples per tick = (5*samprate)
	MOV	DX,5			;   / (2*beatspermin)
	MUL	DX
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	_samplespertick,AX
EFPSETUP_DONE:
	POP	DX
	POP	BX
	POP	AX
	RET
;
; EFQSETUP routine, the effect setup routine for effect Q (fine vibrato),
; ScreamTracker effect U.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFQSETUP:
#IF M_I286
	PUSHA
#ELSE
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
#ENDIF
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameters in AL
	MOV	DL,AL			; AL = first parameter, DL = second
#IF M_I286
        SHR     AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	JZ	EFQSETUP_CHKDEPTH	; first parameter 0?
	CBW				; 1st parm <> 0 - set vibrato increment
	MOV	[BX].CHANVIBINCR,AX
EFQSETUP_CHKDEPTH:
	AND	DL,0Fh			; second parameter 0?
	JZ	EFQSETUP_CHKNOTE
	XOR	DH,DH			; 2nd parm <> 0 - set vibrato depth
	MOV	[BX].CHANVIBDEPTH,DX
EFQSETUP_CHKNOTE:
	CMP	ES:[DI].NOTEPERIOD,0	; is it a new note?
	JZ	EFQSETUP_SAMENOTE	; if new note:
	MOV	SI,[BX].CHANVIBPOS	; SI is position in vibrato wave table
	CMP	[BX].CHANVIBNORETRG,1	; is retrigger disabled?
	JE	EFQSETUP_NORETRG
	XOR	SI,SI			; retrigger vibrato (set position = 0)
	MOV	[BX].CHANVIBPOS,SI
	JMP	SHORT EFQSETUP_LOOKUP
	EVEN
EFQSETUP_NORETRG:
	CMP	SI,33			; position < 33? use it directly then
	JB	EFQSETUP_LOOKUP
	NEG	SI			; position >= 33, use 64-position
	ADD	SI,64
EFQSETUP_LOOKUP:
	SHL	SI,1			; get waveform table index
	ADD	SI,[BX].CHANVIBWAVE	; SI addresses period adjustment
	MOV	AX,[SI]			; AX is fixed-point adjustment
	MUL	[BX].CHANVIBDEPTH	; multiply by vibrato depth
	SHL	AX,1			; round to nearest
	ADC	DX,0			; DX is (positive) adjustment
	CMP	[BX].CHANVIBPOS,32	; position < 32? adjust up
	JB	EFQSETUP_SETPERIOD
	NEG	DX			; position >= 32, adjust down
EFQSETUP_SETPERIOD:
	ADD	DX,[BX].CHANEFFPERIOD	; DX is period for first tick
	JG	EFQSETUP_HIGHENUF	; if < 1, set to 1
	MOV	DX,1
	JMP	SHORT EFQSETUP_SETSTEP
	EVEN
EFQSETUP_HIGHENUF:
	CMP	DX,27392		; if > 27392, set to 27392
	JLE	EFQSETUP_SETSTEP
	MOV	DX,27392
EFQSETUP_SETSTEP:
	PUSH	BX			; save pointer to channel data
	MOV	CX,DX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; compute sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
EFQSETUP_SAMENOTE:
	MOV	[BX].CHANEFFTICKPROC,OFFSET EF4TICK ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET EF4END ; set end procedure
#IF M_I286
	POPA
#ELSE
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	AX
#ENDIF
	RET
;
; EFRSETUP routine, the effect setup routine for effect R (set global volume),
; ScreamTracker effect V.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFRSETUP:
	PUSH	AX
	MOV	AL,ES:[DI].NOTEEFFPARM	; get parameter
	CMP	AL,64			; if > 64, set to 64
	JBE	EFRSETUP_DOSET
	MOV	AL,64
EFRSETUP_DOSET:
	MOV	_globalvol,AL		; set global volume
	POP	AX
	RET
;
; EFSSETUP routine, the effect setup routine for effect S (ScreamTracker
; extended effect).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFSSETUP:
	PUSH	AX
	PUSH	SI
	MOV	AL,ES:[DI].NOTEEFFPARM
#IF M_I286
        SHR     AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	CWD
	SHL	AX,1
	MOV	SI,AX
	MOV	AL,ES:[DI].NOTEEFFPARM
	AND	AL,0Fh
	JMP	EFSSETUPS[SI]
;
; EXTENDED EFFECT SETUP ROUTINES *************************************
;
; EFESETUP jumps to these routines after saving the AX and SI registers on
; the stack, so these routines need to POP SI, POP AX (in that order) before
; returning.  On entry, DS is the default data segment, BX addresses the
; channel data structure, ES:DI addresses the note structure for the current
; note, and AL contains the parameter (low nibble of major effect parameter).
; No registers should be modified.
;
; EFENONE routine.  Used when an extended effect does not exist or is not
; to be implemented.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFENONE:
	POP	SI
	POP	AX
	RET
;
; EFE1SETUP routine, the effect setup routine for effect E1 (fineslide up).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE1SETUP:
	PUSH	CX
	PUSH	DX
	PUSH	DI
	CBW				; AX is slide increment
	SHL	AX,1
	SHL	AX,1
	NEG	AX			; make increment negative
	JZ	EFE1SETUP_DONE		; if the parameter is 0, nothing to do
	ADD	AX,[BX].CHANEFFPERIOD	; add to period
	JG	EFE1SETUP_SETPERIOD	; if less than 1, set to 1
	MOV	AX,1
EFE1SETUP_SETPERIOD:
	MOV	[BX].CHANEFFPERIOD,AX	; save new effective period
	PUSH	BX			; save pointer to channel data
	MOV	CX,AX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; get sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
EFE1SETUP_DONE:
	POP	DI
	POP	DX
	POP	CX
	POP	SI
	POP	AX
	RET
;
; EFE2SETUP routine, the effect setup routine for effect E2 (fineslide down).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE2SETUP:
	PUSH	CX
	PUSH	DX
	PUSH	DI
	CBW				; AX is slide increment
	SHL	AX,1
	SHL	AX,1
	OR	AX,AX			; if the parameter is 0, nothing to do
	JZ	EFE2SETUP_DONE
	ADD	AX,[BX].CHANEFFPERIOD	; add to period
	CMP	AX,27392		; if greater than 27392, set to 27392
	JLE	EFE2SETUP_SETPERIOD
	MOV	AX,27392
EFE2SETUP_SETPERIOD:
	MOV	[BX].CHANEFFPERIOD,AX	; save new effective period
	PUSH	BX			; save pointer to channel data
	MOV	CX,AX			; get period in CX
	MOV	BX,[BX].CHANC4SPD	; get C4SPD in BX
	CALL	GETSTEP			; get sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
EFE2SETUP_DONE:
	POP	DI
	POP	DX
	POP	CX
	POP	SI
	POP	AX
	RET
;
; EFE3SETUP routine, the effect setup routine for effect E3 (glissando
; control).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE3SETUP:
	CMP	AL,1			; if parameter > 1, ignore the effect
	JA	EFE3SETUP_DONE
	MOV	[BX].CHANGLISS,AL	; set glissando flag
EFE3SETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFE4SETUP routine, the effect setup routine for effect E4 (set vibrato
; waveform).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE4SETUP:
	CMP	AL,7			; if parameter > 7, effect invalid
	JA	EFE4SETUP_DONE
	MOV	AH,AL			; AH = retrigger disable flag
	SHR	AH,1
	SHR	AH,1
	MOV	[BX].CHANVIBNORETRG,AL	; set vibrato retrigger
	AND	AL,3			; AL = waveform (0-3)
	CMP	AL,3			; if 3, waveform is random
	JB	EFE4SETUP_SETWAVE
	PUSH	ES
	MOV	AX,40h			; get low byte of clock count
	MOV	ES,AX
	MOV	AL,ES:[6Ch]
	POP	ES
	SHR	AL,1			; take top 6 bits
	SHR	AL,1
	MOV	AH,AL			; multiply by 3
	SHL	AL,1
	ADD	AL,AH
	ROL	AL,1			; get waveform (0-2) in AL
	ROL	AL,1
	AND	AL,3
EFE4SETUP_SETWAVE:
	CMP	AL,1			; is waveform 0, 1 or 2?
	JE	EFE4SETUP_RAMP
	JA	EFE4SETUP_SQUARE
	MOV	[BX].CHANVIBWAVE,OFFSET SINETBL
	JMP	SHORT EFE4SETUP_DONE
	EVEN
EFE4SETUP_RAMP:
	MOV	[BX].CHANVIBWAVE,OFFSET RAMPTBL
	JMP	SHORT EFE4SETUP_DONE
	EVEN
EFE4SETUP_SQUARE:
	MOV	[BX].CHANVIBWAVE,OFFSET SQUARETBL
EFE4SETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFE5SETUP routine, the effect setup routine for effect E5 (set finetune).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE5SETUP:
	PUSH	CX
	PUSH	DX
	PUSH	DI
	XOR	AL,8			; convert finetune to _c4spds index
	CBW
	SHL	AX,1
	MOV	SI,AX
	MOV	AX,_c4spds[SI]		; get C4SPD for finetune
	MOV	[BX].CHANC4SPD,AX	; save new C4SPD
	PUSH	BX			; save channel data pointer
	MOV	CX,[BX].CHANEFFPERIOD	; get period in CX
	MOV	BX,AX			; get C4SPD in BX
	CALL	GETSTEP			; get sample step (BX, CX, DI gone)
	POP	BX			; get back channel data pointer
	MOV	[BX].CHANSTEPFRAC,AX	; save sample step
	MOV	[BX].CHANSTEPINT,DX
	POP	DI
	POP	DX
	POP	CX
	POP	SI
	POP	AX
	RET
;
; EFE6SETUP routine, the effect setup routine for effect E6 (loop pattern).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE6SETUP:
	OR	AL,AL		; is the parameter 0?
	JNZ	EFE6SETUP_NOTSTART
	MOV	AL,_currentrow	; set pattern loop start to current row
	MOV	_patloopstart,AL
	JMP	SHORT EFE6SETUP_DONE
	EVEN
EFE6SETUP_NOTSTART:
	CMP	_patloopcount,0	; is pattern loop count 0? (loop not started)
	JNE	EFE6SETUP_INLOOP
	MOV	_patloopcount,AL ; loop starting - set count to parameter
	JMP	SHORT EFE6SETUP_SETROW
	EVEN
EFE6SETUP_INLOOP:
	DEC	_patloopcount	; in loop - finished yet?
	JZ	EFE6SETUP_DONE	; if zero, loop is done, go on
EFE6SETUP_SETROW:
	MOV	AL,_patloopstart ; looping still, go back to loop start
	MOV	_nextrow,AL
EFE6SETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFE7SETUP routine, the effect setup routine for effect E7 (set tremolo
; waveform).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE7SETUP:
	CMP	AL,7			; if parameter > 7, effect invalid
	JA	EFE7SETUP_DONE
	MOV	AH,AL			; AH = retrigger disable flag
	SHR	AH,1
	SHR	AH,1
	MOV	[BX].CHANTREMNORETRG,AL	; set tremolo retrigger
	AND	AL,3			; AL = waveform (0-3)
	CMP	AL,3			; if 3, waveform is random
	JB	EFE7SETUP_SETWAVE
	PUSH	ES
	MOV	AX,40h			; get low byte of clock count
	MOV	ES,AX
	MOV	AL,ES:[6Ch]
	POP	ES
	SHR	AL,1			; take top 6 bits
	SHR	AL,1
	MOV	AH,AL			; multiply by 3
	SHL	AL,1
	ADD	AL,AH
	ROL	AL,1			; get waveform (0-2) in AL
	ROL	AL,1
	AND	AL,3
EFE7SETUP_SETWAVE:
	CMP	AL,1			; is waveform 0, 1 or 2?
	JE	EFE7SETUP_RAMP
	JA	EFE7SETUP_SQUARE
	MOV	[BX].CHANTREMWAVE,OFFSET SINETBL
	JMP	SHORT EFE7SETUP_DONE
	EVEN
EFE7SETUP_RAMP:
	MOV	[BX].CHANTREMWAVE,OFFSET RAMPTBL
	JMP	SHORT EFE7SETUP_DONE
	EVEN
EFE7SETUP_SQUARE:
	MOV	[BX].CHANTREMWAVE,OFFSET SQUARETBL
EFE7SETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFE9SETUP routine, the effect setup routine for effect E9 (retrigger
; sample).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFE9SETUP:
	OR	AL,AL			; invalid if parameter 0
	JZ	EFE9SETUP_DONE
	MOV	[BX].CHANRETRIGSPEED,AL	; set retrigger speed and count to
	MOV	[BX].CHANRETRIGCOUNT,AL	;   the parameter
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFE9TICK ; set tick procedure
EFE9SETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFEASETUP routine, the effect setup routine for effect EA (fine volume
; slide up).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFEASETUP:
	ADD	AL,[BX].CHANEFFVOL	; add effective volume to parameter
	CMP	AL,64			; if result > 64, set to 64
	JBE	EFEASETUP_SETVOL
	MOV	AL,64
EFEASETUP_SETVOL:
	MOV	[BX].CHANEFFVOL,AL	; set effective and mixing volume
	MOV	[BX].CHANMIXVOL,AL
	POP	SI
	POP	AX
	RET
;
; EFEBSETUP routine, the effect setup routine for effect EB (fine volume
; slide down).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFEBSETUP:
	NEG	AL		; subtract parameter from effective volume
	ADD	AL,[BX].CHANEFFVOL
	JGE	EFEBSETUP_SETVOL	; if result < 0, set to 0
	XOR	AL,AL
EFEBSETUP_SETVOL:
	MOV	[BX].CHANEFFVOL,AL	; set effective and mixing volume
	MOV	[BX].CHANMIXVOL,AL
	POP	SI
	POP	AX
	RET
;
; EFECSETUP routine, the effect setup routine for effect EC (cut sample).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFECSETUP:
	OR	AL,AL			; if parameter is 0, cut sample now
	JNZ	EFECSETUP_NOTNOW
	MOV	[BX].CHANEFFVOL,AL	; set effective and mixing volume to 0
	MOV	[BX].CHANMIXVOL,AL
	JMP	SHORT EFECSETUP_DONE
	EVEN
EFECSETUP_NOTNOW:
	MOV	[BX].CHANCUTCOUNT,AL	; set cut count to parameter
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFECTICK ; set tick procedure
EFECSETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFEDSETUP routine, the effect setup routine for effect ED (delay sample).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFEDSETUP:
	OR	AL,AL		; if parameter is 0, no delay so do nothing
	JZ	EFEDSETUP_DONE
	CMP	ES:[DI].NOTEPERIOD,0 ; if no new note, nothing to delay
	JE	EFEDSETUP_DONE
	MOV	[BX].CHANDELAYCOUNT,AL ; set delay count equal to parameter
	MOV	[BX].CHANEFFTICKPROC,OFFSET EFEDTICK ; set tick procedure
EFEDSETUP_DONE:
	POP	SI
	POP	AX
	RET
;
; EFEESETUP routine, the effect setup routine for effect EE (delay pattern).
;
; Returns nothing.  Destroys nothing.
;
	EVEN
EFEESETUP:
	MOV	_patdelayparm,AL ; save the pattern delay parameter
	POP	SI
	POP	AX
	RET
