{ *******************************************************************
  *			    This file is part of the WMFH package				*
  ******************************************************************* }


{

	  Hlp_unit.pas	A Turbo Pascal unit for displaying help	files


			 Copyright (c) 1997   Gianfranco Boggio-Togna                              *

							C.P. 14021                                   
					  I-20140 Milano (Italy)                             

					 e-mail: gbt@computer.org                                                     *


    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

}

UNIT 	HLP_Unit ;

{DEFINE DEBUG}

{ ************************************************************************
  *																		 *
  *				        U N I T    I N T E R F A C E  					 *
  *																		 *
  ************************************************************************ }

INTERFACE 

USES	Dos, Crt, HLP_Type, WND_Unit, KB_Unit, MS_Unit ;

TYPE

	HLP_palette =	ARRAY [0..15] OF Byte ;

VAR

	HLP_first_line:			Integer ;	
	HLP_last_line:			Integer ;	
	HLP_first_column: 		Integer ;	
	HLP_last_column:		Integer ;	

	HLP_window_shadow: 		WND_Shadow_type ;
	HLP_window_frame:  		WND_Frame_type ;

	HLP_text_window_frame:	WND_Frame_type ;	
	
	HLP_window_BG,
	HLP_window_FG,
	HLP_text_BG,
	HLP_text_FG:            Byte ;

	HLP_button_BG,
	HLP_button_FG,
	HLP_button_high_FG:     Byte ;

	HLP_title_line_FG,
	HLP_title_line_BG:  	Byte ;

	HLP_reference_FG,
	HLP_reference_BG: 	 	Byte ;

	HLP_highlighted_reference_FG,
	HLP_highlighted_reference_BG: 	 	Byte ;

	HLP_popup_reference_FG,
	HLP_popup_reference_BG: Byte ;

 	HLP_popup_FG,
	HLP_popup_BG,
	HLP_popup_text_FG,
	HLP_popup_text_BG,
	HLP_popup_title_line_FG,
	HLP_popup_title_line_BG:  		Byte ;

	HLP_text_scroll_bar_BG:			Byte ;
	HLP_text_scroll_bar_button_BG:	Byte ;
	HLP_text_scroll_bar_button_FG:	Byte ;

	{ The spacing corresponding to the TAB character }	

	HLP_tab_spacing:		Integer ;


{ -------------------------------
  Initialize the help environment
  ------------------------------- }

FUNCTION   HLP_init (file_name: String; p: HLP_palette) : Integer ;

{		file_name	 the name of the help file created by MakeHelp

		p			 each element of this 16-bytes array contains a number
					 identifying a palette register in THE CALLING PROGRAM's
					 palette. The array is used for translating attribute
					 definitions within the text; if the text has \(n)
					 the foreground colour (i.e. palette register) used
					 is NOT "n" but "p[n]".

		HLP_init	 the number of areas defined in the help file or
					 zero if an error has occurred
}



{ --------------------------------
  Display an area of the help text
  -------------------------------- }

PROCEDURE  HLP_display_area (area_number: Integer; title: String) ;

{		area_number
		title		 the title for the help window
}



{ --------------------------------------------------
  Display the section corresponding to an identifier
  -------------------------------------------------- }

PROCEDURE  HLP_display_section (section_id: Integer; title: String) ;

{		section_id	 the section identifier
		title		 the title for the help window
}



{ ------------------------------------------------
  Free all dynamically allocated space for an area
  ------------------------------------------------ }

PROCEDURE  HLP_free (area_number: Integer) ;

{		The unit keeps all displayed sections in memory, for faster
		access.  If the calling program runs short of memory, it can
		use this procedure to free all memory used by an area
}




{ ************************************************************************
  *																		 *
  *					 U N I T	I M P L E M E N T A T I O N				 *
  *																		 *
  ************************************************************************ }

IMPLEMENTATION

CONST
		Copyright:	String =
					'WMFH 1.0 - Copyright 1997 Gianfranco Boggio-Togna' ;

CONST

	max_stacked 	  		= 8 ;
	max_sections_list 		= 32 ;
	max_references_per_page	= 50 ;
    max_zones 				= 10 ;

	min_depth  = 18 ;	  { lines  }
	min_width  = 70 ;	  { colums }

	hex_digit: String = '0123456789abcdef' ;

TYPE

	Buffer 	   = ARRAY  [0..MAX_SECTION_LENGTH] OF Byte ;
	Buffer_ptr = ^ Buffer ;

	String_ptr = ^ String ;

	Section_table_space  	 = ARRAY [0..MAX_SECTIONS-1] OF Section ;
	Section_pointer_space 	 = ARRAY [0..MAX_SECTIONS-1] OF Buffer_ptr ;
	Section_lines_space 	 = ARRAY [0..MAX_SECTION_LINES-1] OF Integer ;

	Area_table_space	 	 = ARRAY [1..MAX_AREAS] OF Area ;

	Reference = RECORD
					section:	Integer ;
					y:			Byte ;
					x_start:	Byte ;
					x_end:		Byte ;
					mode:		Char ;
					highlight:	Boolean ;
				END ;

	Mouse_zone = RECORD
					min_y:     Byte ;
					max_y:     Byte ;
					min_x:     Byte ;
					max_x:     Byte ;
					mode:      Byte ;
					key:       Word ;
				 END ;
	
	Video_location = RECORD
						video_character : Char ;
						video_attribute : Byte ;
					 END ;

	Section_info = RECORD
						number:		Integer ;
						top_line:	Integer ;
						y, x:		Byte ;
				   END ;
VAR

	HLP_window_y,
	HLP_window_x,
	HLP_window_width,
	HLP_window_depth:  		Byte ;

	HLP_title_line_y,
	HLP_title_line_x:  		Byte ;

	HLP_text_window_y,
	HLP_text_window_x,
	HLP_text_window_width,
	HLP_text_window_depth:	Byte ;

	HLP_buttons_y,				  
	HLP_button_backtrack_x,
	HLP_button_previous_x,
	HLP_button_next_x,
	HLP_button_cancel_x:	Byte ;

	HLP_text_y,
	HLP_text_x,
	HLP_text_width,
	HLP_text_depth,
	HLP_text_line_x,
	HLP_text_line_width:  		Byte ;

	HLP_popup_y,
	HLP_popup_x,
	HLP_popup_width,
	HLP_popup_text_y,
	HLP_popup_text_x,
	HLP_popup_text_width,
	HLP_popup_text_line_x,
	HLP_popup_text_line_y,
	HLP_popup_text_line_width: 	Byte ;

	palette:			HLP_palette ;

	help_file: 			File ;

	help_window_title:	String ;


	section_count:	 	Integer ;
	area_count: 		Integer ;
	max_section_lines:	Integer ;

	section_table:		^ Section_table_space ; 
	section_pointer:	^ Section_pointer_space ;
	section_lines:		^ Section_lines_space ;


	current_section:		Buffer_ptr ;
	current_section_lines:	Integer	;

	area_table:			^ Area_table_space ;

	sections_list:	 	ARRAY [1..max_sections_list] OF Section_info ;
	last_list_entry:	Integer ;

	section_buffer:		Buffer_ptr ;

	section_index: 		Integer ;
	area_index: 		Integer ;

	reference_table:	ARRAY [1..max_references_per_page] OF Reference ;
	ref_index:			Integer ;

	scroll_bar:     	Integer ;
	no_save:        	Integer ;
	full_screen_save:  	Integer;

	des:				Descriptor ;

	previous_attribute:	Byte ;

	mouse_zones:        ARRAY [1..max_zones] OF Mouse_zone ;
	n_zones:            Integer ;

	video_buffer:  		Word ;


{$IFDEF DEBUG}

PROCEDURE	dump_init ;
VAR
	i:	Integer ;
BEGIN
	Writeln ;
	Writeln ('Start address: ', des.start_address,
			 '   Areas: ', des.area_count,
			 '   Sections: ', des.section_count,
			 '   Lines: ', des.max_section_lines ) ;
	writeln ;
	writeln ('AREAS') ;
	writeln ;
	FOR  i := 1  TO  area_count  DO
		writeln (i, ' ', area_table^[i].area_first,
					' ', area_table^[i].area_last) ;

	writeln ;
	writeln ('SECTIONS') ;
	writeln ;
	FOR  i := 0  TO  section_count  DO
		writeln (i, ' ', section_table^[i].section_address, ' ',
						 section_table^[i].section_length, ' ',
						 section_table^[i].section_identifier ) ;
END ;
{$ENDIF}


{ ************************************************************************
  *																		 *
  *						  Service procedures							 *
  *																		 *
  ************************************************************************ }
			

{ ************************************************************************
  *																		 *
  *		Register a screen area where mouse action is significant 		 *
  *																		 *
  ************************************************************************ }

PROCEDURE  add_mouse_zone (y_min, x_min, y_max, x_max, m, k:  Word) ;
BEGIN
	Inc (n_zones) ;
	WITH  mouse_zones [n_zones]  DO
	  BEGIN
		min_y := y_min ;
		max_y := y_max ;
		min_x := x_min ;
		max_x := x_max ;
		mode  := m ;
		key   := k ;
	  END ;
END ;


{ ************************************************************************
  *																		 *
  *			   		Free the space used by a section  					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	free_section (s: Integer) ;
BEGIN
	IF  (s <= section_count)  AND  (section_pointer^[s] = Nil)  THEN
	  BEGIN
		FreeMem (section_pointer^[s], section_table^[s].section_length) ;
		section_pointer^[s] := Nil ;
	  END ;
END ;


{ ************************************************************************
  *																		 *
  *					Prepare a section for processing					 *
  *																		 *
  ************************************************************************ }
			
FUNCTION	get_section (s:	Integer) : Buffer_ptr ;
VAR
	p:		Buffer_ptr ;
	n, l:	Integer ;
BEGIN
	IF  s > section_count  THEN
		RunError (201) ;			{ Range check error }

	IF  section_pointer^[s] = Nil  THEN
	  BEGIN
		GetMem (section_pointer^[s], section_table^[s].section_length) ;
{$i-}							   
		Seek (help_file, section_table^[s].section_address) ;				
{$i+}							   
    	IF  IOResult <> 0  THEN				   
		  BEGIN
			get_section := Nil ;
			Exit ;
		  END ;
	
{$i-}							   
		BlockRead (help_file, section_pointer^[s]^,
							  section_table^[s].section_length) ;				
{$i+}							   
	    IF  IOResult <> 0  THEN				   
		  BEGIN
			free_section (s) ;
			get_section := Nil ;
			Exit ;
		  END ;
	  END ;

	  IF  section_pointer^[s] <> Nil  THEN
		BEGIN
			get_section :=  section_pointer^[s] ;
			n := 0 ;
			l := 0 ;
			WHILE  n < section_table^[s].section_length	 DO
			  BEGIN
			  	section_lines^[l] := n ;
				Inc (l) ;
				Inc (n, (section_pointer^[s])^[n] + 1) ;
			  END ;
			IF  l > max_section_lines  THEN
				RunError (201) ;			{ Range check error }
			current_section_lines := l ;
	  END ;
END ;

			
{ ************************************************************************
  *																		 *
  *				 Convert a number from ASCII to binary					 *
  *																		 *
  ************************************************************************ }

FUNCTION  read_number (s: String; VAR i: Integer) : Integer ;
VAR
	n:	Integer ;
BEGIN
		n := 0 ;
		WHILE  s[i] IN ['0'..'9']  DO
		BEGIN
			n := n * 10 + Ord(s[i]) - Ord('0') ;
			Inc (i) ;
		END ;
		read_number := n ;
END ;


{ ************************************************************************
  *																		 *
  *				 Process character(s) definition						 *
  *																		 *
  ************************************************************************ }

FUNCTION  	get_characters (VAR s: String; VAR i: Integer) : String ;

VAR
	c, k, n: 	Integer ;
	w:			String ;
BEGIN
		w := '' ;
		Inc (i) ;	

		WHILE  s[i] <> '}'  DO
		  BEGIN

			c := read_number (s, i);
			w := w + Chr(c) ;

			IF  s[i] = '*'  THEN
			  BEGIN
				Inc (i) ;
				n := read_number (s, i) ;
				FOR  k := 1  TO  n-1  DO
					w := w + Chr(c) ;
				END ;

			IF  s[i] = '-'  THEN
			  BEGIN
				Inc (i) ;
				n := read_number (s, i) ;
				FOR  c := c+1  TO  n  DO
					w := w + Chr(c) ;
			  END ;

			IF  s[i] = ','  THEN
				Inc (i) ;

		  END ;

		Inc (i) ;
		get_characters := w ;

END ;


{ ************************************************************************
  *																		 *
  *				Check whether highlighting is needed					 *
  *																		 *
  ************************************************************************ }

PROCEDURE  check_highlight ;
VAR
	i, old, new:	Integer ;
	x, y:			Integer ;

PROCEDURE  change_attribute (i: Integer; on: Boolean) ;
VAR
	attribute:		Byte ;
	buffer_offset:	Word ;
	video_ptr:		^ Video_location ;
	x:				Integer ;
BEGIN

	reference_table[i].highlight := on ;
	IF  on  THEN
		attribute := HLP_highlighted_reference_FG +
					 (HLP_highlighted_reference_BG shl 4) 
	ELSE
	  IF  reference_table[i].mode = '['  THEN
			attribute := HLP_popup_reference_FG + (HLP_popup_reference_BG shl 4) 
	  ELSE
			attribute := HLP_reference_FG + (HLP_reference_BG shl 4) ;

	{ the offset within the video buffer of a screen position is:
	  			2 * ( 80 * (y-1) + (x-1) )							}

	buffer_offset := 2 * 80 * (reference_table[i].y - 1) - 2 ;

	FOR  x := reference_table[i].x_start TO  reference_table[i].x_end  DO
	  BEGIN
	  	video_ptr := Ptr (video_buffer, buffer_offset + x + x) ;
		video_ptr^.video_attribute := attribute ;
	  END ;
END ;

BEGIN
		x := WhereX ;
		y := WhereY ;
		old := 0 ;
		new := 0 ;
		FOR  i := 1  TO  ref_index  DO
		  BEGIN
		  	IF  reference_table[i].highlight  THEN
 				old := i ;
			IF  (y = reference_table[i].y)        AND
				(x >= reference_table[i].x_start) AND
				(x <= reference_table[i].x_end)   THEN   
		  		new := i ;
		  END ;
		IF  old <> new  THEN
		  BEGIN
			IF  old <> 0  THEN
				change_attribute (old, False) ;
			IF  new <> 0  THEN
				change_attribute (new, True) ;
		  END ;
END ;


{ ************************************************************************
  *																		 *
  *					 		Display a line 								 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  display_line (s: 	 String ;	{ the line }
						 ly, lx: Byte ;		{ screen coordinates }
						 title:  Boolean ;	{ is this a title line ? }
						 popup:  Boolean) ;	{ within a popup section ? }
VAR
	i, n, limit:  Integer ;
	w:	   		  String ;
	reference:	  Boolean ;
	
BEGIN

	GotoXY (lx, ly) ;
	IF  popup  THEN
		limit := HLP_popup_text_line_x + HLP_popup_text_line_width - 1 
	ELSE
		limit := HLP_text_line_x + HLP_text_line_width ;

	reference := False ;
	i := 1 ;
	WHILE  (i <= Length(s))  AND  (WhereX < limit)  DO
	  BEGIN
		IF  s[i] = ESC  THEN 
		  BEGIN
		    Inc(i) ;
		    CASE  s[i]  OF

		   	'(':
		  	  BEGIN
		  	    Inc (i) ;	
		  		IF  s[i] = ')'  THEN
				  TextAttr := previous_attribute
		  		ELSE
		  		  BEGIN
		  		    n := read_number (s, i) ;
		  			TextColor (palette[n]) ;
		  		    IF  s[i] = ','  THEN
		  		  	  BEGIN
		  			    Inc (i) ;
		  			    n := read_number (s, i) ;
		  				TextBackground (palette[n]) ;
		  			  END ;
		  		  END ;
		  		Inc (i) ;
		  	  END ;

		   	'{':
		 	  BEGIN
		 		w := get_characters (s, i) ;
		 		IF  WhereX + Length(w) <= limit  THEN
		 			write (w) ;
		 	  END ;

		   	'<', '[':
		 	  BEGIN
				reference := True ;
				IF  popup  THEN
					Inc (i, 2)
				ELSE
				  BEGIN
					IF  s[i] = '<'  THEN
					  BEGIN
						TextColor (HLP_reference_FG) ;
						TextBackground (HLP_reference_BG) ;
					  END
					ELSE
					  BEGIN
						TextColor (HLP_popup_reference_FG) ;
						TextBackground (HLP_popup_reference_BG) ;
					  END ;
		 			Inc (ref_index) ;
		 			IF  ref_index > max_references_per_page  THEN
						RunError (201) ;			{ Range check error }
		 			reference_table[ref_index].mode := s[i] ;
		 			reference_table[ref_index].highlight := False ;
		 			reference_table[ref_index].y := WhereY ;
		 			Inc (i) ;
		 			reference_table[ref_index].section := read_number (s, i);
		 			Inc (i) ;
		 			reference_table[ref_index].x_start := WhereX ;
				  END ;
		 	  END ;
		 	ELSE
			  BEGIN
			  	write (s[i]) ;
			    Inc (i) ;
			  END ;
		    END	;
		  END 
		ELSE
		 BEGIN
		   IF  (s[i] = '>')  OR  (s[i] = ']')  THEN
		 	  BEGIN
				IF  NOT reference  THEN
					write (s[i]) 
				ELSE
				  BEGIN
					reference := False ;
					IF  NOT  popup  THEN
					  BEGIN
						TextBackground (HLP_text_BG) ; 
						TextColor (HLP_text_FG) ; 
				 		reference_table[ref_index].x_end := WhereX - 1;
					  END ;
				  END ;
			  END 
		   ELSE
			 IF  s[i] = Chr(KB_Tab)  THEN
				 FOR  n := 1  TO
				 	  HLP_tab_spacing - (WhereX - lx) MOD HLP_tab_spacing DO
 				 	write (' ') 
			 ELSE
		 	   	 write (s[i]) ;
		   Inc (i) ;
		 END ;
	  END;	

	IF  i <= Length(s)  THEN
	  BEGIN
	  	HighVideo ;
		write ('') ;
	  	LowVideo ;
	  END ;

	IF  NOT  popup  THEN
	  BEGIN
		IF  title THEN
	  	  BEGIN
			TextBackground (HLP_window_BG) ; 
			TextColor (HLP_window_FG) ; 
	  	  END 
		ELSE
	  	  BEGIN
			TextBackground (HLP_text_BG) ; 
			TextColor (HLP_text_FG) ; 
	  	  END ;

		FOR  i := WhereX  TO  limit  DO
	  		write (' ') ;
	  END ;

END ;


{ ************************************************************************
  *																		 *
  *						Display a page of text							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	display_text (section: Integer ;
						  first_line: Integer ;
						  total_lines: Integer) ;

VAR
	x, y, x_last:	Integer ;
	i:	  			Integer ;	
	line:			String_ptr ;

BEGIN

	WND_Save_Cursor (False) ;
	MS_Hide ;
	WND_show_scroll_bar (scroll_bar,
				 	 	 HLP_text_y ,
					 	 HLP_text_x + HLP_text_width - 1,
					 	 total_lines, first_line) ;

	TextBackground (HLP_text_BG) ; 
	TextColor (HLP_text_FG) ; 
	previous_attribute := TextAttr ;
	ref_index := 0 ;
	y := HLP_text_y ;
	IF  first_line <> 0  THEN
	  BEGIN
	  	x_last :=  first_line + HLP_text_depth - 1 ;
		IF  x_last > total_lines   THEN
	  		x_last := total_lines  ;
		FOR  x := first_line  TO  x_last DO 
		  BEGIN
		  	line := Addr(section_pointer^[section]^[section_lines^[x]]) ;
		  	display_line (line^, y, HLP_text_line_x, False, False) ;
		  	Inc (y) ;
		  END ;
	  	FOR  x := x_last + 1  TO  first_line + HLP_text_depth - 1  DO 
		  BEGIN
			display_line ('  ', y, HLP_text_line_x, False, False) ;
		  	Inc (y) ;
		  END ;
	  END ;
	MS_Show ;
	WND_Restore_Cursor ;
END ; 


{ ************************************************************************
  *																		 *
  *					 Display a popup section							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	display_popup (section: Integer; old_section: Integer ) ;

VAR
	save, nosave: Integer ;
	x, y:		  Integer ;
	n:			  Integer ;
	depth:		  Integer ;
	w:			  String ;
	line:		  String_ptr ;
	dummy:		  Word ;
	
BEGIN
		WND_Save_Cursor (False) ;
		MS_Hide ;

		section_pointer^[section] := get_section (section) ;

		depth := current_section_lines + 2 ;
		IF  depth > HLP_text_depth - 3 THEN
		  depth := HLP_text_depth - 3 ;

		save := 1 ;
		WND_open_window (HLP_popup_y,
						 HLP_popup_x ,
						 HLP_popup_y + depth - 1,
						 HLP_popup_x + HLP_popup_width - 1,
						 HLP_popup_FG, HLP_popup_BG,
						 WND_no_shadow, WND_no_frame,
						 '',
						 save) ;
		nosave := 0 ;
		WND_open_window (HLP_popup_text_y,
						 HLP_popup_text_x,
						 HLP_popup_text_y + depth - 1,
						 HLP_popup_text_x + HLP_popup_text_width - 1,
						 HLP_popup_FG, HLP_popup_BG,
						 WND_no_shadow, WND_single_frame,
						 '',
						 nosave) ;

		y := HLP_popup_text_line_y ;

    	line := Addr(section_pointer^[section]^[section_lines^[0]]) ;
		n :=  Pos('\section ', line^) ;
		IF  n <> 0  THEN
	  	  BEGIN
	  		w := line^ ;
			Delete (w, n, Length('\section ')) ;
			TextColor (HLP_popup_title_line_FG) ;
			TextBackground (HLP_popup_title_line_BG) ;
			previous_attribute := TextAttr ;
			display_line (w, y, HLP_popup_text_line_x, True, True) ;
			Inc (y) ;
			n := 1 ;
	  	  END ;


		TextBackground (HLP_popup_text_BG) ; 
		TextColor (HLP_popup_text_FG) ; 
		previous_attribute := TextAttr ;

		FOR  x := 1  TO  depth - 2 - n DO 
		  BEGIN
		  	line := Addr(section_pointer^[section]^[section_lines^[x]]) ;
		  	display_line (line^, y, HLP_popup_text_line_x, False, True) ;
		  	Inc (y) ;
		END ;

		REPEAT  UNTIL  KeyPressed  OR  MS_LeftPressed  OR  MS_RightPressed  ;

	  	IF  KeyPressed  THEN
			dummy := KB_read ;
		IF	MS_LeftPressed  THEN
			REPEAT UNTIL NOT MS_LeftDown ;
		IF	MS_RightPressed  THEN
			REPEAT UNTIL NOT MS_RightDown ;

		WND_close_window (save) ;
		section_pointer^[old_section] := get_section (old_section) ;

		WND_Restore_Cursor ;
		MS_Show ;
END ;


{ ************************************************************************
  *																		 *
  *					   Display a normal section							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	display_section (VAR s: Section_info; area:  Integer) ;


VAR
	top_line:		Integer ;
	section:		Integer ;
	next_section:	Integer ;
	cursor_x,
	cursor_y:		Integer ;

	line:	 		String_ptr ;
	w:				String ;
	n:				Integer ;
	total_lines:	Integer ;
	key:			Word ;
	move:			Integer ;
	x, y:			Integer ;
	update_display:	Boolean ;


PROCEDURE	check_reference (y, x: Integer) ;
VAR
	i, j:	Integer ;
BEGIN

	i := 1 ;
	WHILE  (i <= ref_index)  AND  NOT ( (y = reference_table[i].y)        AND
					  					(x >= reference_table[i].x_start) AND
					  					(x <= reference_table[i].x_end) ) DO   
	  	Inc (i) ;


	IF  i <= ref_index  THEN
	  BEGIN
		IF  reference_table[i].section = 32767  THEN
			Exit ;
		IF  reference_table[i].mode = '['  THEN
		  display_popup (reference_table[i].section, section)
		ELSE
  		  BEGIN
			Inc (last_list_entry) ; 
			IF  last_list_entry > max_sections_list  THEN
	  	  	BEGIN
	  			FOR  j := 2  TO	max_sections_list  DO
					sections_list [j-1]	:= sections_list [j] ;
				Dec (last_list_entry) ; 
	  	  	END ;
			sections_list [last_list_entry].number := section ;
			sections_list [last_list_entry].top_line := top_line ;
			sections_list [last_list_entry].y := y ;
			sections_list [last_list_entry].x := x ;
		  	next_section := reference_table[i].section ;
		  	top_line := 1 ;
			cursor_x := 0 ;
			cursor_y := 0 ;
		  END
	  END ;
END ;


FUNCTION  interpret_key (k: Word) : Boolean ;
VAR
	x, y, n:		Integer ;
	update_display:	Boolean ;
	ref:			Integer ;
BEGIN

	update_display := True ;

	CASE  k  OF

	KB_Left:   BEGIN
				 x := WhereX ;
				 IF  x > HLP_text_x THEN
				 	 GotoXY (x - 1, WhereY) ;
				 check_highlight ;
				 update_display := False ;
			   END ;

	KB_Right:  BEGIN
				 x := WhereX ;
				 IF  x < HLP_text_x + HLP_text_width - 1 THEN
				 	 GotoXY (x + 1, WhereY) ;
				 check_highlight ;
				 update_display := False ;
			   END ;

	KB_Up:	   BEGIN
				 y := WhereY ;
				 IF  y > HLP_text_y THEN
				   BEGIN
				 	 GotoXY (WhereX, y -1) ;
				 	 check_highlight ;
					 update_display := False ;
				   END
				 ELSE
				   	 IF  top_line > 1  THEN
				   	 	 Dec (top_line) ;
			   END ;

	KB_PgUp:   IF  top_line - HLP_text_depth >= 1   THEN
				 Dec (top_line, HLP_text_depth) 
			   ELSE
			   	 top_line := 1 ;

	KB_Down:   BEGIN
				 y := WhereY ;
				 IF  y < HLP_text_y + HLP_text_depth - 1 THEN
				   BEGIN
				 	 GotoXY (WhereX, y + 1) ;
					 check_highlight ;
					 update_display := False ;
				   END 
				 ELSE
				     IF  top_line + HLP_text_depth <= total_lines  THEN
				 	    Inc (top_line) ;
			   END ;

	KB_PgDown: IF  top_line + HLP_text_depth <= total_lines  THEN
				 BEGIN
				   n := total_lines - (top_line + HLP_text_depth) + 1 ;
				   IF  n > HLP_text_depth  THEN
				   	 n := HLP_text_depth ;
				   Inc (top_line, n) ;
			     END 
			   ELSE
				   update_display := False ;

	KB_Tab:	   BEGIN
				update_display := False ;
				IF  ref_index > 0  THEN
				  BEGIN
				  	x := WhereX ;
				  	y := WhereY ;
				  	ref := 0 ;
				  	n := 1 ;
				  	WHILE  (n <= ref_index)  AND  (ref = 0)  DO 
				      BEGIN
				   		IF  y = reference_table[n].y  THEN
						  BEGIN
					  	  	IF  x < reference_table[n].x_start  THEN
   						  		ref := n ;
						  END
						ELSE
				   		  IF  y < reference_table[n].y  THEN
   						  	  ref := n ;
				  		Inc (n) ;
					  END ;
					IF  ref = 0	 THEN
						ref := 1 ;
					GotoXY (reference_table[ref].x_start,
							reference_table[ref].y) ;
				   END ;
				 check_highlight ;
			     END ;

	KB_ShiftTab: 
			  BEGIN
				update_display := False ;
				IF  ref_index > 0  THEN
				  BEGIN
				  	x := WhereX ;
				  	y := WhereY ;
				  	ref := 0 ;
				  	n := ref_index ;
				  	WHILE  (n > 0)  AND  (ref = 0)  DO 
				      BEGIN
				   		IF  y = reference_table[n].y  THEN
						  BEGIN
					  	  	IF  x > reference_table[n].x_end  THEN
   						  		ref := n ;
						  END
						ELSE
				   		  IF  y > reference_table[n].y  THEN
   						  	  ref := n ;
				  		Dec (n) ;
					  END ;
					IF  ref = 0	 THEN
						ref := ref_index ;
					GotoXY (reference_table[ref].x_start,
							reference_table[ref].y) ;
				   END ;
				 check_highlight ;
			     END ;


	KB_Enter:  BEGIN
				   check_reference (WhereY, WhereX) ;
				   update_display := False ;
			   END ;

	KB_NN,
	KB_AltN:
			   IF  section + 1 <= area_table^[area].area_last  THEN
				 BEGIN
			    	next_section := section + 1 ;			 	
					top_line := 1 ;
					cursor_x := 0 ;
					cursor_y := 0 ;
				 END ;

	KB_PP,
	KB_AltP:	
			   IF  section - 1 >= area_table^[area].area_first  THEN
				 BEGIN
				    next_section := section - 1 ;			 	
					top_line := 1 ;
					cursor_x := 0 ;
					cursor_y := 0 ;
				 END ;

	KB_BB,
	KB_AltB:   IF  last_list_entry > 0  THEN
				 BEGIN
				    next_section := sections_list [last_list_entry].number ;
				    top_line := sections_list [last_list_entry].top_line ;
					cursor_x :=	sections_list [last_list_entry].x ;
					cursor_y :=	sections_list [last_list_entry].y ;
					Dec (last_list_entry) ;
				 END ;
	KB_CC,
	KB_AltC,
	KB_Esc:
			   next_section := -9999 ;

	ELSE
	END	;

	interpret_key := update_display ;

END ;



BEGIN

	section  := s.number ;
	top_line :=	s.top_line ;
	cursor_x :=	s.x ;
	cursor_y :=	s.y ;

    line := Addr(section_pointer^[section]^[section_lines^[0]]) ;
	n :=  Pos('\section ', line^) ;
	IF  n <> 0  THEN
	  BEGIN
	  	w := line^ ;
		Delete (w, n, Length('\section ')) ;
		MS_Hide ;
		TextColor(HLP_title_line_FG) ;
		TextBackground(HLP_title_line_BG) ;
		display_line (w, HLP_title_line_y, HLP_title_line_x, True, False) ;
		MS_Show ;
		n := 1 ;
	  END ;

	total_lines := current_section_lines - n ;

	next_section := section ;

	IF  cursor_x <> 0  THEN
		GotoXY (cursor_x, cursor_y) 
	ELSE
		GotoXY (HLP_text_line_x, HLP_text_y) ;

	update_display := True ;

	REPEAT

		IF  update_display  THEN
		  BEGIN
			display_text (section, top_line, total_lines) ;
			IF  cursor_x <> 0  THEN
	    		check_highlight ;
			update_display := False ;
		  END ;

		REPEAT
		UNTIL  KeyPressed OR MS_LeftPressed OR MS_RightPressed ;

		IF  NOT  KeyPressed  THEN
		  BEGIN
		  	x := MS_WhereX ;
			y := MS_WhereY ;
		  END ;
		IF  MS_RightPressed OR
			( MS_LeftPressed AND
		      ( (x < HLP_first_column)  OR  (x > HLP_last_column)  OR
	    		(y < HLP_first_line)    OR  (y > HLP_last_line) ) )  THEN
	  	  BEGIN
			REPEAT UNTIL NOT MS_RightDown ;
			next_section := -9999 ;
	  	  END 
		ELSE
	  	  IF  MS_LeftPressed  THEN
			BEGIN
		  	  n := 1 ;
		  	  WHILE  (n <= n_zones) AND ( (x < mouse_zones[n].min_x)  OR
		   		 	 (x > mouse_zones[n].max_x)  OR
		   		 	 (y < mouse_zones[n].min_y)  OR
		   		 	 (y > mouse_zones[n].max_y)  )  DO
	  				Inc (n) ;
		  	  IF  n <= n_zones  THEN
		  		BEGIN
			  	  IF  mouse_zones[n].mode  = 0  THEN
			  	  	update_display := interpret_key (mouse_zones[n].key)
			  	  ELSE
					BEGIN
					  REPEAT UNTIL NOT MS_LeftDown ;
					  check_reference (y, x) ;
					END ;
				END 
		  	  ELSE
				BEGIN
			  	  WHILE  MS_LeftDown  AND  WND_check_scroll_bar (scroll_bar, move)  DO
					BEGIN
				  	  IF  move = -2  THEN        { actually +1 }
						BEGIN
					      IF  top_line + HLP_text_depth <= total_lines  THEN
							BEGIN
				 	      	  Inc (top_line) ;
						  	  MS_Delay ;
						  	  display_text (section, top_line, total_lines) ;
							END ;
						END
				  	  ELSE
						IF  move = -1  THEN
					  	  BEGIN
						   	IF  top_line > 1  THEN
							  BEGIN
				   	 	 		Dec (top_line) ;
						    	MS_Delay ;
							    display_text (section, top_line, total_lines) ;
							  END ;
					  	  END
						ELSE
					  	  BEGIN
							top_line := move ;
							MS_Delay ;
						    display_text (section, top_line, total_lines) ;
					  	  END ;
					 END ;
				END	;
			  IF MS_LeftPressed THEN
				REPEAT UNTIL NOT MS_LeftDown ;
			END
	  	  ELSE
			update_display := interpret_key (KB_read) ;

	UNTIL	next_section <> section ;

	s.number := next_section ;
	s.top_line := top_line ;
	s.x := cursor_x ;
	s.y := cursor_y ;
END ;


{ ************************************************************************
  *																		 *
  *			   Display an area, starting at specified section 			 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	display_area (area_number: Integer; section_number: Integer) ;
VAR
	s_info:	Section_info ;
BEGIN
	s_info.number := section_number ;
	s_info.top_line := 1 ;
	s_info.x := 0 ;
	s_info.y := 0 ;
	WHILE  s_info.number <> -9999  DO
	  BEGIN
		section_pointer^[s_info.number] := get_section (s_info.number) ;
		display_section (s_info, area_number) ;
	  END ;
END ;


{ ************************************************************************
  *																		 *
  *				 Put action buttons on screen							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  show_action_buttons (y, x:  Integer) ;

VAR
	i, j, yy:  Integer ;
BEGIN
	
	WND_open_window (HLP_buttons_y, HLP_button_cancel_x,
					 HLP_buttons_y,	HLP_button_cancel_x + 9,
					 HLP_button_FG, HLP_button_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [HLP_button_high_FG + 1] + 'C\xancel',
					 no_save) ;

	
	WND_open_window (HLP_buttons_y, HLP_button_backtrack_x, 
					 HLP_buttons_y, HLP_button_backtrack_x + 12, 
					 HLP_button_FG, HLP_button_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [HLP_button_high_FG + 1] + 'B\xacktrack',
					 no_save) ;


	WND_open_window (HLP_buttons_y, HLP_button_previous_x,
					 HLP_buttons_y, HLP_button_previous_x + 11,
					 HLP_button_FG, HLP_button_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [HLP_button_high_FG + 1] + 'P\xrevious',
					 no_save) ;


	WND_open_window (HLP_buttons_y, HLP_button_next_x,
					 HLP_buttons_y, HLP_button_next_x + 7,
					 HLP_button_FG, HLP_button_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [HLP_button_high_FG + 1] + 'N\xext',
					 no_save) ;

END ;


{ ************************************************************************
  *																		 *
  *					  Initialize the screen								 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	init_screen (title: String) ;

BEGIN
	MS_Hide ;
	full_screen_save := 1 ;
	WND_open_window (HLP_window_y,
					 HLP_window_x,
					 HLP_window_y + HLP_window_depth - 1,
					 HLP_window_x + HLP_window_width - 1,
					 HLP_window_FG, HLP_window_BG,
					 HLP_window_shadow, HLP_window_frame,
					 title,
					 full_screen_save) ;

	no_save := 0 ;
	WND_open_window (HLP_text_window_y,
					 HLP_text_window_x,
					 HLP_text_window_y + HLP_text_window_depth - 1,
					 HLP_text_window_x + HLP_text_window_width - 1,
					 HLP_window_FG, HLP_window_BG,
					 WND_no_shadow, HLP_text_window_frame,
					 '',
					 no_save) ;

	WND_open_window (HLP_text_y,
					 HLP_text_x,
					 HLP_text_y + HLP_text_depth - 1,
					 HLP_text_x + HLP_text_width - 1,
					 HLP_text_FG, HLP_text_BG,
					 WND_no_shadow, WND_no_frame,
					 '',
					 no_save) ;


	show_action_buttons (22, 10) ;
	MS_Show;
END ;


{ ************************************************************************
  *																		 *
  *				   Compute the screen coordinates						 *
  *																		 *
  ************************************************************************ }

FUNCTION  set_coordinates: Boolean ;
VAR
	depth,
	h, w,
	width,
	h_slack_sides,
	h_slack_buttons_1,
	h_slack_buttons_2,
	v_slack_title: 	Integer ;

BEGIN

	depth  := HLP_last_line - HLP_first_line + 1 ;
	width  := HLP_last_column - HLP_first_column + 1 ;

	IF  (depth < min_depth)  OR  (width < min_width) THEN
	  BEGIN
	  	set_coordinates := False ;
		Exit ;
	  END ;

	HLP_window_y 	   := HLP_first_line ;
	HLP_window_x 	   := HLP_first_column ;
	HLP_window_depth   := depth ; 
	HLP_text_window_depth := 20 ; 

	v_slack_title		  := 1 ;	
	h_slack_sides		  := 4 ;
	h_slack_buttons_1	  := 26 ;
	h_slack_buttons_2	  := 12 ;

	w := 80 ;
	IF  width < w  THEN  
	  BEGIN
		h_slack_sides := 2 ;
		Dec (w,2)
	  END ;
	IF  width < w  THEN  
	  BEGIN
		h_slack_sides := 0 ;
		Dec (w,2)
	  END ;

	HLP_window_width := w ; 
	HLP_text_window_width := HLP_window_width - h_slack_sides ; 

	WHILE  width < w  DO  
	  BEGIN
		Dec (HLP_text_window_width, 2) ;
		Dec (HLP_window_width, 2) ;
	  	Dec(h_slack_buttons_1) ;
	  	Dec(h_slack_buttons_2) ;
		Dec (w,2)
	  END ;


	h := 25 ;
	IF  depth < h  THEN
	  BEGIN
		Dec (v_slack_title) ;
		Dec (h) ;
	  END ;
	WHILE  depth < h  DO
	  BEGIN
		Dec (HLP_text_window_depth) ;
		Dec (h) ;
	  END ;

	HLP_title_line_y	  := HLP_window_y + v_slack_title + 1 ;
	HLP_title_line_x	  := HLP_window_x + h_slack_sides DIV 2 + 2 ;

	HLP_text_window_y 	  := HLP_window_y + v_slack_title + 2 ;	 
	HLP_text_window_x 	  := HLP_window_x + h_slack_sides DIV 2 ;

	HLP_buttons_y		  	:= HLP_text_window_y + HLP_text_window_depth ;
	HLP_button_backtrack_x	:= HLP_text_window_x + 2 ;
	HLP_button_previous_x	:= HLP_button_backtrack_x + h_slack_buttons_1 ;
	HLP_button_next_x		:= HLP_button_previous_x + 16 ;
	HLP_button_cancel_x		:= HLP_button_next_x + 8 + h_slack_buttons_2 ;

	HLP_text_y					:= HLP_text_window_y + 1 ;
	HLP_text_x					:= HLP_text_window_x + 2 ;
	HLP_text_width				:= HLP_text_window_width - 4 ;	
	HLP_text_depth				:= HLP_text_window_depth - 2 ;	
	HLP_text_line_x 			:= HLP_text_x + 1 ;
	HLP_text_line_width 		:= HLP_text_width - 3 ;

	HLP_popup_y					:= HLP_text_window_y + 3 ;
	HLP_popup_x					:= HLP_text_line_x ;
	HLP_popup_width				:= HLP_text_width - 3 ;
	HLP_popup_text_y			:= HLP_popup_y  ;
	HLP_popup_text_x			:= HLP_popup_x + 1 ;
	HLP_popup_text_width		:= HLP_popup_width - 2 ;
	HLP_popup_text_line_x		:= HLP_popup_text_x + 2;
	HLP_popup_text_line_y		:= HLP_popup_text_y + 1;
	HLP_popup_text_line_width 	:= HLP_popup_text_width -2 ;

END ;


{ ************************************************************************
  *																		 *
  *				   Initialize the help environment						 *
  *																		 *
  ************************************************************************ }
			
FUNCTION   HLP_init (file_name: String;	 p: HLP_palette) : Integer ;
VAR
	i:			Integer ;
	address:	Longint ;
	
BEGIN

	HLP_init := 0 ;
	IF  NOT set_coordinates  THEN
		Exit ;

	Assign (help_file, file_name) ;	     	   
{$i-}  
	Reset (help_file, 1) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

{$i-}							   
	Seek (help_file, FileSize(help_file) - SizeOf(descriptor)) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

{$i-}							   
	BlockRead (help_file, des, SizeOf(descriptor)) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

{$i-}							   
	Seek (help_file, des.start_address) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

	area_count 			:= des.area_count ;
	section_count 		:= des.section_count ;
	max_section_lines 	:= des.max_section_lines ;

	GetMem (area_table, Sizeof(area) * area_count) ;

{$i-}							   
	BlockRead (help_file, area_table^, Sizeof(area) * area_count) ;
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

{$i-}							   
	Seek (help_file, des.start_address + Sizeof(area) * area_count) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

	GetMem (section_table, Sizeof(section) * (section_count+1)) ;

{$i-}							   
	BlockRead (help_file, section_table^, Sizeof(section) * (section_count+1)) ;
{$i+}							   
    IF  IOResult <> 0  THEN				   
		Exit ;

	GetMem (section_pointer, Sizeof(String_Ptr) * (section_count+1)) ;
	FOR  i := 0  TO  section_count  DO
		section_pointer^[i] := Nil ;

	GetMem (section_lines, Sizeof(Integer) * max_section_lines) ;

	palette := p ;

	scroll_bar := WND_new_scroll_bar  (HLP_text_scroll_bar_BG,			
									   HLP_text_scroll_bar_button_BG,
									   HLP_text_scroll_bar_button_FG,
									   HLP_text_depth ) ;

	n_zones := 0 ;

	add_mouse_zone (HLP_buttons_y, HLP_button_cancel_x,
					HLP_buttons_y, HLP_button_cancel_x + 9,
					0, KB_CC) ;
	
	add_mouse_zone (HLP_buttons_y, HLP_button_backtrack_x, 
					HLP_buttons_y, HLP_button_backtrack_x + 12,
					0, KB_BB) ;

	add_mouse_zone (HLP_buttons_y, HLP_button_previous_x,
					HLP_buttons_y, HLP_button_previous_x + 11,
					0, KB_PP) ;

	add_mouse_zone (HLP_buttons_y, HLP_button_next_x,
					HLP_buttons_y, HLP_button_next_x + 7,
					0, KB_NN) ;

	add_mouse_zone (HLP_text_y,
					HLP_text_line_x,
					HLP_text_y + HLP_text_depth - 1,
					HLP_text_line_x + HLP_text_line_width - 1,
					1, 0) ;

    HLP_init := area_count ;

	MS_Show ;
{$IFDEF DEBUG}
	dump_init ;
{$ENDIF}

END ;


{ ************************************************************************
  *																		 *
  *					   		Display an area	  							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE   HLP_display_area (area_number: Integer; title: String) ;
BEGIN
	IF  (area_number < 1)  OR (area_number > area_count) THEN
		Exit ;
	WND_Save_Cursor (True) ;
	init_screen (title) ;
	last_list_entry := 0 ;
	display_area (area_number, area_table^[area_number].area_first) ;
	WND_Close_window (full_screen_save) ;
	WND_Restore_Cursor ;
END ;


{ ************************************************************************
  *																		 *
  *			    Display a section, given the section identifier			 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  HLP_display_section (section_id: Integer; title: String) ;
VAR
	a, s:	Integer ;
BEGIN
	s := 0 ;
	WHILE  (s <= section_count)  AND 
		   (section_id <> section_table^[s].section_identifier)  DO
		Inc (s) ;

	IF  s <= section_count  THEN 
	  BEGIN
		a := 1 ;
		WHILE  (a <= area_count)  AND 
			   ( (s < area_table^[a].area_first)  OR
				 (s > area_table^[a].area_last) ) DO
			Inc (a) ;

		IF  a <= area_count  THEN 
		  BEGIN
			WND_Save_Cursor (True) ;
			init_screen (title) ;
			last_list_entry := 0 ;
			display_area (a, s) ;
			WND_Close_window (full_screen_save) ;
			WND_Restore_Cursor ;
		  END ;
	  END ;
END ;


{ ************************************************************************
  *																		 *
  *				 Free all dynamically allocated memory					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE   HLP_free (area_number: Integer) ;
VAR
	i:	Integer ;
BEGIN
	IF  (area_number > 0)  AND (area_number <= area_count)	THEN
	  BEGIN
	  	FOR  i := area_table^[area_number].area_first  
	 		 TO   area_table^[area_number].area_last  DO
 			 	free_section (i) ;

	  END ;
END ;


{ ************************************************************************
  *																		 *
  *				 U N I T   I N I T I A L I Z A T I O N					 *
  *																		 *
  ************************************************************************ }
			
BEGIN
	no_save := 0 ;

	IF LastMode = 7  THEN
	  video_buffer := $b000
	ELSE
	  video_buffer := $b800 ;

	HLP_tab_spacing := 8 ;

	HLP_window_BG	:= LightGray ;
	HLP_window_FG	:= Black ;

	HLP_text_BG 	:= Black ;
	HLP_text_FG 	:= Green ;

	HLP_reference_FG :=	Brown ;
	HLP_reference_BG := Black ;

	HLP_popup_reference_FG := Magenta ;
	HLP_popup_reference_BG := Black ;

	HLP_highlighted_reference_FG :=	LightGray ;
	HLP_highlighted_reference_BG := Cyan ;

 	HLP_popup_FG	:= LightGray ;
	HLP_popup_BG	:= Brown ;

	HLP_popup_text_BG := Brown ;
	HLP_popup_text_FG := LightGray ;

	HLP_popup_title_line_FG := White ;
	HLP_popup_title_line_BG := Brown ;

	HLP_text_scroll_bar_BG			:=	Brown ;
	HLP_text_scroll_bar_button_BG	:=	Green ;
	HLP_text_scroll_bar_button_FG	:=	White ;

	HLP_button_BG 	   := Red ;
	HLP_button_FG 	   := LightGray ;
	HLP_button_high_FG := White ;

	HLP_title_line_FG  := HLP_button_BG ;
	HLP_title_line_BG  := HLP_window_BG	;

	HLP_first_line		    := 1 ;	
	HLP_last_line		    := 25 ;
	HLP_first_column		:= 1 ;
	HLP_last_column		    := 80 ;

	HLP_window_shadow     := WND_no_shadow ;	
	HLP_window_frame      := WND_no_frame ;	
	HLP_text_window_frame := WND_single_frame ;	

END.







