(*****************************************************************************

  TextMem
    Version 1.0

  Purpose:
    This unit is designed to incorporate all the necessary routines to handle
    text structures ( such as ASCII text files ) in memory.
    This unit is not intended as a stand alone used.

  Features:
    Can handles very large files ( memory permitting ) of up to 2 billion
      lines of up to 60 thousand characters each. ( Line size is determined
      by TextLine )
    Extensive error checking.
    Stores data is a dynamic memory structure to conserve resources.
    Memory storage provides extremely quick processing.
    Blank lines are stored as nul pointers which take up no extra memory.

  Limitations:
    This unit should not be overlaid.
    The size of the text file is severely limited by available memory.

  CopyRight 1994, All rights reserved.
    By Paul Renaud

  Compiler:
    Turbo Pascal versions 5.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit TextMem;

  Interface

    Uses
      TextLine;

    Const
      { Defines a segment of the file for buffer use only. }
      File_Length = 1000;

    Type
      { Defines the line data pointer. }
      Text_Info_Type = Pointer;
      { Defines the file segment array structure. }
      Text_Array_Type = Packed array [ 1 .. File_Length ] of Text_Info_Type;
      { Defines a pointer to the data storage. }
      Text_Buffer_Pointer_Type = ^Text_Buffer_Type;
      { Defines the file segment storage record. }
      Text_Buffer_Type = Record
                           Line_Start: LongInt;
                           Data: Text_Array_Type;
                           Next,
                           Previous: Text_Buffer_Pointer_Type;
                         End;
      { Defines text data storage structure. }
      Text_Type = Record
                    Where: Text_Buffer_Pointer_Type;
                    FileSize: LongInt;
                  End;

    Var
      { This data structure provides working space for some of the routines. }
      Work_Space: Line_Pointer_Type;
      { Warning!!!!! }
      { These procedure must be initialize, or system failure will result. }
      Read_Status: Procedure( Where: LongInt );
      Write_Status: Procedure( Where, Limit: LongInt );
      { This is a substitute for the getting routine. }
      Peek_Text_Line: Procedure( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );

(***********************************************************

  Function: Allocate text.

    This function allocates the text data structure to hold
    the text.  If it fails, it returns false.

***********************************************************)

    Function Allocate_Text( Var Text: Text_Type ): Boolean;

(***********************************************************

  Function: Check allocation.

    This function returns true if the text data structure
    has been properly allocated.

***********************************************************)

    Function Check_Allocation( Var Text: Text_Type ): Boolean;

(***********************************************************

  Procedure: Get text line.

    This procedure returns the text line of Text, specified
    by Row in Line.

***********************************************************)

    Procedure Get_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );

(***********************************************************

  Function: Put text line.

    This procedure puts the given text line in Text, at the
    specified row.  It returns false if there isn't any
    more memory to do so.

***********************************************************)

    Function Put_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type; Reduce: Boolean ): Boolean;

(***********************************************************

  Function: Read text.

    This function reads the text out of a given file and
    stores it in the Text data structure.  It returns false
    if an input error occurs or when there is no more memory
    to store the structure.

***********************************************************)

    Function Read_Text( Var InFile: Text; Var Text: Text_Type ): Boolean;

(***********************************************************

  Function: Write text.

    This function attempts to store the text structure in
    the given text file.  It returns false if an output
    error occurs.

***********************************************************)

    Function Write_Text( Var OutFile: File; Var Text: Text_Type ): Boolean;

(***********************************************************

  Procedure: Dispose text.

    This procedure disposes of the text within the Text
    storage structure.  The structure is still left
    allocated so that data can be stored in it.

***********************************************************)

    Procedure Dispose_Text( Var Text: Text_Type );

(***********************************************************

  Function: Delete text line.

    This function attempts to delete the given row from the
    text storage structure.  All succeeding rows are moved
    forward after the row is deallocated.

***********************************************************)

    Function Delete_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;

(***********************************************************

  Function: Split text.

    This function attempts to splits the text structure at
    the given location.  This means that the text line is
    split in half and the remaining text on the line is
    converted into a new line.  If the function fails, it
    returns false.  Data loss is highly unlikely.

***********************************************************)

    Function Split_Text( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type; Reduce: Boolean ): Boolean;

(***********************************************************

  Function: Remove return.

    This function attempts combine the line specified by
    Data with the succeeding line.  It undoes what split
    text does.  If the function fails, it returns false.
    Data loss is highly unlikely.

***********************************************************)

    Function Remove_Return( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type ): Boolean;

(***********************************************************

  Function: Get text size.

    This function returns the amount of lines in the text
    data structure.

***********************************************************)

    Function Get_Text_Size( Var Text: Text_Type ): LongInt;

(***********************************************************

  Function: Swap lines.

    This function attempts to swap the data of the two given
    text rows.  It fails only under very unusual conditions.

***********************************************************)

    Function Swap_Lines( Var Text: Text_Type; Row1, Row2: LongInt ): Boolean;

(***********************************************************

  Procedure: Initialize Text.

    This procedure initializes the text variable for storing
    the data.  The text should always be initialized only
    once in the program, before it is used.

***********************************************************)

    Procedure Initialize_Text( Var Text: Text_Type );

(***********************************************************

  Procedure: Erase text.

    This procedure disposes of the text variable entirely
    and should only be after Dispose_Text.

***********************************************************)

    Procedure Erase_Text( Var Text: Text_Type );

{----------------------------------------------------------------------------}

  Implementation

    {$DEFINE Quick} { Select alternate code for faster processing speed. }

{----------------------------------------------------------------------------}

(*************************************************

  Function: Valid data.
    This function returns true if the line is
    allocated.

*************************************************)

    Function Valid_Data( Info: Text_Info_Type ): Boolean;
      Begin
         Valid_Data := ( Info <> Nil );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Deallocate data pointer.
    This procedure releases the memory used by
    the line back to the free list and sets the
    line to a blank line.

*************************************************)

    Procedure Deallocate_Data_Pointer( Var Info: Text_Info_Type );
      Var
        Information: Line_Pointer_Type absolute Info;
      Begin
        If ( Info <> Nil )
          then
            Begin
              FreeMem( Info, Calculate_Size( Information^ ) );
              Info := Nil;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize node.
    This procedure initialized a new part of the
    text storage structure to blank lines.

*************************************************)

    Procedure Initialize_Node( Var Buffer: Text_Buffer_Type );
      Var
        Count: Word;
      Begin
        For Count := 1 to File_Length do
          Buffer.Data[ Count ] := Nil;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal initialize.
    This function allocates a portion of the text
    storage structure and sets it up for later
    use.

*************************************************)

    Function Internal_Initialize( Var Buffer: Text_Buffer_Pointer_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        New( Buffer );
        Okay := ( Buffer <> Nil );
        If Okay
          then
            Begin
              Initialize_Node( Buffer^ );
              Buffer^.Line_Start := 1;
              Buffer^.Previous := Nil;
              Buffer^.Next := Nil;
            End;
        Internal_Initialize := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal dispose.
    This procedure deallocates all the lines in
    a portion of the text structure.

*************************************************)

    Procedure Internal_Dispose( Var Buffer: Text_Buffer_Type );
      Var
        Count: Word;
      Begin
        For Count := 1 to File_Length do
          If Valid_Data( Buffer.Data[ Count ] )
            then
              Deallocate_Data_Pointer( Buffer.Data[ Count ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal dispose text.
    This procedure disposes of the entire text
    structure, except the allocated block.

*************************************************)

    Procedure Internal_Dispose_Text( Var Buffer: Text_Buffer_Pointer_Type );
      Begin
        While ( Buffer^.Next <> Nil ) do
          Buffer := Buffer^.Next;
        While ( Buffer^.Previous <> Nil ) do
          Begin
            Internal_Dispose( Buffer^ );
            Buffer := Buffer^.Previous;
            Dispose( Buffer^.Next );
          End;
        Buffer^.Next := Nil;
        Internal_Dispose( Buffer^ );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand text.
    This function expands the text structure so
    that more lines can be added to it when
    necessary.

*************************************************)

    Function Expand_Text( Var Buffer: Text_Type ): Boolean;
      Begin
        While ( Buffer.Where^.Next <> Nil ) do
          Buffer.Where := Buffer.Where^.Next;
        If ( Buffer.Where^.Next = Nil )
          then
            Begin
              Expand_Text := Internal_Initialize( Buffer.Where^.Next );
              Buffer.Where^.Next^.Previous := Buffer.Where;
              Buffer.Where^.Next^.Line_Start := ( Buffer.Where^.Line_Start + File_Length );
            End
          else
            RunError;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Page row.
    This function tries to shift the text
    structure block to the appropriate one
    containing the given line.  Then Row is
    altered to reflect the correct line of the
    current block.

*************************************************)

    Function Page_Row( Var Text: Text_Type; Var Row: LongInt ): Boolean;
      Begin
        Page_Row := True;
        If ( Row < Text.Where^.Line_Start )
          then
            If ( Text.Where^.Line_Start > 1 )
              then
                While ( Text.Where^.Line_Start > Row ) do
                  If ( Text.Where^.Previous <> Nil )
                    then
                      Text.Where := Text.Where^.Previous
                    else
                      RunError
              else
                RunError
          else
            While ( Row >= ( Text.Where^.Line_Start + File_Length ) ) do
              If ( Text.Where^.Next = Nil )
                then
                  Begin
                    Page_Row := Expand_Text( Text );
                    Text.Where := Text.Where^.Next;
                  End
                else
                  Text.Where := Text.Where^.Next;
        Row := Succ( Row - Text.Where^.Line_Start );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get text size.
    As previously defined.

*************************************************)

    Function Get_Text_Size( Var Text: Text_Type ): LongInt;
      Begin
        Get_Text_Size := Text.FileSize;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get text line.
    As previously defined.

*************************************************)

    Procedure Get_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );
      Begin
        If ( Row > 0 )
          then
            If Page_Row( Text, Row ) and Valid_Data( Text.Where^.Data[ Row ] )
              then
                Retrieve_Line( Text.Where^.Data[ Row ], Line )
              else
                Line.Size := 0
          else
            Line.Size := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Swap lines.
    As previously defined.

*************************************************)

    Function Swap_Lines( Var Text: Text_Type; Row1, Row2: LongInt ): Boolean;
      Var
        Temporary: LongInt;
        Hold1,
        Hold2: Text_Info_Type;
      Begin
        Swap_Lines := True;
        Temporary := Row1;
        If Page_Row( Text, Temporary )
          then
            Begin
              Hold1 := Text.Where^.Data[ Temporary ];
              If Page_Row( Text, Row2 )
                then
                  Begin
                    Hold2 := Text.Where^.Data[ Row2 ];
                    Text.Where^.Data[ Row2 ] := Hold1;
                    If Page_Row( Text, Row1 )
                      then
                        Text.Where^.Data[ Row1 ] := Hold2
                      else
                        Swap_Lines := False;
                  End
                else
                  Swap_Lines := False;
            End
          else
            Swap_Lines := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove.
    This procedure removes the given line from
    the structure.

*************************************************)

    Procedure Remove( Var Where: Text_Info_Type );
      Begin
        If Valid_Data( Where )
          then
            Deallocate_Data_Pointer( Where );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Put text line.
    As previously defined.

*************************************************)

    Function Put_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type; Reduce: Boolean ): Boolean;
      Var
        Okay: Boolean;
        Hold: LongInt;
        Hold_Length: Word;
        Temporary_Pointer: Pointer;
      Begin
        Hold := Row;
        Okay := Page_Row( Text, Row );
        Hold_Length := Line.Size;
        If Reduce
          then
            Reduce_Line( Line );
        If ( ( Line.Size > 0 ) and Okay )
          then
            Begin
              If Valid_Data( Text.Where^.Data[ Row ] )
                then
                  Deallocate_Data_Pointer( Text.Where^.Data[ Row ] );
              Temporary_Pointer := Allocate_Line( Line );
              If ( Temporary_Pointer <> Nil )
                then
                  Text.Where^.Data[ Row ] := Temporary_Pointer
                else
                  Okay := False;
            End
          else
            Remove( Text.Where^.Data[ Row ] );
        If ( ( Hold > Text.FileSize ) and Okay )
          then
            Text.FileSize := Hold;
        Line.Size := Hold_Length;
        Put_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Read text.
    As previously defined.

*************************************************)

    Function Read_Text( Var InFile: Text; Var Text: Text_Type ): Boolean;
      Var
        Okay: Boolean;
        Counter: LongInt;
      Begin
        Okay := True;
        If ( Text.Where = Nil )
          then
            Okay := Allocate_Text( Text );
        Counter := 0;
        While ( ( not EOF( InFile ) ) and Okay ) do
          Begin
            Inc( Counter );
            Okay := Read_Line( InFile, Work_Space^ );
            If Okay
              then
                Begin
                  Okay := Put_Text_Line( Text, Counter, Work_Space^, True );
                  If Okay
                    then
                      Read_Status( Counter );
                End;
          End;
        Read_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Write text.
    As previously defined.

*************************************************)

    Function Write_Text( Var OutFile: File; Var Text: Text_Type ): Boolean;
      Var
        Okay: Boolean;
        Counter: LongInt;
      Begin
        Okay := True;
        For Counter := 1 to Text.FileSize do
          If Okay
            then
              Begin
                Get_Text_Line( Text, Counter, Work_Space^ );
                Okay := Write_Line_Fast( OutFile, Work_Space^ );
                If ( Okay and Odd( Counter ) )
                  then
                    Write_Status( Counter, Text.FileSize );
              End;
        Write_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move back.
    This procedure moves the lines back in a text
    block so that a new one can be added.

*************************************************)

    Procedure Move_Back( Var Buffer: Text_Array_Type; Start, Finish: Word );
     {$IFDEF Quick}
      Begin
        If ( Start < Finish )
          then
            Move( Buffer[ Start ], Buffer[ Succ( Start ) ], ( Finish - Start ) * SizeOf( Text_Info_Type ) );
      End;
     {$ELSE}
      Var
        Count: Word;
      Begin
        For Count := Finish downto Succ( Start ) do
          Buffer[ Count ] := Buffer[ Pred( Count ) ];
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move forward.
    This procedure moves the lines forward in a
    text block so that a one can be deleted.

*************************************************)

    Procedure Move_Forward( Var Buffer: Text_Array_Type; Start, Finish: Word );
     {$IFDEF Quick}
      Begin
        If ( Start < Finish )
          then
            Move( Buffer[ Succ( Start ) ], Buffer[ Start ], ( Finish - Start ) * SizeOf( Text_Info_Type ) );
      End;
     {$ELSE}
      Var
        Count: Word;
      Begin
        For Count := Start to Pred( Finish ) do
          Buffer[ Count ] := Buffer[ Succ( Count ) ];
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Insert text line.
    This function inserts a text line into the
    text structure at the given row.

*************************************************)

    Function Insert_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
        Point: LongInt;
      Begin
        Point := Succ( Text.FileSize );
        Okay := Page_Row( Text, Point );
        If Okay
          then
            Begin
              While ( Row < Text.Where^.Line_Start ) do
                Begin
                  Move_Back( Text.Where^.Data, 1, File_Length );
                  Text.Where := Text.Where^.Previous;
                  Text.Where^.Next^.Data[ 1 ] := Text.Where^.Data[ File_Length ];
                End;
              Okay := Page_Row( Text, Row );
              If Okay
                then
                  Begin
                    Move_Back( Text.Where^.Data, Row, File_Length );
                    Text.Where^.Data[ Row ] := Nil;
                    Inc( Text.FileSize );
                  End;
            End;
        Insert_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal delete text line.
    This function deletes the given text line from
    the text structure.

*************************************************)

    Function Internal_Delete_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := Page_Row( Text, Row );
        If Okay
          then
            Begin
              Move_Forward( Text.Where^.Data, Row, File_Length );
              While ( Text.Where^.Next <> Nil ) do
                Begin
                  Text.Where^.Data[ File_Length ] := Text.Where^.Next^.Data[ 1 ];
                  Text.Where := Text.Where^.Next;
                  Move_Forward( Text.Where^.Data, 1, File_Length );
                End;
              Text.Where^.Data[ File_Length ] := Nil;
              Dec( Text.FileSize );
              If ( Text.FileSize < Text.Where^.Line_Start )
                then
                  If ( Text.Where^.Previous <> Nil )
                    then
                      Begin
                        Internal_Dispose( Text.Where^ );
                        Text.Where := Text.Where^.Previous;
                        Dispose( Text.Where^.Next );
                        Text.Where^.Next := Nil;
                      End;
            End;
        Internal_Delete_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Delete text line.
    As previously defined.

*************************************************)

    Function Delete_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
        New_Row: LongInt;
      Begin
        New_Row := Row;
        Okay := Page_Row( Text, New_Row );
        If Okay
          then
            Begin
              Remove( Text.Where^.Data[ New_Row ] );
              Okay := Internal_Delete_Text_Line( Text, Row );
            End;
        Delete_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Split text.
    As previously defined.

*************************************************)

    Function Split_Text( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type; Reduce: Boolean ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := Insert_Text_Line( Text, Succ( Data.Row ) );
        If Okay
          then
            Begin
              Append_Line( Buffer, ' ' );
              Delete_Line( Buffer, 1, Pred( Data.Column ) );
              Okay := Put_Text_Line( Text, Succ( Data.Row ), Buffer, True );
              If Okay
                then
                  Begin
                    Get_Text_Line( Text, Data.Row, Buffer );
                    Buffer.Size := Pred( Data.Column );
                    Okay := Put_Text_Line( Text, Data.Row, Buffer, Reduce );
                  End;
            End;
        Split_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Remove return.
    As previously defined.

*************************************************)

    Function Remove_Return( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Get_Text_Line( Text, Succ( Data.Row ), Work_Space^ );
        Combine_Lines( Buffer, Work_Space^ );
        Okay := Put_Text_Line( Text, Data.Row, Buffer, True );
        If Okay
          then
            Okay := Delete_Text_Line( Text, Succ( Data.Row ) );
        Remove_Return := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate text.
    As previously defined.

*************************************************)

    Function Allocate_Text( Var Text: Text_Type ): Boolean;
      Begin
        Allocate_Text := Internal_Initialize( Text.Where );
        Text.FileSize := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dispose text.
    As previously defined.

*************************************************)

    Procedure Dispose_Text( Var Text: Text_Type );
      Begin
        Internal_Dispose_Text( Text.Where );
        Text.FileSize := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize text.
    As previously defined.

*************************************************)

    Procedure Initialize_Text( Var Text: Text_Type );
      Begin
        Text.Where := Nil;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Erase text.
    As previously defined.

*************************************************)

    Procedure Erase_Text( Var Text: Text_Type );
      Var
        Temporary_Pointer: Text_Buffer_Pointer_Type;
      Begin
        While( Text.Where <> Nil ) do
          Begin
            Temporary_Pointer := Text.Where;
            Text.Where := Text.Where^.Next;
            Dispose( Temporary_Pointer );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Check allocation.
    As previously defined.

*************************************************)

    Function Check_Allocation( Var Text: Text_Type ): Boolean;
      Begin
        If ( Text.Where = Nil )
          then
            Check_Allocation := Allocate_Text( Text )
          else
            Check_Allocation := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Allocate the work space.

*************************************************)

  Begin
    New( Work_Space );
    If ( Work_Space = Nil )
      then
        RunError( 203 );
    Peek_Text_Line := Get_Text_Line;
  End.

