(*****************************************************************************

  Fractions
    Version 1.0

    This unit holds procedures designed to perform calculations using
    fractions.

    Purpose:
      To make it easy to use fractions in programs.  Using fractions can
      result in possible higher precision in calculations and eliminate
      the accuracy errors that can result with real numbers.

    How it works:
      The operations are performed on the fractions just like in regular
      mathematics, then the results are reduced to lowest terms.

    Features:
      Uses long integers internally to enhance precision.

    Limitations:
      Results are limited to what can be represented by words.

    CopyRight 1989, All rights reserved.
      Paul R. Renaud

     Compilers:
       Turbo Pascal versions 4.0 to 6.0
       Speed Pascal/2 version 1.5

     Systems:
       MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Fraction;

  Interface

(***********************************************************

  Fractions are stored in long integer data forms but with
    certain restrictions.  The long integer forms are
    redefined in terms of two double byte integers or
    integers of word size.

***********************************************************)

    Type
      Number_Type = LongInt;

(***********************************************************

  Function: Equals.

    This function takes the numerator and denominator and
    assigns them together into the Number_Type data
    structure.

***********************************************************)

    Function Equals( Numerator, Denominator: Integer ): Number_Type;

(***********************************************************

  Function: Convert to string.

    This function takes a fraction type value and converts
    it into a string value for output.

***********************************************************)

    Function Convert_To_String( Number: Number_Type ): String;

(***********************************************************

  Function: Convert to real.

    This function takes a fraction type value and converts
    it into a real value for other uses.

***********************************************************)

    Function Convert_To_Real( Number: Number_Type ): Real;

(***********************************************************

  Procedure: Separate the value.

    This procedure takes a fraction type value and separates
    it for your own possible use.

***********************************************************)

    Procedure Separate( Number: Number_Type; Var Numerator, Denominator: Integer );

(***********************************************************

  Function: Add two fractional numbers.

    This function takes two fractional numbers and adds them
    together for you.

***********************************************************)

    Function Add( Number_1, Number_2: Number_Type ): Number_Type;

(***********************************************************

  Function: Subtract two fractional numbers.

    This function subtracts the second fractional number
    from the first.

***********************************************************)

    Function Subtract( Number_1, Number_2: Number_Type ): Number_Type;

(***********************************************************

  Function: Multiply two fractional numbers.

    This function takes two fractional numbers and
    multiplies them together.

***********************************************************)

    Function Multiply( Number_1, Number_2: Number_Type ): Number_Type;

(***********************************************************

  Function: Divide two fractional numbers.

    This function divides the second fractional number into
    the first.

***********************************************************)

    Function Divide( Number_1, Number_2: Number_Type ): Number_Type;

{----------------------------------------------------------------------------}

  Implementation

    Type
      { This type is used for internal operations to ensure accuracy. }
      Fraction_Type   = Record
                          Numerator,
                          Denominator: LongInt;
                        End;
      { This type is used to redefine the number type. }
      Rational_Number = Record
                          Numerator: Integer;
                          Denominator: Word;
                        End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Greatest common denominator.
    This function calculates the greatest common
    denominator for the two given numbers.

*************************************************)

    Function GCD( Number_1, Number_2: LongInt ): LongInt;
      Var
        Examine: LongInt;
      Begin;
        While ( Number_2 <> 0 ) do
          Begin;
            Examine := Number_1 Mod Number_2;
            Number_1 := Number_2;
            Number_2 := Examine;
          End;
        GCD := Number_1;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Lowest terms.
    This procedure reduces the given fraction to
    it's lowest terms.

*************************************************)

    Procedure Lowest_Terms( Var Number: Fraction_Type );
      Var
        Greatest_Common_Divisor: LongInt;
      Begin;
        If ( Number.Denominator < 0 )
          then
            Begin;
              Number.Denominator := -Number.Denominator;
              Number.Numerator := -Number.Numerator;
            End;
        Greatest_Common_Divisor := GCD( Number.Numerator, Number.Denominator );
        If ( ( Greatest_Common_Divisor <> 0 ) and ( Abs( Greatest_Common_Divisor ) <> 1 ) )
          then
            Begin;
              Number.Numerator := ( Number.Numerator div Greatest_Common_Divisor );
              Number.Denominator := ( Number.Denominator div Greatest_Common_Divisor );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Addition.
    This procedure adds the two given fractions
    and returns the result in Result.

*************************************************)

    Procedure Addition( Number_1, Number_2: Fraction_Type; Var Result: Fraction_Type );
      Var
        Number: Fraction_Type;
      Begin;
        Number.Numerator := ( ( Number_1.Numerator * Number_2.Denominator ) +
                              ( Number_2.Numerator * Number_1.Denominator ) );
        Number.Denominator := ( Number_1.Denominator * Number_2.Denominator );
        Lowest_Terms( Number );
        Result := Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Subtraction.
    This procedure subtracts the second fraction
    from the first and returns the result in
    Result.

*************************************************)

    Procedure Subtraction( Number_1, Number_2: Fraction_Type; Var Result: Fraction_Type );
      Var
        Number: Fraction_Type;
      Begin;
        Number.Numerator := ( ( Number_1.Numerator * Number_2.Denominator ) -
                              ( Number_2.Numerator * Number_1.Denominator ) );
        Number.Denominator := ( Number_1.Denominator * Number_2.Denominator );
        Lowest_Terms( Number );
        Result := Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Multiplication.
    This procedure multiplies the first fraction
    and the second and returns the result in
    Result.

*************************************************)

    Procedure Multiplication( Number_1, Number_2: Fraction_Type; Var Result: Fraction_Type );
      Var
        Number: Fraction_Type;
      Begin;
        Number.Numerator := ( Number_1.Numerator * Number_2.Numerator );
        Number.Denominator := ( Number_1.Denominator * Number_2.Denominator );
        Lowest_Terms( Number );
        Result := Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Division.
    This procedure divides the second fraction
    into the first and returns the result in
    Result.

*************************************************)

    Procedure Division( Number_1, Number_2: Fraction_Type; Var Result: Fraction_Type );
      Var
        Number: Fraction_Type;
      Begin;
        Number.Numerator := ( Number_1.Numerator * Number_2.Denominator );
        Number.Denominator := ( Number_1.Denominator * Number_2.Numerator );
        Lowest_Terms( Number );
        Result := Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add.
    As previously defined.

*************************************************)

    Function Add( Number_1, Number_2: Number_Type ): Number_Type;
      Var
        Num_1,
        Num_2,
        Num_3: Fraction_Type;
        R_Number: Rational_Number;
        T_Number: Number_Type absolute R_Number;
        R_Number_1: Rational_Number absolute Number_1;
        R_Number_2: Rational_Number absolute Number_2;
      Begin;
        Num_1.Denominator := R_Number_1.Denominator;
        Num_1.Numerator := R_Number_1.Numerator;
        Num_2.Denominator := R_Number_2.Denominator;
        Num_2.Numerator := R_Number_2.Numerator;
        Addition( Num_1, Num_2, Num_3 );
        R_Number.Denominator := Num_3.Denominator;
        R_Number.Numerator := Num_3.Numerator;
        Add := T_Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Subtract.
    As previously defined.

*************************************************)

    Function Subtract( Number_1, Number_2: Number_Type ): Number_Type;
      Var
        Num_1,
        Num_2,
        Num_3: Fraction_Type;
        R_Number: Rational_Number;
        T_Number: Number_Type absolute R_Number;
        R_Number_1: Rational_Number absolute Number_1;
        R_Number_2: Rational_Number absolute Number_2;
      Begin;
        Num_1.Denominator := R_Number_1.Denominator;
        Num_1.Numerator := R_Number_1.Numerator;
        Num_2.Denominator := R_Number_2.Denominator;
        Num_2.Numerator := R_Number_2.Numerator;
        Subtraction( Num_1, Num_2, Num_3 );
        R_Number.Denominator := Num_3.Denominator;
        R_Number.Numerator := Num_3.Numerator;
        Subtract := T_Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply.
    As previously defined.

*************************************************)

    Function Multiply( Number_1, Number_2: Number_Type ): Number_Type;
      Var
        Num_1,
        Num_2,
        Num_3: Fraction_Type;
        R_Number: Rational_Number;
        T_Number: Number_Type absolute R_Number;
        R_Number_1: Rational_Number absolute Number_1;
        R_Number_2: Rational_Number absolute Number_2;
      Begin;
        Num_1.Denominator := R_Number_1.Denominator;
        Num_1.Numerator := R_Number_1.Numerator;
        Num_2.Denominator := R_Number_2.Denominator;
        Num_2.Numerator := R_Number_2.Numerator;
        Multiplication( Num_1, Num_2, Num_3 );
        R_Number.Denominator := Num_3.Denominator;
        R_Number.Numerator := Num_3.Numerator;
        Multiply := T_Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Divide.
    As previously defined.

*************************************************)

    Function Divide( Number_1, Number_2: Number_Type ): Number_Type;
      Var
        Num_1,
        Num_2,
        Num_3: Fraction_Type;
        R_Number: Rational_Number;
        T_Number: Number_Type absolute R_Number;
        R_Number_1: Rational_Number absolute Number_1;
        R_Number_2: Rational_Number absolute Number_2;
      Begin;
        Num_1.Denominator := R_Number_1.Denominator;
        Num_1.Numerator := R_Number_1.Numerator;
        Num_2.Denominator := R_Number_2.Denominator;
        Num_2.Numerator := R_Number_2.Numerator;
        Division( Num_1, Num_2, Num_3 );
        R_Number.Denominator := Num_3.Denominator;
        R_Number.Numerator := Num_3.Numerator;
        Divide := T_Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Equals.
    As previously defined.

*************************************************)

    Function Equals( Numerator, Denominator: Integer ): Number_Type;
      Var
        Number: Fraction_Type;
        R_Number: Rational_Number;
        T_Number: Number_Type absolute R_Number;
      Begin;
        If ( Denominator = 0 )
          then
           {$IFNDEF VER40}
            RunError( 201 );
           {$ELSE}
            Halt;
           {$ENDIF}
        Number.Numerator := Numerator;
        Number.Denominator := Denominator;
        Lowest_Terms( Number );
        R_Number.Numerator := Number.Numerator;
        R_Number.Denominator := Number.Denominator;
        Equals := T_Number;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Convert to string.
    As previously defined.

*************************************************)

    Function Convert_To_String( Number: Number_Type ): String;
      Var
        Data_1,
        Data_2: String;
        R_Number: Rational_Number absolute Number;
      Begin;
        Str( R_Number.Numerator, Data_1 );
        Str( R_Number.Denominator, Data_2 );
        Convert_To_String := Data_1 + '/' + Data_2;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Convert to real.
    As previously defined.

*************************************************)

    Function Convert_To_Real( Number: Number_Type ): Real;
      Var
        R_Number: Rational_Number absolute Number;
      Begin;
        Convert_To_Real := ( R_Number.Numerator / R_Number.Denominator );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Separate.
    As previously defined.

*************************************************)

    Procedure Separate( Number: Number_Type; Var Numerator, Denominator: Integer );
      Var
        R_Number: Rational_Number absolute Number;
      Begin;
        Numerator := R_Number.Numerator;
        Denominator := R_Number.Denominator;
      End;

{-----------------------------------------------------------------------------}

  End.

