{ EFLIB | Extended Function Library (C) Johan Larsson, 1992 - 1997
          All rights reserved. E-mail to jola@ts.umu.se.

          EXAMPLE PROGRAM                  [x] Real mode
        | Sample/Classrg2.pas              [x] Protected mode

  This is a sample class register that maintains arbitrary tObject
  instances by encapsulating them in carrier instances. Carriers
  (tCarrier) can be connected to each other forming a doubly linked
  data structure.

  tClassReg is a sample component that shows you how to create
  class registers.

  EFLIB IS PROTECTED BY THE COPYRIGHT LAW AND MAY NOT BE MANIPULATED,
  DISTRIBUTED OR COPIED. THIS DEMONSTRATION PROGRAM MAY FREELY BE USED
  AND DISTRIBUTED.                                                        }


type pClassReg = ^tClassReg;
     tClassReg = object (tObject)

       { This is a class register, ie. a simple, un-organized database
         with tObject instances, all owned by this class. This is a
         very primitive ADT. Registers are intercepted together with
         their instances. }

       public

         {#Z Constructors and destructors }
         constructor Initialize;                                { Constructs an empty class register. }
         destructor  Intercept; virtual;                        { Intercepts a class register and all registered instances. }

         {#Z Miscellaneous methods }
         procedure   Put (Instance : pObject); virtual;         { Puts the specified instance into the register. }
         function    Get : pObject; virtual;                    { Gets an instance from the register. }
         function    GetClass (Class : pointer) : pObject;
                     virtual;                                   { Gets an instance derived from the class (TypeOf-address). }

         {#Z Methods for stream storage }
         constructor StreamLoad (Stream : pStream);             { Loads an instance from a stream. Abstract method. }
         procedure   StreamStore (Stream : pStream); virtual;   { Stores an instance to a stream. }

         {#Z Status methods }
         function    IsEmpty : boolean; virtual;                { Is the register empty? }
         function    IsInside (Class : pointer) : boolean;
                     virtual;                                   { Is the an instance derived from this class registered? }
         function    Instances : word; virtual;                 { Returns the number of registered instances. }

       private

         {#Z Fields }
         fContainers    : pCarrier;                             { The first carrier (followed by a doubly linked structure). }

     end;


{ *****************************************
  *               tClassReg               *
  *****************************************
  Derived from: tObject
  Parent for: None                          }

{ Constructs an empty class register. }
constructor tClassReg.Initialize;
begin
     Inherited Initialize;
     fContainers := NIL;
end;

{ Intercepts a class register and all registered instances. }
destructor tClassReg.Intercept;
begin
     Inherited Intercept;

     { Release all container instances. }
     if Assigned (fContainers) then fContainers^.First^.FreeAll;
     fContainers := NIL;
end;


{ Registers a new tObject instance - that is, attachs a carrier
  of the specified instance to the container structure. }
procedure tClassReg.Put (Instance : pObject);
var Carrier : pCarrier;
begin
     { Create a carrier for this instance. }
     Carrier := New ( pCarrier,
                      Initialize ( Instance, NIL ) );

     { Assert that the carrier actually was
       constructed. }
     Assert ( Assigned (Carrier), Error_Damaged );

     if not Assigned (fContainers)
        then fContainers := Carrier { First instance to be registered. }
        else with fContainers^ do begin

                  { Attach new carrier first in the
                    container structure. }
                  AttachBefore (Carrier);
                  fContainers := Carrier;

             end;
end;

{ Gets an instance from the register. }
function tClassReg.Get : pObject;
var Result : pObject;
begin
     { Take an element from the front carrier
       and assign new head. }

     if Assigned (fContainers) then

        with fContainers^ do begin

             { Assign new head if needed }
             fContainers := pCarrier(fContainers^.Successor);

             { Take the instance }
             Take (Result);
             Free; { Detach and release }

        end

     else Result := NIL; { Empty }

     Get := Result;
end;

{ Gets an instance derived from the class (TypeOf-address). }
function tClassReg.GetClass (Class : pointer) : pObject;
var Iterator : pCarrier; Result : pObject;
begin
     { Start searching at the first carrier. }
     Iterator := fContainers;

     while Assigned (Iterator) and not (Iterator^.IsAllocated and
           Iterator^.fInstance^.IsParent (Class)) do { Continue }
           Iterator := pCarrier(Iterator^.Successor);

     { Take instance if it was found, or return NIL. }
     if Assigned (Iterator) then

        with Iterator^ do begin

             { Assign new head if needed }
             if Iterator = fContainers
                then fContainers := pCarrier(Iterator^.Successor);

             { Take the instance }
             Take (Result);
             Free; { Detach and release }

        end

     else Result := NIL; { Not found }

     GetClass := Result; { Return instance or NIL }
end;


{ Loads the object from a stream. }
constructor tClassReg.StreamLoad (Stream : pStream);
var NumberOfInstances, Index : word; Instance : pObject;
begin
     { Asserts that the operation is permitted. }
     Assert ( Assigned (Stream), Error_Resource );
     Assert ( Stream^.IsAllocated, Error_Stream );

     Stream^.Read (NumberOfInstances, SizeOf(NumberOfInstances));

     for Index := 1 to NumberOfInstances do begin

         { Load instance and put it into
           the register. }
         Stream^.Load ( Instance );
         Put ( Instance );

     end;
end;

{ Stores this instance to a stream, ie. writes the element contents to
  the stream. }
procedure tClassReg.StreamStore (Stream : pStream);
var NumberOfInstances : word; Iterator : pCarrier;
begin
     { Asserts that the operation is permitted. }
     Assert ( Assigned (Stream), Error_Resource );
     Assert ( Stream^.IsAllocated, Error_Stream );

     NumberOfInstances := Instances; { Get size }
     Stream^.Write (NumberOfInstances, SizeOf(NumberOfInstances));

     Iterator := fContainers; { Start at the first carrier. }

     while Assigned (Iterator) do begin
           { Assert that this instance is valid }
           Assert ( Iterator^.IsAllocated, Error_Damaged );

           { Store it to the stream }
           Stream^.Store ( Iterator^.fInstance );
           Iterator := pCarrier(Iterator^.Successor);
     end;
end;


{ Is the register empty? }
function tClassReg.IsEmpty : boolean;
begin
     IsEmpty := not Assigned (fContainers);
end;

{ Is the an instance derived from this class registered? }
function tClassReg.IsInside (Class : pointer) : boolean;
var Iterator : pCarrier;
begin
     { Start searching at the first carrier. }
     Iterator := fContainers;

     while Assigned (Iterator) and not (Iterator^.IsAllocated and
           Iterator^.fInstance^.IsParent (Class)) do { Continue }
           Iterator := pCarrier(Iterator^.Successor);

     { If the iterator is still assigned, the loop terminated
       when a match was found. }
     IsInside := Assigned (Iterator);
end;

{ Returns the number of registered instances. }
function tClassReg.Instances : word;
var Iterator : pCarrier; Counter : word;
begin
     { Start counting at the first carrier. }
     Iterator := fContainers; Counter := 0;

     while Assigned (Iterator) do begin
           Inc (Counter); { Continue }
           Iterator := pCarrier(Iterator^.Successor);
     end;

     Instances := Counter; { Return results }
end;



var MyRegister : tClassReg;

begin
     with MyRegister do begin

          Initialize;r

          { Demonstration (does nothing, really). }

          Register ( New (pObject, Initialize) );
          Register ( New (pObject, Initialize)) );
          Register ( New (pStream, Initialize) );

          if IsInside (TypeOf(tStream))
             then WriteLn ('We have a tStream in our register!');

          Intercept;

     end;
end.