{ ---------------------------------------------------------------------------
  program TTY.PAS

  Sample terminal emulation using simple teletype protocol to be used with
  the unit COM.PAS for serial communnication.

  Features:

  - switching between COM1 and COM2
  - baudrates up to 115200 baud
  - RTS/CTS and XON/XOFF flow control
  - debug mode to display control characters

  Version 3.0 - May 1994

  Copyright 1994, Willem van Schaik - Oirschot - Netherlands

  --------------------------------------------------------------------------- }

  program Tty;

  uses Crt, Com;

  const
    Ascii : array [0..255] of string [5] =
      ('<NUL>','<SOH>','<STX>','<ETX>','<EOT>','<ENQ>','<ACK>','<BEL>',
       '<BS>','<HT>','<LF>','<VT>','<FF>','<CR>','<SO>','<SI>',
       '<DLE>','<DC1>','<DC2>','<DC3>','<DC4>','<NAK>','<SYN>','<ETB>',
       '<CAN>','<EM>','<SUB>','<ESC>','<FS>','<GS>','<RS>','<US>',
       ' ','!','"','#','$','%','&','''','(',')','*','+',',','-','.','/',
       '0','1','2','3','4','5','6','7','8','9',':',';','<','=','>','?',
       '@','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O',
       'P','Q','R','S','T','U','V','W','X','Y','Z','[','\',']','^','_',
       '`','a','b','c','d','e','f','g','h','i','j','k','l','m','n','o',
       'p','q','r','s','t','u','v','w','x','y','z','{','|','}','~','<DEL>',
       '<128>','<129>','<130>','<131>','<132>','<133>','<134>','<135>',
       '<136>','<137>','<138>','<139>','<140>','<141>','<142>','<143>',
       '<144>','<145>','<146>','<147>','<148>','<149>','<150>','<151>',
       '<152>','<153>','<154>','<155>','<156>','<157>','<158>','<159>',
       '<160>','<161>','<162>','<163>','<164>','<165>','<166>','<167>',
       '<168>','<169>','<170>','<171>','<172>','<173>','<174>','<175>',
       '<176>','<177>','<178>','<179>','<180>','<181>','<182>','<183>',
       '<184>','<185>','<186>','<187>','<188>','<189>','<190>','<191>',
       '<192>','<193>','<194>','<195>','<196>','<197>','<198>','<199>',
       '<200>','<201>','<202>','<203>','<204>','<205>','<206>','<207>',
       '<208>','<209>','<210>','<211>','<212>','<213>','<214>','<215>',
       '<216>','<217>','<218>','<219>','<220>','<221>','<222>','<223>',
       '<224>','<225>','<226>','<227>','<228>','<229>','<230>','<231>',
       '<232>','<233>','<234>','<235>','<236>','<237>','<238>','<239>',
       '<240>','<241>','<242>','<243>','<244>','<245>','<246>','<247>',
       '<248>','<249>','<250>','<251>','<252>','<253>','<254>','<255>');

  var
    TtyPort : PortType;
    TtyBaud : BaudType;
    TtyParity : ParityType;
    TtyLength : LengthType;
    TtyStop : StopType;
    TtyFlow : FlowType;

    ChCom, ChKey : char;
    DoDebug : boolean;
    GoExit : boolean;

{ ---------------------------------------------------------------------------
  TtyGetPars

  Procedure to handle alt-key combinations that are used to change the
  settings of the terminal emulation protocol.
  --------------------------------------------------------------------------- }

  procedure TtyGetPars (AltKey : char);

  var
    ParsInput : string[16];

  begin
    case AltKey of

      #120:  { alt-1 }
      begin
        if WhereX > 1 then Writeln;
        Writeln ('TTY:  port = COM1:');
        if TtyPort <> COM1 then
        begin
          ExitCom (TtyPort);
          TtyPort := COM1;
          InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow)
        end;
      end;

      #121:  { alt-2 }
      begin
        if WhereX > 1 then Writeln;
        Writeln ('TTY:  port = COM2:');
        if TtyPort <> COM2 then
        begin
          ExitCom (TtyPort);
          TtyPort := COM2;
          InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow)
        end;
      end;

      #48:  { alt-B }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  baudrate = ');
        Readln (ParsInput);
        if (ParsInput = '3') or (ParsInput = '300') then TtyBaud := B300
        else if (ParsInput = '6') or (ParsInput = '600') then TtyBaud := B600
        else if (ParsInput = '12') or (ParsInput = '1200') then TtyBaud := B1200
        else if (ParsInput = '24') or (ParsInput = '2400') then TtyBaud := B2400
        else if (ParsInput = '48') or (ParsInput = '4800') then TtyBaud := B4800
        else if (ParsInput = '96') or (ParsInput = '9600') then TtyBaud := B9600
        else if (ParsInput = '192') or (ParsInput = '19200') then TtyBaud := B19200
        else if (ParsInput = '384') or (ParsInput = '38400') then TtyBaud := B38400
        else
          Writeln ('      baudrate = 300,600,1200,2400,4800,9600,19200,38400');
        ExitCom (TtyPort);
        InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow);
      end;

      #38:  { alt-L }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  word length = ');
        Readln (ParsInput);
        case ParsInput[1] of
          '5': TtyLength := D5;
          '6': TtyLength := D6;
          '7': TtyLength := D7;
          '8': TtyLength := D8;
        else
          Writeln ('      word length = 5,6,7,8');
        end;
        ExitCom (TtyPort);
        InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow);
      end;

      #25:  { alt-P }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  parity bit = ');
        Readln (ParsInput);
        case ParsInput[1] of
          'n', 'N': TtyParity := None;
          'o', 'O': TtyParity := Odd;
          'e', 'E': TtyParity := Even;
          'm', 'O': TtyParity := Mark;
          's', 'O': TtyParity := Space;
        else
          Writeln ('      parity bit = none,odd,even,mark,space');
        end;
        ExitCom (TtyPort);
        InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow);
      end;

      #31:  { alt-S }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  stop bits = ');
        Readln (ParsInput);
        case ParsInput[1] of
          '1': TtyStop := S1;
          '2': TtyStop := S2;
        else
          Writeln ('      stop bits = 1,2');
        end;
        ExitCom (TtyPort);
        InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow);
      end;

      #33:  { alt-F }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  flow control = ');
        Readln (ParsInput);
        case ParsInput[1] of
          'n', 'N': TtyFlow := No;
          'r', 'R': TtyFlow := RtsCts;
          'x', 'X': TtyFlow := XonXoff;
        else
          Writeln ('      flow control = no,rts/cts,xon/xoff');
        end;
        ExitCom (TtyPort);
        InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow);
      end;

      #23:  { alt-I }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  port = COM', ord(TtyPort)+1, ':      ');
        case TtyBaud of
          B110: Write ('baudrate = 110          ');
          B150: Write ('baudrate = 150          ');
          B300: Write ('baudrate = 300          ');
          B600: Write ('baudrate = 600          ');
          B1200: Write ('baudrate = 1200         ');
          B2400: Write ('baudrate = 2400         ');
          B4800: Write ('baudrate = 4800         ');
          B9600: Write ('baudrate = 9600         ');
          B19200: Write ('baudrate = 19200        ');
          B38400: Write ('baudrate = 38400        ');
          B57600: Write ('baudrate = 57600        ');
          B115200: Write ('baudrate = 115200       ');
        end;
        case TtyParity of
          None: Writeln ('parity bit = none');
          Odd: Writeln ('parity bit = odd');
          Even: Writeln ('parity bit = even');
          Mark: Writeln ('parity bit = mark');
          Space: Writeln ('parity bit = space');
        end;
        case TtyFlow of
          No: Write ('      flow = no         ');
          RtsCts: Write ('      flow = rts/cts    ');
          XonXoff: Write ('      flow = xon/xoff   ');
        end;
        Write ('word length = ', ord(TtyLength)+5, '         ');
        Writeln ('stop bits = ', ord(TtyStop)+1);
      end;

      #35:  { alt-H }
      begin
        if WhereX > 1 then Writeln;
        Write ('TTY:  alt-1 - COM1      ');
        Write ('alt-B - baudrate        ');
        Write ('alt-I - info');
        Writeln;
        Write ('      alt-2 - COM2      ');
        Write ('alt-L - word length     ');
        Write ('alt-H - help');
        Writeln;
        Write ('      alt-F - flow      ');
        Write ('alt-P - parity bit');
        Writeln;
        Write ('      alt-D - debug     ');
        Write ('alt-S - stop bits       ');
        Write ('alt-X - exit');
        Writeln;
      end;

      #32:  { alt-D }
      begin
        DoDebug := not DoDebug;
        if WhereX > 1 then Writeln;
        if DoDebug then
          Writeln ('TTY:  debug = on')
        else
          Writeln ('TTY:  debug = off');
      end;

      #45:  { alt-X }
      begin
        if WhereX > 1 then Writeln;
        Writeln ('TTY:  exit');
        GoExit := true;
      end;

    end;  { case ChKey }
  end;  { procedure TtyGetPars }

{ ---------------------------------------------------------------------------
  main program

  In a while-loop that can be ended by pressing alt-X, constantly the com-
  buffer is checked with ComAvailable and the keyboard buffer is checked with
  KeyPressed. Any byte received from the COM port is displayed on screen and
  any key read is send to the COM port with WriteCom.

  The program may be started with a parameter to select port COM2 (allowed
  formats COM2, com2 or just 2).
  --------------------------------------------------------------------------- }

  begin
    TtyPort := COM1;
    TtyBaud := B9600;
    TtyParity := None;
    TtyLength := D8;
    TtyStop := S1;
    TtyFlow := No;

    if (ParamCount = 1) then
      if ((ParamStr(1) = '2') or
          (ParamStr(1) = 'com2') or
          (ParamStr(1) = 'com2:') or
          (ParamStr(1) = 'COM2') or
          (ParamStr(1) = 'COM2:')) then
        TtyPort := COM2;

    InitCom (TtyPort, TtyBaud, TtyParity, TtyLength, TtyStop, TtyFlow);
    TextColor (LightGray);
    Writeln;
    TtyGetPars (#35); 				{ display help }

    DoDebug := false;
    GoExit := false;
    repeat
    begin
      if ComReceived (TtyPort) then
      begin
        ChCom := ReadCom (TtyPort);
        if not DoDebug then			{ terminal mode }
        begin
          if (ChCom = ^M) then			{ <CR> }
            GotoXY (1, WhereY)
          else
          if (ChCom = ^J) or (ChCom = ^L) then	{ <LF> or <FF> }
            writeln
          else if (ChCom = ^H) or (ord(ChCom) = 127) then
          begin					{ <BS> or <DEL> }
  	  if (WhereX > 1) then
              GotoXY (WhereX - 1, WhereY)
          end
          else if (ChCom = ^I) then		{ <HT> (tab) }
          begin
  	  if (WhereX <= 72) then
            begin
              Write ('       ');
              GotoXY ((WhereX div 8) * 8 + 1, WhereY)
            end
          end
          else if (ChCom >= ' ') and (ChCom <= '~') then
            write (ChCom);
        end
        else { debug mode }
          write (Ascii[ord(ChCom)]);
      end
      else
        if KeyPressed then
        begin
          ChKey := ReadKey;
          if (ChKey <> #0) then
          begin
            if ComAllowed (TtyPort) then
              WriteCom (TtyPort, ChKey);
  	end
          else
          begin
            ChKey := ReadKey;
            TtyGetPars (ChKey);
          end;
        end;
      Delay (1); { this delay is there to give schedulers like OS/2 a chance }
    end  { repeat }
    until GoExit;

    ExitCom (TtyPort);
  end.

{ ---------------------------------------------------------------------------
  end of TTY.PAS
  --------------------------------------------------------------------------- }
