procedure COLORMOD (Shade: real; Systemm: integer; Color: integer;
  var Pcolor, Fmod: integer);
{ Produce a "plotting color" and "mod number", given the requested
  color and shade.  The mod number is used to produce a simple dither
  pattern.  Pcolormod works as follows:

  -- For a system with less than 3 colors, it halts the program with
     an error message because this procedure should never have been
     called (dithering should be used).
  -- For a system with 3-6 colors, Pcolormod ignores the requested
     color and generates a plotting color and mod number that work
     in conjunction to produce 7 unique shades of grey on a monochrome
     monitor.
  -- For a system with 7-14 colors, Pcolormod does basically the same
     thing as with 3-6 colors, but produces 11 unique shades of grey.
     This is based on intensity tests on the Sanyo MBC-555.
  -- For a system with 15-63 colors, it is suggested the user come
     up with a good intensity table for monochrome monitors that will
     produce many unique shades of grey without using the mod number.
  -- For a system with 64 colors or more, an example is given of a
     fictitious system that allows the user to request colors in the
     range of 1-8, but returns a plotting color in the range 1-Ncolors
     by assuming the system has (Ncolors/8) intensities available for
     each of the 8 colors.  If the ordering is not right, you will have
     to form your own lookup table.

**************************************************************************
NOTE that Colormod is only called if the dithering option is not selected.
**************************************************************************
}
var Tcolor: integer;        { temp for Color }
    Nshade: integer;        { #available shades of each color }
    Pshade: integer;        { shade # within a color }
begin
  if (Ncolors < 3) then begin
    writeln ('Error: COLORMOD called with Ncolors = ',Ncolors);
    writeln ('Internal error; get help!');
    halt;
  end else if (Ncolors < 7) then begin
    { IBM Color Graphics Adapter in lo-res 4-color mode }
    { This routine uses 3 colors and some simple dithering to create
      7 shades of grey on a monochrome screen }
    if (Shade = 0.0) then begin
      Pcolor := 0;
      Fmod := 1;
    end else case trunc (Shade * 6.0) of
      0: begin
        Pcolor := 1;
        Fmod := 3;
      end;
      1: begin
        Pcolor := 1;
        Fmod := 2;
      end;
      2: begin
        Pcolor := 1;
        Fmod := 1;
      end;
      3: begin
        Pcolor := 3;
        Fmod := 3;
      end;
      4: begin
        Pcolor := 3;
        Fmod := 2;
      end
      else begin
        Pcolor := 3;
        Fmod := 1;
      end;
    end; { case }
{ end else if (Ncolors < 15) then begin }
{ The above line should be substituted for the line below if someone
  comes up with a good color intensity table for a 15-color system.
  Note that if you are adding code for a 15-color table, Fmod can
  always be 1; dithering is not necessary if you have that many
  intensities available.
}
  end else if (Ncolors < 64) then begin
    { This routine uses 7 colors and some simple dithering to create
      11 shades of grey on a monochrome screen }
    if (Shade = 0.0) then begin
      Pcolor := 0;
      Fmod := 1;
    end else case trunc (Shade * 10.0) of
      0: begin
        Pcolor := 1;
        Fmod := 2;
      end;
      1: begin
        Pcolor := 4;
        Fmod := 2;
      end;
      2: begin
        Pcolor := 2;
        Fmod := 2;
      end;
      3: begin
        Pcolor := 2;
        Fmod := 1;
      end;
      4: begin
        Pcolor := 5;
        Fmod := 1;
      end;
      5: begin
        Pcolor := 7;
        Fmod := 2;
      end;
      6: begin
        Pcolor := 6;
        Fmod := 1;
      end;
      7: begin
        Pcolor := 7;
        Fmod := -3;
      end;
      8: begin
        Pcolor := 7;
        Fmod := -4;
      end
      else begin
        Pcolor := 7;
        Fmod := 1;
      end;
    end; { case }
  end else begin
    Fmod := 1;
    { A fictitious system of Ncolors colors: }
    if (Color > 8) then
      Tcolor := 8
    else
      Tcolor := Color;
    Nshade := Ncolors div 8 - 1;
    Pshade := round (Shade * Nshade);
    if (Pshade < 0) then
      Pshade := 0
    else if (Pshade > Nshade) then
      Pshade := Nshade;
    Pcolor := (Tcolor-1) * (Nshade+1) + Pshade;
  end; { if Ncolors }
end; { procedure COLORMOD }
