;;******************************************************************************
;;                         point.inc      point.inc
;;******************************************************************************
;;
;;  Copyright (C) 1989 Vance Morrison
;;
;;
;; Permission to view, compile, and modify for LOCAL (intra-organization) 
;; USE ONLY is hereby granted, provided that this copyright and permission 
;; notice appear on all copies.  Any other use by permission only.
;;
;; Vance Morrison makes no representations about the suitability 
;; of this software for any purpose.  It is provided "as is" without expressed 
;; or implied warranty.  See the copywrite notice file for complete details.
;;
;;*****************************************************************************
;;
;;  point.asm contains the DL_IP interface for the point to point line 
;;  protocols.  This interface expects a serial IF interface and simply 
;;  converts it to a DL_IP interface (which really is quite easy).
;;
;;   PP_DECLARE name, serial_if, task
;;   PP_DEFINE name, ip_address, ip_mask
;;   PP_DL_IP_R_READ name, code_label
;;   PP_DL_IP_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES name, ok
;;   PP_DL_IP_RETURN name
;;   PP_DL_IP_W_ACCESS_in_CX_out_DI_ES_const_CX_BP name, fail
;;   PP_DL_IP_W_WRITE_in_AX_CX_const_BP name, broadcast
;;   PP_DL_IP_IS_BROADCAST_in_BX_ES_const_AX_BX_CX_DX_BP_DI_ES name
;;   PP_DL_IP_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES name
;;
;;  Variables Provided by this module (READ ONLY!!!!)
;;
;;      pp_&name&_declared        1 if this object has been declared
;;      dl_ip_&name&_ip             the IP address
;;      dl_ip_&name&_mask           the network mask
;;      dl_ip_&name&_net            the network (IP addr bitwize AND ip_mask)
;;      dl_ip_&name&_broad          the network broadcast address
;;      dl_ip_&name&_flags          A word exclusively for IP use
;;      dl_ip_&name&_metric         The interface metric
;;      dl_ip_&name&_mtu            the Maximum transmission unit (packet size)
;;
;;*****************************************************************************
 

;;*****************************************************************************
;; PP_DECLARE name, serial_if, task
;;    PP_DECLARE declares the serial DL_IP object
;;
PP_DECLARE MACRO name, serial_if, task
    .errb <serial_if>
    .errb <task>
 
    pp_&name&_declared = 1
    pp_&name&_if = serial_if
    pp_&name&_task = task

    dl_ip_&name&_mtu       = if_&serial_if&_mtu

    .DATA
    global dl_ip_&name&_ip:dword
    global dl_ip_&name&_mask:dword
    global dl_ip_&name&_net:dword
    global dl_ip_&name&_broad:dword
    global dl_ip_&name&_flags:word
    global dl_ip_&name&_metric:word

    global pp_&name&_read:word

    .CODE
    global pp_&name&_continue:near
    global pp_&name&_real_define:near
ENDM
 

;;*****************************************************************************
;;   PP_DEFINE name, ip_address, ip_mask
;;      PP_DEFINE defines any memory and runs any initialization code.
;;
PP_DEFINE MACRO name, ip_address, ip_mask
 
ifdef pp_&name&_declared
    mov AX, word ptr ip_address                 ;; copy over params
    mov BX, 0FFFFH                              ;; Force our assumtion
    mov word ptr dl_ip_&name&_ip, AX
    mov word ptr dl_ip_&name&_net, AX
    mov word ptr dl_ip_&name&_broad, AX
    mov word ptr dl_ip_&name&_mask, BX
 
    mov AX, word ptr ip_address+2
    mov BX, word ptr ip_mask+2
    mov word ptr dl_ip_&name&_ip+2, AX
    mov word ptr dl_ip_&name&_mask+2, BX
    and AX, BX
    mov word ptr dl_ip_&name&_net+2, AX
    not BX
    or AX, BX
    mov word ptr dl_ip_&name&_broad+2, AX
 
    call pp_&name&_real_define
endif
ENDM
 
PP_REAL_DEFINE MACRO name
    local start, around

ifdef pp_&name&_declared
    .DATA
    dl_ip_&name&_ip         DD ?
    dl_ip_&name&_mask       DD ?
    dl_ip_&name&_net        DD ?
    dl_ip_&name&_broad      DD ?
    dl_ip_&name&_flags      DW ?
    dl_ip_&name&_metric     DW ?

    pp_&name&_read dw ?                             ;; The code to call

    .CODE
    jmp around
        start:
        PP_TASK name
            ;; this does NOT fall through
    around:

    pp_&name&_real_define:
    mov pp_&name&_read, offset pp_&name&_continue
    TASK_DEFINE %pp_&name&_task, start
    RET
endif
ENDM



;;******************************************************************************
;;   DL_IP_R_READ name, code_label
;;       DL_IP_R_READ declares that the code starting at 'code_label'
;;       should be called whenever a IP packet is read.  BX:ES is initilized
;;       to the begining of the IP packet before 'code_label' is called
;;       The code at 'code_label' should call PP_DL_IP_R_RETURN when it
;;       is done processing the packet.
;;       This procedure can only be called once per 'name'
;;
PP_DL_IP_R_READ MACRO name, code_label
    .errb <code_label>
 
    mov word ptr pp_&name&_read, offset code_label
ENDM
 
;;******************************************************************************
;;   DL_IP_R_CONT_in_BX_CX_ES name, ok
;;       DL_IP_R_CONT determines if the packet returned by R_READ in BX:ES
;;       of length CX is continuous.  If it is it jumps to 'ok' otherwise
;;       it just returns
;;
PP_DL_IP_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES MACRO name, ok
    .errb <ok>
 
    jmp ok                      ;; it is always continuous
ENDM
 
;;******************************************************************************
;;   DL_IP_R_RETURN name
;;       DL_IP_R_RETURN should be executed by the READ routine to signal
;;       that it is done processing the packet.
;;
PP_DL_IP_R_RETURN MACRO name
    local done
    .errb <name>
 
    jmp pp_&name&_continue
ENDM
 

;;******************************************************************************
;;   DL_IP_IS_BROADCAST_in_BX_ES name
;;      DL_IP_IS_BROADCAST_in_BX_ES determines if the packet pointed to
;;      by BX:ES is a broadcast and sets the zero flag if it is NOT a
;;      broadcast
;;
PP_DL_IP_IS_BROADCAST_in_BX_ES_const_AX_BX_CX_DX_BP_DI_ES MACRO name
    .errb <name>
    
        ;; if the IP destination is on the local network AND it is
        ;; not specifically for me, then I will consider it a broadcast.

        ;; this fakery is needed, because SLIP in particular does not
        ;; have true DL addressing, it will read in ANY packet off the
        ;; serial line, whether it is for 'me' (at a DL level) or not
    mov SI, word ptr [BX+ip_dst+2]          ;; is it on my subnet
    and SI, word ptr dl_ip_&name&_mask+2
    cmp SI, word ptr dl_ip_&name&_net+2
    jnz not_broad                           
    mov SI, word ptr [BX+ip_dst]        
    cmp SI, word ptr dl_ip_&name&_net
    jnz not_broad
        mov SI, word ptr [BX+ip_dst+2]      ;; and not me
        cmp SI, word ptr dl_ip_&name&_ip+2
        jnz done
    not_broad:
        cmp AX, AX
    done:

ENDM
 

;;******************************************************************************
;;   DL_IP_W_ACCESS returns a pointer to an output buffer for a IP (network)
;;   packet.  The buffer is at least min(CX, dl_ip_mtu) bytes long.  The
;;   pointer is returned in DI.  If there is no buffer available
;;   this routine jumps to 'no_buffer'
;;
PP_DL_IP_W_ACCESS_in_CX_out_DI_ES_const_CX_BP MACRO name, no_buffer
    local fail, got_buffer
    .errb <name>
    .errb <no_buffer>
 
    IF_W_ACCESS_in_CX_out_DI_ES_const_BX_CX_BP %pp_&name&_if, no_buffer
ENDM
 

;;******************************************************************************
;; DL_IP_W_WRITE_in_AX_CX name, broadcast
;;    DL_IP_W_WRITE actually signals the link layer to write a packet to the
;;    network.  The packet is assumed to be in the buffer returned by
;;    DL_IP_W_ACCESS.  CX is the length of the packet to send.   AX holds the
;;    last two bytes of the IP address to send the packet to.  (notice we are
;;    assuming a host portion of less than 16 bits).  if 'broadcast' is not
;;    blank, then the packet is written to the broadcast address.  AX is
;;    ignored in this case
;;
PP_DL_IP_W_WRITE_in_AX_CX_const_BP MACRO name, broadcast 
    local done, bad 
    .errb <name> 

        ;; just send it out, we don't care about the address
    IF_W_WRITE_in_CX_const_BX_BP_ES %pp_&name&_if 
ENDM
 

;;******************************************************************************
;;   DL_IP_COPY_in_CX_SI_DI_ES_out_SI_DI name
;;      DL_IP_COPY_in_CX_SI_DI_ES copys a packet from the input buffer (pointed
;;      to by SI and the segment register given if IF_DECLARE) to an output
;;      buffer (pointed to by DI and dest_reg) of length CX.  It assumes the
;;      output buffer is contiguous.  (and the caller shouldn't care if the
;;      input buffer is contiguous)  COPY updates the pointers SI and DI
;;      to the end of the packet, and COPY could be called again if CX is not
;;      the total packet length (Note that CX MUST be even if you care about
;;      SI, and DI being updated properly)
;;
PP_DL_IP_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES MACRO name
    .errb <name>
 
    inc CX
    shr CX, 1
    rep
    movsw
ENDM
 

;;******************************************************************************
;;   PP_TASK is the read task that reads the next packet from the interface
;;   and dispatches it to the next higher protocol layer
;;
PP_TASK MACRO name
    local done
    .errb <name>
 
    IF_R_ACCESS_out_BX_CX_ES %pp_&name&_if, done
    jmp word ptr pp_&name&_read
 
    pp_&name&_continue:
        IF_R_FREE_const_BX_CX_BP_SI_DI_ES %pp_&name&_if

    done:
    TASK_RETURN %pp_&name&_task
ENDM

