/*
 * MultiMail offline mail reader
 * Interface, ShadowedWin, ListWindow

 Copyright (c) 1996 Kolossvary Tamas <thomas@tvnet.hu>
 Copyright (c) 1997 John Zero <john@graphisoft.hu>
 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "interfac.h"
#include "mysystem.h"

extern mmail		mm;
extern Welcome		welcome;
extern PacketListWindow	packets;
extern AreaListWindow	areas;
extern LetterListWindow	letters;
extern LetterWindow	letterwindow;
extern HelpWindow	helpwindow;
extern AddressBook	addresses;
extern file_list	*packetList;
extern letter_list	*letterList;
extern LittleAreaListWindow *littleareas;

ShadowedWin::ShadowedWin(int y1, int x1, int y2, int x2, chtype backg, 
			int border)
{
#ifdef USE_SHADOWS
	int i, sx, sy;
	chtype *right, *lower;
# ifndef NCURSES_VERSION
#  ifdef __PDCURSES__
	WINDOW *newscr = curscr;
#  else
	WINDOW *newscr = stdscr;
#  endif
# endif
	int limit = 2 + border;
	sx = x1;
	sy = y1;
	while ((sx + x2) > (COLS - limit))
		sx--;
	while ((sy + y2) > (LINES - limit))
		sy--;

	right = new chtype[(sy - 1) << 1];
	lower = new chtype[sx + 1];
	for (i = 0; i < (sy - 1); i++) {
		right[i << 1] = (mvwinch(newscr, (y2 + i + 1),
			(x2 + sx)) & (A_CHARTEXT | A_ALTCHARSET));
		right[(i << 1) + 1] = (mvwinch(newscr, (y2 + i + 1),
			(x2 + sx + 1)) & (A_CHARTEXT | A_ALTCHARSET));
	}
	mvwinchnstr(newscr, (y2 + sy), (x2 + 2), lower, sx);
	shadow = newwin(sy, sx, y2 + 1, x2 + 2);
	leaveok(shadow, TRUE);
	for (i = 0; i < (sy - 1); i++) {
		mvwaddch(shadow, i, sx - 2, (right[i << 1] |
			C_SHADOW | A_BOLD));
		mvwaddch(shadow, i, sx - 1, (right[(i << 1) + 1] |
			C_SHADOW | A_BOLD));
	}
	for (i = 0; i < sx; i++)
		mvwaddch(shadow, sy - 1, i, (lower[i] & (A_CHARTEXT |
			A_ALTCHARSET)) | C_SHADOW | A_BOLD);
	wnoutrefresh(shadow);
#endif
	win = newwin(y1, x1, y2, x2);
	wbkgdset(win, backg | ' ');
	werase(win);
	wbkgdset(win, ' ');
	wattrset(win, backg);
	box(win, 0, 0);
	keypad(win, TRUE);
	leaveok(win, TRUE);

#ifdef USE_SHADOWS
	delete lower;
	delete right;
#endif
}

ShadowedWin::~ShadowedWin()
{
	delwin(win);
#ifdef USE_SHADOWS
	delwin(shadow);
#endif
}

void ShadowedWin::touch()
{
#ifdef USE_SHADOWS
	touchwin(shadow);
	wnoutrefresh(shadow);
#endif
	touchwin(win);
	wnoutrefresh(win);
}

int ShadowedWin::getstring(int y, int x, char *string, int n, int bg_color, 
		int fg_color)
{
	int i, j, end, c;
	char *tmp = new char[n];

	cropesp(string);

	wattrset(win, fg_color);
	for (i = 0; i < n; i++) {
		mvwaddch(win, y, x + i, ACS_BOARD);
		tmp[i] = '\0';
	}
	mvwaddstr(win, y, x, string);

	leaveok(win, FALSE);
	keypad(win, TRUE);
	nodelay(win, FALSE);

	wrefresh(win);		//to have cursor

	i = end = 0;
	while (!end) {
		do
			c = wgetch(win);
		while (c == ERR);
		switch (c) {
		case '\r':
		case '\n':
			end = 2;
			break;
		case '\033':
			end = 1;
			break;
		case KEY_UP:
			end = 3;
			break;
		case KEY_DOWN:
			end = 4;
			break;
		case KEY_LEFT:
			if (!tmp[0]) {
				strcpy(tmp, string);
				i = strlen(string);
			}
			if (i > 0)
				i--;
			break;
		case KEY_RIGHT:
			if ((i < n - 1) && tmp[i])
				i++;
			break;
		case 127:
		case KEY_DC:	// Delete key 
			strncpy(&tmp[i], &tmp[i + 1], n - i - 1);
			tmp[n - 1] = '\0';
			break;
		case 8:	// Ctrl-H
		case KEY_BACKSPACE:
			if (i > 0) {
				strncpy(&tmp[i - 1], &tmp[i], n - i - 1);
				tmp[n - 1] = '\0';
				i--;
			}
			break;
		default:
			for (j = n - 1; j > i; j--)
				tmp[j] = tmp[j - 1];
			tmp[i] = c;
			if (i < n - 1)
				i++;
		}
		for (j = 0; j < n; j++)
			mvwaddch(win, y, x + j, (tmp[j] ? tmp[j] : ACS_BOARD));
		wmove(win, y, x + i);
		wrefresh(win);
	}

	if (tmp[0])
		strcpy(string, tmp);

	wattrset(win, bg_color);

	j = strlen(string);
	for (i = 0; i < n; i++)
		mvwaddch(win, y, x + i, ((i < j) ? string[i] : ' '));

	wrefresh(win);
	leaveok(win, TRUE);

	delete tmp;

	return end - 1;
}

InfoWin::InfoWin(int y1, int x1, int y2, int x2, chtype backg, int border,
	int bottx, int topx) : ShadowedWin(y1, x1, y2, x2, backg, border)
{
	info = newwin(y1 - bottx, x1 - 2, y2 + topx, x2 + 1);
	wbkgdset(info, backg | ' ');
	werase(info);
	wbkgdset(info, ' ');
	keypad(info, TRUE);
	leaveok(info, TRUE);
}

InfoWin::~InfoWin()
{
	delwin(info);
}

void InfoWin::touch()
{
	ShadowedWin::touch();
	touchwin(info);
	wnoutrefresh(info);
}

ListWindow::ListWindow()
{
	position = active = 0;
	oldPos = oldActive = -100;
}

void ListWindow::checkPos(int limit)
{
	if (active < 0)
		active = 0;
	else
		if (active >= limit)
			active = limit - 1;

	if (active < position)
		position = active;
	else
		if (active - position >= list_max_y)
			position = active - list_max_y + 1;

	if (limit > list_max_y) {
		if (position < 0)
			position = 0;
		else
			if (position > limit - list_max_y)
				position = limit - list_max_y;
	} else
		position = 0;
}

void ListWindow::Draw()
{
	int i, j, limit;

	limit = NumOfItems();
	checkPos(limit);

	i = position - oldPos;
	switch (i) {
	case -1:
	case 1:
		scrollok(list->info, TRUE);
		wscrl(list->info, i);
		scrollok(list->info, FALSE);
	case 0:
		if (active != oldActive) {
			j = oldActive - position;
			if ((j >= 0) && (j < list_max_y))
				oneLine(j);
		}
		oneLine(active - position);
		wnoutrefresh(list->info);
		break;
	default:
		for (j = 0; j < list_max_y; j++) {
			oneLine(j);
			if (position + j == limit - 1)
				j = list_max_y;
		}
		ReDraw();
	}
	oldPos = position;
	oldActive = active;
}

void ListWindow::Move(direction dir)
{
	int limit = NumOfItems();

	switch (dir) {
	case UP:
		active--;
		break;
	case DOWN:
		active++;
		break;
	case PGUP:
		position -= list_max_y;
		active -= list_max_y;
		break;
	case PGDN:
		position += list_max_y;
		active += list_max_y;
		break;
	case HOME:
		active = 0;
		break;
	case END:
		active = limit - 1;
		break;
	}
	checkPos(limit);
}

void ListWindow::ReDraw()
{
	list->touch();
}

void ListWindow::ForceRedraw()
{
	oldPos = -100;
	Draw();
}

Interface::Interface()
{
	screen_init();
	welcome.MakeActive();
	packets.MakeActive();
	state = packetlist;
	currList = &packets;
	doupdate();
}

Interface::~Interface()
{
	delwin(screen);
	touchwin(stdscr);
	refresh();
	leaveok(screen, FALSE);
	echo();
	endwin();
}

ListWindow *Interface::activeList()
{
	return currList;
}

void Interface::init_colors()
{
	int x, y;

	for (y = COLOR_BLACK; y <= COLOR_WHITE; y++)
		for (x = COLOR_BLACK; x <= COLOR_WHITE; x++)
			init_pair(((x << 3) + y), x, y);

	// Color pair 0 cannot be set (argh!), so swap:

	init_pair(((COLOR_WHITE << 3) + COLOR_WHITE),
		COLOR_BLACK, COLOR_BLACK);

	// Of course, now I can't do white-on-white (grey). :-(
}

void Interface::screen_init()
{
	initscr();
	refresh();
	start_color();
	init_colors();
	cbreak();
	noecho();
	nonl();
	screen = newwin(0, 0, 0, 0);
	leaveok(screen, TRUE);
	keypad(screen, TRUE);
	wbkgdset(screen, C_SBACK | ACS_BOARD);
	werase(screen);
	wbkgdset(screen, ' ');
	wattrset(screen, C_SBORDER | A_BOLD);
	box(screen, 0, 0);
	mvwaddch(screen, 0, 2, ACS_RTEE);
	wattrset(screen, C_SBACK | A_BOLD);
	wprintw(screen, MM_TOPHEADER, MM_NAME, MM_MAJOR, MM_MINOR);
	wattrset(screen, C_SBORDER | A_BOLD);
	waddch(screen, ACS_LTEE);
	wattrset(screen, C_SSEPBOTT);
	wmove(screen, LINES - 5, 1);
	whline(screen, ACS_HLINE, COLS - 2);
	wnoutrefresh(screen);
}

void Interface::screentouch()
{
	touchwin(screen);
	wnoutrefresh(screen);
}

int Interface::WarningWindow(const char *warning, const char *yes,
				const char *no)
{
	int x, c;
	int def_val = 1;	//possible values: 1 or 0
	int result = 0;

	x = strlen(warning) + 4;

	ShadowedWin warn(7, x, LINES / 2 - 4, (COLS - x) / 2, C_WTEXT
		| A_BOLD);

	keypad(warn.win, TRUE);
	mvwaddstr(warn.win, 2, 2, (char *) warning);
	mvwprintw(warn.win, 4, x / 3 - 1, "  %s ", &yes[1]);
	mvwprintw(warn.win, 4, 2 * x / 3 - strlen(no) - 1, "  %s ", &no[1]);
	wattrset(warn.win, C_WTEXTHI | A_BOLD);
	mvwaddch(warn.win, 4, x / 3, yes[0]);
	mvwaddch(warn.win, 4, 2 * x / 3 - strlen(no), no[0]);

	while (!result) {
		mvwaddch(warn.win, 4, x / 3 - 1, def_val ? '[' : ' ');
		mvwaddch(warn.win, 4, x / 3 + strlen(yes), def_val
			? ']' : ' ');
		mvwaddch(warn.win, 4, 2 * x / 3 - strlen(no) - 1, def_val
			? ' ' : '[');
		mvwaddch(warn.win, 4, 2 * x / 3, def_val ? ' ' : ']');
		wrefresh(warn.win);

		do
			c = wgetch(warn.win);
		while (c == ERR);
		if ((c == tolower(yes[0])) || (c == toupper(yes[0]))) {
			def_val = 1;
			result = 1;
		} else if ((c == tolower(no[0])) || (c == toupper(no[0]))) {
			def_val = 0;
			result = 1;
		} else
			switch (c) {
			case KEY_LEFT:
			case KEY_RIGHT:
			case 9:
				def_val = !def_val;
				break;
			case '\r':
			case '\n':
				result = 1;
			}
	}
	return def_val;
}

statetype Interface::active()
{
	return state;
}

void Interface::select()
{
	switch (state) {
	case packetlist:
		unsaved_reply = 0;
		any_read = 0;
		packets.Select();
		packets.Delete();
		welcome.Delete();
		mm.selectPacket(packetList->getName());
		screentouch();
		areas.MakeActive();
		areas.FirstUnread();
		state = arealist;
		currList = &areas;
		break;
	case arealist:
		areas.Select();
		if (mm.areaList->getNoOfLetters() > 0) {
			letterList = mm.areaList->getLetterList();
			areas.Delete();
			screentouch();
			letters.MakeActive();
			letters.FirstUnread();
			state = letterlist;
			currList = &letters;
		}
		break;
	case letterlist:
		letters.Select();
		letters.Delete();
		screentouch();
		letterwindow.MakeActive();
		any_read = 1;
		state = letter;
		break;
	case letter:
		letterwindow.Next();
		break;
	case littlearealist:
		delete littleareas;
		if (mm.areaList->isNetmail())
			if ((Key == 'N') || (Key == 'E'))
				addresses.MakeActive();
			else {
				net_address nm = letterwindow.PickNetAddr();
				letterwindow.set_Letter_Params(&nm, NULL);
			}
		Key = '\0';
		areas.Select();
		letterwindow.ReDraw();
		state = letter;
		unsaved_reply |= letterwindow.EnterLetter();
	default:;
	}
}

int Interface::back()
{
	switch (state) {
	case packetlist:
		if (WarningWindow("Do you really want to quit?",
				  "Yes", "No")) {
			packets.Delete();
			welcome.Delete();
			return 1;
		} else {
			screentouch();
			welcome.ReDraw();
			packets.ReDraw();
			helpwindow.redraw();
		}
		break;
	case arealist:
		if (any_read) {
			if (WarningWindow("Save lastread pointers?",
					  "Yes", "No"))
				mm.saveRead();
			any_read = 0;
		}
		if (unsaved_reply)
			if (WarningWindow(
		"The content of the reply area has changed. Create new packet?",
						 "Yes", "No"))
				create_reply_packet();
		areas.Delete();
		mm.Delete();
		screentouch();
		packets.MakeActive();
		welcome.MakeActive();
		state = packetlist;
		currList = &packets;
		break;
	case letterlist:
		letters.Delete();
		delete letterList;
		screentouch();
		areas.MakeActive();
		state = arealist;
		currList = &areas;
		break;
	case letter:
		letterwindow.Delete();
		screentouch();
		letters.MakeActive();
		letters.Draw();
		state = letterlist;
		currList = &letters;
		break;
	case littlearealist:
		delete littleareas;
		areas.Select();
		state = letter;
	default:;
	}
	return 0;
}

#ifdef SIGWINCH
void Interface::resize()
{
		delwin(screen);
		endwin();
		screen_init();
}

void Interface::sigwinch()
{
	switch (state) {
	case packetlist:
		welcome.Delete();
		packets.Delete();
		resize();
		welcome.MakeActive();
		packets.MakeActive();
		break;
	case arealist:
		areas.Delete();
		resize();
		areas.MakeActive();
		break;
	case letterlist:
		letters.Delete();
		resize();
		letters.MakeActive();
		letters.Draw();
		break;
	case letter:
		letterwindow.Delete();
		resize();
		letterwindow.MakeActive();
		break;
	case letter_help:
		helpwindow.Delete();
		letterwindow.Delete();
		resize();
		letterwindow.MakeActive();
		if (!mm.areaList->isReplyArea())
			helpwindow.letter();
		else
			helpwindow.reply_area_letter();
		break;
	case littlearealist:
		delete littleareas;
		letterwindow.Delete();
		resize();
		letterwindow.MakeActive();
		littleareas = new LittleAreaListWindow(Key);
		currList = littleareas;
	default:;
	}
	if (!currAnsi)
		doupdate();
}
#endif

void Interface::ReplyPacket()
{
	if (state == arealist) {
		if (WarningWindow(
		"This will overwrite any existing reply packet. Continue?",
				  "Yes", "No")) {
			create_reply_packet();
		} else
			areas.ReDraw();
	}
}

void Interface::create_reply_packet()
{
	mm.areaList->makeReply();
	unsaved_reply = 0;
	screentouch();
	areas.ReDraw();
	helpwindow.redraw();
}

void Interface::RIGHT_ARROW()
{
	switch (state) {
	case arealist:
		do {
			areas.Move(DOWN);
			areas.Select();
		} while (!mm.areaList->getNoOfLetters() &&
			 (mm.areaList->getAreaNo() <
			  mm.areaList->noOfAreas() - 1));
		areas.Draw();
		break;
	case letterlist:
		letters.NextUnread();
		break;
	case letter:
		letterwindow.Next();
		break;
	case packetlist:
	case littlearealist:
		currList->Move(DOWN);
		currList->Draw();
	default:;
	}
}

void Interface::LEFT_ARROW()
{
	switch (state) {
	case arealist:
		do {
			areas.Move(UP);
			areas.Select();
		} while (!mm.areaList->getNoOfLetters() &&
			 (mm.areaList->getAreaNo() > 0));
		areas.Draw();
		break;
	case letterlist:
		letters.PrevUnread();
		break;
	case letter:
		letterwindow.Previous();
		break;
	case packetlist:
	case littlearealist:
		currList->Move(UP);
		currList->Draw();
	default:;
	}
}

void Interface::ikeypad(direction dir)
{
	switch (state) {
	case packetlist:
	case arealist:
	case letterlist:
	case littlearealist:
		currList->Move(dir);
		currList->Draw();
		break;
	case letter:
		letterwindow.Move(dir);
	default:;
	}
}

void Interface::redraw()
{
	switch (state) {
	case arealist:
	case letterlist:
		currList->ForceRedraw();
		break;
	case letter:
		letterwindow.DrawHeader();
		letterwindow.MakeChain(COLS);
		letterwindow.DrawBody();
	default:;
	}
}

void Interface::help()
{
	char c;

	if (state == letter) {
		if (!mm.areaList->isReplyArea())
			helpwindow.letter();
		else
			helpwindow.reply_area_letter();
		doupdate();
		state = letter_help;
		do
			c = wgetch(screen);
		while (c == ERR);
		helpwindow.Delete();
		letterwindow.ReDraw();
		state = letter;
	}
}

void Interface::enterletter(int key)
{
	switch (state) {
	case littlearealist:
	case packetlist:
		if (key == ' ') {
			ikeypad(PGDN);
			break;
		}
		break;
	case arealist:
		switch (key) {
		case ' ':
			ikeypad(PGDN);
			break;
		case 'E':
			if (!mm.areaList->isCollection()) {
				areas.Select();
				if (mm.areaList->isNetmail())
					addresses.MakeActive();
				letterwindow.set_Letter_Params(
					mm.areaList->getAreaNo(), 'E');
				unsaved_reply |= letterwindow.EnterLetter();
				areas.ForceRedraw();
			}
			break;
		case 'L':
			areas.Select();
			areas.Delete();
			mm.areaList->relist();
			areas.ResetActive();
			areas.MakeActive();
		}
		break;
	case letterlist:
		switch (key) {
		case '\t':
			letters.NextUnread();
			break;
		case ' ':
			ikeypad(PGDN);
			break;
		case 'U':
		case 'M':
			// Toggle read/unread and marked from letterlist

			letters.Select();
			letterList->setStatus(letterList->getStatus() ^
				((key == 'U') ? MS_READ : MS_MARKED));
			any_read = 1;
			ikeypad(DOWN);
			break;
		case 'E':
			if (mm.areaList->isReplyArea())
				unsaved_reply |= letterwindow.EditLetter();
			else
			    if (!mm.areaList->isCollection()) {
				if (mm.areaList->isNetmail())
					addresses.MakeActive();
				letterwindow.set_Letter_Params(
					mm.areaList->getAreaNo(), 'E');
				unsaved_reply |= letterwindow.EnterLetter();
			    }
			break;
		}
		break;
	case letter:
		if (key == ' ') {
			letterwindow.NextDown();
			break;
		}
		if (mm.areaList->isReplyArea()) {
			switch (key) {
			case 'E':
				unsaved_reply |= letterwindow.EditLetter();
				break;
			case 'V':
			case 1:	// CTRL-A
			case 22: // CTRL-V
				letterwindow.ansi_dump();
			}
		} else {
			switch (key) {
			case 'M':
			case 'U':
				letterwindow.StatToggle((key == 'M') ? 
					MS_MARKED : MS_READ);
				break;
			case 'R':
			case 'O':
			case 'E':
				Key = key;
				littleareas = new LittleAreaListWindow(key);
				currList = littleareas;
				state = littlearealist;
				break;
			case 'N':
				Key = key;
				net_address nm = letterwindow.PickNetAddr();
				letterwindow.set_Letter_Params(
					  mm.areaList->getAreaNo(), 'N');
				letterwindow.set_Letter_Params(&nm, NULL);
				unsaved_reply |= letterwindow.EnterLetter();
				break;
			case 'T':
				letterwindow.GetTagline();
				break;
			case 'V':
			case 1:	// CTRL-A
			case 22: // CTRL-V
				letterwindow.ansi_dump();
			}
		}
		break;
	default:;
	}
}

void Interface::save()
{
	switch (state) {
	case letterlist:
		letterwindow.Save(1);
		screentouch();
		letters.ReDraw();
		helpwindow.redraw();
		break;
	case letter:
		letterwindow.Save(0);
		letterwindow.ReDraw();
		break;
	default:;
	}
}

void Interface::search()
{
}

void Interface::kill_letter()
{
	if (mm.areaList->isReplyArea())
		if (WarningWindow(
			"Are you sure you want to delete this letter?",
				  "Yes", "No")) {
			if (state == letterlist) {
				letters.Select();
				letters.Delete();
				screentouch();
			}
			mm.areaList->killLetter(letterList->getMsgNum());
			unsaved_reply = 1;
			if (state == letter)
				back();
			else {
				letters.MakeActive();
				letters.Draw();
			}
			if (!mm.areaList->getNoOfLetters())
				back();
			else
				ikeypad(UP);

		} else if (state == letter)
			letterwindow.ReDraw();
		else {
			letters.Delete();
			screentouch();
			letters.MakeActive();
			letters.Draw();
		}
}

void Interface::kill_packet()
{
	char tmp[128];

	packets.Select();
	sprintf(tmp, "Do you really want to delete %s?",
		packetList->getName());
	if (WarningWindow(tmp, "Yes", "No")) {
		mychdir(mm.resourceObject->get(PacketDir));
		remove(packetList->getName());
		packets.ReList();
	}
	screentouch();
	welcome.ReDraw();
	packets.ReDraw();
	helpwindow.redraw();
}

void Interface::kill()
{
	switch (state) {
	case packetlist:
		kill_packet();
		break;
	case letterlist:
	case letter:
		kill_letter();
	default:;
	}
}

void Interface::addressbook()
{
	if (state != littlearealist)
		addresses.MakeActive(1);	//1 means NOENTER 
}

void Interface::KeyHandle()		// Main loop
{
	int c, end = 0;
	while (!end) {
		do
			c = wgetch(screen);
		while (c == ERR);
		switch (c) {
		case 'q':
		case 'Q':
		case '\033':	//the escape key
		case KEY_BACKSPACE:
			end = back();
			break;
		case '\n':
		case '\r':
			select();
			break;
		case '/':
			search();
			break;
		case KEY_DOWN:
			ikeypad(DOWN);
			break;
		case KEY_UP:
			ikeypad(UP);
			break;
		case KEY_HOME:
			ikeypad(HOME);
			break;
		case KEY_END:
			ikeypad(END);
			break;
		case 'b':
		case 'B':
		case KEY_PPAGE:
			ikeypad(PGUP);
			break;
		case 'f':
		case 'F':
		case KEY_NPAGE:
			ikeypad(PGDN);
			break;
		case '+':
		case KEY_RIGHT:
			RIGHT_ARROW();
			break;
		case '-':
		case KEY_LEFT:
			LEFT_ARROW();
			break;
		case '?':
		case KEY_F(1):
			help();
			break;
		case 's':
		case 'S':
			save();
			break;
		case 'k':
		case 'K':
			kill();
			break;
		case 'a':
		case 'A':
			addressbook();
			break;
		case '!':
		case KEY_F(2):
			ReplyPacket();
			break;
		case 'c':
		case 'C':
			isoconvert = !isoconvert;
			redraw();
			break;
		default:
			enterletter(toupper(c));
		}
		if (!end)
			doupdate();
	}
}
