; 
; irit.el - Definitions of IRIT mode for emacs editor.
; 
; Author:	Gershon Elber
; 		Computer Science Dept.
; 		University of Utah
; Date:	Tue May 14 1991
; Copyright (c) 1991, Gershon Elber
;
; This file defines an environment to run edit and execute IRIT programs.
; Such a program should have a '.irt' extension in order it to be in
; irit-mode major mode. Two new functions are provided to communicate
; between the editted file and the solid modeller:
; 1. send-line-to-irit - sends a single line to the solid modeller for
;    execution. A line is defined from current position to the next
;    semicolon ';'. If however several commands exists on the same line
;    they will all be send as one line.
;    Bounded to Meta-E by default.
; 2. send-region-to-irit - sends the region from the current mark (mark-marker)
;    to current position (point-marker) to the solid modeller. This function
;    is convenient for sending a large block of commands.
;    Bounded to Meta-R by default.
; Both functions checks for existance of a buffer named irit-solid-modeller
; and a process named "irit" hooked to it, and will restart a new process
; or buffer if none exists. The program to execute as process "irit" is
; defined by the irit-program constant below.
;

(defvar irit-program "irit"
  "*The executable to run for irit-solid-modeller buffer.")

(defvar irit-echo-program nil
  "*Control echo of executed commands to irit-solid-modeller buffer.")

(defvar irit-mode-map nil "")
(if irit-mode-map
    ()
  (setq irit-mode-map (make-sparse-keymap))
  (define-key irit-mode-map "\M-e" 'send-line-to-irit)
  (define-key irit-mode-map "\M-r" 'send-region-to-irit))

;;;
;;; Define the irit-mode
;;;
(defun irit-mode ()
  "Major mode for editing and executing IRIT files.

see send-line-to-irit and send-region-to-irit for more."
  (interactive)
  (use-local-map irit-mode-map)
  (setq major-mode 'irit-mode)
  (setq mode-name "Irit")
  (run-hooks 'irit-mode-hook))

;;;
;;; Define send-line-to-irit - send from current cursor position to next
;;; semicolin detected.
;;;
(defun send-line-to-irit ()
  "Sends one line of code from current buffer to the IRIT program.

Use to execute a line in the IRIT solid modeller. A line is anything
that is terminated by a semicolon, but is at least one line of text so
multiple commands per line (with several semicolons) are still
considered a single line.

The IRIT solid modeller buffer name is irit-solid-modeller and the 
process name is 'irit'. If none exists, a new one is created.

The name of the irit program program to execute is stored in irit-program
and may be changed."
  (interactive)
  (if (equal major-mode 'irit-mode)
    (progn
      (make-irit-buffer)        ; In case we should start a new one.
      (let ((start-mark (point-marker)))
	(search-forward ";")
	(let ((end-one-mark (point-marker)))
	  (goto-char start-mark)
	  (beginning-of-line)
	  (next-line 1)
	  (let* ((crnt-buffer (buffer-name))
	         (end-two-mark (point-marker))
	         (end-max-mark (max end-one-mark end-two-mark))
		 (string-copy (buffer-substring start-mark end-max-mark)))
	    (switch-to-buffer-other-window (get-buffer "irit-solid-modeller"))
	    (end-of-buffer)
	    (if irit-echo-program
	      (insert string-copy))
	    (set-marker (process-mark (get-process "irit")) (point-marker))
	    (if (not (pos-visible-in-window-p))
	      (recenter 3))
	    (switch-to-buffer-other-window (get-buffer crnt-buffer))
	    (process-send-region "irit" start-mark end-max-mark)
	    (goto-char end-max-mark)
	    (if (equal "\n" (buffer-substring (point-marker)
					      (+ 1 (point-marker))))
	      (process-send-string "irit" "\n"))  
	    (if (> end-one-mark end-two-mark)
	      (forward-char 1))))))
    (message "Should be invoked in irit-mode only.")))

;;;
;;; Define send-region-to-irit - send from current cursor position to
;;; current marker.
;;;
(defun send-region-to-irit ()
  "Sends a region of code from current buffer to the IRIT program.

When this function is invoked on an IRIT file it send the region from current
point to current mark to the irit solid modeller.

The IRIT solid modeller buffer name is irit-solid-modeller and the
process name is 'irit'. If none exists, a new one is created.

The name of the irit program program to execute is stored in irit-program
and may be changed."
  (interactive)
  (if (equal major-mode 'irit-mode)
    (progn
      (make-irit-buffer)     ; In case we should start a new one.
      (copy-region-as-kill (mark-marker) (point-marker))
      (let ((crnt-buffer (buffer-name)))
	(switch-to-buffer-other-window (get-buffer "irit-solid-modeller"))
	(end-of-buffer)
	(if irit-echo-program
	  (yank))
	(set-marker (process-mark (get-process "irit")) (point-marker))
	(if (not (pos-visible-in-window-p))
	  (recenter 3))
	(switch-to-buffer-other-window (get-buffer crnt-buffer))
	(process-send-region "irit" (mark-marker) (point-marker))))
    (message "Should be invoked in irit-mode only.")))

;;;
;;; Switch to "irit-solid-modeller" buffer if exists. If not, creates one and
;;; execute the program defined by irit-program.
;;;
(defun make-irit-buffer ()
  "Switch to iris-solid-modeller buffer or create one if none exists"
  (interactive)
  (if (get-buffer "irit-solid-modeller")
    (if (not (get-process "irit"))
      (progn
	(message "Starting IRIT solid modeller...")
	(start-process "irit" "irit-solid-modeller" irit-program)
	(process-send-string "irit" "\n")
	(message "Done.")))
    (progn
      (message "Starting IRIT solid modeller...")
      (start-process "irit" "irit-solid-modeller" irit-program)
      (process-send-string "irit" "\n")
      (message "Done."))))

;;;
;;; Autoload irit-mode on any file with irt extension. 
;;;
(setq auto-mode-alist (append '(("\\.irt$" . irit-mode))
			      auto-mode-alist))
