/* QIC_DRV.C - QIC-02 / PC-36 device driver for MS-DOS */

/*
 * $Author:   Phlash  $
 * $Revision:   1.8  $
 * $Date:   04 Nov 1994 21:10:38  $
 */

#include "qic_02.h"
#include "qic_drv.h"

// Actual device name to set up in the header during Init()
char gDevName[] = "QIC_TAPE";

// Debugging flag (set/reset via IOCTL Write)
int gDebug = 0;

// Asyncronous IO flag
int gAsync = 0;

// Device open / close control flag
#define DEV_CLOSE 0
#define DEV_OPEN  1
#define DEV_READ  2
#define DEV_WRITE 3
int gOpen = DEV_CLOSE;

// Special value to identify end of _BSS segment (and also this driver)
static BYTE endData;

// Utility function, to display a string on screen via the BIOS
int Display(char far *str)
{
int i;

   for(i=0; str[i] != 0; i++)
      BIOSChr(str[i]);
   return i;
}

// Utility function to decode a hex string into an int
int htoi(char far *str)
{
int i;
int rv = 0;

   for(i=0; i<4; i++)
   {
      if(str[i] >= 'a')
         str[i] += 'A'-'a';
      if(str[i] >= 'A' && str[i] <= 'F')
         rv = (rv << 4) | (str[i] - 'A' + 10);
      else if(str[i] >= '0' && str[i] <= '9')
         rv = (rv << 4) | (str[i] - '0');
      else
         break;
   }
   return rv;
}

// Utility function to display an int as a hex string
int DisplayHex(int hex)
{
int i, nibble;

   for(i=3; i>=0; i--)
   {
      nibble = (hex >> (4*i)) & 0xF;
      if(nibble > 9)
         BIOSChr(nibble - 10 + 'A');
      else
         BIOSChr(nibble + '0');
   }
   return 4;
}

// utility function to copy a string
int bcopy(char far *src, char far *dst, int len)
{
int i;

   for(i=0; i<len; i++)
      dst[i] = src[i];
}

// utility function to compare a string
int bcmp(char far *src, char far *dst, int len)
{
int i;

   for(i=0; i<len; i++)
      if(dst[i] != src[i])
         return 1;
   return 0;
}

//------------------------- The driver functions -------------------------

int Init(void)
{
char far *cmdLine;
int i,j;
int port = 0x300;
int intr = 3;
int dma = 1;

// Say hi.
   Display("\r\n\n");
   Display(gDevName);
   Display(": $Revision:   1.8  $ (c) (AshbySoft *) 1994\r\n");

// Change the device name
   for(i=0; i<8; i++)
      DHPtr->name[i] = gDevName[i];

// Display command line & check for debug switch
   cmdLine = *((char far * far *)&RHPtr->count);
   Display(gDevName);
   Display(": DEVICE=");
   for(i=0; cmdLine[i] != '\n'; i++)
   {
      BIOSChr(cmdLine[i]);
      if(cmdLine[i] == '/' || cmdLine[i] == '-')
      switch(cmdLine[i+1])
      {
      case 'X':
         gDebug = 1;
         break;
      case 'Q':
         qicDebug = 1;
         break;
      case 'P':
         port = htoi(&cmdLine[i+2]);
         break;
      case 'I':
         intr = htoi(&cmdLine[i+2]);
         break;
      case 'D':
         dma = htoi(&cmdLine[i+2]);
         break;
      }
   }
   Display("\r\n");
   Display(gDevName);
   Display(": port=0x");
   DisplayHex(port);
   Display(" intr=");
   DisplayHex(intr);
   Display(" dma=");
   DisplayHex(dma);
   Display("\r\n");
   if(gDebug)
   {
      Display(gDevName);
      Display(": Debugging enabled.\r\n");
   }

// See if we can find a tape drive at the specified location
   if(qicInit(port, intr, dma, 0))
   {
   // No drive, so abort installation of driver
      Display(gDevName);
      Display(": Not loaded, cannot find tape drive.\r\n");
      RHPtr->address = (BYTE far *)Init;
      *((WORD far *)&RHPtr->address) = 0;
      RHPtr->unit = 0;
      DHPtr->attributes &= 0x7FFF;
      return 0;
   }

// Set the break address for DOS loader
   RHPtr->address = &endData;
   return 0;
}

int MediaChk(void)
{
   if(gDebug)
      Display("Media Chk\r\n");
   return 0;
}

int BuildBPB(void)
{
   if(gDebug)
      Display("Build BPB\r\n");
   return 0;
}

int IoctlRd(void)
{
qicDrvIoctl_t far *pio;

   if(gDebug)
      Display("Ioctl Rd\r\n");

// Fill out ioctl data
   if(RHPtr->count >= sizeof(qicDrvIoctl_t))
   {
      pio = (qicDrvIoctl_t far *)RHPtr->address;
      bcopy(QIC_CHK_VALUE, pio->chk, 4);
      pio->debug = gDebug;
      pio->async = gAsync;
      pio->command = QIC_NOCOMMAND;
      RHPtr->count = sizeof(qicDrvIoctl_t);
   }
   else
   {
      Display(gDevName);
      Display(": IoctlRd buffer too small\r\n");
      RHPtr->count = 0;
   }
   return 0;
}

int Read(void)
{
WORD i;

   if(gDebug)
      Display("Read\r\n");

// Check driver mode
   if(gOpen != DEV_OPEN && gOpen != DEV_READ)
      return 0x8003;
   gOpen = DEV_READ;

// Check for read request a multiple of 512 bytes
   if(RHPtr->count < 512 || RHPtr->count & 0x1FF)
   {
      Display(gDevName);
      Display(": Error, invalid transfer size: ");
      DisplayHex(RHPtr->count);
      Display("\r\n");
      return 0x800B;
   }

// Transfer direct to application buffer
   i = RHPtr->count/512;
   if(qicReadBlocks(RHPtr->address, i) < 0)
      return 0x800B;
   if(!gAsync)
   {
      if((i=qicComplete()) < 0)
         return 0x800B;
   }
   RHPtr->count = i*512;
   return 0;
}

int NdRead(void)
{
   if(gDebug)
      Display("Nd Read\r\n");

// Return with BUSY bit set (nothing to read)
   return 0x0200;
}

int InpStat(void)
{
   if(gDebug)
      Display("Inp Stat\r\n");
   return 0;
}

int InpFlush(void)
{
   if(gDebug)
      Display("Inp Flush\r\n");
   return 0;
}

int Write(void)
{
WORD i;

   if(gDebug)
      Display("Write\r\n");

// Check device mode
   if(gOpen != DEV_OPEN && gOpen != DEV_WRITE)
      return 0x8003;
   gOpen = DEV_WRITE;

// Check for write request a multiple of 512
   if(RHPtr->count < 512 || RHPtr->count & 0x1FF)
   {
      Display(gDevName);
      Display(": Error, invalid transfer size: ");
      DisplayHex(RHPtr->count);
      Display("\r\n");
      return 0x800A;
   }

// Transfer direct from application buffer
   i = RHPtr->count/512;
   if(qicWriteBlocks(RHPtr->address, i) < 0)
      return 0x800A;
   if(!gAsync)
   {
      if((i=qicComplete()) < 0)
         return 0x800A;
   }
   RHPtr->count = i*512;
   return 0;
}

int WriteVfy(void)
{
   if(gDebug)
      Display("Write Vfy\r\n");

// Pass to normal write
   return Write();
}

int OutStat(void)
{
   if(gDebug)
      Display("Out Stat\r\n");

// Device not busy
   return 0;
}

int OutFlush(void)
{
   if(gDebug)
      Display("Out Flush\r\n");
   return 0;
}

int IoctlWt(void)
{
qicDrvIoctl_t far *pio;
unsigned int rv;

   if(gDebug)
      Display("Ioctl Wt\r\n");

// Copy in ioctl data
   if(RHPtr->count >= sizeof(qicDrvIoctl_t))
   {
      pio = (qicDrvIoctl_t far *)RHPtr->address;
      if(bcmp(pio->chk, QIC_CHK_VALUE, 4))
      {
         Display(gDevName);
         Display(": Invalid IOCTL check value\r\n");
         return 0x800C;
      }
      gDebug = pio->debug;
      if(!gDebug || gDebug > 1)
         qicDebug = gDebug;
      gAsync = pio->async;
      switch(pio->command)
      {
      case QIC_NOCOMMAND:
         break;
      case QIC_COMPLETE:
         if((rv=qicComplete()) < 0)
            return 0x800C;
         rv *= 512;
         (unsigned int)pio->command = rv;
         break;
      case QIC_ERASE:
      case QIC_RETENSION:
         Display(gDevName);
         Display(": IoctlWt command not implemented\r\n");
         break;
      default:
         Display(gDevName);
         Display(": IoctlWt: invalid command\r\n");
         return 0x800C;
      }
      RHPtr->count = sizeof(qicDrvIoctl_t);
   }
   else
   {
      Display(gDevName);
      Display(": IoctlWt buffer too small\r\n");
      RHPtr->count = 0;
   }
   return 0;
}

int DevOpen(void)
{
   if(gDebug)
      Display("Dev Open\r\n");
   if(gOpen)
      return 0x800C;
   gOpen = DEV_OPEN;
   return 0;
}

int DevClose(void)
{
int i;

   if(gDebug)
      Display("Dev Close\r\n");
   if(!gOpen)
      return 0x800C;

// Rewind tape if any I/O has occurred
   switch(gOpen)
   {
   case DEV_READ:
   case DEV_WRITE:
      if(gAsync)
         qicComplete();
      qicRewind();
      break;
   }
   gOpen = DEV_CLOSE;
   gAsync = 0;
   return 0;
}

int Unknown(void)
{
   Display(gDevName);
   Display(": Unknown device request\r\n");
   return 0x8003;
}

/* End */
