/*
 *	sbifeps.c  --  simple binary image file to epson printer format.
 *
 *	25 july 1989  Olle Olsson.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
/*#include <conio.h>*/
#include <io.h>
#include <string.h>
#include <process.h>
#include <bios.h>
#include "sbif.h"

/* max number of dots in an epson bit image */
#define MAXDOTS 480

/* the functions */
void main( int argc, char *argv[] );
void beep( void );
void error( char *s, ... );
void warning( char *s, ... );
void usage( void );
int prbytes( char *buffer, int size, FILE *outfile );
int weps( FILE *outfile, char *obuf, int obuf_size, int indent );
int wepsbegin( FILE *outfile );
int wepsend( FILE *outfile );

static void usage( void )
{
fprintf( stderr, "Usage:\n" );
fprintf( stderr, "\tsbifeps [options] sbif_file\n" );
fprintf( stderr, "Options are:\n" );
fprintf( stderr, "\t-o string\toutput file name\n" );
fprintf( stderr, "\t-b\tuse bios to print (on lpt1:)\n" );
fprintf( stderr, "\t-i num\tindent amount (characters)\n" );
fprintf( stderr, "\t-x num\tx offset\n" );
fprintf( stderr, "\t-y num\ty offset\n" );
fprintf( stderr, "\n" );
}


void main( argc, argv )
int argc;
char *argv[];
{
register unsigned char mask;	/* pixel bit mask */
register int x, y;		/* current x, y */
unsigned char *vp;		/* pixel byte pointer */
FILE *inf;			/* data file */
FILE *outf;			/* output file */
int xlines, ylines;		/* x and y pixel count */
int xbytes;			/* x byte count */
unsigned char *pmat;		/* the pixel matrix (part of it) */
int xoffs, yoffs;		/* display offsets */
int indent;			/* line indent amount */
int trace;			/* trace flag */
int i, c;			/* tmp */
char datafile[100];		/* data file name */
char outfile[100];		/* output file name */
char *ap, **p;			/* argument pointers */
char hbuf[SBHSIZE];		/* buffer for the sbif header */
char nbuf[100];			/* buffer for the name in the sbif header */
char obuf[MAXDOTS];		/* output conversion buffer */
register unsigned char omask;	/* output dot bit mask */
char *op;			/* output byte pointer */
int olen;			/* output dot count */
int biospr;			/* flag for bios print */


/* read arguments */
datafile[0] = outfile[0] = '\0';
trace = 0;
biospr = indent = xoffs = yoffs = 0;
for (p = argv + 1, c = 1; c < argc; ++c)
	{
	if (trace) printf( "%s:\n", *p );
	ap = *p++;

	if (*ap == '-')	for (i = 1; ap[i]; ++i) switch (ap[i])
		{
		case 'b':
			biospr++;
			continue;

		case 'o':
			if (++c >= argc)
				error( "-o: output file name missing");

			if (outfile[0])
				error( "-o: too many output files" );

			strcpy( outfile, *p++ );
			continue;

		case 'i':	/* indent */
			if (++c >= argc)
				error( "-i: indent value missing");

			indent = atoi( *p++ );

			if (indent < 0)
				error( "-i: indent < 0" );

			continue;

		case 'x':	/* x offset */
			if (++c >= argc)
				error( "-x: offset missing");

			xoffs = atoi( *p++ );

			if (xoffs <= 0)
				error( "-x: offset <= 0" );

			continue;

		case 'y':	/* y offset */
			if (++c >= argc)
				error( "-y: offset missing");

			yoffs = atoi( *p++ );

			if (yoffs <= 0)
				error( "-y: offset <= 0" );

			continue;

		case 't':
			trace++;
			continue;

		default:
			usage();
			error( "don't understand flag '-%c'", ap[i] );
		}
	else if (!datafile[0])
		{
		strcpy( datafile, ap );
		}
	else
		{
		usage();
		error( "too many data files specified" );
		}
	}

/* open the data file */
if (!datafile[0])
	error( "No SBIF data file specified" );

if ((inf = fopen( datafile, "rb" )) == NULL)
	error( "Can't open input file '%s'", datafile );

/* write to stdout or a named file? */
/* or use biosprint() */
if (biospr)
	outf = NULL;
else if (!outfile[0])
	{
	/* stdout, make it binary mode */
	outf = stdout;
	setmode( fileno( outf ), O_BINARY );
	}
else if ((outf = fopen( outfile, "wb" )) == NULL)
	error( "Can't create output file '%s'", outfile );

/* read the header */
if (fread( hbuf, sizeof( hbuf ), 1, inf ) < 1)
	error( "premature end of file '%s'", datafile );

/* read the sizes */
xlines = ylines = 0;
sscanf( hbuf, "%s %d %d", nbuf, &xlines, &ylines );
xbytes = XBYTES( xlines );

if (trace) fprintf( stderr, "%s x:%d y:%d (offs %d, %d xbytes:%d)\n",
			 nbuf, xlines, ylines, xoffs, yoffs, xbytes );

if (xlines <= 0 || ylines <= 0)
	error( "don't understand contents of file '%s'", datafile );

/* get a buffer for one row of the pixel matrix */
if (!(pmat = (unsigned char *) calloc( xbytes, 1 )))
	error( "out of memory for the pixel matrix" );

/* skip the offset */
for (y = 0; y < yoffs; ++y)
	if (fread( pmat, xbytes, 1, inf ) < 1)
		error( "premature end of file '%s'", datafile );

/* start output */
if (!wepsbegin( outf ))
	error( "output write failure" );

/* clear the output buffer */
memset( obuf, 0, sizeof (obuf) );

/* determine the output dot count (dot line length) */
olen = ((xlines - xoffs) < sizeof (obuf)) ? (xlines - xoffs) : sizeof (obuf);

/* convert the picture */
/* (flip the bytes from lying down to standing up) */
for (y = 0, omask = 0; y < ylines; ++y, omask >>= 1)
	{
	/* read one line */
	if (fread( pmat, xbytes, 1, inf ) < 1)
		{
		if (trace) fprintf( stderr, "end at line %d (%d bytes) ",
						 y + 1, xbytes * (y + 1) );
		error( "premature end of file '%s'", datafile );
		}

	if (trace) fprintf( stderr, " y:%d ", y );

	/* restart at the output buffer first (top) line? */
	if (!omask)
		{
		omask = 0x80;

		/* write the buffer */
		if (y) if (!weps( outf, obuf, olen, indent ))
			error( "output write failure" );

		/* clear the output buffer for the next lines */
		memset( obuf, 0, sizeof (obuf) );
		}

	/* scan one line */
	for (x = 0, vp = pmat, op = obuf, mask = 0x80;
		x < xlines && op < &obuf[sizeof (obuf)];
			++x, ++op, mask >>= 1)
		{
		if (!mask)
			{
			/* next byte */
			++vp;
			mask = 0x80;
			}

		/* check x offset */
		if (x < xoffs)
			continue;

		/* copy the pixel only if it is on */
		if (*vp & mask)
			{
			/* (y is upside down, but so is the printer y coord.) */
			*op |= omask;
			}
		}
	}

/* write odd lines */
if (omask != 0x80) if (!weps( outf, obuf, olen, indent ))
	error( "output write failure" );

/* end */
if (!wepsend( outf ))
	error( "output write failure" );

/* ready */
exit( 0 );
}

int prbytes( bp, size, of )
char *bp;
int size;
FILE *of;
{
int port = 0;		/* biosprint() port, 0 --> lpt1: etc. */

/* ordinary write or bios print? */
if (of == NULL)
	{
	while (size--) if (biosprint( 0, *bp++, port ) & 0x01)
				return (0);
	}
else if (fwrite( bp, size, 1, of) < 1)
	return (0);

/* ok */
return (1);
}

int wepsbegin( outf )
FILE *outf;
{
char *ist = "\033A\010\0332";	/* the secret code */

/* set the line feed to eight dots */
if (!prbytes( ist, strlen( ist ), outf ))
	return (0);

/* ok */
return (1);
}

int wepsend( outf )
FILE *outf;
{
/* restore the line feed length */
if (!prbytes( "\0332", 2, outf ))
	return (0);

/* print a formfeed */
if (!prbytes( "\f", 1, outf ))
	return (0);

/* ok */
return (1);
}


int weps( outf, obuf, olen, indent )
FILE *outf;
char *obuf;
int olen;
int indent;
{
char sbuf[5];

/* indent */
while (indent-- > 0) if (!prbytes( " ", 1, outf ))
			return (0);

/* bit image header (60 dots/inch) */
if (!prbytes( "\033K", 2, outf ))
	return (0);

/* lsb of bit image length */
sbuf[0] = olen & 0xff;
if (!prbytes( sbuf, 1, outf ))
	return (0);

/* msb of bit image length */
sbuf[0] = olen >> 8;
if (!prbytes( sbuf, 1, outf ))
	return (0);

/* write the bit image */
if (!prbytes( obuf, olen, outf ))
	return (0);

/* newline */
if (!prbytes( "\r\n", 2, outf ))
	return (0);

/* ok */
return (1);
}


void error( s, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10 )
char *s;
int v1, v2, v3, v4, v5, v6, v7, v8, v9, v10;
{
fprintf( stderr, "\nError:" );
fprintf( stderr, s, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10 );
fprintf( stderr, "\n" );

exit( 2 );
}

void warning( s, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10 )
char *s;
int v1, v2, v3, v4, v5, v6, v7, v8, v9, v10;
{
fprintf( stderr, "\nWarning:" );
fprintf( stderr, s, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10 );
fprintf( stderr, "\n" );
}


