/* Power Render Material Interface */

#ifndef _PRMATERIAL_
#define _PRMATERIAL_

#ifdef __cplusplus
 extern "C" {
#endif

typedef struct {
  char          name[19];
  PR_DWORD      render_method;
  PR_DWORD      texture_number;
  PR_UCHAR      base_color;
  PR_DWORD      render_shades;
  PR_UCHAR      shadetable;
  PR_UCHAR      environment_map;
  PR_UCHAR      environment_axis;

  PR_UCHAR      MipMap;
  PR_WORD       MipMaterial1;
  PR_WORD       MipMaterial2;
  PR_WORD       MipMaterial3;
  PR_WORD       MipMaterial4;
  PR_UCHAR      MipShift1;
  PR_UCHAR      MipShift2;
  PR_UCHAR      MipShift3;
  PR_UCHAR      MipShift4;
 
  PR_REAL       r,g,b,a;         /* Truecolor support */
} PR_MATERIAL;


#define MIP_LEVEL1 1
#define MIP_LEVEL2 2
#define MIP_LEVEL3 3
#define MIP_LEVEL4 4

extern PR_DWORD PR_NumMaterials;
extern PR_MATERIAL *PR_ObjectMaterialList;



PR_DWORD PR_AddMaterial (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_AddMaterial
   Purpose:    Adds a material to the PR_ObjectMaterialList array 
   Parameters: mat - Pointer to the material
   Return:     Material number
   ----------------------------------------------------------------------- */


void PR_AllocMaterials (PR_DWORD size);
/* -----------------------------------------------------------------------
   Function:   PR_AllocMaterialList
   Purpose:    Allocates space for a number of materials 
   Parameters: size - Number of materials to allocate
   Return:     None
   ----------------------------------------------------------------------- */


void PR_DeleteAllMaterials (void);
/* -----------------------------------------------------------------------
   Function:   PR_DeleteAllMaterials
   Purpose:    Deletes all the material from the PR_ObjectMaterialList array 
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PR_DeleteMaterial (char *name);
/* -----------------------------------------------------------------------
   Function:   PR_DeleteMaterial
   Purpose:    Deletes a material from the PR_ObjectMaterialList array 
   Parameters: name - Name of the material
   Return:     MATERIAL_NOT_FOUND, or MATERIAL_DELETED
   ----------------------------------------------------------------------- */


PR_DWORD PR_FindMaterial (char *name);
/* -----------------------------------------------------------------------
   Function:   PR_FindMaterial
   Purpose:    Finds a material number given its name
   Parameters: name - Name of the material
   Return:     MATERIAL_NOT_FOUND if the name is not in the
               PR_ObjectMaterialList array, or
               material number
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialAlpha (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialAlpha
   Purpose:    Gets the alpha value of the material 
   Parameters: mat - Pointer to the material
   Return:     Alpha (0-255)
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialBaseColor (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialBaseColor
   Purpose:    Gets the base color of the material 
   Parameters: mat - Pointer to the material
   Return:     Base color
   ----------------------------------------------------------------------- */


void PR_GetMaterialColor (PR_MATERIAL *mat, PR_REAL *r, PR_REAL *g, PR_REAL *b);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialColor
   Purpose:    Gets the RGB color of the material 
   Parameters: mat - Pointer to the material
               r,g,b - Pointers to the RGB components
   Return:     None
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialEnvironmentAxis (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialEnvironmentAxis
   Purpose:    Gets the environment mapping axis flag of the material 
   Parameters: mat - Pointer to the material
   Return:     XAXIS, YAXIS, ZAXIS
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialEnvironmentMap (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialEnvironmentMap
   Purpose:    Gets the environment mapping flag of the material 
   Parameters: mat - Pointer to the material
   Return:     Environment mapping state (TRUE or FALSE)
   ----------------------------------------------------------------------- */


PR_DWORD PR_GetMaterialMethod (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialMethod
   Purpose:    Gets the rendering method of the material 
   Parameters: mat - Pointer to the material
   Return:     Rendering method number
   ----------------------------------------------------------------------- */


PR_WORD PR_GetMaterialMipMap (PR_MATERIAL *mat, PR_UCHAR level);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialMipMap
   Purpose:    Gets the mip mapping material 
   Parameters: mat - Pointer to the material
               level - Mipmap level
   Return:     Mipmap material for the given level
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialMipMapShift (PR_MATERIAL *mat,
                                    PR_UCHAR level);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialMipMapShift
   Purpose:    Gets the mip mapping shift value 
   Parameters: mat - Pointer to the material
               level - Mipmap level
   Return:     Amount to shift texture coordinates for this level
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialMipMapState (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialMipMapState
   Purpose:    Gets the mip mapping material flag 
   Parameters: mat - Pointer to the material
   Return:     State of mipmapping for the material (TRUE or FALSE)
   ----------------------------------------------------------------------- */


void PR_GetMaterialName (PR_MATERIAL *mat, char *name);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialName
   Purpose:    Gets the name of the material 
   Parameters: mat - Pointer to the material
               name - Buffer for name
   Return:
   ----------------------------------------------------------------------- */


PR_DWORD PR_GetMaterialShades (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialShades
   Purpose:    Gets the number of shades of the material 
   Parameters: mat - Pointer to the material
   Return:     Number of shades used
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMaterialTable (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialTable
   Purpose:    Gets the shadetable number of the material 
   Parameters: mat - Pointer to the material
   Return:     Shade table number
   ----------------------------------------------------------------------- */


PR_DWORD PR_GetMaterialTexture (PR_MATERIAL *mat);
/* -----------------------------------------------------------------------
   Function:   PR_GetMaterialTexture
   Purpose:    Gets the texture number of the material 
   Parameters: mat - Pointer to the material
   Return:     Texture number
   ----------------------------------------------------------------------- */


PR_REAL PR_GetMipMapDepth (PR_UCHAR level);
/* -----------------------------------------------------------------------
   Function:   PR_GetMipMapDepth
   Purpose:    Gets the global mip map depth level 
   Parameters: level - Mipmap level
   Return:     The depth for the mipmap level given
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMipMapShrink (void);
/* -----------------------------------------------------------------------
   Function:   PR_GetMipMapShrink
   Purpose:    Gets the global mip map shrink flag 
   Parameters: None
   Return:     Returns the state of mipmap texture shrinking (TRUE or FALSE)
   ----------------------------------------------------------------------- */


PR_UCHAR PR_GetMipMapState (void);
/* -----------------------------------------------------------------------
   Function:   PR_GetMipMapState
   Purpose:    Gets the global mip map flag 
   Parameters: None
   Return:     The global state of mipmapping (TRUE or FALSE)
   ----------------------------------------------------------------------- */


char *PR_ReturnMethodName (PR_DWORD method);
/* -----------------------------------------------------------------------
   Function:   PR_ReturnMethodName
   Purpose:    Returns the rendering name given its number
   Parameters: method - Number of rendering method
   Return:     The name of the rendering method as a string 
   ----------------------------------------------------------------------- */


void PR_SetMaterialAlpha (PR_MATERIAL *mat, PR_UCHAR alpha);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialAlpha
   Purpose:    Sets the alpha value of the material
   Parameters: mat - Pointer to the material
               alpha - Alpha value (0-255)
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialBaseColor (PR_MATERIAL *mat, PR_UCHAR col);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialBaseColor
   Purpose:    Sets the base color of the material
   Parameters: mat - Pointer to the material
               col - Base color for indexed color modes
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialColor (PR_MATERIAL *mat, PR_REAL r, PR_REAL g, PR_REAL b);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialColor
   Purpose:    Sets the RGB color of the material 
   Parameters: mat - Pointer to the material
               r,g,b - RGB components of the material
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialEnvironmentAxis (PR_MATERIAL *mat, PR_UCHAR envaxis);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialEnvironmentAxis
   Purpose:    Sets the environment mapping axis flag of the material 
   Parameters: mat - Pointer to the material
               envaxis - XAXIS, YAXIS, ZAXIS
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialEnvironmentMap (PR_MATERIAL *mat, PR_UCHAR envmap);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialEnvironmentMap
   Purpose:    Sets the environment mapping flag of the material 
   Parameters: mat - Pointer to the material
               envmap - Environment map state - TRUE or FALSE
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialMethod (PR_MATERIAL *mat, PR_DWORD method);
/* -----------------------------------------------------------------------
   Function:   PR_SetMAterialMethod
   Purpose:    Sets the rendering method of the material 
   Parameters: mat - Pointer to the material
               method - Render method number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialMipMap (PR_MATERIAL *mat, PR_UCHAR level,
                           PR_WORD mip);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialMipMap
   Purpose:    Sets the mip mapping material 
   Parameters: mat - Pointer to the material
               level - Mipmap level
               mip - Mipmap material number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialMipMapShift (PR_MATERIAL *mat, PR_UCHAR level,
                                PR_UCHAR shift);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialMipMapShift
   Purpose:    Sets the mip mapping shift value 
   Parameters: mat - Pointer to the material
               level - Mipmap level
               shift - Shift amount
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialMipMapState (PR_MATERIAL *mat, PR_UCHAR mip);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialMipMapState
   Purpose:    Sets the mip mapping material flag 
   Parameters: mat - Pointer to the material
               mip - Mipmap state (TRUE or FALSE)
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialName (PR_MATERIAL *mat, char *name);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialName
   Purpose:    Sets the name of the material
   Parameters: mat - Pointer to the material
               name - New name 
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialShades (PR_MATERIAL *mat, PR_DWORD shades);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialShades
   Purpose:    Sets the number of shades of the material 
   Parameters: mat - Pointer to the material
               shades - Number of shades used in color indexed mode
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialTable (PR_MATERIAL *mat, PR_UCHAR table);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialTable
   Purpose:    Sets the shadetable number of the material
   Parameters: mat - Pointer to the material
               table - Shade table number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMaterialTexture (PR_MATERIAL *mat, PR_DWORD texture);
/* -----------------------------------------------------------------------
   Function:   PR_SetMaterialTexture
   Purpose:    Sets the texture number of the material 
   Parameters: mat - Pointer to the material
               texture - Texture number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMipMapDepth (PR_UCHAR level, PR_REAL depth);
/* -----------------------------------------------------------------------
   Function:   PR_SetMipMapDepth
   Purpose:    Sets the global mip map depth level 
   Parameters: level - Mipmap level
               depth - Z distance where mipmap is used
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMipMapShrink (PR_UCHAR mip);
/* -----------------------------------------------------------------------
   Function:   PR_SetMipMapShrink
   Purpose:    Sets the global mip map shrink flag 
   Parameters: mip - Texure Shrink state (TRUE or FALSE)
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetMipMapState (PR_UCHAR mip);
/* -----------------------------------------------------------------------
   Function:   PR_SetMipMapState
   Purpose:    Sets the global mip map flag 
   Parameters: mip - Global mipmap state (TRUE or FALSE)
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetObjectMaterial (PR_OBJECT *obj, PR_DWORD mat);
/* -----------------------------------------------------------------------
   Function:   PR_SetObjectMaterial
   Purpose:    Sets the material of an entire object 
   Parameters: obj - Pointer to the object
               mat - Material number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetSegmentMaterial (PR_SEGMENT *seg, PR_DWORD mat);
/* -----------------------------------------------------------------------
   Function:   PR_SetSegmentMaterial
   Purpose:    Sets the material of an entire segment 
   Parameters: seg - Pointer to the segment
               mat - Material number
   Return:     None
   ----------------------------------------------------------------------- */


#ifdef __cplusplus
}
#endif

#endif



