/* Power Render Camera Interface */

#ifndef _PRCAM_
#define _PRCAM_

#ifdef __cplusplus
  extern "C" {
#endif

/* Global Variables */
extern PR_CAMERA *PR_CurrentCamera;
extern PR_CAMERA *PR_CameraList;
extern PR_MATRIX PR_ViewMatrix;
extern PR_MATRIX PR_ProjectionMatrix;
extern PR_REAL   PR_ConstantZ;

void PR_AddCamera (PR_CAMERA *newcamera);
/* -----------------------------------------------------------------------
   Function:   PR_AddCamera
   Purpose:    Adds a camera to the world 
   Parameters: newcamera - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


PR_CAMERA *PR_AllocCamera (void);
/* -----------------------------------------------------------------------
   Function:   PR_AllocCamera
   Purpose:    Allocates space for a new camera 
   Parameters: None
   Return:     Pointer to the new camera structure
   ----------------------------------------------------------------------- */


void PR_AnimateCamera (PR_CAMERA *cam, PR_DWORD frame);
/* -----------------------------------------------------------------------
   Function:   PR_AnimateCamera
   Purpose:    Uses keyframe animation
   Parameters: cam - Pointer to the camera
               frame - Frame number
   Return:     None
   ----------------------------------------------------------------------- */


void PR_AttachCameraEntity (PR_CAMERA *cam, PR_ENTITY *ent);
/* -----------------------------------------------------------------------
   Function:   PR_AttachCameraEntity
   Purpose:    Tells which entity a camera should look at with keyframed
               animation.
   Parameters: cam - Pointer to the camera
               ent - Pointer to the entity
   Return:     None
   ----------------------------------------------------------------------- */


void PR_CameraDirection (PR_CAMERA *camdir);
/* -----------------------------------------------------------------------
   Function:   PR_CameraDirection
   Purpose:    Make a normalized vector from source to dest of camera 
   Parameters: camdir - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


void PR_DeleteCamera (PR_CAMERA *delcamera);
/* -----------------------------------------------------------------------
   Function:   PR_DeleteCamera
   Purpose:    Deletes a camera from the world, given its pointer 
   Parameters: delcamera - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


void PR_DollyCamera (PR_CAMERA *setcam, PR_REAL steps);
/* -----------------------------------------------------------------------
   Function:   PR_DollyCamera
   Purpose:    Move a camera in the direction it is facing 
   Parameters: setcam - Pointer to the camera
               steps - Movement multiplier
   Return:     None
   ----------------------------------------------------------------------- */


PR_CAMERA *PR_FindClosestCamera (PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_FindClosestCamera
   Purpose:    Returns a pointer to the camera closest to a coordinate.
   Parameters: x,y,z - Reference location
   Return:     Pointer to the closest camera
   ----------------------------------------------------------------------- */


void PR_FindDirectionVector (PR_POINT *direct,
                        PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_FindDirectionVector
   Purpose:    Finds a direction vector given a rotation value 
   Parameters: direct - Pointer to a normalized direction vector
               x,y,z - Rotation values to rotate vector
   Return:     None
   ----------------------------------------------------------------------- */


void PR_FreeCamera (PR_CAMERA *camera);
/* -----------------------------------------------------------------------
   Function:   PR_FreeCamera
   Purpose:    Frees the space used by a camera structure 
   Parameters: camera - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


PR_CAMERA *PR_GetActiveCamera (void);
/* -----------------------------------------------------------------------
   Function:   PR_GetActiveCamera
   Purpose:    Returns a pointer to the current viewing camera 
   Parameters: None
   Return:     Pointer to the active camera
   ----------------------------------------------------------------------- */


PR_CAMERA *PR_GetCamera (char *name);
/* -----------------------------------------------------------------------
   Function:   PR_GetCamera
   Purpose:    Returns a pointer to the camera, given its name 
   Parameters: name - Name of the camera
   Return:     Pointer to the camera with the name given
   ----------------------------------------------------------------------- */


PR_CAMERA *PR_GetFirstCamera (void);
/* -----------------------------------------------------------------------
   Function:   PR_GetFirstCamera
   Purpose:    Returns a pointer to the first camera in the world
   Parameters: None
   Return:     Pointer to the first camera in the world
   ----------------------------------------------------------------------- */


void PR_InitializeCamera (PR_CAMERA *initcam);
/* -----------------------------------------------------------------------
   Function:   PR_InitializeCamera
   Purpose:    Initializes a camera to some standard values 
   Parameters: initcam - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


void PR_MakeViewMatrix (PR_CAMERA *cam);
/* -----------------------------------------------------------------------
   Function:   PR_MakeViewMatrix
   Purpose:    Makes a matrix for a given camera which will transform
               vertices from world space into view space.
   Parameters: cam - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


void PR_MoveCameraSource (PR_CAMERA *setcam, PR_REAL x, PR_REAL y,
                          PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_MoveCameraSource
   Purpose:    Move a camera's source coordinate 
   Parameters: setcam - Pointer to the camera
               x,y,z - Amount to move in each axis
   Return:     None
   ----------------------------------------------------------------------- */


void PR_MoveCameraTarget (PR_CAMERA *setcam, PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_MoveCameraTarget
   Purpose:    Move a camera's target coordinate 
   Parameters: setcam - Pointer to the camera
               x,y,z - Amount to move in each axis
   Return:     None
   ----------------------------------------------------------------------- */


void PR_PositionCameraSource (PR_CAMERA *setcam, PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_SetCameraSource
   Purpose:    Sets the location of a camera, given its pointer 
   Parameters: setcam - Pointer to the camera
               x,y,z - Location in world coordinates
   Return:     None
   ----------------------------------------------------------------------- */


void PR_PositionCameraTarget (PR_CAMERA *setcam, PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_SetCameraTarget
   Purpose:    Sets the focus of a camera, given its pointer 
   Parameters: setcam - Pointer to the camera
               x,y,z - Location in world coordinates
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetActiveCamera (PR_CAMERA *cam);
/* -----------------------------------------------------------------------
   Function:   PR_SetActiveCamera
   Purpose:    Sets the viewing camera
   Parameters: cam - Pointer to the camera
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetCameraMode (PR_CAMERA *cam, PR_DWORD mode);
/* -----------------------------------------------------------------------
   Function:   PR_SetCameraMode
   Purpose:    Sets the viewing mode for the camera. 
   Parameters: cam - Pointer to the camera
               mode - Camera mode
   Return:     None
   ----------------------------------------------------------------------- */


void PR_SetCameraUp (PR_CAMERA *setcam, PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PR_SetCameraUp
   Purpose:    Sets the up vector of a camera, given its pointer 
   Parameters: setcam - Pointer to the camera
               x,y,z - Normalized up vector
   Return:
   ----------------------------------------------------------------------- */


void PR_SetConstantZ (PR_REAL zval);
/* -----------------------------------------------------------------------
   Function:   PR_SetConstantZ
   Purpose:    Sets the Z value used for 2D primitives
   Parameters: zval - 1 to 65535
   Return:     None
   ----------------------------------------------------------------------- */


void PR_TransformCamera (PR_CAMERA *cam, PR_REAL x, PR_REAL y, PR_REAL z,
                         PR_REAL distance);
/* -----------------------------------------------------------------------
   Function:   PR_TransformCamera
   Purpose:    Transforms the source location of a camera
   Parameters: cam - Pointer to the camera
               x,y,z - Amount to rotate
               distance - Distance from origin 
   Return:     None
   ----------------------------------------------------------------------- */


#ifdef __cplusplus
}
#endif

#endif
