//===================================================================
// test.cpp
//
// Version 1.
//
// Written by:
//   Brent Worden
//   WordenWare
//   email:  Brent@Worden.org
//
// Copyright (c) 1998-1999 WordenWare
//
// Created:  May 17, 1999
// Revised:  
//===================================================================

#include <iostream>
#include <string>
#include <list>

#include "algorthm.hpp"
#include "contdist.h"
#include "correlat.hpp"
#include "descript.hpp"
#include "deviate.h"
#include "discdist.h"
#include "eigensys.hpp"
#include "linalg.hpp"
#include "matrix.hpp"
#include "rootfind.hpp"
#include "vector.hpp"

static const int SIZE = 5;

int randomValue(int n)
{
	return (int)((double)rand() / ((double)RAND_MAX + 1) * n);
}

template<class T>
void initArray(T first, T last)
{
	while(first != last){
		*first = randomValue(SIZE);
		++first;
	}
}
template<class T>
void outputArray(T array[], int n)
{
	std::cout << array[0];
	for(int i = 1; i < n; ++i){
		std::cout << ' ' << array[i];
	}
	std::cout << std::endl;
}

template<class T>
std::ostream& operator<<(std::ostream& os, num::Vector<T>& vec)
{
	for(int i = 0; i < vec.size(); ++i){
		os << vec[i] << ' ';
	}
	os << std::endl;
	return os;
}

template<class T>
std::ostream& operator<<(std::ostream& os, num::Matrix<T>& mat)
{
	for(int i = 0; i < mat.rows(); ++i){
		os << mat[i];
	}
	return os;
}

void testAlgorithm()
{
	double x[SIZE];
	num::Vector<double> y(SIZE);

	initArray(x, x + SIZE);
	initArray(y.begin(), y.end());

	std::cout << "x: "; outputArray(x, SIZE);
	std::cout << "y: " << y;

	std::cout << "average(x[0], x[1]): " << num::average(x[0], x[1]) << std::endl << std::endl;

	double pair[SIZE*SIZE];
	num::pairwdiff(x, x + SIZE, y.begin(), y.end(), pair);
	std::cout << "pairwdiff(x, y): "; outputArray(pair, SIZE*SIZE); std::cout << std::endl;
	
	double walsh[SIZE*(SIZE+1)/2];
	num::walshavg(x, x + SIZE, walsh);
	std::cout << "walshavg(x): "; outputArray(walsh, SIZE*(SIZE+1)/2); std::cout << std::endl;
}

void testContDist()
{
	double p;
	double x;

	p = num::betap(.5, 1, 2); x = num::betav(p, 1, 2);
	std::cout << "betap(.5, 1, 2):     " << p << "\t betav(" << p << ", 1, 2):     " << x << std::endl;

	p = num::cauchyp(.5, 1, 2); x = num::cauchyv(p, 1, 2);
	std::cout << "cauchyp(.5, 1, 2):   " << p << "\t cauchyv(" << p << ", 1, 2):   " << x << std::endl;

	p = num::dblexpp(.5, 1, 2); x = num::dblexpv(p, 1, 2);
	std::cout << "dblexpp(.5, 1, 2):   " << p << "\t dblexpv(" << p << ", 1, 2):   " << x << std::endl;

	p = num::expp(.5, 1); x = num::expv(p, 1);
	std::cout << "expp(.5, 1):         " << p << "\t expv(" << p << ", 1):         " << x << std::endl;

	p = num::gammap(.5, 1, 2); x = num::gammav(p, 1, 2);
	std::cout << "gammap(.5, 1, 2):    " << p << "\t gammav(" << p << ", 1, 2):    " << x << std::endl;

	p = num::logisticp(.5, 1, 2); x = num::logisticv(p, 1, 2);
	std::cout << "logisticp(.5, 1, 2): " << p << "\t logisticv(" << p << ", 1, 2): " << x << std::endl;

	p = num::lognormp(.5, 1, 2); x = num::lognormv(p, 1, 2);
	std::cout << "lognormp(.5, 1, 2):  " << p << "\t lognormv(" << p << ", 1, 2):  " << x << std::endl;

	p = num::rayleighp(.5, 1); x = num::rayleighv(p, 1);
	std::cout << "rayleighp(.5, 1):    " << p << "\t rayleighv(" << p << ", 1):    " << x << std::endl;

	p = num::uniformp(1.5, 1, 2); x = num::uniformv(p, 1, 2);
	std::cout << "uniformp(1.5, 1, 2): " << p << "\t uniformv(" << p << ", 1, 2):  " << x << std::endl;

	p = num::weibullp(.5, 1, 2); x = num::weibullv(p, 1, 2);
	std::cout << "weibullp(.5, 1, 2):  " << p << "\t weibullv(" << p << ", 1, 2):  " << x << std::endl;
}

void testCorrelation()
{
	double x[SIZE];
	num::Vector<double> y(SIZE);

	initArray(x, x + SIZE);
	initArray(y.begin(), y.end());

	std::cout << "x: "; outputArray(x, SIZE);
	std::cout << "y: " << y;

	std::cout << "covariance(x, y):  " << num::covariance(x, x + SIZE, y.begin()) << std::endl;
	std::cout << "correlation(x, y): " << num::correlation(y.begin(), y.end(), x) << std::endl;
	std::cout << "kendall(x, y):     " << num::kendall(x, x + SIZE, y.begin()) << std::endl;
	std::cout << "spearman(x, y):    " << num::spearman(y.begin(), y.end(), x) << std::endl;
}

void testDescript()
{
	double x[SIZE];
	num::Vector<double> y(SIZE);
	std::list<double> z(SIZE);

	initArray(x, x + SIZE);
	initArray(y.begin(), y.end());
	initArray(z.begin(), z.end());

	std::cout << "centmon(x, 3):        " << num::centmom(x, x + SIZE, 3) << std::endl;
	std::cout << "coeffvar(y):          " << num::coeffvar(y.begin(), y.end()) << std::endl;
	std::cout << "geomean(z):           " << num::geomean(z.begin(), z.end()) << std::endl;
	std::cout << "harmean(y):           " << num::harmean(y.begin(), y.end()) << std::endl;
	std::cout << "iqr(x):               " << num::iqr(x, x + SIZE) << std::endl;
	std::cout << "kurtosis(y):          " << num::kurtosis(y.begin(), y.end()) << std::endl;
	std::cout << "mean(z):              " << num::mean(z.begin(), z.end()) << std::endl;
	std::cout << "meandev(y):           " << num::meandev(y.begin(), y.end()) << std::endl;
	std::cout << "meddev(x):            " << num::meddev(x, x + SIZE) << std::endl;
	std::cout << "medain(y):            " << num::median(y.begin(), y.end()) << std::endl;
	std::cout << "quantile(z):          " << num::quantile(z.begin(), z.end(), .1) << std::endl;
	std::cout << "quartile1(y):         " << num::quartile1(y.begin(), y.end()) << std::endl;
	std::cout << "quartile3(x):         " << num::quartile3(x, x + SIZE) << std::endl;
	std::cout << "range(y):             " << num::range(y.begin(), y.end()) << std::endl;
	std::cout << "rms(z):               " << num::rms(z.begin(), z.end()) << std::endl;
	std::cout << "skewness(y):          " << num::skewness(y.begin(), y.end()) << std::endl;
	std::cout << "stddev(x):            " << num::stddev(x, x + SIZE) << std::endl;
	std::cout << "stderrmean(y):        " << num::stderrmean(y.begin(), y.end()) << std::endl;
	std::cout << "sum(x):               " << num::sum(x, x + SIZE) << std::endl;
	std::cout << "sum2(y):              " << num::sum2(y.begin(), y.end()) << std::endl;
	std::cout << "sump(x):              " << num::sump(x, x + SIZE, y.begin()) << std::endl;
	std::cout << "sumsquare(y):         " << num::sumsquare(y.begin(), y.end()) << std::endl;
	std::cout << "trimmean(x, .05):     " << num::trimmean(x, x + SIZE, .05) << std::endl;
	std::cout << "trimmean(y, .05, .1): " << num::trimmean(y.begin(), y.end(), .05, .1) << std::endl;
	std::cout << "variance(z):          " << num::variance(z.begin(), z.end()) << std::endl;
}

void testDeviate()
{
	int n = SIZE;
	int i;
	long seed = 10203040L;

	std::cout << "ran0: ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::ran0(&seed) << ' '; } std::cout << std::endl;
	
	std::cout << "ran1: ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::ran1(&seed) << ' '; } std::cout << std::endl;
	
	std::cout << "ran2: ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::ran2(&seed) << ' '; } std::cout << std::endl;
	
	std::cout << "ran3: ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::ran3(&seed) << ' '; } std::cout << std::endl;
	
	std::cout << "berdev(.5): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::berdev(.5, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "betadev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::betadev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "bnldev(10, .5): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::bnldev(10, .5, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "caudev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::caudev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "chidev(1): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::chidev(1, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "dexpdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::dexpdev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "expdev(1): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::expdev(1, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "fdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::fdev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "gamdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::gamdev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "gasdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::gasdev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "logdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::logdev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "nchidev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::nchidev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "nfdev(1, 2, 3): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::nfdev(1, 2, 3, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "poisdev(1): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::poisdev(1, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "tdev(1): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::tdev(1, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "unifdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::unifdev(1, 2, &seed) << ' '; } std::cout << std::endl;
	
	std::cout << "weibdev(1, 2): ";
	for(i = 0; i < SIZE; ++i){ std::cout << num::weibdev(1, 2, &seed) << ' '; } std::cout << std::endl;
}

void testDiscDist()
{
	double p;
	int x1, x2;

	p = num::binomialp(5, 10, .5); num::binomialv(p, 10, .5, &x1, &x2);
	std::cout << "binomialp(5, 10, .5): " << p <<
		"\t binomialv(" << p << ", 10, .5): " << x1 << ", " << x2 << std::endl;

	p = num::geomp(2, .5); num::geomv(p, .5, &x1, &x2);
	std::cout << "geomp(2, .5):     " << p <<
		"\t geomv(" << p << ", .5):     " << x1 << ", " << x2 << std::endl;

	p = num::hyperp(5, 10, 20, 30); num::hyperv(p, 10, 20, 30, &x1, &x2);
	std::cout << "hyperp(5, 10, 20, 30):    " << p <<
		"\t hyperv(" << p << ", 10, 20, 30):    " << x1 << ", " << x2 << std::endl;

	p = num::negbnlp(5, 10, .5); num::negbnlv(p, 10, .5, &x1, &x2);
	std::cout << "negbnlp(5, 10, .5):   " << p <<
		"\t negbnlv(" << p << ", 10, .5):   " << x1 << ", " << x2 << std::endl;

	p = num::poissonp(5, 10); num::poissonv(p, 10, &x1, &x2);
	std::cout << "poissonp(5, 10):  " << p <<
		"\t poissonv(" << p << ", 10):  " << x1 << ", " << x2 << std::endl;
}

void testEigenSys()
{
	num::Matrix<double> a(SIZE, SIZE);
	num::Vector<double> d;
	num::Vector<double> e;
	int i, j;
	double aij;

	for(i = 0; i < SIZE; ++i){
		for(j = i + 1; j < SIZE; ++j){
			aij = randomValue(SIZE);
			a[i][j] = aij;
			a[j][i] = aij;
		}
		a[i][i] = randomValue(SIZE);
	}

	std::cout << "a: " << std::endl << a << std::endl;
	num::tridred(a, d, e);
	std::cout << "tridred(a):" << std::endl;
	std::cout << "Diagonal: " << d;
	std::cout << "Off Diag: " << e << std::endl;

	num::trideig(d, e);
	std::cout << "trideig(d, e):" << std::endl;
	std::cout << "Eigenvalues: " << d;
}

void testLinAlg()
{
	num::Matrix<double> a(SIZE, SIZE);
	num::Vector<double> p(SIZE);
	num::Matrix<double> aa(SIZE, SIZE);
	int aij;

	for(int i = 0; i < SIZE; ++i){
		for(int j = i + 1; j < SIZE; ++j){
			aij = randomValue(SIZE);
			aa[i][j] = aij;
			aa[j][i] = aij;
		}
		aa[i][i] = randomValue(SIZE);
		p[i] = randomValue(SIZE);
	}

	a = aa;
	std::cout << "a: " << std::endl << a;
	std::cout << "determinant(a): " << num::determinant(a) << std::endl;

	std::cout << "inverse(a):" << std::endl;
	num::inverse(a);
	std::cout << a;

	a = aa;
	std::cout << "linsolve(a, b): " << std::endl;
	std::cout << "b: " << p;
	num::linsolve(a, p, p);
	std::cout << "x: " << p;
}

void testRanking()
{
	num::Vector<double> x(SIZE);
	num::Vector<double> rank(SIZE);
	num::Vector<double> cp(SIZE);

	initArray(cp.begin(), cp.end());

	x = cp;
	std::cout << "x: " << x << std::endl;

	num::index(x.begin(), x.end(), rank.begin());
	std::cout << "index(x): " << rank;

	x = cp;
	num::rank(x.begin(), x.end(), rank.begin());
	std::cout << "rank(x): " << rank;

	x = cp;
	num::arank(x.begin(), x.end(), rank.begin());
	std::cout << "arank(x): " << rank;

	x = cp;
	num::abrank(x.begin(), x.end(), rank.begin());
	std::cout << "abrank(x): " << rank;

	std::cout << "countGreater(x, 5): " << num::countGreater(x.begin(), x.end(), 5) << std::endl;
}

void testResidual()
{
	num::Vector<double> x(SIZE);
	num::Vector<double> cp(SIZE);

	initArray(cp.begin(), cp.end());

	x = cp;
	std::cout << "x: " << x << std::endl;

	num::resid(x.begin(), x.end(), 3.0);
	std::cout << "resid(x, 3): " << x;

	x = cp;
	num::aresid(x.begin(), x.end(), 3.0);
	std::cout << "aresid(x, 3): " << x;

	x = cp;
	num::nresid(x.begin(), x.end(), 5, 1.5);
	std::cout << "nresid(x, 5, 1.5): " << x;
}

double function1(double x)
{
	return sin(x) * cos(x) / exp(x);
}

double function2(double x)
{
	return sin(x) - cos(x) / exp(x);
}

double function2Prime(double x)
{
	return cos(x) + (cos(x) + sin(x)) / exp(x);
}

void testRootfind()
{
	double x = num::bisection(function1, 1.0, 2.0);
	double y = function1(x);

	std::cout << "bisection(function1, 1, 2): " << x << std::endl;
	std::cout << "function1(" << x << "): " << y << std::endl << std::endl;

	x = num::newton(function2, function2Prime, 3.0);
	y = function2(x);
	std::cout << "newton(function2, function2Prime, 3): " << x << std::endl;
	std::cout << "function2(" << x << "): " << y << std::endl << std::endl;

	x = num::secant(function1, 3.0, 4.0);
	std::cout << "secant(function1, 3, 4): " << x << std::endl;
	std::cout << "function1(" << x << "): " << y << std::endl;
}

void testSort()
{
	num::Vector<double> x(SIZE * 10);
	num::Vector<double> cpx(SIZE * 10);
	num::Vector<int> y(SIZE * 10);
	num::Vector<int> cpy(SIZE * 10);

	initArray(cpx.begin(), cpx.end());
	initArray(cpy.begin(), cpy.end());

	x = cpx;
	y = cpy;
	std::cout << "x: " << x << "y: " << y << std::endl;

	num::isort(x.begin(), x.end());
	std::cout << "isort(x): " << std::endl << x << std::endl;

	x = cpx;
	num::isort2(x.begin(), x.end(), y.begin());
	std::cout << "isort2(x, y): " << std::endl << x << y << std::endl;

	x = cpx;
	num::qsort(x.begin(), x.end());
	std::cout << "qsort(x): " << std::endl << x << std::endl;

	x = cpx;
	y = cpy;
	num::qsort2(x.begin(), x.end(), y.begin());
	std::cout << "qsort2(x, y): " << std::endl << x << y << std::endl;

	x = cpx;
	y = cpy;
	num::qsort2key(x.begin(), x.end(), y.begin());
	std::cout << "qsort2key(x, y): " << std::endl << x << y << std::endl;
}

int main()
{
	srand(10203040L);

	std::cout << "Testing Algorithms:" << std::endl;
	testAlgorithm();

	std::cout << std::endl << std::endl << "Testing Continuous Distributions:" << std::endl;
	testContDist();

	std::cout << std::endl << std::endl << "Testing Correlation:" << std::endl;
	testCorrelation();

	std::cout << std::endl << std::endl << "Testing Descriptive Statistics:" << std::endl;
	testDescript();

	std::cout << std::endl << std::endl << "Testing Deviate:" << std::endl;
	testDeviate();

	std::cout << std::endl << std::endl << "Testing Discrete Distributions:" << std::endl;
	testDiscDist();

	std::cout << std::endl << std::endl << "Testing Eigen Systems:" << std::endl;
	testEigenSys();

	std::cout << std::endl << std::endl << "Testing Linear Algebra:" << std::endl;
	testLinAlg();

	std::cout << std::endl << std::endl << "Testing Ranking:" << std::endl;
	testRanking();

	std::cout << std::endl << std::endl << "Testing Residuals:" << std::endl;
	testResidual();

	std::cout << std::endl << std::endl << "Testing Rootfind:" << std::endl;
	testRootfind();

	std::cout << std::endl << std::endl << "Testing Sorting:" << std::endl;
	testSort();

	return 0;
}

//===================================================================
// Revision History
//
// Version 1.0 - 05/17/1999 - New.
//===================================================================
