//--------------------------------------------------------------------------
//
//      EXAMPLE3.CPP: example program 3 for DOS coroutine library.
//      Copyright (c) J.English 1993.
//      Author's address: je@unix.brighton.ac.uk
//
//      Permission is granted to use copy and distribute the
//      information contained in this file provided that this
//      copyright notice is retained intact and that any software
//      or other document incorporating this file or parts thereof
//      makes the source code for the library of which this file
//      is a part freely available.
//
//--------------------------------------------------------------------------

#include <iostream.h>
#include <bios.h>
#include "coroutine.h"

#define ESC     0x1B        // code for the ESC key

//--------------------------------------------------------------------------
//
//      Class Example3.
//
//      This is a coroutine which displays a series of messages.  It
//      illustrates how "terminate" can be used as well as the use of
//      "wait" in a coroutine destructor.
//
class Example3 : public Coroutine
{
  public:
    Example3 (int n)        { num = n; }
    ~Example3 ();

  protected:
    virtual void main ();   // code to be executed by coroutine

  private:
    int num;                // coroutine identification number
};

static int quit = 0;        // flag to request coroutine termination

//--------------------------------------------------------------------------
//
//      Example3::~Example3.
//
//      The coroutine class destructor.  Note how "wait" is called at the
//      beginning of the destructor to ensure the coroutine has terminated
//      before any other finalisation is done.
//
Example3::~Example3 ()
{
    wait ();
    cout << "\nCoroutine E" << num << " finished\n";
}

//--------------------------------------------------------------------------
//
//      Example3::main.
//
//      This is the code executed by each instance of class Example3.
//      It displays a startup message, executes a loop (pausing each
//      time) until the main program sets the "quit" flag, and then
//      displays a termination message before exiting.
//
void Example3::main ()
{
    cout << "\nCoroutine E" << num << " started\n";

    while (!quit)
    {   cout << num << " ";
        pause ();
    }
}

//--------------------------------------------------------------------------
//
//      The main program.
//
//      This creates three instances of class Example3, starts them
//      running and then waits until ESC is pressed.  It then sets
//      the "quit" flag to signal the coroutines to terminate before
//      exiting.  If 1, 2 or 3 is pressed, the corresponding coroutine
//      is immediately terminated.  Note that the loop which waits for
//      a key to be pressed calls "pause" each time to allow the coroutines
//      to execute, and that it uses bioskey(1) (test if a key has been
//      pressed) rather than just bioskey(0) which would wait for a key
//      to be pressed and would not allow the coroutines to run between
//      keypresses.
//
void main ()
{
    Example3 e1 (1), e2 (2), e3 (3);

    //--- announce startup
    cout << "Press ESC to exit, 1 or 2 or 3 to terminate the corresponding\n"
         << "coroutine.  Press any key to start.\n";
    bioskey (0);

    //--- start the coroutines running
    if (!e1.run ())
        cout << "Couldn't start e1\n";
    if (!e2.run ())
        cout << "Couldn't start e2\n";
    if (!e3.run ())
        cout << "Couldn't start e3\n";

    //--- wait for a suitable key to be pressed
    for (;;)
    {   while (bioskey (1) == 0)
            Example3::pause ();
        switch (bioskey (0) & 0xFF)
        {
            case ESC:
                quit = 1;
                return;
            case '1':
                e1.terminate ();
                break;
            case '2':
                e2.terminate ();
                break;
            case '3':
                e3.terminate ();
                break;
        }
    }
}   //--- destructors called here: wait for coroutines to finish, then exit

