/*
    getargs - command line argument processor for C programs.

    Original Author: Allen Hollub
    Modified By:     Virgilio So
    
    History:

    05  1985 - published in Dr. Dobbs's Journal #103
    05191985 - transcribed by James R. Van Zandt
    03102004 - added new features

    Future:
             - Add float functionality
             - Revise the - or / char format to string
             - Put back the procedure fuinctionality
             - ???
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "getargs.h"

/* PRIVATE GLOBAL VARIABLES */
static char arg_switcher = ARG_DASH;
int arg_options = 0;

/* PRIVATE FUNCTION PROTOTYPES */
int validate_argument(int type, char *input);
void set_arguments(ARGUMENTS *argp, char *input);
int is_number(char *number);
int stoi(  register char **instr);
int has_next_argument(int type);
static ARGUMENTS *find_arguments(char *flag_name, int tab_size, ARGUMENTS *tabp);
char *get_flag_name(char *flag);
void parameter_usage(ARGUMENTS *settings, int size, char *program_name);
int get_arguments(int argc, char **argv, ARGUMENTS *settings, int size, ARG_OPTIONS *chain);


/* PRIVATE FUNCTION DEFINITIONS */

void parameter_usage(ARGUMENTS *settings, int size, char *program_name)
{
    int index;

    if((arg_options & SHOW_FLAGS)== SHOW_FLAGS)
    {
        fprintf(stderr, "\nUsage: %s [OPTION]...\n\n", program_name);
        for(index = 0; index < size; index++)
        {
            switch(settings->type)
            {
                case INTEGER:   fprintf(stderr, " %c%-20s %s\n", arg_switcher,
                                        settings->flag,
                                        settings->description); 
                                break;
                case BOOLEAN:   fprintf(stderr, " %c%-20s %s\n", arg_switcher,
                                        settings->flag,
                                        settings->description); 
                                break;
                case CHARACTER: fprintf(stderr, " %c%-20s %s\n", arg_switcher,
                                        settings->flag,
                                        settings->description); 
                                break;
                case STRING:   fprintf(stderr, " %c%-20s %s\n", arg_switcher,
                                       settings->flag,
                                       settings->description); 
                               break;
            }
            settings++;
        }
    }
    printf("\n");
}


char *get_flag_name(char *flag)
{
    int length;
    int index;

    length = strlen(flag);
    for(index = 0; index < length; index++)
    {
        if(isspace(flag[index]))
        {
            return &flag[index];
        }
    }
    return flag;
}

static ARGUMENTS *find_arguments(char *flag_name, 
                                 int tab_size,
                                 ARGUMENTS *tabp)
{
    while(tab_size > 0)
    {
        if(strcmp(flag_name, tabp->flag) == 0)
        {
            return tabp;
        }
        tabp++;
        tab_size--;
    }
    return 0;
}

int has_next_argument(int type)
{
    if((type == INTEGER) || (type == CHARACTER) || (type == STRING))
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

int stoi(  register char **instr)
{    /*    Convert string to integer.  If string starts with 0x it is
           interpreted as a hex number, else if it starts with a 0 it
           is octal, else it is decimal.  Conversion stops on encountering
           the first character which is not a digit in the indicated
           radix.  *instr is updated to point past the end of the number.    */

    register int num=0;
    register char *str;
    int sign=1;

    str=*instr;
    while(*str==' ' || *str=='\t' || *str=='\n')
    {
        str++;
    }
    if(*str=='-')
    {
        sign=-1; str++;
    }
    if(*str=='0')
    {
        ++str;
        if(*str=='x' || *str=='X')
        {
            str++;
            while(('0'<=*str && *str<='9') ||
                  ('a'<=*str && *str<='f') ||
                  ('A'<=*str && *str<='F'))
            {
                num = num*16+ (('0'<=*str && *str<='9') ?
                      *str-'0' : 
                      toupper(*str)-'A'+10);
                str++;
            }
        }
        else 
        {
            while('0'<=*str && *str<='7') 
            {
                num = num*8+ *str++ - '0';
            }
        }
    }
    else 
    {
        while('0'<=*str && *str<='9') 
        {
            num = num*10+ *str++-'0';
        }
    }
    *instr = str;
    return (num * sign);
}

int is_number(char *number)
{
    int index;
    int length;

    length = strlen(number);

    for(index = 0; index < length; index++)
    {
        if(!isdigit(number[index]))
        {
            return 0;
        }
    }
    return 1;
}

void set_arguments(ARGUMENTS *argp, char *input)
{
    switch(argp->type)
    {
        case INTEGER: *(int *)argp->variable = stoi(&input);
                      break;
        case BOOLEAN:   *(int *)argp->variable = 1; 
                        break;
                        
        case CHARACTER: *(char *)argp->variable = input[0]; 
                        break;
        case STRING:   // *(char **)argp->variable = input; 
                        strcpy(argp->variable, input);
                        input = ""; 
                        break;
    }
}

int validate_argument(int type, char *input)
{
    int status = 1;
    
    switch(type)
    {
        case INTEGER:   
            status = is_number(input);
            break;
        case BOOLEAN:   break;
        case CHARACTER: 
            if(strlen(input) > 1)
            {
                status = 0;
            }
    }
    return status;
}

void check_chain(char **argv, int current_index, int last_index)
{
    int index;

    if((arg_options & CHAIN_STR) == CHAIN_STR)
    {
        for(index = current_index; index < last_index; index++)
        {
            /* 
               BUG NOTE: add checking of chains here
             */
        }
    }
}

int get_arguments(int argc,
                  char **argv,
                  ARGUMENTS *settings,
                  int size,
                  ARG_OPTIONS *chain)
{
    int index;
    char *program_name;
    char *flag_name;
    int flag = 1;
    ARGUMENTS *argp;
    int has_flags = 0;

    if((arg_options & CHAIN_STR) == CHAIN_STR)
    {
        if(chain == NULL)
        {
            return 8;
        }
        chain->end = -1;
    }
    program_name = get_program_name(argv[0]);
    /* exits if there were no arguments passed */
    if(argc == 1)
    {
        parameter_usage(settings, size, program_name);
        return 1;
    }

    for(index = 1; index < argc; index++)
    {
        if(flag == 1)
        {
            if(argv[index][0] != arg_switcher)
            {
                if((arg_options & CHAIN_STR) == CHAIN_STR)
                {
                    if(has_flags == 0)
                    {
                        return 2;
                    }
                    chain->start = index;
                    chain->end = argc;
                    check_chain(argv, index, argc);
                    return 0;
                }
                else
                {
                    parameter_usage(settings, size, program_name);
                    return 3;
                }
            }
            else
            {
                has_flags = 1;
                flag_name = get_flag_name(&argv[index][1]);
#ifdef __DEBUG__                
                printf("DEBUG: flag_name = %s\n", flag_name);
#endif                
                argp = find_arguments(flag_name, size, settings);
                if(argp != NULL)
                {
                    if(has_next_argument(argp->type) == 1)
                    {
                        flag = 0;
                    }
                    else
                    {
                        /* BUG NOTE: 2nd argument is useless if boolean */
                        set_arguments(argp, argv[index]);
                    }
                }
                else
                {
                    parameter_usage(settings, size, program_name);
                    return 3;
                }
            }
        }
        else
        {
            if(validate_argument(argp->type, argv[index]))
            {
                set_arguments(argp, argv[index]);
            }
            else
            {
                parameter_usage(settings, size, program_name);
                return 5;
            }
            flag = 1;
        }
    }
    if(flag == 0)
    {
        parameter_usage(settings, size, program_name);
        return 6;
    }
    if((arg_options & CHAIN_STR) == CHAIN_STR)
    {
        if(chain->end == -1)
        {
            parameter_usage(settings, size, program_name);
            return 7;
        }
    }
    return 0;
}

/* PUBLIC FUNCTION DEFINITIONS */

int init_arg(int argc, char **argv, ARGUMENTS *settings, int size, char switcher, ARG_OPTIONS *chain)
{
    int status = 0;
    
    arg_switcher = switcher;
    if(chain != NULL)
    {
        arg_options = chain->option;
    }
    status = get_arguments(argc, argv, settings, size, chain);

    return status;
}

char *arg_error(int error_number)
{
    int max_error_number = 9;
    
    char *arg_error_msg[] = {"", 
                              "Invalid number of arguments", /* 1 */
                              "Needed argument missing",     /* 2 */
                              "Invalid argument",            /* 3 */
                              "Internal error",              /* 4 */
                              "Pair invalid",                /* 5 */
                              "No pair found",               /* 6 */
                              "Other arguments missing",     /* 7 */
                              "ARG_OPTIONS is not initialized",/* 8 */
                              "Unknown error"                /* 9 */
    };

    
    if((error_number >= 1) && (error_number <= max_error_number))
    {
        return arg_error_msg[error_number];
    }
    if(error_number > max_error_number)
    {
        return arg_error_msg[max_error_number];
    }
    return "";
}

char *get_program_name(char *argv)
{
    int length;
    int index;

    length = strlen(argv);
    for(index = 0; index < length; index++)
    {
        if(isalpha(argv[index]))
        {
            return &argv[index];
        }
    }
    return argv;
}
