/*-----------------------------------------------------------------------*
 *   File: datelib.h							 *
 *   Date: 1-26-97							 *
 *   Author: Brady Tippit						 *
 *   Compiler: Borland (C++ 4.5, using small memory model)		 *
 *  									 *
 *   References:  Peter Meyer, "Julian and Gregorian Calendars."	 *
 *                Dr. Dobb's Journal, March 1993.			 *
 *               							 *
 *									 *
 *-----------------------------------------------------------------------*/

#ifndef DATELIB_H
#define DATELIB_H

typedef struct td {
  int td_year;
  int td_mon;
  int td_mday;
  int td_hour;
  int td_min;
  int td_sec;   
}tdtype;




/*-------------[ Prototypes for the date and time routines ]--------------*/ 





/*-----------------------------------------------------------------------*
 *  Function: TimeDate()						 *
 *									 *
 *  Creates a date/time structure and initializes all fields with the	 *
 *  current system date AND time info.					 *
 *									 *
 *  Parameters: none.							 *
 *  Returns: a pointer to the date/time structure.			 *
 *-----------------------------------------------------------------------*/

tdtype * TimeDate(void);




/*-----------------------------------------------------------------------*
 *  Function: CreateDate()						 *
 *									 *
 *  Dynamically allocates a td (time/date) structure.			 *
 *									 *
 *  NOTE: Neither this function nor LoadDate() are needed if all	 *
 *  you want to work with is the current date/time.			 *
 *  If this function is used, call LoadDate() to initialize it's         *
 *  fields with the month, day, and year you want.			 *
 *  Call FreeDate() to return the memory back to the system.             *
 *                                                                       *
 *  Parameters: none.							 *
 *  Returns: a pointer to the newly allocated td structure.		 *
 *-----------------------------------------------------------------------*/

tdtype * CreateDate(void);




/*-----------------------------------------------------------------------*
 *  Function: LoadDate()						 *
 *									 *
 *  Loads a previously created td structure with a user specified year,	 *
 *  month, and day.							 *
 *  NOTE: All functions expect months from 1-12, years from 1-32766, and *
 *  days from 1-31.							 *
 *                                                                       *
 *  Parameters: pointer to the td struct, integer dates to load into the *
 *  struct.						                 *
 *  Returns: -1 on invalid date, 0 on success. 				 *
 *-----------------------------------------------------------------------*/

int LoadDate(tdtype *, int, int, int);




/*-----------------------------------------------------------------------*
 *  Function: FreeDate()						 *
 *									 *
 *  Makes more memory available to the system by freeing the memory	 *
 *  that was allocated by the user calling CreateDate(). 		 *
 *  The pointer/s in the calling environment are set to NULL and should  *
 *  not be used again without first calling CreateDate() again. duh!     *
 *       								 *
 *  Parameters: a pointer to the address of the previously 	         *
 *  allocated td structure.						 *
 *  Returns: nothing.							 *
 *-----------------------------------------------------------------------*/

void FreeDate(tdtype **);




/*-----------------------------------------------------------------------*
 *  Function: CreateTime()						 *
 *									 *
 *  Dynamically allocates a td time structure.			         *
 *									 *
 *  NOTE: Neither this function nor LoadTime() are needed if all	 *
 *  you want to work with is the current time.                           *
 *  ( Just call TimeDate() )					         *
 *									 *
 *  If this function is used, call LoadTime() to initialize it's         *
 *  fields with the hours, mins, and secs you want.			 *
 *  Call FreeTime() to return the memory back to the system.             *
 *                                                                       *
 *  Parameters: none.							 *
 *  Returns: a pointer to the newly allocated td structure, which is now *
 *  ready to be filled with the time info. that you supply.              *
 *-----------------------------------------------------------------------*/

tdtype * CreateTime(void);




/*-----------------------------------------------------------------------*
 *  Function: LoadTime()						 *
 *									 *
 *  Initializes the hour, min, and sec fields of a td time structure.	 *
 *  Used after calling CreateTime().					 *
 *                                                                       *
 *  Parameters: pointer to a td structure, hours, mins, secs.		 *
 *  Returns: 0 on success, -1 on invalid time.                           *
 *-----------------------------------------------------------------------*/

int LoadTime(tdtype *, int, int, int);




/*-----------------------------------------------------------------------*
 *  Function: FreeTime()						 *
 *									 *
 *  Makes more memory available to the system by freeing the memory	 *
 *  that was allocated by the user calling CreateTime(). 		 *
 *  The pointer/s in the calling environment are set to NULL and should  *
 *  not be used again without first calling CreateTime() again.          *
 *       								 *
 *  Parameters: a pointer to the address of the previously allocated td  *
 *  structure.						                 *
 *  Returns: nothing.							 *
 *-----------------------------------------------------------------------*/

void FreeTime(tdtype **);




/*-----------------------------------------------------------------------*
 *  Function: IsDateValid()						 *
 *            								 *
 *  Checks for valid dates. Earliest date allowed is Jan 1, 1 A.D.	 *
 *  Maximum date is Dec. 31, 32766                                       *
 *  This is done automatically but CAN be called.                        *
 *                                                                       *
 *  Parameters: pointer to the td struct.			         *
 *  Returns: 0 if date is not valid, otherwise 1.			 *
 *-----------------------------------------------------------------------*/

int IsDateValid(tdtype *);




/*-----------------------------------------------------------------------*
 *  Function: IsTimeValid()						 *
 *									 *
 *  Checks the time fields for validity.			         *
 *  This is done automatically but CAN be called.                        *
 *                                                                       *
 *  Returns: 1 if valid, otherwise 0.					 *
 *-----------------------------------------------------------------------*/

int IsTimeValid(tdtype *);




/*-----------------------------------------------------------------------*
 *  Function: DateDiff()						 *
 *									 *
 *  Calculates the difference, in days, between two given dates. Two td	 *
 *  structures are used.						 *
 *  									 *
 *  Parameters: pointers to the td structures.		                 *
 *  Returns: the number of days from date1 to date2. A negative result	 *
 *  indicates a decrease in the number of days.				 *
 *  (No error check, you can check this yourself.)			 *
 *-----------------------------------------------------------------------*/

long DateDiff(tdtype *, tdtype *);




/*-----------------------------------------------------------------------*
 *  Function: DateToDays()						 *
 *									 *
 *  Calculates the number of days from Jan 1, 1 A.D. to the date found	 *
 *  in the td structure.						 *
 *									 *
 *  Parameters: pointer to the td struct.			         *
 *  Returns: the number of days or -1 if date not valid.	         *
 *-----------------------------------------------------------------------*/

long DateToDays(tdtype *);





/*-----------------------------------------------------------------------*
 *  Function: DaysToDate()						 *
 *									 *
 *  Given some number of days from Jan 1, 1 A.D. , this function 	 *
 *  calculates a year, month, and day-of-month for that many days	 *
 *  and loads the td struture with that year, month, and                 *
 *  day-of-the-month.						         *
 *									 *
 *  Parameters: pointer to the td struct,			         *
 *             the number of days since Jan 1, 1 A.D.			 *
 *  Returns: -1 on error, 0 on success.					 *
 *-----------------------------------------------------------------------*/

int DaysToDate(tdtype *, long int);




/*-----------------------------------------------------------------------*
 *  Function: DayOfWeek()						 *
 *									 *
 *  This function expects the YEAR, MONTH AND DAY fields in the          *
 *  td structure to be the actual dates you want, NOT year - 1900! etc.  *
 *									 *
 *  Parameters: pointer to the td struct.			         *
 *  Returns:  -1 on error, otherwise the day of the week, (0-6).	 *
 *-----------------------------------------------------------------------*/

int DayOfWeek(tdtype *);




/*-----------------------------------------------------------------------*
 *  Function: DayOfYear()						 *
 *									 *
 *  This function calculates the day of any given year starting from	 *
 *  Jan. 1st of that year. The td structure should be loaded normally	 *
 *  using LoadDate() ( or TimeDate() ) with a 1-12 month, 1-31 month-day *
 *  and the full year desired.						 *
 *									 *
 *  Parameters: pointer to the td struct.			         *
 *  Returns: -1 on error or the day of the year from Jan 1.		 *
 *-----------------------------------------------------------------------*/

int DayOfYear(tdtype *);




/*-----------------------------------------------------------------------*
 *  Function: IsLeapYear()						 *
 *									 *
 *  Determines whether a year is a leap year or not.			 *
 *  This is done automatically but CAN be called.                        *
 *									 *
 *  Parameters: the year to check.				         *
 *  Returns: 0 if not a leap year, otherwise 1.			         *
 *-----------------------------------------------------------------------*/

int IsLeapYear(int);




/*-----------------------------------------------------------------------*
 *  Function: TimeDiff()						 *
 *									 *
 *  Calculates the difference, in seconds, between two calendar dates	 *
 *  or times.								 *
 *									 *
 *  The advantage this function has over the ANSI difftime() is that	 *
 *  this function works for any date from 1 A.D.			 *
 *  And the same is true for my DateDiff() function.			 *
 *									 *
 *  Parameters: pointers to the two td structures.	                 *
 *									 *
 *  Returns: The difference, in seconds, between two dates or times.	 *
 *  If the seconds returned are negative, this reflects a decrease in	 *
 *  in time and positive for an increase in time.			 *
 *  The times or dates must be valid for the result to be valid.	 *
 *  No error is returned.						 *
 *-----------------------------------------------------------------------*/

double TimeDiff(tdtype *, tdtype *);




/*-----------------------------------------------------------------------*
 *  Function: PrintCalendar()						 *
 *									 *
 *  Prints a calendar month for any year from Jan 1, 1 A.D. to 32766 	 *
 *  Parameters: month and year of calendar to print.	                 *
 *  Returns: nothing.							 *
 *-----------------------------------------------------------------------*/

void PrintCalendar(int, int);





/*-----------------------------------------------------------------------*
 *  Function: TDFormat()   (see howto.txt for format options)            *
 *									 *
 *  Converts dates to various formats. For the "%Y" option, the year	 *
 *  should be entered in the precise form you desire, ie, 1996 for 1996	 *
 *  and 96 for the year 96 A.D. The upper limit for full year formatting *
 *  is the year 32,766 (the maximum integer size - 1 on 2 byte systems). *
 *									 *
 *  Unlike the hindered ANSI strftime(), which only works for dates from *
 *  1900 or later, this function works for any date from Jan 1, 1 A.D.!	 *
 *									 *
 *  All format options are ANSI compliant with the exception of the two	 *
 *  format options %r and %n, which I use most often.			 *
 *  None of the format options are THE ANSI format options.		 *
 *									 *
 *  Parameters: a pointer to the td structure,			         *
 *              a pointer to the output string,				 *
 *              the size of the output string (800 bytes max),		 *
 *              the format of the output string.			 *
 *									 *
 *  Returns: -1 if out of memory, or the number of characters formatted, *
 *  or 0 if all of the characters could not be formatted, or an error	 *
 *  message if the user specifies an illegal format option.		 *
 *-----------------------------------------------------------------------*/

int TDFormat(tdtype *, char *, int, char *);


#endif

