; COM_PAX2.ASM
; Communications Package for the IBM PC, XT, AT, Clones
; Extensively modified for DLX -- not for general use
; 32 line version -- both com/8 and mc/8

; port1	port2
; 3f8	2f8	com1=line1  /  com2=line2
; 3f8	other	com1=line1  /  com2 com/x lines 2-9
; other	2f8	com1 com/x lines 1-8  /  com2=line9
; other	other	com1 com/x lines 1-8,17-24  /  com2 com/x lines 9-16,25-32

; DLX Bulletin Board System V7.0

; FREEWARE NOTICE

; DLX V7.0 is placed in the public domain by its author, Richard Gillmann.
; Anyone who wishes to may run the program, copy it, or modify it for
; any purpose, including commercial gain.

; o Special features for BBSes
; o Uses MS Pascal calling conventions
; o Supports all serial ports simultaneously
; o All speeds to 19200 baud
; o Compatible with PC, XT, AT (not PCjr or PS/2 alas)
; o Built in XON/XOFF flow control
; o I/O buffers not in DGROUP
 
; boolean constants
true		equ	1		; what MS Pascal
false		equ	0		; believes
; params
wrapcol		equ	74		; hard wrap at this column
nlines		equ	32		; number of lines
; maximum buffer sizes
r_size		equ     360		; size of receive buffers
s_size		equ     180		; size of transmit buffers
; ascii
bs		equ	8		; backspace
tab		equ	9		; tab
lf		equ	10		; linefeed
cr		equ	13		; carriage return
del		equ	127		; del
; interrupt numbers
int_com1 	equ	0ch		; com1: from 8259
int_com2 	equ	0bh		; com2: from 8259
; 8259 ports
inta00  	equ     20h		; 8259a port, a0 = 0
inta01  	equ     21h		; 8259a port, a0 = 1
; com1: level 4
irq4		equ	2*2*2*2		; 8259a ocw1 mask, m4=1, a0=0
nirq4		equ	not irq4 and 0ffh ; complement of above
eoi4		equ	4 or 01100000b	; 8259a ocw2 specific irq4 eoi, a0=0
; com2: level 3
irq3		equ	2*2*2		; 8259a ocw1 mask, m3=1, a0=0
nirq3		equ	not irq3 and 0ffh ; complement of above
eoi3		equ	3 or 01100000b	; 8259a ocw2 specific irq3 eoi, a0=0
; flow control characters
control_q 	equ	11h		; xon
control_s 	equ	13h		; xoff
; misc.
control_c 	equ	3h		; control c
control_o 	equ	0fh		; control o
dos		equ	21h		; dos function calls

pushem		macro
		push	bp
		mov	bp,sp
		push	ds
		mov	ax,data2
		mov	es,ax
		endm

popem		macro
		pop	ds
		pop	bp
		endm

;
; rom bios data area
;
rbda		segment	at 40h
		org	6ch
timer_low	dw	?		; low word of bios timer count
rbda		ends

;
; table for each serial port
;
sp_tab		struc
port		db	?		; 1..nlines
; buffer displacements
di_tdata	dw	?		; offset to our part of tdata
di_rdata	dw	?		; offset to our part of rdata
; attributes
baud_rate	dw	?		; 300 to 9600 baud ok
parity		db	?		; n(one) o(dd) e(ven) s(pace) m(ark)
stop_bits	db	?		; 1, 2
echo_state	db	?		; 1=echo 0=no echo
chars_this_line	db	?		; chars on this line so far
woscol		db	?		; wrap on space column
; flow control state
host_off	db	?		; host xoffed (1=yes,0=no)
pc_off		db	?		; pc xoffed (1=yes,0=no)
cancelled	db	?		; flag if control c seen
skipped		db	?		; flag if control o seen
timeout		dw	?		; timeout clock
warnout		db	?		; warning flag
binary		db	?		; xmodem flag
soft_cr		db	?		; soft carriage return (1=yes,0=no)
; 8250 ports
datreg		dw	?		; data register
ier		dw	?		; interrupt enable register
iir		dw      ?		; interrupt identification register
lcr		dw      ?		; line control register
mcr		dw      ?		; modem control register
lsr		dw      ?		; line status register
msr		dw      ?		; modem status register
; buffer pointers
start_tdata     dw      ?       	; index to 1st character in xmit buf
end_tdata       dw      ?       	; index to 1st free space in xmit buf
start_rdata     dw      ?       	; index to 1st character in rec buf
end_rdata       dw      ?		; index to 1st free space in rec buf
; buffer counts
size_tdata      dw      ?		; number of characters in x-mit buffer
size_rdata      dw      ?       	; number of characters in rec. buffer
sp_tab		ends
; sp_tab equates
; we have to use these because of problems with struc
dll		equ	datreg		; low divisor latch
dlh		equ	ier		; high divisor latch

data2	segment	public 'DATA'
; i/o buffers (not in DGROUP)
tdata		db      nlines*s_size dup(?) ; send buffers
rdata		db	nlines*r_size dup(?) ; receive buffers
data2	ends

	extrn	to_warn:word,to_kill:word,hostecho:byte,dcd_ok:byte
	extrn	ps2mc8:byte
	public	com1_port,com2_port,status_port1,status_port2,bits8,eskape
data    segment public 'DATA'
com1_port	dw	3f8h		; port address / int4
com2_port	dw	2f8h		; port address / int3
status_port1	dw	140h		; digiboard status / int4
status_port2	dw	140h		; digiboard status / int3
old_com_off1	dw	?		; old handler's offset level 4
old_com_seg1	dw	?		; old handler's segment level 4
old_com_off2	dw	?		; old handler's offset level 3
old_com_seg2	dw	?		; old handler's segment level 3
temp		dw	0		; temp used by echo_com
bpp		db	8		; bytes per port in i/o address space
bits8		db	7fh		; mask applied to incoming characters
div50		dw	2304		; pc/xt/at divisor for 50 baud
area_length	dw	type sp_tab	; length of param block
current_area	dw	area1		; currently selected area
channel		db	0		; current mc/8 channel

; data areas for each port
atabl	macro	x
area&x	sp_tab	<x,(x-1)*s_size,(x-1)*r_size>
	endm

y	=	0
	rept	nlines
y	=	y + 1
	atabl	%y
	endm

; filter for incoming characters (0 means ignore them)
; disallows certain characters, also strips parity
filter	db	3 dup(0)
	db	7fh		; control-c
	db	4 dup(0)
	db	7fh		; bs
	db	7fh		; tab
	db	7fh		; lf
	db	2 dup(0)
	db	7fh		; cr
	db	7fh		; double note for music
	db	7fh		; control-o
	db	0
	db	7fh		; control-q
	db	0
	db	7fh		; control-s
	db	7 dup(0)
eskape	db	0		; escape
	db	4 dup(0)
	db	95 dup(7fh)	; printable ascii
	db	7fh		; del
	db	128 dup(0ffh)	; upper chars for europe
tabsiz	db	8		; tab spacing
data    ends
dgroup	group	data

auxhndlr segment 'CODE'
	assume  cs:auxhndlr,ds:dgroup,ss:dgroup,es:data2
; 8259
	public	save_com	; save old interrupt vector
	public  installc	; install our interrupt vector
	public  restorec	; restore old interrupt vector
; 8250
	public	select_port	; select communications port
	public  open_com	; initialize port
	public  close_com       ; turn off interrupts from com port
	public  dtr_off		; turn off dtr
	public  dtr_on		; turn on dtr
	public	rlsd		; return rlsd (still on-line if true)
	public	warning		; return true if time to warn of timeout
	public	unwarn		; cancel timeout warning
	public  r_count		; return number of characters in input buffer
	public  receive		; read next character in input buffer
	public  s_free		; return no. of free bytes in output buffer
	public  s_working	; return no. of chars waiting in output buffer
	public  send		; write a character to output buffer
	public	echo_com	; echo to sender again
	public	purge_com	; empty buffers
	public	unskip_com	; stop skipping
	public	cancelled_com	; have we been control c-ed?
	public	force_cancel	; simulate control-C
	public	skipped_com	; have we been control o-ed?
	public	wrap		; set wrap column
	public	binary_mode	; toggle binary mode
	public	was_soft	; was that a soft carriage return?
	page
;
; internal buffer routines
; at this point, si is already set up
;   gst = get from start of tdata (to al)
;   gsr = get from start of rdata (to al)
;   pet = put (from al) to end of tdata (preserves al)
;   per = put (from al) to end of rdata (preserves al)
;   pst = put (from al) to start of tdata (preserves al)
; to be safe, call these with interrupts off
;
gst	proc	near			; *call with interrupts off*
	mov	al,0			; in case of underflow
	cmp	size_tdata[si],0	; any characters?
	je	gst_x			; jump if not
	mov     bx,start_tdata[si]	; bx points to next char. to be sent
	mov	di,di_tdata[si]
	mov     al,tdata[di][bx]	; get data from buffer
	inc     bx			; increment start_tdata
	cmp     bx,s_size		; see if gone past end
	jb      gst_1			; if not then skip
	mov	bx,0			; reset to beginning
gst_1:	mov     start_tdata[si],bx	; save start_tdata
	dec     size_tdata[si]		; one less character in x-mit buffer
gst_x:	ret				; done
gst	endp
;
gsr	proc	near
	mov	ax,0			; in case of underflow
	cmp	size_rdata[si],0	; any characters?
	je	gsr_x			; abort if not
	mov     bx,start_rdata[si]	; get pointer to oldest char
	mov	di,di_rdata[si]
	mov     al,rdata[di][bx]	; get char from buffer
	inc     bx			; bump start_rdata
	cmp     bx,r_size		; see if past end
	jb	gsr_1			; if not then skip
	mov	bx,0			; adjust to beginning
gsr_1:	mov     start_rdata[si],bx	; save the new start_rdata value
	dec     size_rdata[si]		; one less character
gsr_x:	ret				; done
gsr	endp
;
pet	proc	near			; *call with interrupts off*
	cmp	size_tdata[si],s_size	; buffer full?
	jb	pet_1			; jump if not
	jmp	short pet_x		; punt
pet_1:	mov     bx,end_tdata[si]	; bx points to free space
	mov	di,di_tdata[si]
	mov     tdata[di][bx],al	; move char to buffer
	inc     bx			; increment end_tdata
	cmp     bx,s_size		; see if past end
	jb	pet_2			; if not then skip
	mov	bx,0			; adjust to beginning
pet_2:	mov     end_tdata[si],bx	; save new end_tdata
	inc     size_tdata[si]		; one more character in x-mit buffer
; re-enable tx interrupts
	push	ax			; save char
	mov     dx,ier[si]		; interrupt enable register
	in      al,dx			; get it
	test	al,2			; see if tx interrupts are enabled
	jnz     pet_3			; jump if so
	mov     al,7			; if not then rcv, tx, line error
	out     dx,al			; are enabled
pet_3:	pop	ax			; recover char
pet_x:	ret				; done
pet	endp
;
per	proc	near			; *call with interrupts off*
	cmp	size_rdata[si],r_size	; buffer full?
	jae	per_x			; punt if so
	mov     bx,end_rdata[si]	; bx points to free space
	mov	di,di_rdata[si]
	mov     rdata[di][bx],al	; move data to buffer
	inc     bx			; increment end_rdata pointer
	cmp     bx,r_size		; see if gone past end
	jb	per_1			; if not then skip
	mov     bx,0			; else adjust to beginning
per_1:	mov	end_rdata[si],bx	; save value
	inc	size_rdata[si]		; got one more character
per_x:	ret				; done
per	endp
;
pst	proc	near			; *call with interrupts off*
	push	dx			; who needs trouble?
	cmp	size_tdata[si],s_size	; buffer full?
	jb	pst_1			; jump if not
; forced to overwrite first character of buffer (this is an emergency!)
	mov	bx,start_tdata[si]	; bx points to first char in buffer
	mov	di,di_tdata[si]
	mov     tdata[di][bx],al 	; clobber first char in buffer
	jmp	short pst_x		; continue
; normal case
pst_1:	mov	bx,start_tdata[si]	; bx points to first char in buffer
	dec	bx			; backup the ptr
	cmp	bx,-1			; before beginning?
	jne	pst_2			; jump if not
	mov	bx,s_size-1		; point to end if so
pst_2:	mov	di,di_tdata[si]
	mov     tdata[di][bx],al 	; move char to buffer
	mov     start_tdata[si],bx	; save new start_tdata
	inc     size_tdata[si]		; one more character in x-mit buffer
; re-enable tx interrupt
	push	ax			; save char
	mov     dx,ier[si]		; interrupt enable register
	in      al,dx			; get it
	test	al,2			; see if tx interrupts are enabled
	jnz     pst_3			; jump if so
	mov     al,7			; if not then rcv, tx, line error
	out     dx,al			; are enabled
pst_3:	pop	ax			; recover char
pst_x:	pop	dx			; not me!
	ret				; done
pst	endp
	page
;
; internal routines for flow control
;
; flow_in - respond to flow control commands from caller
; flow_out - issue flow control commands to caller
;
flow_in	proc	near
	cmp	al,control_s		; stop command received?
	jne	fi_1			; jump if not
	mov	pc_off[si],1		; we must shut up
	jmp	short fi_3		; continue
fi_1:	cmp	pc_off[si],1		; are we stifled
	jne	fi_2			; jump if not
	mov	pc_off[si],0		; start talking again
	mov	al,control_q		; treat restart character as ctrl-q
; re-enable tx interrupts
	cmp	size_tdata[si],0	; anything waiting?
	je	fi_2			; jump if nothing waiting
	push	ax			; save char
	mov     dx,ier[si]		; interrupt enable register
	in      al,dx			; get it
	test	al,2			; see if tx interrupts are enabled
	jnz     fi_1a			; jump if so
	mov     al,7			; if not then rcv, tx, line error
	out     dx,al			; are enabled
fi_1a:	pop	ax			; recover char
fi_2:	cmp	al,control_q		; control-q?
	je	fi_3			; jump if so
	clc				; else we've processed a normal char
	jmp	short fi_x		; done
fi_3:	stc				; flow control char
fi_x:	ret				; done
flow_in	endp
;
flow_out proc	near
	cmp	host_off[si],1		; host turned off?
	je	fo_x			; jump if so
	cmp	size_rdata[si],r_size/2	; receive buffer nearly full?
	jle	fo_x			; done if not
	mov	al,control_s		; turn off host if so
	call	pst			; send immediately
	mov	host_off[si],1		; host is now off
fo_x:	ret				; done
flow_out endp
	page
;
; b_and_e: buffer and echo incoming character in al
;
b_and_e	proc	near
; pseudo word wrap
	cmp	echo_state[si],1	; echoing?
	jne	bne_1			; jump if not echoing
	cmp	al,' '			; blank?
	je	bne_0			; jump if so
	cmp	al,tab			; tab?
	jne	bne_1			; jump if not
bne_0:	mov	bl,woscol[si]		; wrap on space column
	cmp	chars_this_line[si],bl	; wrap on space?
	jb	bne_1			; jump if not
	mov	al,cr			; substitute a carriage return
bne_0a:	mov	soft_cr[si],true	; a soft one
; buffer character
bne_1:	call	per			; put to end of rdata
	call	flow_out		; issue flow control commands to host
; possible echo to caller
	cmp	echo_state[si],1	; are we echoing?
	jne	bne_x			; done with this interrupt if not
	inc	chars_this_line[si]	; bump character count for this line
	cmp	al,bs			; backspace?
	je	bne_2			; jump if so
	cmp	hostecho,false		; should we echo?
	je	bne_1a			; skip if not
	call	pet			; echo incoming character
bne_1a:	cmp	al,cr			; carriage return?
	je	bne_4			; jump if so
	mov	al,cr			; carriage return
	mov	bl,chars_this_line[si]	; no. of characters so far
	cmp	bl,wrapcol		; line too long?
	jb	bne_x			; done if not
	cmp	bl,woscol[si]		; line way too long?
	jae	bne_0a			; do fake cr if so
	cmp	woscol[si],wrapcol	; wrap suppressed?
	ja	bne_4			; avoid line overflow if so
	jmp	bne_0a			; else do fake cr
; backspace
bne_2:	sub	chars_this_line[si],2	; adjust character count for this line
	cmp	chars_this_line[si],0	; before the beginning?
	jge	bne_3			; jump if not
	mov	chars_this_line[si],0	; else clear character count
	jmp	short bne_x		; and don't echo the bs
; echo backspace
bne_3:	cmp	hostecho,false		; should we echo?
	je	bne_x			; jump if not
	call	pet			; echo backspace to caller
	mov	al,' '			; space
	call	pet			; echo to caller
	mov	al,bs			; backspace
	call	pet			; echo to caller
	jmp	short bne_x		; back for more
; stop echoing after carriage return
bne_4:	mov	echo_state[si],0	; turn off echoing after cr
bne_x:	ret				; keep listening
b_and_e	endp
	page
;
; int_hndlr1 - handles interrupts generated by com1: (level 4)
; lines 1-8	board 1
; lines 17-24	board 2
;
dataseg	dw	0
	public	int_hndlr1
int_hndlr1 proc  far
	push	ax			; save regs
	push	bx
	push	cx
	push	dx
	push	bp
	push	si
	push	di
	push	ds
	push	es

; setup
	mov	ax,cs:dataseg		; our data seg
	mov	ds,ax			; to ds
	mov	ax,data2		; setup our iobufs
	mov	es,ax

; clear the interrupt controller flag
	mov	dx,inta00		; 8259 control port
	mov	al,eoi4			; specific end of interrupt
	out	dx,al			; clear flag

; decide if we're dealing with a Digiboard card or just a normal port
	cmp	com1_port,3f8h		; com/x installed?
	jne	digi1			; jump if so
	lea	si,area1		; data area for normal com1
	jmp	int_common		; continue

; we're dealing with a Digiboard port interrupting on irq4 (com1)
digi1:	cmp	ps2mc8,0		; mc/8?
	jne	digi1m			; jump if so
; com/8 for pc/at
	mov	dx,status_port1		; com/x status port
	inc	dx			; plus one for even status (irq4)
	in	al,dx			; get status from digicom card
	mov	ah,al			; copy of status
	and	ah,18h			; leave just the board id
	lea	bx,area1		; assume board 1 = lines 1-8
	cmp	ah,08h			; board 1?
	je	@F			; jump if so
	lea	bx,area17		; else board 2 = lines 17-24
@@:	jmp	digi2a			; continue
; mc/8 for ps/2
digi1m:	mov	dx,status_port1		; status port of first mc/8
	inc	dx			; plus one for even status
	in	al,dx			; get status from mc/8
	cmp	al,0			; at least one interrupt indicated?
	jne	@F			; continue if so
	jmp	int_oops		; if not, it's glitch
@@:	lea	bx,area1		; assume lines 1-8 (16 channels)
; round robin to pick channel
	mov	cl,channel		; last channel used
	inc	cl			; round robin
	ror	al,cl			; next channel bit now low
@@:	test	al,1			; interrupt on this channel?
	jnz	@F			; jump if so
	shr	al,1			; next bit
	inc	cl			; next channel
	jmp	@B			; loop back
@@:	mov	al,cl			; channel to al
	mov	channel,al		; remember channel
	jmp	digi2a			; continue

;
; int_hndlr2 - handles interrupts generated by com2: (level 3)
; lines 9-16 (or 2-9)	board 0
; lines 25-32   	board 3
;
	public	int_hndlr2
int_hndlr2 proc  far
	push	ax			; save regs
	push	bx
	push	cx
	push	dx
	push	bp
	push	si
	push	di
	push	ds
	push	es

; setup
	mov	ax,cs:dataseg		; our data seg
	mov	ds,ax			; to ds
	mov	ax,data2		; setup our iobufs
	mov	es,ax

; clear the interrupt controller flag
	mov	dx,inta00		; 8259 control port
	mov	al,eoi3			; specific end of interrupt
	out	dx,al			; clear flag

; decide if we're dealing with a Digiboard card or just a normal port
	cmp	com2_port,2f8h		; com/x installed?
	jne	digi2			; jump if so
	lea	si,area2		; data area for normal com2
	cmp	com1_port,3f8h		; com1 normal also?
	je	int_common		; jump if so
	lea	si,area9		; else 1-8 on com1
	jmp	short int_common	; continue

; we're dealing with a Digiboard port interrupting on irq3 (com2)
digi2:	cmp	ps2mc8,0		; mc/8?
	jne	digi2m			; jump if so
; com/8 for pc/at
	mov	dx,status_port2		; com/x status port (ODD for IRQ3)
	in	al,dx			; get status from digicom card
	lea	bx,area2		; assume board 0, lines 2-9
	cmp	com1_port,3f8h		; com1 normal?
	je	digi2a			; jump if so
	mov	ah,al			; copy of status
	and	ah,18h			; leave just the board id
	lea	bx,area9		; assume board 0, lines 9-16
	cmp	ah,0			; board 0?
	je	digi2a			; jump if so
	lea	bx,area25		; else board 3 = lines 25-32
	jmp	short digi2a
; mc/8 for ps/2
digi2m:	mov	dx,status_port2		; status port of second mc/8
	in	al,dx			; get status from mc/8
	cmp	al,0			; at least one interrupt indicated?
	je	int_oops		; if not, it's glitch
	lea	bx,area2		; assume lines 2-9
	cmp	com1_port,3f8h		; com1 normal?
	je	@F			; skip if so
	lea	bx,area9		; else we're lines 9-16
@@:
; round robin to pick channel
	mov	cl,channel		; last channel used
	inc	cl			; round robin
	ror	al,cl			; next channel bit now low
@@:	test	al,1			; interrupt on this channel?
	jnz	@F			; jump if so
	shr	al,1			; next bit
	inc	cl			; next channel
	jmp	@B			; loop back
@@:	mov	al,cl			; channel to al
	mov	channel,al		; remember channel

; channel number now in al, bx points to base area
digi2a:	and	ax,7			; ax = 0..7
	mul	area_length		; times length of param block in bytes
	add	bx,ax			; plus displacement
	mov	si,bx			; equals offset to desired block
;
; body of interrupt handler
;
int_common:

; find out where interrupt came from and jump to routine to handle it
repoll:
	mov     dx,iir[si]		; read interrupt status register
	in      al,dx
	cmp     al,4
	je	rx_int			; if from the receiver
	cmp     al,2
	je      tx_int			; if from the transmitter
	cmp     al,6
	je	lstat_int       	; interrupt because of line status
	cmp     al,0
	je	mstat_int       	; interrupt because of modem status
int_oops:jmp	far ptr int_end		; done, exit (don't fix far ptr stuff)

lstat_int:
	mov     dx,lsr[si]		; read and ignore line status
	in      al,dx			; this resets the 8250 chip
	jmp     repoll			; see if any more interrupts

mstat_int:
	mov     dx,msr[si]		; read and ignore modem status
	in      al,dx			; this resets the 8250 chip
	jmp     repoll			; see if any more interrupts

tx_int:
	cmp	pc_off[si],1		; have we been told to shut up?
	jne	goodtx1			; jump if not
	cmp	host_off[si],1		; has host also shut up?
	jne	send_no_more		; jump if not

; clear xon/xoff deadlock
	mov	pc_off[si],0		; we speak
	mov	host_off[si],0		; they reply
	mov	al,control_q		; but only when
	call	pst			; we let them

goodtx1:
	cmp     size_tdata[si],0	; see if any more data to send
	jg	have_data       	; if positive then data to send

; if no data to send then reset tx interrupt and return
send_no_more:
	mov     dx,ier[si]
	mov     al,1			; just receive
	out     dx,al			; is set
	jmp     repoll

have_data:
	cmp	warnout[si],1		; are we in the final 30 seconds?
	je	hd1			; if so, continue timeout
	push	es
	mov	ax,rbda			; rom bios data area
	mov	es,ax			; to es
	assume	es:rbda
	mov	ax,timer_low		; bios clock
	pop	es
	assume	es:data2
	mov	timeout[si],ax		; to our area
hd1:
; dsr/cts handshake would go here
	call	gst			; get from start of tdata
	mov     dx,datreg[si]		; dx equals port to send data to
	out     dx,al			; send data
	jmp     repoll

rx_int:
	push	es
	mov	dx,rbda			; rom bios data area
	mov	es,dx			; to es
	assume	es:rbda
	mov	ax,timer_low		; bios clock
	pop	es
	assume	es:data2
	mov	timeout[si],ax		; to our area
	mov	warnout[si],0		; clear warning, too
	mov     dx,datreg[si]		; 8250 data register
	in      al,dx			; get data
	cmp	binary[si],true		; xmodem?
	jne	rx_int1			; jump if not
	call	per			; just stash char, no processing
	jmp	repoll			; and continue
rx_int1:
	and	al,bits8		; strip parity bit for flow control
	call	flow_in			; respond to remote flow control
	jc	repoll2			; treat c-q/c-s locally only
	mov	bl,al			; get character received
	xor	bh,bh			; in bx
	and	al,filter[bx]		; make zero if not to be kept
	jz	repoll2			; ignore bad control characters
	cmp	al,control_c		; control c?
	jne	skippy			; jump if not
	mov	cancelled[si],true	; note request for cancellation
	jmp	repoll			; ignore control-c otherwise
skippy:	cmp	al,control_o		; control o?
	jne	good_rx1		; jump if not
	mov	skipped[si],true	; note request for skip
repoll2:jmp	repoll			; back for more
good_rx1:
	cmp	al,del			; del character?
	jne	good_rx2		; skip if not
	mov	al,bs			; replace with backspace
good_rx2:
	cmp	al,tab			; horizontal tab character?
	jne	grx_4			; jump if not
; process tabs
	mov	al,chars_this_line[si]	; current column
	mov	ah,0			; to ax
	div	tabsiz			; compute
	mov	cl,tabsiz		; equivalent
	sub	cl,ah			; number of spaces
	mov	ch,0			; to cx
grx_3:	push	cx			; save loop index
	mov	al,' '			; blank
	call	b_and_e			; buffer and echo it
	pop	cx			; restore loop index
	loop	grx_3			; tab expansion loop
	jmp	repoll			; loop back to get all characters
grx_4:
	call	b_and_e			; our special processing
	jmp	repoll			; loop back to get all characters

int_end:
	pop	es			; restore regs
	pop	ds
	pop	di
	pop	si
	pop	bp
	pop     dx
	pop     cx
	pop     bx
	pop     ax

	iret
int_hndlr2 endp
int_hndlr1 endp
	page
;
; procedure save_com(iv : integer);
; save original com vector
;
save_com proc	far
	pushem
	mov	ax,6[bp]		; get arg
	cmp	ax,1			; com1 interrupt level?
	jne	svc1			; jump if not

; save com1 vector
	mov	ah,35h			; fetch interrupt vector contents
	mov	al,int_com1		; interrupt number
	int	dos			; dos 2 function
	mov	old_com_off1,bx		; save
	mov	bx,es			; es:bx
	mov	old_com_seg1,bx		; for later restoration
	jmp	short svcx		; continue

; save com2 vector
svc1:	mov	ah,35h			; fetch interrupt vector contents
	mov	al,int_com2		; interrupt number
	int	dos			; dos 2 function
	mov	old_com_off2,bx		; save
	mov	bx,es			; es:bx
	mov	old_com_seg2,bx		; for later restoration
; done
svcx:	popem
	ret	2			; done, toss arg
save_com endp
	page
;
; procedure installc(iv : integer);
; installc:  install our com vector
;
installc proc far
	pushem
	mov	ax,6[bp]		; get arg
	cmp	ax,1			; com1 interrupt level?
	jne	inst1			; jump if not

; install our com1 handler
	mov	ah,25h			; set interrupt vector contents
	mov	al,int_com1		; interrupt number
	lea	dx,int_hndlr1		; our interrupt handler
	push    ds			; save data segment
	push	cs			; copy cs
	pop	ds			; to ds
	int	dos			; dos function
	pop     ds			; recover data segment
	in      al,inta01		; set enable bit on 8259
	and     al,nirq4
	jmp	$+2			; i/o delay for at
	out     inta01,al
	jmp	short instx		; continue

; install our com2 handler
inst1:	mov	ah,25h			; set interrupt vector contents
	mov	al,int_com2		; interrupt number
	lea	dx,int_hndlr2		; our interrupt handler
	push    ds			; save data segment
	push	cs			; copy cs
	pop	ds			; to ds
	int	dos			; dos function
	pop     ds	    	 	; recover data segment
	in      al,inta01		; set enable bit on 8259
	jmp	$+2			; i/o delay for at
	and     al,nirq3
	out     inta01,al

; done
instx:	popem
	ret	2			; done, toss arg
installc endp
	page
;
; procedure restorec(iv : integer);
; restore original interrupt vector
;
restorec proc far
	pushem
	mov	ax,6[bp]		; get arg
	cmp	ax,1			; com1 interrupt level?
	jne	rsc1			; jump if not

; restore old com1 vector
	mov     dx,inta01		; turn off 8259
	in      al,dx
	or      al,irq4
	jmp	$+2			; delay for at
	out     dx,al
	mov	ah,25h			; set interrupt vector function
	mov	al,int_com1		; interrupt number
	mov	dx,old_com_off1		; old offset to dx
	mov	bx,old_com_seg1		; old seg
	push	ds			; save ds
	mov	ds,bx			; to ds
	int	dos			; dos function
	pop	ds			; recover ds
	jmp	short rscx		; continue

; restore old com2 vector
rsc1:	mov     dx,inta01		; turn off 8259
	in      al,dx
	or      al,irq3
	jmp	$+2			; delay for at
	out     dx,al
	mov	ah,25h			; set interrupt vector function
	mov	al,int_com2		; interrupt number
	mov	dx,old_com_off2		; old offset to dx
	mov	bx,old_com_seg2		; old seg
	push	ds			; save ds
	mov	ds,bx			; to ds
	int	dos			; dos function
	pop	ds			; recover ds

; done
rscx:	popem
	ret	2			; done, toss arg
restorec endp
	page
;
; select which port is to be "active"
; 6[bp] = port number 1..nlines
;
select_port proc far
	pushem
	mov	ax,ds		; our dataseg
	mov	cs:dataseg,ax	; to save area
	mov	ax,6[bp]	; get arg
	dec	al		; 0..nlines-1
	cmp	al,nlines-1	; range check
	ja	sp_x		; jump if bad
	mov	ah,0		; al to ax
	mul	area_length	; times length in bytes
	lea	bx,area1	; offset of first area
	add	ax,bx		; plus displacement
	mov	current_area,ax	; set selection in memory
sp_x:	popem
	ret	2		; done, toss arg
select_port endp
	page
;
; open_com on current port
; 10[bp] = baud rate
; 8[bp] = parity:     N(one), O(dd), E(ven), S(pace), M(ark)
; 6[bp]  = stop bits:  1, 2
; modified to work with digicom card
;
; clear buffers
; set 8250 ports
; re-initialize the Intel 8250 uart
; enable interrupts on the Intel 8259 interrupt control chip
;
open_com proc   far
	pushem
	mov	si,current_area		; si points to data area

	mov	ax,10[bp]		;
	mov	baud_rate[si],ax	;
	mov	ax,8[bp]		;
	mov	parity[si],al		;
	mov	ax,6[bp]		;
	mov	stop_bits[si],al	;

; reset flow control
	cli				; interrupts off
	mov	echo_state[si],0	; no echo to caller
	mov	chars_this_line[si],0	; reset char count
	mov	host_off[si],0		; host flowing
	mov	pc_off[si],0		; pc flowing
	mov	cancelled[si],false	; no control c
	mov	skipped[si],false	; no control o
	push	si
	int	1ah			; bios clock
	pop	si
	mov	timeout[si],dx		; update timeout clock
	mov	warnout[si],0		; clear warning flag
	mov	woscol[si],wrapcol-10	; default soft wrap column
	mov	binary[si],false	; binary mode off
	mov	soft_cr[si],false	; no soft CRs yet

; reset buffer counts and pointers
	mov	start_tdata[si],0
	mov	end_tdata[si],0
	mov	start_rdata[si],0
	mov	end_rdata[si],0
	mov	size_tdata[si],0
	mov	size_rdata[si],0

; set 8250 port addresses
oc1:	cmp	com1_port,3f8h		; normal com1?
	jne	oc1a			; jump if not
	cmp	com2_port,2f8h		; normal com2?
	jne	oc1b			; jump if not

; 2 ports only
	mov	ax,com1_port		; com1?
	cmp	port[si],1		; are we right
	je	oc2			; jump if so
	mov	ax,com2_port		; else com2
	jmp	short oc2		; continue

; com1 split
oc1a:	cmp	com2_port,2f8h		; com1 also split?
	jne	oc1c			; jump if so
	mov	ax,com2_port		; com2?
	cmp	port[si],9		; are we right?
	je	oc2			; jump if so
	mov	al,port[si]		; ports 1-8
	sub	al,1			; map to 0-7
c1stuff:mul	bpp			; times bytes per port
	add	ax,com1_port		; plus com1 base
	jmp	short oc2		; continue

; com2 split
oc1b:	mov	ax,com1_port		; com1?
	cmp	port[si],1		; are we right?
	je	oc2			; jump if so
	mov	al,port[si]		; ports 2-9
	sub	al,2			; map to 0-7
c2stuff:mul	bpp			; times bytes per port
	add	ax,com2_port		; plus com2 base
	jmp	short oc2		; continue

; both split = 16+ lines, all DigiBoard system
oc1c:	mov	al,port[si]		; DLX port number
	cmp	al,8			; port 1-8?
	ja	oc9up			; jump if not
	sub	al,1			; map to 0-7
	jmp	c1stuff			; based on com1
oc9up:	cmp	al,16			; port 9-16?
	ja	oc17up			; jump if not
	sub	al,9			; map to 0-7
	jmp	c2stuff			; based on com2
oc17up:	cmp	al,24			; port 17-24?
	ja	oc25up			; jump if not
	sub	al,9			; map 17-24 to 8-15
	jmp	c1stuff			; do com1 stuff
oc25up:	sub	al,17			; map 25-32 to 8-15
	jmp	c2stuff			; do com2 stuff

; do all seven for each 8250
oc2:	mov	bx,datreg		; offset of table of ports
	mov	cx,7			; loop six times
oc3:	mov	word ptr [si][bx],ax	; set port address
	inc	ax			; next port
	add	bx,2			; next word address
	loop	oc3			; rs232 base loop

; reset the 8250
oc4:	mov     al,0
	mov     dx,mcr[si]
	out     dx,al
	jmp	$+2			; i/o delay for at

	mov     dx,lsr[si]		; reset line status condition
	in      al,dx
	jmp	$+2			; i/o delay for at
	mov     dx,datreg[si]		; reset receive data condition
	in      al,dx
	jmp	$+2			; i/o delay for at
	mov     dx,msr[si]		; reset modem deltas and conditions
	in      al,dx
	jmp	$+2			; i/o delay for at

; convert passed baud rate to 8250 divisor
	mov	ax,50			; 50 baud
	mul	div50			; times its divisor
	div	baud_rate[si]		; other speeds are proportional
	mov	bx,ax			; result to bx

; set 8250 divisor
	mov     dx,lcr[si]		; line control register
	mov     al,80h			; hi bit on
	out     dx,al			; set dlab = 1
	jmp	$+2			; i/o delay for at
	mov     dx,word ptr dll[si]	; least significant byte
	mov     al,bl			; lsb from table
	out     dx,al			; set lsb on 8250
	jmp	$+2			; i/o delay for at
	mov     dx,word ptr dlh[si]	; most significant byte
	mov     al,bh			; msb from table
	out     dx,al			; set msb on 8250
	jmp	$+2			; i/o delay for at

; set parity and number of stop bits
	mov	al,03h		; none or space parity is the default
	cmp	parity[si],'O'	; odd parity requested?
	jne	p1		; jump if not
	mov	al,0ah		; select odd parity
	jmp	short p3	; continue
p1:	cmp	parity[si],'E'	; even parity requested?
	jne	p2		; jump if not
	mov	al,1ah		; select even parity
	jmp	short p3	; continue
p2:	cmp	parity[si],'M'	; mark parity requested?
	jne	p3		; jump if not
	mov	al,2ah		; select mark parity
p3:	test	stop_bits[si],2	; 2 stop bits requested?
	jz	stop1		; no
	or	al,4		; yes
stop1:	mov     dx,lcr[si]	; line control register
	out     dx,al		; set 8250 parity mode and dlab=0
	jmp	$+2		; i/o delay for at

; enable interrupts on 8250 chip
	mov     dx,ier[si]	; enable interrupts on 8250
	mov     al,1		; receive only
	out     dx,al
	jmp	$+2		; i/o delay for at
	mov     dx,mcr[si]	; set dtr and enable int driver
	mov     al,0bh
	out     dx,al

; open complete
ocx:	sti			; interrupts on
	popem
	ret	6		; done, toss three args
open_com endp
	page
;
; close_com - turns off interrupts from the communications port
;
close_com proc  far
	pushem
	mov	si,current_area		; si points to data area

; turn off 8250
	mov     dx,ier[si]
	mov     al,0
	out     dx,al

ccx:	popem
	ret
close_com endp
	page
;
; dtr_off - turns off dtr to tell modems that the terminal has gone away
;	   and to hang up the phone
;
dtr_off proc    far
	pushem
	mov	si,current_area	; si points to data area

	mov     dx,mcr[si]
	mov     al,08h		; dtr off, rts off, out2 on
	out     dx,al
dfx:	popem
	ret
dtr_off endp
;
; dtr_on - turns dtr on
;
dtr_on  proc    far
	pushem
	mov	si,current_area	; si points to data area

	mov     dx,mcr[si]
	mov     al,0bh
	out     dx,al
dnx:	popem
	ret			; done
dtr_on  endp
	page
;
; function rlsd : boolean;
; is user still on the line?  online=true offline=false
;
rlsd	proc	far
	pushem
	mov	si,current_area	; si points to data area

	cmp	dcd_ok,0	; pay attention to DCD line?
	je	rlsd_usc	; (USC needs this)
	mov	dx,msr[si]	; modem status register
	in	al,dx		; get modem status
	test	al,80h		; rlsd bit on?
	jz	rlsd_f		; jump if not

rlsd_usc:
	mov	bx,timeout[si]	; avoid bad timing
	mov	ah,0		; read clock
	push	si
	int	1ah		; bios time-of-day routine
	pop	si
	mov	cx,dx		; copy time to cx
	sub	dx,bx		; compute elapsed time
	cmp	dx,to_kill	; has kill timeout passed?
	jbe	rlsd_t		; jump if not
	cmp	al,0		; midnight problem?
	je	rlsd_f		; no, we've got a deadbeat on our hands
	mov	timeout[si],cx	; new value after midnight

rlsd_t:	mov	ax,true		; carrier present = all is well
	jmp	short rlsd_x	; finish
rlsd_f:	mov	ax,false	; carrier not present = no one there
rlsd_x:	popem
	ret			; done, no args
rlsd	endp
	page
;
; function warning : boolean;
; 30 seconds from timeout?
;
warning	proc	far
	pushem
	mov	si,current_area	; si points to data area

	cmp	warnout[si],1	; already warned?
	je	warn_f		; don't repeat it

	mov	bx,timeout[si]	; avoid bad timing
	mov	ah,0		; read clock
	push	si
	int	1ah		; bios time-of-day routine
	pop	si
	mov	cx,dx		; copy time to cx
	sub	dx,bx		; compute elapsed time
	cmp	dx,to_warn	; has warning timeout passed?
	jbe	warn_f		; jump if not
	cmp	al,0		; midnight problem?
	je	warn_t		; jump if not
	mov	timeout[si],cx	; new value after midnight
	jmp	short warn_f	; continue

warn_t:	mov	ax,true		; warning
	mov	warnout[si],1	; de-bounce
	jmp	short warn_x	; finish
warn_f:	mov	ax,false	; no warning
warn_x:	popem
	ret			; done, no args
warning	endp
	page
;
; unwarn - cancel pending warning
;
unwarn	proc	far
	pushem
	mov	si,current_area	; si points to data area
	push	si
	int	1ah		; bios clock
	pop	si
	mov	timeout[si],dx	; update timeout clock
	mov	warnout[si],0	; clear warning flag
unwarnx:popem
	ret			; done, no args
unwarn	endp
	page
;
; r_count - returns number of bytes in the receive buffer in ax
;	   returns total size in bx
;
r_count proc    far
	pushem
	mov	si,current_area	; si points to data area
	mov	bx,r_size	; get total size
	mov     ax,size_rdata[si] ; get number of bytes used
rcx:	popem
	ret
r_count endp
	page
;
; receive - returns the next character from the receive buffer in al
;	    and removes it from the buffer
;	    the parity bit is stripped off
;
receive	proc    far
	pushem
	mov	si,current_area		; si points to data area

	call	gsr			; get from start of rdata

	cmp	host_off[si],1		; host turned off?
	jne	rcvx			; jump if not
	cmp	size_rdata[si],r_size/20 ; receive buffer nearly empty?
	jge	rcvx			; done if not
	mov	host_off[si],0		; turn on host if so
	push	ax			; save received char
	mov	al,control_q		; tell him to talk
	cli				; interrupts off
	call	pst			; send immediately internal
	sti				; interrupts back on
	pop	ax			; restore received char

; done
rcvx:	popem
	ret				; done
receive	endp
	page
;
; s_free - returns in ax the amount of free space
;	  remaining in the transmit buffer
;
s_free	proc    far
	pushem
	mov	si,current_area	; si points to data area

	mov     ax,s_size	; get the size of the x-mit buffer
	sub     ax,size_tdata[si] ; subtract the number of bytes used

sfx:	popem
	ret
s_free	endp
	page
;
; s_working - returns in ax the number of characters
;	     waiting in the transmit buffer
;
s_working proc    far
	pushem
	mov	si,current_area	; si points to data area
	mov	ax,size_tdata[si] ; the number of bytes in use
swx:	popem
	ret
s_working endp
	page
;
; send - send a character
; 6[bp] = char to write
;
send    proc    far
	pushem
	mov	al,6[bp]
	mov	si,current_area		; si points to data area
	cli				; interrupts off
	call	pet			; put to end of tdata
	sti				; interrupts back on
l44:	popem
	ret	2			; done, toss arg
send    endp
	page
;
; procedure echo_com;
;
; re-enable echo to caller
;
echo_com proc	far
	pushem
	mov	si,current_area		; si points to data area
; echo characters typed ahead of time
	mov	chars_this_line[si],0	; reset character count
	cmp	size_rdata[si],0	; any characters typed ahead?
	je	ecx_on			; jump if not
	mov	cx,size_rdata[si]	; count of characters in buffer
	mov	temp,cx			; stash count
	mov     bx,start_rdata[si]	; get pointer to oldest char
ecx_1:	mov	di,di_rdata[si]
	mov     al,rdata[di][bx]	; get char from buffer
	push	bx			; save bx ptr
	inc	chars_this_line[si]	; bump character count
	cmp	al,bs			; backspace?
	jne	ecx_1a			; jump if not
	sub	chars_this_line[si],2	; adjust count
	cmp	chars_this_line[si],0	; attempt to backspace too far?
	jge	ecx_1a			; jump if not
	mov	chars_this_line[si],0	; reset count
	jmp	short ecx_1b		; ignore extra backspace
ecx_1a:	cmp	hostecho,false		; should we echo?
	je	ecx_1b			; jump if not
	cli				; interrupts off
	call	pet			; echo to caller
	sti				; interrupts back on
ecx_1b:	pop	bx			; restore bx ptr
	cmp	al,cr			; another carriage return?
	je	ecx_off			; break out of loop if so
	inc     bx			; bump start_rdata
	cmp     bx,r_size		; see if past end
	jb	ecx_2			; if not then skip
	mov	bx,0			; adjust to beginning
ecx_2:	loop	ecx_1			; typeahead echo loop
	mov	ax,size_rdata[si]	; count of characters now in buffer
	mov	cx,ax			; copy it to cx
	sub	cx,temp			; less count of those echoed
	mov	temp,ax			; stash new count
	ja	ecx_1			; loop back and echo newcomers
ecx_on:	mov	echo_state[si],1	; enable echo
	jmp	short ecx		; finish up
ecx_off:mov	echo_state[si],0	; disable echo
ecx:	popem
	ret				; done, no args
echo_com endp
	page
;
; procedure purge_com;
; empty buffers, stop sending
;
purge_com proc	far
	pushem
	mov	si,current_area		; si points to data area

; reset buffer counts and pointers
	cli				; no interrupts
	mov	pc_off[si],0		; pc flowing
	mov	cancelled[si],false	; no control c
	mov	skipped[si],false	; no control o
	mov	start_tdata[si],0	; tx
	mov	end_tdata[si],0		;    buffer
	mov	start_rdata[si],0	; rx
	mov	end_rdata[si],0		;    buffer
	mov	size_tdata[si],0	; sizes,
	mov	size_rdata[si],0	;	too

; turn off xmit interrupts
	mov     dx,ier[si]		; interrupt enable register
	mov     al,1			; just receive
	out     dx,al			; are set
	sti				; interrupts back on

pur_x:	popem
	ret				; done, no args
purge_com endp
	page
;
; procedure unskip_com;
; stop skipping
;
unskip_com proc	far
	pushem
	mov	si,current_area		; si points to data area

	mov	skipped[si],false	; no control o

uns_x:	popem
	ret				; done, no args
unskip_com endp
	page
;
; function cancelled_com : boolean;
; have we been control c-ed?
;
cancelled_com proc far
	pushem
	mov	si,current_area		; si points to data area

	mov	al,cancelled[si]	; get cancellation state
	mov	cancelled[si],false	; reset state

cc_x:	popem
	ret				; done, no args
cancelled_com endp
	page
;
; procedure force_cancel;
; simulate a control-C
;
force_cancel proc far
	pushem
	mov	si,current_area		; si points to data area

	mov	cancelled[si],true	; indicate cancellation

fc_x:	popem
	ret				; done, no args
force_cancel endp
	page
;
; function skipped_com : boolean;
; have we been control o-ed?
;
skipped_com proc far
	pushem
	mov	si,current_area		; si points to data area

	mov	al,skipped[si]		; get skip state

sc_x:	popem
	ret				; done, no args
skipped_com endp
	page
;
; procedure wrap(col : byte);
; wrap - set wrap column
;
; 6[bp] = column to wrap on spaces
;
wrap	proc    far
	pushem
	mov	si,current_area		; si points to data area

	mov	al,6[bp]		; column
	mov	woscol[si],al		; to data area

wx:	popem
	ret	2			; done, toss arg
wrap	endp
	page
;
; procedure binary_mode;
;
; 6[bp] = 1 for binary mode, 0 for ordinary
;
binary_mode proc far
	pushem
	mov	si,current_area		; si points to data area

	mov	ax,6[bp]		; copy arg
	mov	binary[si],al		; to flag

bmx:	popem
	ret	2			; done, toss arg
binary_mode endp
	page
;
; function was_soft : binary;
;
; returns true if last cr was soft, else false
;
was_soft proc	far
	mov	si,current_area		; si points to data area
	mov	ah,0			; clear top of anser
	mov	al,soft_cr[si]		; get data value
	mov	soft_cr[si],false	; reset
	ret				; done, no args
was_soft endp
auxhndlr ends
	end
