;--------------------
; cmd-tail.asm - Reading command line parameters from Jacob Stedman

 ; demonstrates how to parse the command line for arguments
 ; try it with the arguments PARAM1 and PARAM2
 ; note: arguments are what you enter on the command line,
 ; parameters are what the program expect

 .model small

 .stack 0200h

 .data

   ; scan this table for each arg until we get a zero word
   tableParams label byte
     dw       offset sParam1
     dw       offset sParam2
     dw       0000h

   ; these params can't contain any international chars
   ; format: length byte, string, offset handler
     sParam1  db 6
              db 'PARAM1'
              dw offset got_param1
     sParam2  db 6
              db 'PARAM2'
              dw offset got_param2

   ; messages to print, not nescessary..
   sGotParam1     db    'PARAM1 found on cmdline', 0Dh, 0Ah, '$'
   sGotParam2     db    'PARAM2 found on cmdline', 0Dh, 0Ah, '$'
   sBadArgument   db    'Illegal argument', 0Dh, 0Ah, '$'

 .code

   entrypoint:
        cld                        ; string instructions forward!

        ; make the whole cmdline uppercase so we can
        ; use the quick string search REP CMPSB

        mov        si, 0080h        ; there's a length *byte* at
        lodsb                       ; [0080h], make that a
        sub        cx, cx           ; *word* in cx (so we can use
        mov        cl, al           ; LOOP)

        mov        di, si       ; ds:si and es:di move in synch

   uppercase_next_char:
        lodsb
        cmp        al, 'a'
        jb         uppercase_store
        cmp        al, 'z'
        ja         uppercase_store
        sub        al, 'a'-'A'
   uppercase_store:
        stosb
        loop       uppercase_next_char        ; works for a-zA-Z

        mov        si, 0081h        ; reload si

        mov        ax, @data        ; make es -> our data segment
        mov        es, ax
        assume     cs:@code, ds:nothing, es:@data

   next_arg:
        call       skip_spaces
        push       si
        call       skip_nonspaces
        mov        dx, si           ; dx = pos after last char -
        pop        si               ;      post at first char
        sub        dx, si           ; (that is, the arg length)

        or         dx, dx
        jz         parse_done        ; if length 0 we got CR: stop

        mov        bx, offset tableParams
        ; length > 0: got argument, see if it's a valid one

   check_next_param:               ; si->argument,
        mov        di, es:[bx]     ; dx=arglen, di->sParamX
        or         di, di          ; if we read 0 from
        jz         arg_not_found   ; tableParams, unknown arg

        inc        bx              ; inc tableParams index
        inc        bx
        sub        cx, cx          ; retrieve length *byte*
        mov        cl, es:[di]     ; and make it a *word*
        inc        di              ; inc sParamX index

        cmp        cx, dx          ; if arg and param of
        jne        check_next_param; different lengths or
        push       si              ; if same but they don't
        repe       cmpsb           ; match: not this one
        pop        si              ; (POP SI doesn't touch
        jne        check_next_param        ; the flags)

        ; if repe cmpsb fails after a partial scan, si would have
        ; been corrupted if we didn't save it. now instead, always
        ; restore it to the beginning of the argument and if we're
        ; done with that argument, use skip_nonspaces to move over
        ; it

        call       es:[di]          ; if still here, right
        call       skip_nonspaces   ; param, call handler
        jmp        short next_arg
        ; no idea to check the remaining params, it was this one

   arg_not_found:                    ; we've scanned the whole
        mov        dx, offset sBadArgument  ; parameter table
        call       write_string      ; for this argument,
        call       skip_nonspaces    ; and we can't find it :-(
        jmp        short next_arg    ; skip that arg, try next

   parse_done:
        mov        ax, 4C00h                ; quit to DOS
        int        21h

   ;------------------------------------------------------------;
   ; in:        es:dx        $-terminated string to print       ;
   ;------------------------------------------------------------;
   write_string proc
        push       ds
        mov        ax, es
        mov        ds, ax
        mov        ah, 09h
        int        21h
        pop        ds
        ret
   write_string endp

   ;------------------------------------------------------------;
   ; do {
   ;   c = read_char();
   ; } while (c == ' ' || c == '\t');
   ; undo_last_read();
   ;-----------------------------------------------------------;
   ; in:         si        first space to skip
   ; out:        si        first nonspace we got
   ;-----------------------------------------------------------;
   skip_spaces proc
     skip_next_space:
        lodsb
        cmp       al, ' '
        je        skip_next_space
        cmp       al, 09h
        je        skip_next_space
        dec       si
        ret
   skip_spaces endp

   ;------------------------------------------------------------;
   ; do {
   ;   c = read_char();
   ; } while (c != ' ' && c != '\t' && c != '\n');
   ; undo_last_read();
   ;------------------------------------------------------------;
   ; in:        si        first char to skip
   ; out:       si        first space we got
   ;------------------------------------------------------------;
   skip_nonspaces proc                ; stops at CR!!
     skip_next_nonspace:
        lodsb
        cmp       al, ' '
        je        got_space
        cmp       al, 09h
        je        got_space
        cmp       al, 0Dh
        jne       skip_next_nonspace
     got_space:
        dec       si
        ret
   skip_nonspaces endp

   ;----------------------------------------------------;
   ; these get called when we recognize that argument
   ;----------------------------------------------------;
   got_param1 proc
        mov        dx, offset sGotParam1
        call       write_string
        ret
   got_param1 endp

   got_param2 proc
        mov        dx, offset sGotParam2
        call       write_string
        ret
   got_param2 endp

 end entrypoint

