WITH Ada.Text_IO;
WITH Ada.Integer_Text_IO;
WITH Simple_Dates;
PROCEDURE Sort_Score_File IS
------------------------------------------------------------------
--| Sorts and displays an array of test score records
--| The records are read from a file scorfile.dat
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: July 1995                                     
------------------------------------------------------------------

  MaxSize : CONSTANT Positive := 250;
  MaxScore : CONSTANT Positive := 100;

  SUBTYPE StudentName IS String(1..20);
  SUBTYPE ClassIndex IS Positive RANGE 1..MaxSize;
  SUBTYPE ClassRange IS Natural  RANGE 0..MaxSize;
  SUBTYPE ScoreRange IS Natural  RANGE 0..MaxScore;

  TYPE ScoreRecord IS RECORD
     Name:  StudentName;
     Score: ScoreRange;
  END RECORD;

  TYPE ScoreArray IS ARRAY (ClassIndex) OF ScoreRecord;

  Scores : ScoreArray;
  ClassSize : ClassRange;

  PROCEDURE GetRecords(Scores: OUT ScoreArray; ClassSize: OUT ClassRange) IS
  -- Pre:  None
  -- Post: Scores contains records read from file; ClassSize
  --   indicates the number of records read
    
    TestScores: Ada.Text_IO.File_Type; -- program variable naming the input file
    TempSize:   ClassRange;
    TempRecord: ScoreRecord;

  BEGIN  -- GetRecords 

    -- Open the file and associate it with the file variable name
    Ada.Text_IO.Open
      (File => TestScores, Mode => Ada.Text_IO.In_File, Name => "scorfile.dat");

    -- Read each data item
    -- and store it in the appropriate element of Scores

    TempSize := 0;                         -- initial class size   
    -- Read each array element until done.   
    WHILE (NOT Ada.Text_IO.End_Of_File(TestScores)) AND 
          (TempSize < MaxSize) LOOP
      -- invariant:
      --   Records remain in the file and
      --   TempSize <= MaxSize
        
      Ada.Text_IO.Get(File => TestScores, Item => TempRecord.Name);
      Ada.Integer_Text_IO.Get(File => TestScores, Item => TempRecord.Score);

      TempSize := TempSize + 1;
      Scores(TempSize) := TempRecord;     -- Save the score   

    END LOOP;      
    -- assert:
    --   End of file reached or
    --   TempSize is MaxSize
      
    IF TempSize = MaxSize THEN
      Ada.Text_IO.Put(Item => "Array is filled.");  
      Ada.Text_IO.New_Line;
    END IF;

    ClassSize := TempSize;

  END GetRecords;


  PROCEDURE Exchange(Student1, Student2: IN OUT ScoreRecord) IS
  -- Pre:  Student1 and Student2 are defined
  -- Post: Student1 and Student2 are interchanged

    TempRecord: ScoreRecord;

  BEGIN

    TempRecord := Student1;
    Student1 :=   Student2;
    Student2 :=   TempRecord;

  END Exchange;

  PROCEDURE SelectSort(Scores: IN OUT ScoreArray; ClassSize: IN ClassRange) IS
  -- Pre:  Scores and ClassSize are defined
  -- Post: The records in the first ClassSize elements of Scores 
  --   are sorted in descending order by score

    IndexOfMax: ClassRange;

  BEGIN

    FOR PositionToFill IN 1..ClassSize - 1 LOOP

      -- Find the element in subarray 1..PositionToFill with largest Score
      IndexOfMax := PositionToFill;
      FOR ItemToCompare IN PositionToFill + 1 .. ClassSize LOOP
        IF Scores(ItemToCompare).Score > Scores(IndexOfMax).Score THEN
          IndexOfMax := ItemToCompare;
        END IF;
      END LOOP;
      -- assert: element at IndexOfMax is largest in subarry

      IF IndexOfMax /= PositionToFill THEN
        Exchange(Scores(PositionToFill),Scores(IndexOfMax));
      END IF;

    END LOOP;

  END SelectSort;

  PROCEDURE DisplayScores(Scores: IN ScoreArray; ClassSize: IN ClassRange) IS
  -- Pre:  Scores and ClassSize are defined
  -- Post: dislays the first ClassSize records in Scores

  BEGIN

    FOR I IN 1..ClassSize LOOP
      Ada.Integer_Text_IO.Put(Item => I, Width => 3);
      Ada.Text_IO.Put(Item => "  ");
      Ada.Text_IO.Put(Item => Scores(I).Name);
      Ada.Integer_Text_IO.Put(Item => Scores(I).Score, Width => 4);
      Ada.Text_IO.New_Line;
    END LOOP;

  END DisplayScores;
  

BEGIN -- Sort_Score_File

  Ada.Text_IO.Put(Item => "Today is ");
  Simple_Dates.Put(Item => Simple_Dates.Today);
  Ada.Text_IO.New_Line;

  GetRecords(Scores => Scores, ClassSize => ClassSize);
  Ada.Text_IO.Put(Item => "Original Test File:");
  Ada.Text_IO.New_Line;
  Ada.Text_IO.New_Line;
  DisplayScores(Scores => Scores, ClassSize => ClassSize);

  SelectSort(Scores => Scores, ClassSize => ClassSize);
  Ada.Text_IO.New_Line;
  Ada.Text_IO.Put(Item => "Sorted Test File:");
  Ada.Text_IO.New_Line;
  Ada.Text_IO.New_Line;
  DisplayScores(Scores => Scores, ClassSize => ClassSize);
  Ada.Text_IO.New_Line;

END Sort_Score_File;
