-----------------------------------------------------------------------------
-- Demonstration of displaying a voxel heightscape                         --
--  by Steven H Don                                                        --
--                                                                         --
-- For questions, feel free to e-mail me.                                  --
--                                                                         --
--    shd@earthling.net                                                    --
--    http://shd.home.ml.org                                               --
-----------------------------------------------------------------------------
--  Adaptation from Pascal to GNAT/DOS Ada:  G. de Montmollin 22.IX.1998   --
--  (32-bit, SVGA, resolution independant)                                 --
--                                                                         --
--  Date / Version:  24.IX.1998                                            --
--  e-mail: Gautier.deMontmollin@Maths.UniNe.CH                            --
--                                                                         --
--  NB: the algorithm here is intended to be more instructive and          --
--  compact than efficient ! Feel free to improve it and extend the        --
--  program, e.g. read the dimensions of the 1st bitmap file and           --
--  allocate dynamic maps, which can be much larger (no 64K limit !).      --
--  For Ada / implementation questions, please contact G. de Montmollin    --
-----------------------------------------------------------------------------

with Ada.Text_IO;                   use Ada.Text_IO;
with Ada.Direct_IO;
with Ada.Numerics;                  use Ada.Numerics;
with Ada.Numerics.Elementary_functions;     
use  Ada.Numerics.Elementary_functions;

-- package by Jerry van Dijk from SVGA pack in EZ2LOAD distribution:

with SVGA;                          use SVGA;

procedure Voxels is

 package IIO is new Integer_IO(integer); use IIO;

 procedure Voxels_in_graphics is

  type Byte is mod 2 ** 8;
  for Byte'Size use 8;

  package DBIO is new Ada.Direct_IO(Byte); use DBIO;

-- Resolution
  resX: constant integer:= X_Size;
  resY: constant integer:= Y_Size;

--   This controls the maximum distance displayed. If you increase this,
--   you will see more of the landscape, but it will also be slower
  Depth: constant:= 130;

  -- This is an off-screen frame buffer
  type Frame_Access is access Screen_Buffer;
  Buffer : Frame_Access;    -- The buffer

  landX: constant:= 255;
  landY: constant:= 255;
  
  type height_map is array(0..landX, 0..landY) of Byte; 
  HMap, AHMap : height_map;                                 -- Height map
  TMap        : array(0..landX, 0..landY) of Color_Type;    -- Texture map

  -- Cosine and Sine tables
  finesse: constant:= 256; -- finesse <-> 1.0 on map
  CosT, SinT : Array (0..2047) of Integer;

  fine_land_X: constant:= finesse * (landX+1);
  fine_land_Y: constant:= finesse * (landY+1);

  -- Distance compensation table
  DComp : Array (1..Depth + 1) of Integer;
  -- Camera information
  x, y, Angle, Height, Base_Height: Integer;
  h_factor: float;
  -- Keyboard input
  c: character;
  use ASCII;

-- Casts a ray from specified position and renders result
procedure Ray (a, x0, y0: integer; sX: X_Loc) is
  x: integer:= x0;
  y: integer:= y0;
  minY : integer:= ResY;
  deltax, deltay, d, hu, y1, xmap, ymap : integer;

begin
  deltax := CosT (a);
  deltay := SinT (a);
  hu:= 64 * Height;
  d := 0;

  loop
    -- New distance
    d:= d+1;
    x:= (x + deltax) mod fine_land_X;
    y:= (y + deltay) mod fine_land_Y;
    xmap:= x / finesse;
    ymap:= y / finesse;
    -- Calculate height
    y1 := DComp (d) - (128 * integer(AHMap(xmap,ymap)) - hu) / d;
    if y1 < 1 then y1:= 1; end if;
    if y1 < minY then
      -- Draw voxel
      Ver_Line(Buffer.all, sX, Y_Loc(y1), Y_Loc(minY - 1), TMap(xmap,ymap) );
      minY := y1;
    end if;
    exit when d >= Depth;
  end loop;
end Ray;

pragma Inline(Ray);

procedure DrawView is
  a: Integer;
begin
  -- Fill every column on-screen
  for i in 0..ResX-1 loop
    -- Calculate ray angle depending on view angle
    a := (Angle + i -ResX/2) mod (CosT'length);
    -- Cast the ray
    Ray (a, x, y, X_Loc(i));
  end loop;
end DrawView;

procedure InitMap is
  RGB: RGB_Color;
  f : DBIO.File_type;
  d: byte;

begin
  -- Load in the file "GROUND.BMP"
  Open(f, in_file, "GROUND.BMP");
  -- The palette starts at byte 54
  Set_index (f, 55);

  -- Read palette
  for DAC in Color_Type loop
    -- .BMP format saves 4 bytes per colour
    -- In reverse order, and in the range of 0..255 instead of 0..63
    Read(f, d); RGB.Blue  := Color_Value(d / 4);
    Read(f, d); RGB.Green := Color_Value(d / 4);
    Read(f, d); RGB.Red   := Color_Value(d / 4);
    Read(f, d);
    -- Set the palette
    Set_Color (DAC, RGB);
  end loop;

  -- Load in both parts of the texture map
  for y in TMap'range(2) loop
    for x in TMap'range(1) loop
      Read(f,d); TMap(x,y):= Color_Type(d);
    end loop;
  end loop;
  -- And close the file
  Close (f);

  -- Load in the file "HEIGHT.BMP"
  Open(f, in_file, "HEIGHT.BMP");
  -- Skip the palette for this one
  Set_index (f, 1079);

  for y in HMap'range(2) loop
    for x in HMap'range(1) loop
      Read(f,HMap(x,y));
    end loop;
  end loop;
  Close (f);

exception
  when DBIO.Name_Error =>
    Text_Mode; Put_Line("GROUND.BMP or HEIGHT.BMP is missing"); Raise;

end InitMap;

-- This procedure calculates some lookup table
procedure InitTables is
f: constant float:= float(finesse);
begin
  for a in CosT'range loop
    -- Precalculate cosine
    CosT (a) := integer(Cos(2.0*pi * float(a) / float(CosT'length)) * f);
    -- and sine
    SinT (a) := integer(Sin(2.0*pi * float(a) / float(SinT'length)) * f);
  end loop;
  -- Precalculate distance compensation table
  for a in DComp'range loop
    DComp(a) := (ResY*5) / a + ResY/2;
  end loop;
end InitTables;

procedure Adjust_height is
  begin
    for x in HMap'range(1) loop
      for y in HMap'range(2) loop
        AHMap(x,y):= byte(float(HMap(x,y)) * h_factor);
      end loop;
    end loop;
  end;

 begin
  InitTables;   -- Precalculate sine and cosine
  InitMap;      -- Create the map
  Buffer:= New Screen_Buffer(resX,resY);

  -- Set up player
  x := 16#8000#;
  y := 16#8000#;
  Angle := 600;
  Base_Height:= 5;
  h_factor:= 0.25;
  Adjust_height;

  loop
    x:= x mod fine_land_X;
    y:= y mod fine_land_Y;

    -- Adjust camera height according to height map
    Height := integer(HMap (x / finesse, y / finesse)) + Base_Height;

    Clear_Screen (Buffer.all);   --  Clear the buffer
    DrawView;                    --  Draw a screen
    Put_Buffer (Buffer.all);     --  Copy the buffer to the screen

    Get_Immediate(c);            --  Get keyboard input
    case c is
      when NUL =>
        Get_Immediate(c);
        case c is
          when 'H'=> -- move forward
             x:= x + CosT (Angle);
             y:= y + SinT (Angle);
          when 'K'=> -- turn left
             Angle := (Angle - 16) mod CosT'length;
          when 'M'=> -- turn right
             Angle := (Angle + 16) mod CosT'length;
          when 'P'=> -- turn backward
             x:= x - CosT (Angle);
             y:= y - SinT (Angle);
          when others=> null;
        end case;
      when 'A'|'a' =>         Base_Height:= Base_Height + 5;
      when 'Y'|'y'|'Z'|'z' => Base_Height:= Base_Height - 5;
      when 'S'|'s' =>         if h_factor < 1.0 then
                                h_factor:= h_factor + 0.05;
                                Adjust_height;
                              end if;
      when 'X'|'x' =>         if h_factor > 0.05 then 
                                h_factor:= h_factor - 0.05;
                                Adjust_height;
                              end if;
      when ESC|'q'|'Q' => exit;
      when others=> null;
    end case;

  end loop;
 end Voxels_in_graphics;

c: integer; 
stop: boolean:= false; can_draw: boolean;

begin
  loop
    Put_Line("Voxel demo");
    New_Line;
    Put_Line("Choose resolution");
    Put_Line("  1)  VESA 640x400,   256 colours");
    Put_Line("  2)  VESA 640x480,   256 colours");
    Put_Line("  3)  VESA 800x600,   256 colours");
    Put_Line("  4)  VESA 1024x768,  256 colours");
    Put_Line("  5)  VESA 1280x1024, 256 colours");
    New_Line;
    Put_Line("  0)  Quit");
     -- Switch to graphics mode
    New_Line;
    can_draw:= true;
    Put("Choice: "); Get(c); Skip_Line; New_Line;
    begin
     case c is
      when 1 => Graphics_Mode (M640x400);
      when 2 => Graphics_Mode (M640x480);
      when 3 => Graphics_Mode (M800x600);
      when 4 => Graphics_Mode (M1024x768);
      when 5 => Graphics_Mode (M1280x1024);
      when 0 => stop:= true;
      when others => Put_Line("Bad choice..."); can_draw:= false;
     end case;
    exception
     when others => Put_Line("Unsupported display type..."); can_draw:= false;
    end;
    exit when stop;
    if can_draw then
      Voxels_in_graphics;
      -- Switch back to text
      Wait_For_Vertical_Retrace;
      Text_Mode;
    end if;
  end loop;
end Voxels;
