-----------------------------------------------------------------------
--
--  File:        conio.adb
--  Description: DJGPP console I/O
--  Rev:         0.2
--  Date:        03/01/97
--  Author:      Jerry van Dijk
--  Mail:        jerry@jvdsys.nextjk.stuyts.nl
--
--  Copyright (c) Jerry van Dijk, 1997
--  Forelstraat 211
--  2037 KV  HAARLEM
--  THE NETHERLANDS
--  tel int + 31 23 540 1052
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------
with Interfaces, Interfaces.C, Interfaces.C.Strings, Unchecked_Conversion;
use  Interfaces, Interfaces.C, Interfaces.C.Strings;

package body Conio is

   -------------------------
   -- The DJGPP interface --
   -------------------------
   Djgpp_Wscroll : int;
   pragma Import (C, Djgpp_Wscroll, "_wscroll");

   type Djgpp_Textinfo is
      record
         Winleft      : Unsigned_8;
         Wintop       : Unsigned_8;
         Winright     : Unsigned_8;
         Winbottom    : Unsigned_8;
         Attribute    : Unsigned_8;
         Normattr     : Unsigned_8;
         Currmode     : Unsigned_8;
         Screenheight : Unsigned_8;
         Screenwidth  : Unsigned_8;
         Curx         : Unsigned_8;
         Cury         : Unsigned_8;
      end record;
   pragma Convention (C, Djgpp_Textinfo);

   procedure Djgpp_Cputs (Cp : in Chars_Ptr);
   pragma Import (C, Djgpp_Cputs, "cputs");

   procedure Djgpp_Cgets (Cp : in Chars_Ptr);
   pragma Import (C, Djgpp_Cgets, "cgets");

   function Djgpp_Kbhit return Int;
   pragma Import (C, Djgpp_Kbhit, "_conio_kbhit");

   procedure Djgpp_Putch(C : in Char);
   pragma Import (C, Djgpp_Putch, "putch");

   procedure Djgpp_Setcursortype(Size : in Integer);
   pragma Import (C, Djgpp_Setcursortype, "_setcursortype");

   procedure Djgpp_Textmode (Mode : in Integer);
   pragma Import (C, Djgpp_Textmode, "textmode");

   procedure Djgpp_Set_Screen_Lines (N : in Integer);
   pragma Import (C, Djgpp_Set_Screen_Lines, "_set_screen_lines");

   procedure Djgpp_Textcolor (Color : in Integer);
   pragma Import (C, Djgpp_Textcolor, "textcolor");

   procedure Djgpp_Textbackground (Color : in Integer);
   pragma Import (C, Djgpp_Textbackground, "textbackground");

   procedure Djgpp_Textattr (Attr : in Int);
   pragma Import (C, Djgpp_Textattr, "textattr");

   procedure Djgpp_Gettextinfo (Info : out Djgpp_Textinfo);
   pragma Import (C, Djgpp_Gettextinfo, "gettextinfo");

   function Djgpp_Movetext
     (X1, Y1, X2, Y2, New_X, New_Y : in Integer) return Integer;
   pragma Import (C, Djgpp_Movetext, "movetext");

   function Djgpp_Gettext
     (X1, Y1, X2, Y2 : in Integer; Cp : in chars_ptr) return int;
   pragma Import (C, Djgpp_Gettext, "gettext");

   function Djgpp_Puttext
     (X1, Y1, X2, Y2 : in Integer; Cp : in chars_ptr) return int;
   pragma Import (C, Djgpp_Puttext, "puttext");

   ---------------------------
   -- Unchecked Conversions --
   ---------------------------
   function To_C is new Unchecked_Conversion (Text_Attribute, char);
   function To_Ada is new Unchecked_Conversion (char, Text_Attribute);
   function To_Unsigned_8 is new
     Unchecked_Conversion (Text_Attribute, Unsigned_8);
   function To_Attr is
     new Unchecked_Conversion (Unsigned_8, Text_Attribute);

   --------------------------------------
   -- Puts the string onto the console --
   --------------------------------------
   procedure Cputs (S : in String) is
      Str : chars_ptr := New_String (S);
   begin
      Djgpp_Cputs (Str);
      Free (Str);
   end Cputs;

   ------------------------------
   -- Read string from console --
   ------------------------------
   procedure Cgets (S : out String; N : out Natural) is
      I   : size_t;
      Str : chars_ptr;
      ST  : String (1 .. S'Length + 3);
   begin
      ST (1) := Character'Val (S'Length);
      Str := New_String (ST);
      Djgpp_Cgets (Str);
      I := Strlen (Str);
      ST (1 .. Integer (I)) := Value (Str);
      Free (Str);
      N := Character'Pos (ST (2));
      S (1 .. N) := ST (3 .. N + 2);
      S (N + 1 .. S'Last) := (others => ' ');
   end Cgets;

   -------------------------------------------------------
   -- Determine if character is waiting at the keyboard --
   -------------------------------------------------------
   function Kbhit return Boolean is
      I      : int;
      Result : Boolean := False;
   begin
      I := Djgpp_Kbhit;
      if I /= 0 then
         Result := True;
      end if;
      return Result;
   end Kbhit;

   ----------------------------------------
   -- Print character at cursor position --
   ----------------------------------------
   procedure Putch (C : in Character) is
   begin
      Djgpp_Putch (To_C (C));
   end Putch;

   ---------------------
   -- Set cursor size --
   ---------------------
   procedure Setcursortype (Size : in Cursor_Size) is
   begin
      Djgpp_Setcursortype (Cursor_Size'Enum_rep (Size));
   end Setcursortype;

   -------------------------
   -- Set a new text mode --
   -------------------------
   procedure Textmode (Mode : in Text_Mode) is
   begin
      Djgpp_Textmode (Text_Mode'Enum_rep (Mode));
   end Textmode;

   ------------------------------------
   -- Set the number of screen lines --
   ------------------------------------
   procedure Set_Screen_Lines (N : in Num_Lines) is
   begin
      Djgpp_Set_Screen_Lines (Num_Lines'Enum_rep (N));
   end Set_Screen_Lines;

   ------------------------------
   -- Set new foreground color --
   ------------------------------
   procedure Textcolor (Color : in Foreground_Color) is
   begin
      Djgpp_Textcolor (Foreground_Color'Enum_rep (Color));
   end Textcolor;

   ------------------------------
   -- Set new background color --
   ------------------------------
   procedure Textbackground (Color : in Background_Color) is
   begin
      Djgpp_Textbackground (Background_Color'Enum_rep (Color));
   end Textbackground;

   ----------------------------
   -- Set new text attribute --
   ----------------------------
   procedure Textattr (Attr : in Text_Attribute) is
   begin
      Djgpp_Textattr (Int (To_Unsigned_8 (Attr)));
   end Textattr;

   -----------------------------
   -- Fill a Text_Info record --
   -----------------------------
   procedure Gettextinfo (Info : out Text_Info) is
      Current_Info : Djgpp_Textinfo;
   begin
      Djgpp_Gettextinfo (Current_Info);
      Info.Curr_X        := X_Pos (Current_Info.Curx);
      Info.Curr_Y        := Y_Pos (Current_Info.Cury);
      Info.Window_Top    := Y_Pos (Current_Info.Wintop);
      Info.Window_Left   := X_Pos (Current_Info.Winleft);
      Info.Window_Right  := X_Pos (Current_Info.Winright);
      Info.Window_Bottom := Y_Pos (Current_Info.Winbottom);
      Info.Norm_Attr     := To_Attr (Current_Info.Normattr);
      Info.Screen_Width  := X_Pos (Current_Info.Screenwidth);
      Info.Curr_Attr     := To_Attr (Current_Info.Attribute);
      Info.Screen_Height := Y_Pos (Current_Info.Screenheight);
   end Gettextinfo;

   --------------------------
   -- Set window scrolling --
   --------------------------
   procedure Wscroll (Scroll : in Boolean) is
   begin
      if Scroll = True then
         Djgpp_Wscroll := 1;
      else
         Djgpp_Wscroll := 0;
      end if;
   end Wscroll;

   --------------------------------------------------
   -- Copy an area on the screen to a new location --
   --------------------------------------------------
   function Movetext (Left     : in X_Pos; Top     : in Y_Pos;
                      Right    : in X_Pos; Bottom  : in Y_Pos;
                      New_Left : in X_Pos; New_Top : in Y_Pos)
                      return Boolean is
      Result : Boolean;
   begin
      if Djgpp_Movetext(Left, Top, Right, Bottom, New_Left, New_Top) = 0 then
         Result := False;
      else
         Result := True;
      end if;
      return Result;
   end Movetext;

   -------------------------------------------
   -- Read screen area into a Screen_Buffer --
   -------------------------------------------
   procedure Gettext (Left   : in     X_Pos; Top    : in Y_Pos;
                      Right  : in     X_Pos; Bottom : in Y_Pos;
                      Buffer :    out Screen_Buffer;
                      Result :    out Boolean) is
      R   : int;
      I   : size_t := 1;
      Buf : char_array
        (1 .. size_t (2 * (Right - Left + 1) * (Bottom - Top + 1)))
        := (others => To_C (Character'Val (0)));
      Str : chars_ptr := New_Char_Array (Buf);
   begin
      Result := False;
      R := Djgpp_Gettext (Left, Top, Right, Bottom, Str);
      if R /= 0 then
         Result := True;
         Buf := Value (Str,
           size_t (2 * (Right - Left + 1) * (Bottom - Top + 1)));
         for X in Left .. Right loop
            for Y in Top .. Bottom  loop
               Buffer (X,Y).Value := To_Ada (Buf (I));
               I := I + 1;
               Buffer (X,Y).Attr := To_Ada (Buf (I));
               I := I + 1;
            end loop;
         end loop;
      end if;
      Free (Str);
   end Gettext;

   ------------------------------------------
   -- Writes a screen buffer on the screen --
   ------------------------------------------
   function Puttext (Left   : in X_Pos; Top    : in Y_Pos;
                     Right  : in X_Pos; Bottom : in Y_Pos;
                     Buffer : in Screen_Buffer) return Boolean is
      R      : int;
      Str    : chars_ptr;
      I      : size_t := 1;
      Result : Boolean := False;
      Buf    : char_array (1 .. 2 * Buffer'Length (1) * Buffer'Length (2));
   begin
      for X in Left .. Right loop
         for Y in Top .. Bottom loop
            Buf (I) := To_C (Buffer (X, Y).Value);
            I := I + 1;
            Buf (I) := To_C (Buffer (X, Y).Attr);
            I := I + 1;
         end loop;
      end loop;
      Str := New_Char_Array (Buf);
      R := Djgpp_Puttext (Left, Top, Right, Bottom, Str);
      Free (Str);
      if R /= 0 then
         Result := True;
      end if;
      return Result;
   end Puttext;

end Conio;
