-----------------------------------------------------------------------
--
--  File:        vgagraph.ads
--  Description: Basic VGA graphics for GNAT/DOS
--  Rev:         0.7
--  Date:        01-feb-98
--  Author:      Jerry van Dijk
--  Mail:        jdijk@acm.org
--
--  Copyright (c) Jerry van Dijk, 1996, 1997, 1998
--  Billie Holidaystraat 28
--  2324 LK Leiden
--  THE NETHERLANDS
--  tel int + 31 71 531 4365
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------


   -----------------------------------------------------------------------
   --                       PACKAGE CONTAINS                            --
   -----------------------------------------------------------------------
   -- Types:                                                            --
   --    Color_Palette        - The color palette                       --
   --    Color_Value          - Individual color within the palette     --
   --    Horizontal_Location  - Horizontal position on screen           --
   --    Put_Rule_Type        - Screen_Buffer put mode                  --
   --    Screen_Buffer        - Screen area buffer                      --
   --    Screen_Color         - The standard colors                     --
   --    Vertical_Location    - Vertical position on screen             --
   --                                                                   --
   -- Constants:                                                        --
   --    Font_Height          - Character height in pixels              --
   --    Font_Width           - Character width in pixels               --
   --    Horizontal_Maximum   - Maximum column number                   --
   --    Horizontal_Size      - Number of horizontal pixels             --
   --    Num_Colors           - The number of avaliable colors          --
   --    Vertical_Maximum     - Maximum line number                     --
   --    Vertical_Size        - Number of vertical pixels               --
   --                                                                   --
   -- Exceptions:                                                       --
   --    Flood_Fill_Failed    - Not full region filled                  --
   --    No_VGA_Error         - Couldn't switch to VGA mode             --
   --    No_TXT_Mode_Error    - Couldn't return to Text mode            --
   --    Not_In_VGA_Error     - Executing VGA command in Text mode      --
   --    Outside_Screen_Error - Position outside screen boundary        --
   --    Screen_Buffer_Error  - Buffer exceeds screen size              --
   --                                                                   --
   -- Functions:                                                        --
   --    Get_Palette          - Gets an individual color from a palette --
   --    Get_Pixel            - Returns the color of a pixel            --
   --    Read_All_Palette     - Returns the current palette             --
   --                                                                   --
   -- Procedures:                                                       --
   --    Clear_Screen         - Clears the screen by filling it         --
   --    Draw_Circle          - Draw a full single line circle          --
   --    Draw_Line            - Draw a single line                      --
   --    Fill_Box             - Fills a rectangular box                 --
   --    Flood_Fill           - Fills a screen region with single color --
   --    Get_Buffer           - Copy a screen area to a Screen_Buffer   --
   --    Horizontal_Line      - Draw a single horizontal line           --
   --    Put_Buffer           - Copy a Screen_Buffer to the screen      --
   --    Put_Pixel            - Set a pixel to the specified color      --
   --    Set_Palette          - Sets a palette color to a new value     --
   --    TXT_Mode             - Return from VGA Mode to text mode       --
   --    Vertical_Line        - Draw a single vertical line             --
   --    Write_All_Palette    - Makes the palette the current palette   --
   --    Write_String         - Writes a string in the system font      --
   --    VGA_Mode             - Switch the display in VGA Mode          --
   -----------------------------------------------------------------------


with Interfaces;       use Interfaces;
with Interfaces.C;     use Interfaces.C;
with Ada.Finalization; use Ada.Finalization;

package VGA_Graphics is

-------------------
--  PUBLIC PART  --
-------------------

   ------------------------------------------------------
   -- NAME:    Package Exceptions                      --
   --                                                  --
   -- PURPOSE: Defines the package specific exceptions --
   ------------------------------------------------------
   Flood_Fill_Failed    : exception;      -- Not full region filled
   No_VGA_Error         : exception;      -- Couldn't switch to VGA mode
   Not_In_VGA_Error     : exception;      -- Executing VGA command in Text mode
   No_TXT_Mode_Error    : exception;      -- Couldn't return to Text mode
   Screen_Buffer_Error  : exception;      -- Buffer exceeds screen size
   Outside_Screen_Error : exception;      -- Position outside screen boundary


   --------------------------------------------------
   -- NAME:    Font Size                           --
   --                                              --
   -- PURPOSE: Defines the size of the system font --
   --------------------------------------------------
   Font_Width  : constant Integer :=  8;  -- Character width in pixels
   Font_Height : constant Integer := 14;  -- Character height in pixels


   ----------------------------------------------------------------
   -- NAME:    Screen Size                                       --
   --                                                            --
   -- PURPOSE: Defines the VGA screen dimensions and coordinates --
   ----------------------------------------------------------------
   Vertical_Maximum   : constant := 479;  -- Maximum line number
   Horizontal_Maximum : constant := 639;  -- Maximum column number

   Vertical_Size   : constant := 480;     -- Number of vertical pixels
   Horizontal_Size : constant := 640;     -- Number of horizontal pixels

   subtype Vertical_Location   is Natural range 0..Vertical_Maximum;
   subtype Horizontal_Location is Natural range 0..Horizontal_Maximum;


   ---------------------------------------------------------------------
   -- NAME:    Color Definitions                                      --
   --                                                                 --
   -- PURPOSE: Defines the standard color names                       --
   --                                                                 --
   -- NOTES:   1. These names refer to the package standard colors,   --
   --             they will not be correct if the palette is changed  --
   --          2. The Border color is only used to set the color of   --
   --             the overscan area and shouldn't normally be used    --
   ---------------------------------------------------------------------
   Num_Colors : constant := 16;  -- The number of avaliable colors

   type Screen_Color is (Black, Blue, Green, Cyan, Red, Magenta, Brown,
                         Light_Gray, Dark_Gray, Light_Blue, Light_Green,
                         Light_Cyan, Light_Red, Light_Magenta, Yellow,
                         White, Border);


   --------------------------------------------------------------
   -- NAME:    Color Palette                                   --
   --                                                          --
   -- PURPOSE: Defines the color palette and color value types --
   --------------------------------------------------------------
   type Color_Palette is private;      -- The color palette
   subtype Color_Value is Unsigned_8;  -- Individual color within the palette


   -------------------------------------------------------------------
   -- NAME:    Screen Buffer                                        --
   --                                                               --
   -- PURPOSE: Defines the Screen_Buffer and its manipulation types --
   --                                                               --
   -- NOTES:   1. Buffer size is defined upon creation              --
   -------------------------------------------------------------------
   type Put_Rule_Type is (Put_Force, Put_And, Put_Or, Put_Xor);
   for Put_Rule_Type use (Put_Force =>  0,   -- Replace buffer area
                          Put_And   =>  8,   -- AND buffer with area
                          Put_Or    => 16,   -- OR buffer with area
                          Put_Xor   => 24);  -- XOR buffer with area

   type Screen_Buffer(Buf_Width  : Horizontal_Location;
                      Buf_Height : Vertical_Location) is limited private;


   ------------------------------------------------
   -- NAME:    VGA_Mode                          --
   --                                            --
   -- PURPOSE: Switch the display in VGA Mode    --
   --                                            --
   -- EXCEPTS: No_VGA_Error - if the call failed --
   ------------------------------------------------
   procedure VGA_Mode;


   ----------------------------------------------------------------
   -- NAME:    TXT_Mode                                          --
   --                                                            --
   -- PURPOSE: Return from VGA Mode to original text mode        --
   --                                                            --
   -- EXCEPTS: Not_In_VGA_Error  - if not in VGA mode            --
   --          No_TXT_Mode_Error - if return to textmode failed  --
   --                                                            --
   -- NOTES:   1. Always returns to the mode from which the last --
   --             VGA_Mode was called, not to a fixed textmode   --
   --          2. Doesn't remember previous mode changes         --
   ----------------------------------------------------------------
   procedure TXT_Mode;


   -------------------------------------------------------------------
   -- NAME:    Clear_Screen                                         --
   --                                                               --
   -- PURPOSE: Clears the screen by filling it with a single color  --
   --                                                               --
   -- INPUTS:  Color - Color to fill screen with, default is black  --
   --                                                               --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                --
   --                                                               --
   -- NOTES:   1. Faster then filling the screen with a loop as it  --
   --             uses a dedicated algorithm                        --
   -------------------------------------------------------------------
   procedure Clear_Screen(Color : in Screen_Color := Black);


   ----------------------------------------------------
   -- NAME:    Put_Pixel                             --
   --                                                --
   -- PURPOSE: Set a pixel to the specified color    --
   --                                                --
   -- INPUTS:  X_Pos - Horizontal pixel position     --
   --          Y_Pos - Vertical pixel position       --
   --          Color - Color to set the pixel to     --
   --                                                --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode --
   --                                                --
   -- NOTES:   1. Position only garanteed if range   --
   --             checking is enabled                --
   ----------------------------------------------------
   procedure Put_Pixel(X_Pos : in Horizontal_Location;
                       Y_Pos : in Vertical_Location;
                       Color : in Screen_Color);


   ----------------------------------------------------
   -- NAME:    Get_Pixel                             --
   --                                                --
   -- PURPOSE: Get the color of the specified pixel  --
   --                                                --
   -- INPUTS:  X_Pos - Horizontal pixel position     --
   --          Y_Pos - Vertical pixel position       --
   --                                                --
   -- RETURNS: The color of the pixel                --
   --                                                --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode --
   --                                                --
   -- NOTES:   1. Result only garanteed if range     --
   --             checking is enabled                --
   ----------------------------------------------------
   function Get_Pixel(X_Pos : in Horizontal_Location;
                      Y_Pos : in Vertical_Location)
                      return Screen_Color;


   ------------------------------------------------------------------------
   -- NAME:    Vertical_Line                                             --
   --                                                                    --
   -- PURPOSE: Draw a single vertical line                               --
   --                                                                    --
   -- INPUTS:  X_Pos  - Constant horizontal position                     --
   --          First  - Vertical starting point                          --
   --          Last   - Vertical ending point                            --
   --          Color  - Line color                                       --
   --                                                                    --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                     --
   --                                                                    --
   -- NOTES:   1. If First is equal to Last, a single pixel is set       --
   --          2. If First is larger then Last the direction is reversed --
   --          3. Position only garanteed if range checking is enabled   --
   --                                                                    --
   -- WARNING: Clashes with Ada.Characters.Latin_1.Vertical_Line !!!     --
   ------------------------------------------------------------------------
   procedure Vertical_Line(X_Pos : in Horizontal_Location;
                           First : in Vertical_Location;
                           Last  : in Vertical_Location;
                           Color : in Screen_Color);


   ------------------------------------------------------------------------
   -- NAME:    Horizontal_Line                                           --
   --                                                                    --
   -- PURPOSE: Draw a single horizontal line                             --
   --                                                                    --
   -- INPUTS:  First  - Horizontal starting point                        --
   --          Last   - Horizontal ending point                          --
   --          Y_Pos  - Constant vertical position                       --
   --          Color  - Line color                                       --
   --                                                                    --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                     --
   --                                                                    --
   -- NOTES:   1. If First is equal to Last, a single pixel is set       --
   --          2. If First is larger then Last the direction is reversed --
   --          3. Position only garanteed if range checking is enabled   --
   ------------------------------------------------------------------------
   procedure Horizontal_Line(First : in Horizontal_Location;
                             Last  : in Horizontal_Location;
                             Y_Pos : in Vertical_Location;
                             Color : in Screen_Color);


   -----------------------------------------------------------------------
   -- NAME:    Draw_Line                                                --
   --                                                                   --
   -- PURPOSE: Draw a single line                                       --
   --                                                                   --
   -- INPUTS:  X1    - Horizontal start position                        --
   --          Y1    - Vertical start position                          --
   --          X2    - Horizontal end position                          --
   --          Y2    - Vertical end position                            --
   --          Color - Line color                                       --
   --                                                                   --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                    --
   --                                                                   --
   -- NOTES:   1. If the line is strictly vertical or horizontal,       --
   --             the thus specialized functions are much faster        --
   --          2. If X1 > X2 or Y1 > Y2 the direction is reversed       --
   --          3. if X1 = X2 and Y1 = Y2 a single pixel is set          --
   --          4. Position only garanteed if range checking is enabled  --
   -----------------------------------------------------------------------
   procedure Draw_Line(X1    : in Horizontal_Location;
                       Y1    : in Vertical_Location;
                       X2    : in Horizontal_Location;
                       Y2    : in Vertical_Location;
                       Color : in Screen_Color);


   ---------------------------------------------------------------
   -- NAME:    Draw_Circle                                      --
   --                                                           --
   -- PURPOSE: Draw a full single line circle                   --
   --                                                           --
   -- INPUTS:  X_Pos  - Horizontal position of center           --
   --          Y_Pos  - Vertical position of center             --
   --          Radius - Circle radius in pixels                 --
   --          Color  - Circle line color                       --
   --                                                           --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode        --
   --          Outside_Screen_Error - writes outside the screen --
   ---------------------------------------------------------------
   procedure Draw_Circle(X_Pos  : in Horizontal_Location;
                         Y_Pos  : in Vertical_Location;
                         Radius : in Positive;
                         Color  : in Screen_Color);


   ----------------------------------------------------------------------
   -- NAME:    Fill_Box                                                --
   --                                                                  --
   -- PURPOSE: Fills a rectangular box with a single color             --
   --                                                                  --
   -- INPUTS:  X1    - Horizontal start position                       --
   --          Y1    - Vertical start position                         --
   --          X2    - Horizontal end position                         --
   --          Y2    - Vertical end position                           --
   --          Color - Fill color                                      --
   --                                                                  --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   --                                                                  --
   -- NOTES:   1. If X1 = X2 and Y1 = Y2 a single pixel is set         --
   --          2. If X1 = X2 or Y1 = Y2 a single line is drawn         --
   --          3. If X1 > X2 or Y1 > Y2 the direction is reversed      --
   --          4. Position only garanteed if range checking is enabled --
   ----------------------------------------------------------------------
   procedure Fill_Box(X1    : in Horizontal_Location;
                      Y1    : in Vertical_Location;
                      X2    : in Horizontal_Location;
                      Y2    : in Vertical_Location;
                      Color : in Screen_Color);


   -----------------------------------------------------------------
   -- NAME:    Flood_Fill                                         --
   --                                                             --
   -- PURPOSE: Fills a screen region with a single color          --
   --                                                             --
   -- INPUTS:  X     - Horizontal position within region          --
   --          Y     - Vertical position within region            --
   --          Color - Fill color                                 --
   --                                                             --
   -- EXCEPTS: Not_In_VGA_Error  - if not in VGA mode             --
   --          Flood_Fill_Failed - Region not completedly filled  --
   --                                                             --
   -- NOTES:   1. Works best on small areas                       --
   --          2. Large areas are slow because of stack unwinding --
   -----------------------------------------------------------------
   procedure Flood_Fill(X     : in Horizontal_Location;
                        Y     : in Vertical_Location;
                        Color : in Screen_Color);


   ---------------------------------------------------------------
   -- NAME:    Write_String                                     --
   --                                                           --
   -- PURPOSE: Writes a string in the system font               --
   --                                                           --
   -- INPUTS:  X_Pos - Horizontal start position                --
   --          Y_Pos - Vertical start position                  --
   --          Color - The text color                           --
   --          Text  - The string to write                      --
   --                                                           --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode        --
   --          Outside_Screen_Error - writes outside the screen --
   --                                                           --
   -- NOTES:   1. X_Pos, Y_Pos is upper-left position           --
   --          2. Position only garanteed if range checking is  --
   --             enabled                                       --
   ---------------------------------------------------------------
   procedure Write_String(X_Pos : in Horizontal_Location;
                          Y_Pos : in Vertical_Location;
                          Color : in Screen_Color;
                          Text  : in String);


   ----------------------------------------------------------------------
   -- NAME:    Read_All_Palette                                        --
   --                                                                  --
   -- PURPOSE: Returns the current palette                             --
   --                                                                  --
   -- RETURNS: The palette                                             --
   --                                                                  --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   --          Storage_Error    - if unable to allocate a conventional --
   --                             memory transfer buffer               --
   ----------------------------------------------------------------------
   function Read_All_Palette return Color_Palette;


   ----------------------------------------------------------------------
   -- NAME:    Write_All_Palette                                       --
   --                                                                  --
   -- PURPOSE: Makes the palette the current palette                   --
   --                                                                  --
   -- INPUTS:  Palette - The color palette to set                      --
   --                                                                  --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   --          Storage_Error    - if unable to allocate a conventional --
   --                             memory transfer buffer               --
   ----------------------------------------------------------------------
   procedure Write_All_Palette(Palette : in Color_Palette);


   ---------------------------------------------------------------------
   -- NAME:    Set_Palette                                            --
   --                                                                 --
   -- PURPOSE: Sets a palette color to a new value                    --
   --                                                                 --
   -- INPUTS:  Palette - The color palette to change                  --
   --          Color   - The color value within the palette to change --
   --          Value   - The new value for the color                  --
   --                                                                 --
   -- OUTPUTS: Palette - Loaded with current color palette            --
   --                                                                 --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                  --
   --                                                                 --
   -- NOTES:   1. See any PC reference how to code a RGB color value  --
   --          2. Changing the palette invalidates screen color names --
   ---------------------------------------------------------------------
   procedure Set_Palette(Palette : in out Color_Palette;
                         Color   : in     Screen_Color;
                         Value   : in     Color_Value);


   ----------------------------------------------------------------------
   -- NAME:    Get_Palette                                             --
   --                                                                  --
   -- PURPOSE: Gets an individual palette color from a palette         --
   --                                                                  --
   -- INPUTS:  Palette - The color palette to read                     --
   --          Color   - The color value within the palette to read    --
   --                                                                  --
   -- RETURNS: The color value                                         --
   --                                                                  --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   --                                                                  --
   -- NOTES:   1. See any PC reference how to decode a RGB color value --
   ----------------------------------------------------------------------
   function Get_Palette(Palette : in Color_Palette;
                        Color   : in Screen_Color)
                        return Color_Value;


   -------------------------------------------------------------------------
   -- NAME:    Get_Buffer                                                 --
   --                                                                     --
   -- PURPOSE: Copy a screen area to a Screen_Buffer                      --
   --                                                                     --
   -- INPUTS:  Buffer - The screen buffer                                 --
   --          Left   - Left position of screen area                      --
   --          Top    - Top position of screen area                       --
   --                                                                     --
   -- OUTPUTS: Buffer - Loaded with area                                  --
   --                                                                     --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode                  --
   --          Outside_Screen_Error - Area outside the screen             --
   --                                                                     --
   -- NOTES:   1. Area size is defined when creating the Screen_Buffer    --
   --          2. Position only garanteed if range checking is enabled    --
   -------------------------------------------------------------------------
   procedure Get_Buffer(Buffer : in out Screen_Buffer;
                        Left   : in     Horizontal_Location;
                        Top    : in     Vertical_Location);


   ----------------------------------------------------------------------
   -- NAME:    Put_Buffer                                              --
   --                                                                  --
   -- PURPOSE: Copy a Screen_Buffer to the screen                      --
   --                                                                  --
   -- INPUTS:  Buffer - The screen buffer                              --
   --          Left   - Left position of screen area                   --
   --          Top    - Top position of screen area                    --
   --          Rule   - Screen buffer put mode                         --
   --                                                                  --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode               --
   --          Outside_Screen_Error - Writin outside the screen        --
   --                                                                  --
   -- NOTES:   1. Area size is defined when creating the Screen_Buffer --
   --          2. Position only garanteed if range checking is enabled --
   ----------------------------------------------------------------------
   procedure Put_Buffer(Buffer : in Screen_Buffer;
                        Left   : in Horizontal_Location;
                        Top    : in Vertical_Location;
                        Rule   : in Put_Rule_Type := Put_Force);


--------------------
--  PRIVATE PART  --
--------------------
private

   ---------------------------------------------------------
   -- NAME:    Planes                                     --
   --                                                     --
   -- PURPOSE: Defines the array that the Screen_Buffer   --
   --          uses to store the screen area color planes --
   ---------------------------------------------------------
   type Planes is array(Natural range <>) of unsigned_8;
   type Planes_Access is access Planes;


   ------------------------------------------------------------------
   -- NAME:    Screen Buffer                                       --
   --                                                              --
   -- PURPOSE: Defines the Screen_Buffer                           --
   --                                                              --
   -- NOTES:   1. Buffer size is defined upon creation             --
   --          2. The Initialize and Finalize procedures           --
   --             take care of the actual allocation and           --
   --             deallocation of buffer memory                    --
   --          2. Size only garanteed if range checking is enabled --
   ------------------------------------------------------------------
   type Screen_Buffer(Buf_Width  : Horizontal_Location;
                      Buf_Height : Vertical_Location)
                      is new Limited_Controlled with
      record
         Width  : Horizontal_Location := Buf_Width;
         Height : Vertical_Location   := Buf_Height;
         Data   : Planes_Access;
      end record;

   procedure Initialize(Buffer : in out Screen_Buffer);
   procedure Finalize(Buffer : in out Screen_Buffer);


   ----------------------------------------
   -- NAME:    Color Palette             --
   --                                    --
   -- PURPOSE: Defines the color palette --
   ----------------------------------------
   type Color_Palette is array (Screen_Color) of Color_Value;


end VGA_Graphics;
