/*
 * File......: TRASHCAN.PRG
 * Author....: Martin Colloby
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Martin Colloby
 * Date......: 18/4/93
 * Revision..: 1.0
 *
 * This is an original work by Martin Colloby and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */


/*  $DOC$
 *  $FUNCNAME$
 *      GT_TRASHCAN()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      Copy a record to a trashcan and then delete it
 *  $SYNTAX$
 *      GT_TrashCan( cTable , lSecond , cReason , cDesc )
 *  $ARGUMENTS$
 *      cTable    - Table to be audited
 *      lSecond   - If .T., use two indexes
 *      cReason   - Reason for deletion ( code - C2 )
 *      cDesc     - Reason if code is other - C40
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      The current record in the table cTable is appended to
 *      TABLES\TRASHCAN\cTable
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *      GT_LIB.CH
 *  $END$
 */
*
#include "GT_LIB.ch"

FUNCTION GT_TrashCan( cTable , lSecond , cReason , cDesc )

/*****************************************************************************
 Purpose - Remove the current record from cTable and place it in a trash can
 Returns - None
 Author  - Log
 Created - 22/10/92
 Edited  - 4/11/92 - Log - Now adds a record to the audit trail marking the
                           record as trashed.
******************************************************************************
 Parameters - cTable    - Table to be audited
              lSecond   - If .T., use two indexes
              cReason   - Reason for deletion ( code - C2 )
              cDesc     - Reason if code is other - C40
 Locals     - aFields   - Array to hold fields in cTable
              nCount    - Counting variable
 PUBLICS    - cUserName - Name of user performing operation
******************************************************************************
 N.B. Each application must maintain a table of cReason codes in the format :

            REASON C 2
            DESC   C 40
*****************************************************************************/

LOCAL aFields  := {}
LOCAL aIndexes := {}
LOCAL nCount   := 0

// Select the table
SELECT ( cTable )

// Try to lock the record
IF GT_RecLock( 0 )
    // Copy the contents of the data table into the array
    FOR nCount := 1 TO FCOUNT()
        AADD( aFields , FIELDGET( nCount ) )
    NEXT

    aIndexes := { "trashcan\" + cTable }

    IF lSecond
        AADD( aIndexes , "trashcan\" + ALLTRIM( LEFT( cTable , 7 ) ) + "1" )
    ENDIF

    // Open the Trashcan table
    GT_UseDbf( "trashcan\" + cTable , aIndexes , "trashcan" , .F. , .F. , .F. , 5 )

    // Add a blank record
    GT_AddRec( 5 )

    // Update the key fields
    REPLACE trashcan->trashdate WITH DATE()
    REPLACE trashcan->trashtime WITH TIME()
    REPLACE trashcan->trashuser WITH cUserName
    REPLACE trashcan->trashreas WITH cReason
    REPLACE trashcan->trashdesc WITH cDesc

    // This code block stuffs the contents of the array a_fields into the current
    // table, but places them five fields to the right of their ordinal position
    // in the array - this gets them past the first five key fields
    AEVAL( aFields , { |value, count| FIELDPUT( count + 5 , value ) } )

    // Unlock the trashcan table record
    UNLOCK

    // Release the trashcan table
    SELECT "trashcan"
    USE

    // Go back to the source table
    SELECT ( cTable )

    // Audit this operation
    GT_Audit( cTable , A_TRASHCAN , .F. )

    // Delete the record, and then unlock it
    DELETE
    SKIP
    UNLOCK
    DBCOMMITALL()
ELSE
    GT_Warning( { "Couldn't lock record to trash" } )
ENDIF

RETURN NIL
*
