/*

 Ŀ
                                                                        
  File Name...: MODEM.PRG                                               
  Author......: Vern Six                                                
  Date created: 06-29-94                Date updated: 09-30-94         
  Time created: 01:16:16pm              Time updated: 10:16:17am       
  CopyRight...: (c) 1994 by FrontLine Software                          
                                                                        
 
  

*/

#include "setcurs.ch"
#include "inkey.ch"
#include "box.ch"
#include "telepath.ch"
#include "BAS_VERN.CH"

static snPalette := 5        // palette for basWind() calls


// Rate between your computer and your modem
static snDTE       := 0

// rate between your modem and the user's modem
static snDCE       := 0


// Statics for md_Progress()
static snStartTime := 0
static slStarted   := .f.
static snStartedAt := 0
static snFileSize  := 0
static snBytesSent := 0
static scStatus    := ""
static scFileName  := ""
static slBox       := .f.
static sacFiles    := {}

// statics for md_Dial()
static scLastMsg   := "None"
static slInitYet   := .f.


// port settings...
static snIrq       := 0
static snAddr      := 0

// modem commands...
static scInitSeq   := ""
static scOnHook    := ""
static scOffHook   := ""
static scDialPre   := ""
static scDialSuf   := ""
static scEscape    := ""
static scAnswerSeq := ""

// modem messages...
static scError     := ""
static scBusy      := ""
static scOk        := ""
static scRing      := ""
static scNoCarrier := ""
static scNoTone    := ""
static scNoAnswer  := ""
static scVoice     := ""
static scConn300   := ""
static scConn1200  := ""
static scConn2400  := ""
static scConn9600  := ""
static scConn14400 := ""
static scConn28800 := ""

// modem settings...
static snMaxBaud   := 0
static slSetBaud   := .f.
static slManual    := .f.
static slInitEach  := .f.

// timing settings...
static snPause     := 0
static snDelay     := 0
static snSlowPoke  := 0
static snDtrDelay  := 0
static snDialTime  := 0
static snRedial    := 0

// buffer sizes...
static snInpBuff   := 0
static snOutBuff   := 0
static snXferBuff  := 0

// flow control settings...
static snCtrlCts   := 0
static snCtrlRts   := 0
static snCtrlDsr   := 0
static snCtrlDtr   := 0
static snCtrlDcd   := 0
static snCtrlX     := 0
static snFifo      := 0


/* HYPERTEXT START
!short: md_Dte()        Get the DTE baud rate
md_Dte()        Get the DTE baud rate

^BDescription: ^B

   md_Dte() allows your programs to determine the DTE baud rate (the rate
   between your computer and your modem).


^BSyntax:^B

   nDte := md_Dte( [nNewDte] )


^BPass:^B

   ^BnNewDte^B is an optional numeric expression that should contain the
   DTE baud rate.


^BReturns:^B

   ^BnDte^B is a numeric expression that will contain the current DTE baud
   rate.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Dte(pnNewDte)

   local nOldDte := snDte

   if pcount() > 0
      snDte := pnNewDte
   endif

   return nOldDte





/* HYPERTEXT START
!short: md_Dce()        Get/Set the DCE baud rate
md_Dce()        Get/Set the DCE baud rate

^BDescription: ^B

   md_Dce() allows your programs to get/set the DCE baud rate (the rate
   between your modem and the user's modem).


^BSyntax:^B

   nDce := md_Dce( [nNewDce] )


^BPass:^B

   ^BnNewDce^B is an optional numeric expression that should be set to the
   new DCE rate.


^BReturns:^B

   ^BnDce^B is a numeric expression that will contain the current DCE baud
   rate


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG



HYPERTEXT END */
function md_Dce( pnNew )

   local nOld := snDce

   if pcount() > 0
      snDce := pnNew
   endif

   return snDce



/* HYPERTEXT START
!short: md_Load()       Load all the modem settings from *.INI file
md_Load()       Load all the modem settings from *.INI file

^BDescription: ^B

   md_Load() is essential because without it the other modem functions
   won't be able to operate.  You ^BMUST^B call md_Load() before you call
   any other modem function.


^BSyntax:^B

   lSuccess := md_Load( [cFileName], [cIniAlias] )


^BPass:^B

   ^BcFileName^B is an optional character expression that should contain
   the name of your MODEM.INI file.  If you don't pass ^BcFileName^B,
   "MODEM.INI" is assumed

   ^BcIniAlias^B is an optional character expression that should contain
   the alias name you want to use for the MODEM.INI file.  If you don't pass
   ^BcIniAlias^B, "MODEM" is assumed

^BReturns:^B

   ^BlSuccess^B is a logical expression that will be set to TRUE if md_Load()
   was successful, otherwise it will be set to FALSE.  Usually it will be set
   to FALSE if you specified an invalid value for ^BcFileName^B.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


   Requires Nanforum ToolKit available on CompuServe in the CLIPPER forum
   and on our BBS at 817-267-9768

^BSource:^B

   MODEM.PRG



HYPERTEXT END */
function md_Load(pcIniFile,pcIniAlias)

   local lSuccess := .f.
   local cAddr    := ""

   assume pcIniFile  is "MODEM.INI" if missing
   assume pcIniAlias is "MODEM"     if missing

   begin sequence

      if .not. iniOpen(pcIniFile,pcIniAlias)
         break
      endif

      // port settings...
      snIrq       := iniGetNmbr("modem","irq")

      cAddr       := upper( iniGetStr("modem","i/o address") )
      if "H" $ cAddr
         snAddr   := sHex2Dec(left(cAddr,len(cAddr)-1))
      else
         snAddr   := val(cAddr)
      endif

      // modem commands...
      scInitSeq   := iniGetStr("modem","init")
      scOnHook    := iniGetStr("modem","onhook")
      scOffHook   := iniGetStr("modem","offhook")
      scDialPre   := iniGetStr("modem","dialprefix")
      scDialSuf   := iniGetStr("modem","dialsuffix")
      scEscape    := iniGetStr("modem","escape")
      scAnswerSeq := iniGetStr("modem","answer")

      // modem messages...
      scError     := iniGetStr("modem","error")
      scBusy      := iniGetStr("modem","busy")
      scOk        := iniGetStr("modem","ok")
      scRing      := iniGetStr("modem","ring")
      scNoCarrier := iniGetStr("modem","no carrier")
      scNoTone    := iniGetStr("modem","no dial tone")
      scNoAnswer  := iniGetStr("modem","no answer")
      scVoice     := iniGetStr("modem","voice")
      scConn300   := iniGetStr("modem","connect 300")
      scConn1200  := iniGetStr("modem","connect 1200")
      scConn2400  := iniGetStr("modem","connect 2400")
      scConn9600  := iniGetStr("modem","connect 9600")
      scConn14400 := iniGetStr("modem","connect 14400")
      scConn28800 := iniGetStr("modem","connect 28800")

      // modem settings...
      snMaxBaud   := iniGetNmbr("modem","max baud rate")
      slSetBaud   := iniGetLog("modem","reset speed to connect speed")
      slManual    := iniGetLog("modem","manual answer")
      slInitEach  := iniGetLog("modem","init before each dial attempt")

      // timing settings...
      snPause     := iniGetNmbr("modem","pause between commands")
      snDelay     := iniGetNmbr("modem","pause between command characters")
      snSlowPoke  := iniGetNmbr("modem","pause before baud change")
      snDtrDelay  := iniGetNmbr("modem","dtr hangup drop length")
      snDialTime  := iniGetNmbr("modem","dial time")
      snRedial    := iniGetNmbr("modem","redial pause")

      // buffer sizes...
      snInpBuff   := iniGetNmbr("modem","input buffer size")
      snOutBuff   := iniGetNmbr("modem","output buffer size")
      snXferBuff  := iniGetNmbr("modem","transfer buffer size")

      // flow control settings...
      snCtrlCts   := iniGetNmbr("modem","clear-to-send")
      snCtrlRts   := iniGetNmbr("modem","request-to-send")
      snCtrlDsr   := iniGetNmbr("modem","data-set-ready")
      snCtrlDtr   := iniGetNmbr("modem","data-terminal-ready")
      snCtrlDcd   := iniGetNmbr("modem","data-carrier-detect")
      snCtrlX     := iniGetNmbr("modem","xon-xoff")
      snFifo      := iniGetNmbr("modem","fifo")

      lSuccess := .t.

   end sequence

   return lSuccess



/* HYPERTEXT START
!short: md_Init()       Initialize the modem
md_Init()       Initialize the modem

^BDescription: ^B

   md_Init() will send the initialization sequence (defined in MODEM.INI) to
   the modem.


^BSyntax:^B

   nError := md_Init()


^BPass:^B

   Nothing


^BReturns:^B

   ^BnError^B is a numeric expression that will be set to one of the
   following values...

       Ŀ
        Value  Description                                    
       Ĵ
                                                              
          0    No Error.  Modem successfully initialized.     
                                                              
         -1    Unable to open communications port.  Check     
               your IRQ and I/O ADDRESS settings in MODEM.INI 
                                                              
         -2    Too many failed initialization attempts.       
                                                              
       


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Init()

   local nAttempts := 1
   local nError    := 0
   local nRetVal   := 0

   begin sequence

      if .not. md_OpenPort(.f.)
         nRetVal := -1
      endif

      // try five times to initialize the modem
      while nAttempts < 5

         md_Send( sCvtCode(scInitSeq) )

         if tp_WaitFor(1,3,sCvtCode(scOk)) > 0
            break
         endif

         tp_Delay(snPause)
         tp_ClearIn(1)
         tp_ClearOut(1)

         md_ToggleDtr()

         nAttempts++

      enddo

      nRetVal := -2

   end sequence

   return nRetVal



/* HYPERTEXT START
!short: md_ToggleDtr()  Drop DTR line momentarily to hangup modem
md_ToggleDtr()  Drop DTR line momentarily to hangup modem

^BDescription: ^B

   md_ToggleDtr() momentarily drops the DTR line between your computer and
   you modem which should tell your modem to hangup (disconnect).


^BSyntax:^B

   md_ToggleDtr()


^BPass:^B

   Nothing


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG



HYPERTEXT END */
function md_ToggleDtr()

   local nTemp := tp_CtrlDtr(1,0)

   tp_Delay( snDtrDelay )

   tp_CtrlDtr(1,nTemp)

   return nil




/* HYPERTEXT START
!short: md_HangUp()     Hangup the phone line
md_HangUp()     Hangup the phone line

^BDescription: ^B

   md_Hangup() tells your modem to hangup using one several different methods.
   md_Hangup() first tries dropping DTR.  If unsuccessful, md_Hangup() will
   send the "escape" sequence and then the "hangup" sequence.  md_HangUp()
   will try each method up to five times.


^BSyntax:^B

   nError := md_Hangup()


^BPass:^B

   Nothing


^BReturns:^B

   ^BnError^B is a numeric expression that will be set to one of the
   following values...

       Ŀ
        Value  Description                                    
       Ĵ
                                                              
          0    No Error.  Modem successfully disconnected     
                                                              
         -1    Unable to force modem to disconnect            
                                                              
       


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_HangUp()

   local nAttempts := 0
   local nRetVal   := 0

   begin sequence

      if md_Dte() == 0 .or. md_Dce() == 0
         break
      endif


      while nAttempts < 5

         // drop DTR first
         md_ToggleDtr()

         if .not. tp_IsDcd(1)
            break
         endif


         // try a little "brute force"
         md_Esc()

         if .not. tp_IsDcd(1)
            break
         endif


         // try the hard way
         md_Send( sCvtCode(scOnHook) )

         if .not. tp_IsDcd(1)
            break
         endif

         tp_Delay(snPause * 3)

         tp_ClearIn(1)
         tp_ClearOut(1)

         nAttempts++

      enddo

      nRetVal := -1

   end sequence

   return nRetVal






/* HYPERTEXT START
!short: md_Send()       Send a command to the modem
md_Send()       Send a command to the modem

^BDescription: ^B

   md_Send() will send a command to your modem.  While this could be
   accomplished with a call to md_Xmit(), some modems cannot handle commands
   sent at full speed.  Therefore, md_Send() inserts a small delay between
   each character that it sends to the modem.


^BSyntax:^B

   md_Send( cString )


^BPass:^B

   ^BcString^B is a character expression that should contain the command you
   want sent to your modem.


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Send( pcCommand )

   local nCntr   := 0
   local nLength := len(pcCommand)

   tp_FlowOn(1)
   tp_Flush(1,0.05)

   for nCntr := 1 TO nLength

      if substr(pcCommand,nCntr,1) == "~"
         tp_Delay(.5)
      else
         tp_Flush(1,0.5)
         tp_Delay(snDelay)
         tp_SendSub(1,pcCommand,nCntr,1)
      endif

   next nCntr

   return nil





/* HYPERTEXT START
!short: md_Esc()        Send "escape" sequence to modem
md_Esc()        Send "escape" sequence to modem

^BDescription: ^B

   md_Esc() will send the "escape" sequence to your modem in an attempt to
   force your modem into command mode.  Normally you will not use this
   function directly, but several modem functions depend on it.


^BSyntax:^B

   md_Esc()


^BPass:^B

   Nothing


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Esc()

   tp_Delay(snPause)

   md_Send(scEscape)

   tp_Delay(snPause)
   tp_ClearIn(1)
   tp_ClearOut(1)

   return nil






/* HYPERTEXT START
!short: md_Chr()        Read a character from the modem
md_Chr()        Read a character from the modem

^BDescription: ^B

   md_Chr() allows you poll the modem for a character.


^BSyntax:^B

   cChar := md_Chr()


^BPass:^B

   Nothing


^BReturns:^B

   ^BcChar^B is a character expression that will contain the character that
   was received via the modem.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Chr()

   return tp_Recv(1,1)






/* HYPERTEXT START
!short: md_OffHook()    Take the modem "offhook"
md_OffHook()    Take the modem "offhook"

^BDescription: ^B

   md_OffHook() instructs your modem to take the phone offhook (ie: make the
   line appear busy to a caller).


^BSyntax:^B

   lSuccess := md_OffHook()


^BPass:^B

   Nothing


^BReturns:^B

   ^BlSuccess^B is a logical value that will be set to TRUE if md_OffHook()
   succeeded, otherwise it will be set to FALSE.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_OffHook()

   tp_Baud(1,snMaxBaud)

   tp_Delay(snPause)
   md_Send( sCvtCode(scOffHook) )

   tp_Delay(snPause)

   if tp_WaitFor(1,3,sCvtCode(scOk)) > 0
      return .t.
   endif

   return .f.






/* HYPERTEXT START
!short: md_CfgPort()    Configure the port after detecting carrier
md_CfgPort()    Configure the port after detecting carrier

^BDescription: ^B

   md_CfgPort() will properly configure your serial port after a carrier is
   detected on the modem.  Calling this function is very important if you
   are handling answering the calls yourself.  If you are using md_Wait4Call()
   you will not need to call this function yourself.


^BSyntax:^B

   md_CfgPort( nBaud )


^BPass:^B

   ^BnBaud^B is a numeric expression that should contain the baud rate at
   which the user connected (ie: the baud rate between the two modems).


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_CfgPort( pnBaud )

   tp_WatchOff()

   tp_Delay(snSlowPoke)

   tp_ClearIn(1)
   tp_ClearOut(1)
   tp_FlowOn(1)

   tp_Delay(snSlowPoke)

   if slSetBaud
      tp_Baud(1,pnBaud)
      snDTE := pnBaud
   else
      tp_Baud(1,snMaxBaud)
      snDTE := snMaxBaud
   endif

   tp_Delay(snSlowPoke)

   tp_CtrlCts(1,snCtrlCts)
   tp_CtrlDsr(1,snCtrlDsr)
   tp_CtrlDtr(1,snCtrlDtr)
   tp_CtrlRts(1,snCtrlRts)
   tp_CtrlX  (1,snCtrlX  )
   tp_CtrlDcd(1,snCtrlDcd)
   tp_Fifo   (1,snFifo   )

   snDCE := pnBaud

   return nil



/* HYPERTEXT START
!short: md_OpenPort()   Open the serial port
md_OpenPort()   Open the serial port

^BDescription: ^B

   md_OpenPort() will open the serial port, establish buffer sizes and
   optionally set handshaking lines.  Normally you will not call this function
   directly, but several modem functions depend on it.


^BSyntax:^B

   lSuccess := md_OpenPort( lHandShake )


^BPass:^B

   ^BlHandShake^B is a logical value that should be set to TRUE if you want
   md_OpenPort() to set the hand shaking lines, otherwise set it to FALSE.
   TRUE is the default if you don't specify anything.


^BReturns:^B

   ^BlSuccess^B is a logical value that will be set to TRUE if md_OpenPort()
   succeeds, otherwise it will be set to FALSE.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_OpenPort(plHandShake)

   local nError := 0

   if valtype(plHandShake) == "U"
      plHandShake := .t.
   endif

   // Set the IRQ and I/O address
   tp_SetPort(1,snAddr,snIrq)

   nError := tp_Open(1,snInpBuff,snOutBuff,snMaxBaud)

   if nError < 0
      return .f.
   endif

   tp_WatchOff()
   tp_Delay   (snPause)
   tp_ClearIn (1)
   tp_ClearOut(1)
   tp_FlowOn  (1)
   tp_Delay   (snPause)

   tp_Baud    (1,snMaxBaud)

   tp_Delay   (snPause)

   if plHandShake
      tp_CtrlCts(1,snCtrlCts)
      tp_CtrlDsr(1,snCtrlDsr)
      tp_CtrlDtr(1,snCtrlDtr)
      tp_CtrlRts(1,snCtrlRts)
      tp_CtrlX  (1,snCtrlX  )
      tp_CtrlDcd(1,snCtrlDcd)
   endif

   tp_Fifo(1,snFifo)

   tp_xDcd(.t.)

   snDTE := snMaxBaud
   snDCE := 0

   return .t.


/* HYPERTEXT START
!short: md_ReOpen()     re-open the serial port after a call to md_Close()
md_ReOpen()     re-open the serial port after a call to md_Close()

^BDescription: ^B

   md_ReOpen() will re-open the serial port after a call to md_Close().


^BSyntax:^B

   md_ReOpen()


^BPass:^B

   Nothing


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_ReOpen()

   tp_ReOpen(1,snInpBuff,snOutBuff)

   tp_WatchOff()
   // tp_Delay   (snPause)
   tp_ClearIn (1)
   tp_ClearOut(1)
   tp_FlowOn  (1)
   // tp_Delay   (snPause)

   tp_Baud(1,snDTE)
   tp_Delay(snPause)

   tp_CtrlCts(1,snCtrlCts)
   tp_CtrlDsr(1,snCtrlDsr)
   tp_CtrlDtr(1,snCtrlDtr)
   tp_CtrlRts(1,snCtrlRts)
   tp_CtrlX  (1,snCtrlX  )
   tp_CtrlDcd(1,snCtrlDcd)

   tp_Fifo(1,snFifo)

   return nil



/* HYPERTEXT START
!short: md_Close()      Close the serial port for shells to DOS etc
md_Close()      Close the serial port for shells to DOS etc

^BDescription: ^B

   md_Close() closes the serial port for things like shelling to DOS, etc.


^BSyntax:^B

   md_Close()


^BPass:^B

   Nothing


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Close()

   tp_Close(1,0,.t.)

   return nil




/* HYPERTEXT START
!short: md_Dcd()        Tests if carrier is present
md_Dcd()        Tests if carrier is present

^BDescription: ^B

   md_Dcd() tests to see if carrier is present on the serial port defined
   in MODEM.INI


^BSyntax:^B

   lPresent := md_Dcd()


^BPass:^B

   Nothing


^BReturns:^B

   ^BlPresent^B is a logical expression that will be set to TRUE if carrier
   is present, otherwise it will be set to FALSE.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Dcd()

   return if( md_Dce() == 0, .t., tp_IsDcd(1) )



/* HYPERTEXT START
!short: md_Xmit()       Write a string out the serial port
md_Xmit()       Write a string out the serial port

^BDescription:^B

   md_Xmit() is one of the main output functions that allows you to
   communicate with the remote user.


^BSyntax:^B

   md_Xmit( cText )


^BPass:^B

   ^BcText^B is a character expression that should contain the text you
   want to send to the remote user.


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Xmit( pcText )

   tp_Send(1,pcText,25)

   return nil


/* HYPERTEXT START
!short: md_XmitLn()     Write a string out the serial port
md_XmitLn()     Write a string out the serial port

^BDescription:^B

   md_XmitLn() is one of the main output functions that allows you to
   communicate with the remote user.


^BSyntax:^B

   md_XmitLn( cText )


^BPass:^B

   ^BcText^B is a character expression that should contain the text you
   want to send to the remote user.  A carriage return (chr(13)) will be
   added to the end of ^BcText^B.


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_XmitLn( pcText )

   tp_Send(1,pcText+chr(13),25)

   return nil



/* HYPERTEXT START
!short: md_Write()      Write a string out the serial port and echo to screen
md_Write()      Write a string out the serial port and echo to screen

^BDescription:^B

   md_Write() is one of the main output functions that allows you to
   communicate with the remote user.


^BSyntax:^B

   md_Write( cText )


^BPass:^B

   ^BcText^B is a character expression that should contain the text you
   want to send to the remote user.


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Write( pcText )

   if snDCE > 0
      tp_Send(1,pcText,25)
   endif

   outstd( strtran(pcText,chr(7)) )       // no bells to local screen EVER!!

   return nil


/* HYPERTEXT START
!short: md_WriteLn()    Write a string out the serial port and echo to screen
md_WriteLn()    Write a string out the serial port and echo to screen

^BDescription:^B

   md_WriteLn() is one of the main output functions that allows you to
   communicate with the remote user.


^BSyntax:^B

   md_WriteLn( cText )


^BPass:^B

   ^BcText^B is a character expression that should contain the text you
   want to send to the remote user.  A carriage return/line feed
   ( chr(13) + chr(10) ) will be added to the end of ^BcText^B.


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_WriteLn( pcText )

   if snDCE > 0
      tp_Send(1,pcText+chr(13)+chr(10),25)
   endif

   // no bells to local screen EVER!!
   outstd( strtran(pcText,chr(7)) + chr(13) + chr(10) )

   return nil







// progress graph for file transfers
function md_Progress( pnStatus, pcFile, pnLength, pnError )


   do case

      case pnStatus = TXS_START // start of transfer

         snStartTime := 0
         slStarted   := .f.
         snStartedAt := 0

         scFilename  := ""
         scStatus    := "Waiting to start transfer"

         snFileSize  := 0
         snBytesSent := 0

         sacFiles    := {}

         sDispBox()



      case pnStatus = TXS_SFILE // waiting to send file

         slStarted   := .f.
         snStartTime := basDateTime()
         snFileSize  := pnLength
         scFileName  := pcFile
         scStatus    := "Waiting to transfer file"
         snBytesSent := 0
         snStartedAt := 0

         sDispBox()



      case pnStatus = TXS_SDATA // start of file data

         if .not. slStarted

            aAdd( sacFiles, pcFile )

            scFileName := pcFile

            if pnLength > 0
               scStatus := "Resuming transfer"
            else
               scStatus := ""
            endif

            slStarted   := .t.
            snStartedAt := pnLength

         endif



      case pnStatus = TXS_BLOCK // Sent Block

         snBytesSent := pnLength

         sDispBox()



      case pnStatus = TXS_ERROR // error

         scStatus := tp_ErrMsg(pnError)

         sDispBox()


      case pnStatus = TXS_ABFILE // aborting file

         scStatus := tp_ErrMsg(pnError)

         adel(sacFiles,len(sacFiles))
         aSize(sacFiles,len(sacFiles)-1)

         sDispBox()



      case pnStatus = TXS_ABORT // aborting transfer

         snStartTime := 0
         slStarted   := .f.
         snStartedAt := 0
         snFileSize  := 0
         snBytesSent := 0
         scStatus    := tp_ErrMsg(pnError)
         scFileName  := ""

         sDispBox()



      case pnStatus = TXS_WEND // waiting to end

         scStatus := "End of file"

         sDispBox()



      case pnStatus = TXS_END// end of transfer

         snStartTime := 0
         slStarted   := .f.
         snStartedAt := 0
         snFileSize  := 0
         snBytesSent := 0
         scStatus    := "End of transfer"
         scFileName  := ""

         sDispBox()

         tp_Delay(2)

         slBox := .f.

         SetCursor(SC_NORMAL)

   endcase


   return 0




static function sPadIt( pcString, pnLength )

   // Grab the right most part of a string

   pcString := right(pcString, pnLength)

   return pcString + space(pnLength - len(pcString))


/*
 7     Ŀ
 8      Filename:                                                   
 9       
10      Transfer Time.: 23:59:45        Approximate CPS..:     1234 
11      Elapsed Time..: 00:00:00        File Size........: 12345678 
12      Time Remaining: 00:00:00        Bytes Transferred: 12345678 
13       
14        00 % Complete 
15       
16      Last Status/Error:                                          
17     
*/


static function sDispBox()

   local nElapSecs   := basElapSecs( snStartTime, basDateTime() )
   local nCps        := max( int( (snBytesSent-snStartedAt) / nElapSecs ), 1 )

   local nXferSecs   := int( (snFileSize-snStartedAt) / nCps )
   local nRemaSecs   := int( (snFileSize-snBytesSent) / nCps )

   local nPercent    := snBytesSent / snFileSize
   local nBlocks     := int( 44 * nPercent )

   local nI          := 1
   local cMode       := ""

   DispBegin()

   if .not. slBox

      // find out what mode we are in
      while .not. empty(procname(nI))

         if left(procname(nI),4) == "TP_R"
            cMode := " Receiving "
            exit
         endif

         if left(procname(nI),4) == "TP_S"
            cMode := " Sending "
            exit
         endif

         nI++

      enddo

      basWind(7,8,17,70,cMode," Press [ESC] to abort ",snPalette)

      @ 08,10 say "Filename:"

      DispBox( 9,10, 9,68, B_SINGLE )

      @ 10,10 say "Transfer Time.:"
      @ 11,10 say "Elapsed Time..:"
      @ 12,10 say "Time Remaining:"

      @ 10,42 say "Approximate CPS..:"
      @ 11,42 say "File Size........:"
      @ 12,42 say "Bytes Transferred:"

      DispBox( 13,10, 13,68, B_SINGLE )

      DispBox( 15,10, 15,68, B_SINGLE )

      @ 16,10 say "Last Status/Error:"

      slBox := .t.

   endif

   @ 08,20 say sPadIt( scFileName, 49 )

   @ 10,26 say sTime(nXferSecs)
   @ 11,26 say sTime(nElapSecs)
   @ 12,26 say sTime(nRemaSecs)

   @ 10,61 say str(nCps,8,0)
   @ 11,61 say str(snFileSize,8,0)
   @ 12,61 say str(snBytesSent,8,0)

   if snBytesSent > 0

      @ 14,10 say replicate(chr(176),44)

      @ 14,10 say replicate(chr(219), nBlocks )

      @ 14,55 say str( int(nPercent*100),3,0) + "% Complete"

   endif

   @ 16,29 say sPadit(scStatus,40)

   DispEnd()

   return nil


// return HH:MM:SS
static function sTime( pnSecs )

   local nHours := int( pnSecs / 3600 )
   local nMins  := int( ( pnSecs - (nHours*3600) ) / 60 )
   local nSecs  := pnSecs - ( (nHours*3600) + (nMins*60) )
   local cTime  := strzero(nHours,2,0) + ":" + ;
      strZero(nMins, 2,0) + ":" + ;
      strZero(nSecs, 2,0)

   if at("*",cTime) > 0
      cTime := "00:00:00"
   endif

   return cTime



/* HYPERTEXT START
!short: md_Files()      Returns a list of files that were just transferred
md_Files()      Returns a list of files that were just transferred

^BDescription:^B

   md_Files() allows your programs to obtain a list of all the files that
   were transferred during the last file transfer.


^BSyntax:^B

   acFiles := md_Files()


^BPass:^B

   ^BacFiles^B is an array of character expressions.  Each element in this
   array will contain the file name of a file that was transferred during the
   last file transfer session.


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Files()

   return sacFiles


// convert ^M to chr(13), ^27 to chr(27), etc
static function sCvtCode( pcString )

   local nPos     := 0
   local cRetVal  := ""
   local cBit     := ""

   while .t.

      nPos := at("^",pcString)

      if nPos <= 0
         exit
      endif

      cRetVal  += substr(pcString,1,nPos-1)

      cBit     := lower (substr(pcString,nPos+1,1) )

      pcString := substr(pcString,nPos+2)

      nPos     := at(cBit,"abcdefghijklmnopqrstuvwxyz[")

      do case

         case nPos > 0
            cRetVal += chr(asc(cBit)-96)

         case cBit == "^"
            cRetVal += "^"

         case cBit $ "0123456789"

            while substr(pcString,1,1) $ "0123456789"

               cBit     += substr(pcString,1,1)
               pcString := substr(pcString,2)

            enddo

            cRetVal += chr(val(cBit))

      endcase

   enddo

   return cRetVal + pcString


// convert a hexidecimal number to decimal
static function sHex2Dec(pcHexNmbr)

   local nDecNmbr := 0
   local nI       := 0
   local nLength  := len(pcHexNmbr)

   for nI := 1 to nLength

      nDecNmbr *= 16
      nDecNmbr += at( substr(pcHexNmbr,nI,1), "0123456789ABCDEF") - 1

   next nI

   return nDecNmbr


/* HYPERTEXT START
!short: md_Wait4Call()  Wait for a call and answer the phone
md_Wait4Call()  Wait for a call and answer the phone

^BDescription:^B

   md_Wait4Call() will wait for a call and answer the phone.  This is usually
   the main function during a host session.


^BSyntax:^B

   nBaud := md_Wait4Call( [ bInterrogate | cPrompt ], [bIdle] )


^BPass:^B

   ^BbInterrogate^B is an optional code block that will be evaluated once a
   carrier has been established.  If this code block returns TRUE,
   md_Wait4Call() will assume there is a valid caller online and will return
   the connection baud rate to the calling program.

   If ^BbInterrogate^B is omitted, a default code block is used which will
   prompt the remote caller to either hangup or enter the "kickoff" character
   (chr(254)).  If the remote user enters chr(254), md_Wait4Call() will
   return the connection baud rate to the calling program.

   If ^BbInterrogate^B returns FALSE, md_Wait4Call() will drop carrier and
   continue waiting for the next call.


   ^BbIdle^B is an optional code block that md_Wait4Call() will execute
   periodically if nothing is happening with the modem.  If this code block
   returns FALSE, md_Wait4Call() will return -2 to the calling program.


   ^BcPrompt^B is an optional character expression that should contain the
   login prompt for your callers.


^BReturns:^B

   ^BnBaud^B is a numeric expression that will be one of the following
   values...

       Ŀ
        Value  Description                                    
       Ĵ
                                                              
           -2  Idle code block aborted md_Wait4Call()         
                                                              
           -1  Unable to initialize modem                     
                                                              
            0  Local user aborted by pressing [ESCAPE]        
                                                              
          300  Remote user connected at 300 baud              
                                                              
         1200  Remote user connected at 1200 baud             
                                                              
         2400  Remote user connected at 2400 baud             
                                                              
         9600  Remote user connected at 9600 baud             
                                                              
        14400  Remote user connected at 14400 baud            
                                                              
        28800  Remote user connected at 28800 baud            
                                                              
       


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_Wait4Call(pbInterrogate,pbIdle)

   local nBaud          := 0
   local nStart         := basDateTime()
   local nRow           := basRandom( 24 )
   local nCol           := basRandom( 54 )
   local anWatches      := {}
   local lInit          := .t.
   local nKey           := 0
   local lInterrogate   := .f.
   local nRetVal        := 0

   basSaveScrn()

   setcolor("W/N")

   SetCursor(SC_NONE)

   while .t.

      // should we initialize the modem?
      if lInit

         cls
         @ nRow,nCol say " Initializing Modem... " color "W+/R"

         // clear the watches
         tp_WatchOff()
         anWatches := {}

         if .not. md_Init() == 0
            nRetVal := -1
            exit
         endif

         cls
         @ nRow,nCol say " Waiting for call... " color "W+/R"

         nBaud := 0
         lInit := .f.

      endif

      // does the pbIdle code block want to abort?
      if valtype(pbIdle) == "B"

         if .not. eval(pbIdle)
            nRetVal := -2
            exit
         endif

      endif


      // time to move the screen saver?
      if basElapSecs(nStart,basDateTime()) >= 1.00

         nStart   := basDateTime()
         nRow     := basRandom(24)
         nCol     := basRandom(59)

         cls
         @ nRow,nCol say " Waiting for call... " color "W+/R"

      endif


      // do we need to set watches?
      if len(anWatches) == 0

         tp_WatchOff()
         anWatches := {}

         aAdd( anWatches, tp_Watch(1,sCvtCode(scError     ),.t.) ) // 1
         aAdd( anWatches, tp_Watch(1,sCvtCode(scRing      ),.t.) ) // 2
         aAdd( anWatches, tp_Watch(1,sCvtCode(scNoCarrier ),.t.) ) // 3
         aAdd( anWatches, tp_Watch(1,sCvtCode(scConn300   ),.t.) ) // 4
         aAdd( anWatches, tp_Watch(1,sCvtCode(scConn1200  ),.t.) ) // 5
         aAdd( anWatches, tp_Watch(1,sCvtCode(scConn2400  ),.t.) ) // 6
         aAdd( anWatches, tp_Watch(1,sCvtCode(scConn9600  ),.t.) ) // 7
         aAdd( anWatches, tp_Watch(1,sCvtCode(scConn14400 ),.t.) ) // 8
         aAdd( anWatches, tp_Watch(1,sCvtCode(scConn28800 ),.t.) ) // 9

      endif

      // stobe the modem
      md_Chr()

      do case

         case tp_IsWatch(anWatches[1]) // error

            lInit := .t.


         case tp_IsWatch(anWatches[2]) // ring

            if slManual
               md_Send( sCvtCode(scAnswerSeq) )
            endif


         case tp_IsWatch(anWatches[3]) // no carrier

            lInit := .t.


         case tp_IsWatch(anWatches[4]) // connect 300

            md_CfgPort(300)

            nBaud := 300


         case tp_IsWatch(anWatches[5]) // connect 1200

            md_CfgPort(1200)

            nBaud := 1200


         case tp_IsWatch(anWatches[6]) // connect 2400

            md_CfgPort(2400)

            nBaud := 2400


         case tp_IsWatch(anWatches[7]) // connect 9600

            md_CfgPort(9600)

            nBaud := 9600


         case tp_IsWatch(anWatches[8]) // connect 14400

            md_CfgPort(14400)

            nBaud := 14400


         case tp_IsWatch(anWatches[9]) // connect 28800

            md_CfgPort(28800)

            nBaud := 28800

      endcase


      // If we have a connection, find out who it is!!
      if nBaud > 0

         tp_WatchOff()
         anWatches := {}

         if valtype(pbInterrogate) == "B"

            lInterrogate := eval(pbInterrogate,nBaud)

         else

            if valtype(pbInterrogate) == "U"
               pbInterrogate := "This is NOT a BBS.  Please hangup!"
            endif

            lInterrogate := sInterrogate(pbInterrogate)

         endif

         if lInterrogate
            nRetVal := nBaud
            exit
         endif

         md_HangUp()

         lInit := .t.

         loop

      endif

      // stobe the local keyboard
      nKey := tp_Inkey()

      // does the local user want to abort?
      if nKey == K_ESC
         nRetVal := 0
         exit
      endif

      // does local user want to reset modem?
      if nKey == asc(" ")
         lInit := .t.
      endif

   enddo

   basRestScrn()

   return nRetVal



// Find out if this is a valid user
static function sInterrogate(pcPrompt)

   local nStart   := basDateTime()
   local cChr     := ""
   local lSuccess := .f.

   tp_Delay(5)

   md_Xmit  ( chr(12) )     // clear the user's screen
   md_XmitLn( "Vern Six's Host Utility"+ chr(10)   )
   md_XmitLn( "CopyRight (c) 1993,94 by FrontLine Software" + chr(10) )
   md_XmitLn( "All Rights Reserved World Wide" + chr(10) )
   md_XmitLn( chr(10) )
   md_XmitLn( "For complete information about this product, please" + chr(10) )
   md_XmitLn( "call 817/267-8944 (voice) or 817/267-9768 (modem)" + chr(10) )
   md_XmitLn( chr(10) )
   md_XmitLn( pcPrompt + chr(10) )

   while .t.

      if .not. md_Dcd()
         exit
      endif

      // give them ten seconds!
      if basElapSecs(nStart,basDateTime()) > 10
         md_HangUp()
         exit
      endif

      cChr := md_Chr()

      if cChr == chr(254)
         lSuccess := .t.
         exit
      endif

   enddo

   return lSuccess





/* HYPERTEXT START
!short: md_Dial()       Dial a list of hosts
md_Dial()       Dial a list of hosts

^BDescription:^B

   md_Dial() attempts to dial a list of hosts.


^BSyntax:^B

   aConnInfo := md_Dial( axHostInfo, [nMaxAttempts] )


^BPass:^B

   ^BaxHostInfo^B is an array with the following structure...

      { { cHostName, cPhoneNmbr }, ... }

      ^BcHostName^B is a character expression that should contain the name
      of the host you are attempting to call.

      ^BcPhoneNmbr^B is a character expression that should contain the
      phone number of the host you are attempting to call.


   ^BnMaxAttempts^B is a numeric value that should contain the maximum number
   of dial attempts md_Dial() should make.  If omitted or zero, md_Dial() will
   try indefinitely or until aborted by the local user.


^BReturns:^B

   ^BaConnInfo^B is an array that will contain two elements...

      { nHost, nBaud }

      ^BnHost^B is the number of the host md_Dial() connected to.  ^BnHost^B
      will be set to zero if the dial attempt was unsuccessful.

      ^BnBaud^B is the baud rate md_Dial() connected at.  ^BnBaud^B will be
      set to zero if the dial attempt was unsuccessful.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */

/*
            1         2         3         4         5         6
   123456789012345678901234567890123456789012345678901234567890
 8Ŀ
 9 Redial Started At 01:22:14         Attempt #002 - 01:23:35 
10                                                            
11                  Calling... ClipBoard Home                 
12                        817-695-0058                        
13                                                            
14                     Last Attempt: BUSY                     
15                                                            
16               Press [Escape] To Abort Redial               
17
   1         2         3         4         5         6         7
  90123456789012345678901234567890123456789012345678901234567890


 8Ŀ
 9 Redial Started At 01:22:14         Attempt #001 - 01:23:35 
10                                                            
11                  Calling... ClipBoard Home                 
12                        817-695-0058                        
13                                                            
14                     Last Attempt: BUSY                     
15                                                            
16   Pausing for 10 seconds.  Press Any Key To Redial Now!    
17

*/
function md_Dial( paxHostInfo, pnMaxAttempts )

   local nBaud      := 0
   local nPos       := 0
   local nAttempts  := 1

   if valtype(pnMaxAttempts) == "U"
      pnMaxAttempts := 0
   endif

   scLastMsg := "None"
   slInitYet := .f.

   basSaveScrn()

   basWind( 08,09, 17,70, "", "", snPalette )


   while .t.

      if pnMaxAttempts > 0

         if nAttempts >= pnMaxAttempts
            exit
         endif

      endif

      nPos++

      if nPos > len(paxHostInfo)
         nPos := 1
         nAttempts++
      endif

      nBaud := sDial( paxHostInfo[nPos], nAttempts )

      if nBaud != 0
         exit
      endif

      if sAbort()
         exit
      endif

   enddo

   basRestScrn()

   if nBaud > 0
      md_CfgPort(nBaud)
   else
      md_XmitLn("")         // make the modem give up
      nPos := 0
   endif

   return { nPos, nBaud }


static function sDial( pacBoard, pnAttempt )

   local anWatch   := {}
   local nStart    := 0
   local nBaud     := 0
   local nElapSecs := 0

   begin sequence

      if .not. slInitYet

         @ 16,11 say padc( "*** Initializing Modem.  Please wait. ***", 58 )

         if .not. md_Init() == 0
            nBaud := -1
            break
         endif

         // set statics for next dial attempt
         if .not. slInitEach
            slInitYet := .t.
         endif

      endif

      @ 09,11 say "Redial Started At " + time()
      @ 09,46 say "Attempt #" + StrZero(pnAttempt,3,0) + " - " + time()

      @ 11,11 say padc( "Calling... " + alltrim(pacBoard[1]), 58 )
      @ 12,11 say padc( alltrim(pacBoard[2]), 58 )

      @ 14,11 say padc( "Last Attempt: " + scLastMsg, 58 )

      @ 16,11 say padc( "Press [Escape] To Abort Redial", 58 )

      md_Send( sCvtCode(scDialPre)+alltrim(pacBoard[2])+sCvtCode(scDialSuf) )

      nStart := basDateTime()

      // Setup all the input watches
      tp_WatchOff()

      aAdd( anWatch, tp_Watch( 1, alltrim(scError     ), .t. ) ) //  1
      aAdd( anWatch, tp_Watch( 1, alltrim(scBusy      ), .t. ) ) //  2
      aAdd( anWatch, tp_Watch( 1, alltrim(scNoCarrier ), .t. ) ) //  3
      aAdd( anWatch, tp_Watch( 1, alltrim(scNoTone    ), .t. ) ) //  4
      aAdd( anWatch, tp_Watch( 1, alltrim(scNoAnswer  ), .t. ) ) //  5
      aAdd( anWatch, tp_Watch( 1, alltrim(scVoice     ), .t. ) ) //  6
      aAdd( anWatch, tp_Watch( 1, alltrim(scConn300   ), .t. ) ) //  7
      aAdd( anWatch, tp_Watch( 1, alltrim(scConn1200  ), .t. ) ) //  8
      aAdd( anWatch, tp_Watch( 1, alltrim(scConn2400  ), .t. ) ) //  9
      aAdd( anWatch, tp_Watch( 1, alltrim(scConn9600  ), .t. ) ) // 10
      aAdd( anWatch, tp_Watch( 1, alltrim(scConn14400 ), .t. ) ) // 11
      aAdd( anWatch, tp_Watch( 1, alltrim(scConn28800 ), .t. ) ) // 12

      while .t.

         nElapSecs := basElapSecs( nStart, basDateTime() )

         @ 09,61 say time()

         md_Chr()

         do case

            case nElapSecs > snDialTime
               scLastMsg := "Timed Out waiting for connection"
               nBaud     := 0
               exit

            case tp_Inkey() == K_ESC
               scLastMsg := "Dial Attempt Aborted!"
               nBaud     := -1
               exit

            case tp_IsWatch( anWatch[1] )
               scLastMsg := "ERROR"
               nBaud     := -2
               exit

            case tp_IsWatch( anWatch[2] )
               scLastMsg := "BUSY"
               nBaud     := 0
               exit

            case tp_IsWatch( anWatch[3] )
               scLastMsg := "NO CARRIER"
               nBaud     := 0
               exit

            case tp_IsWatch( anWatch[4] )
               scLastMsg := "NO DIAL TONE"
               nBaud     := 0
               exit

            case tp_IsWatch( anWatch[5] )
               scLastMsg := "NO ANSWER"
               nBaud     := 0
               exit

            case tp_IsWatch( anWatch[6] )
               scLastMsg := "VOICE"
               nBaud     := 0
               exit

            case tp_IsWatch( anWatch[7] )
               scLastMsg := "CONNECT 300"
               nBaud     := 300
               exit

            case tp_IsWatch( anWatch[8] )
               scLastMsg := "CONNECT 1200"
               nBaud     := 1200
               exit

            case tp_IsWatch( anWatch[9] )
               scLastMsg := "CONNECT 2400"
               nBaud     := 2400
               exit

            case tp_IsWatch( anWatch[10] )
               scLastMsg := "CONNECT 9600"
               nBaud     := 9600
               exit

            case tp_IsWatch( anWatch[11] )
               scLastMsg := "CONNECT 14400"
               nBaud     := 14400
               exit

            case tp_IsWatch( anWatch[12] )
               scLastMsg := "CONNECT 28800"
               nBaud     := 28800
               exit

         endcase

      enddo

      @ 14,11 say padc( "Last Attempt: " + scLastMsg, 58 )

   end sequence


   return nBaud




static function sAbort()

   local nStart    := basDateTime()
   local nKey      := 0
   local nElapSecs := 0

   while .t.

      @ 09,61 say time()

      nElapSecs := basElapSecs( nStart, basDateTime() )

      if nElapSecs > snRedial
         exit
      endif

      @ 16,11 say padc( "Pausing For " + ;
         alltrim( str(snRedial-nElapSecs,3,0) ) + ;
         " Seconds.  Press Any Key To Redial Now!", 58 )

      nKey := Inkey(1)

      if .not. nKey == 0
         exit
      endif

   enddo


   // did the user abort during the redial pause?
   if nKey == K_ESC
      return .t.
   endif

   return .f.


/* HYPERTEXT START
!short: md_ClearOut()   Clear output buffer
md_ClearOut()   Clear output buffer

^BDescription:^B

   md_ClearOut() immediately disgards any characters in the output buffer


^BSyntax:^B

   md_ClearOut()


^BPass:^B

   Nothing


^BReturns:^B

   Nothing


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_ClearOut()
   return tp_ClearOut(1)













/* HYPERTEXT START
!short: md_rZmodem()    Receive files using Zmodem protocol
md_rZmodem()    Receive files using Zmodem protocol

^BDescription:^B

   md_rZmodem() receives one or more files using the Zmodem protocol.


^BSyntax:^B

   nError := md_rZmodem( [cDir] )


^BPass:^B

   ^BcDir^B is an optional character expression that should contain the
   name of the directory where you want the received files to be located.
   If you omit ^BcDir^B, the current working directory is assumed.


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring files.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_rZmodem(pcDir)

   local nError := 0

   if valtype(pcDir) == "U"
      pcDir := ""
   endif

   basSaveScrn()

   nError := tp_rZmodem(1,pcDir,"md_Progress",3)

   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError






/* HYPERTEXT START
!short: md_sZmodem()    Send files using Zmodem protocol
md_sZmodem()    Send files using Zmodem protocol

^BDescription:^B

   md_sZmodem() sends one or more files using the Zmodem protocol.


^BSyntax:^B

   nError := md_sZmodem( acFiles )


^BPass:^B

   ^BacFiles^B is an array of character expressions.  Each element in
   ^BacFiles^B should contain the full path and name of a file you want sent.


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring files.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_sZmodem(pacFiles)

   local nError := 0

   basSaveScrn()

   nError := tp_sZmodem(1,pacFiles,"md_Progress",3)
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError










/* HYPERTEXT START
!short: md_rYmodem()    Receive files using Ymodem protocol
md_rYmodem()    Receive files using Ymodem protocol

^BDescription:^B

   md_rYmodem() receives one or more files using the Ymodem protocol.


^BSyntax:^B

   nError := md_rYmodem( [cDir] )


^BPass:^B

   ^BcDir^B is an optional character expression that should contain the
   name of the directory where you want the received files to be located.
   If you omit ^BcDir^B, the current working directory is assumed.


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring files.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_rYmodem(pcDir)

   local nError := 0

   if valtype(pcDir) == "U"
      pcDir := ""
   endif

   basSaveScrn()

   nError := tp_rYmodem(1,pcDir,"md_Progress")
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError






/* HYPERTEXT START
!short: md_sYmodem()    Send files using Ymodem protocol
md_sYmodem()    Send files using Ymodem protocol

^BDescription:^B

   md_sYmodem() sends one or more files using the Ymodem protocol.


^BSyntax:^B

   nError := md_sYmodem( acFiles )


^BPass:^B

   ^BacFiles^B is an array of character expressions.  Each element in
   ^BacFiles^B should contain the full path and name of a file you want sent.


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring files.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_sYmodem(pacFiles)

   local nError := 0

   basSaveScrn()

   nError := tp_sYmodem(1,pacFiles,"md_Progress")
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError









/* HYPERTEXT START
!short: md_rX1k()       Receive file using Xmodem-1k protocol
md_rX1k()       Receive file using Xmodem-1k protocol

^BDescription:^B

   md_rX1k() receives one file using the Xmodem-1k protocol.


^BSyntax:^B

   nError := md_rX1k( cFile )


^BPass:^B

   ^BcFile^B is a character expression that should contain the name of
   the file you want to receive (including path).


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring the file.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_rX1k(pcFile)

   local nError := 0

   basSaveScrn()

   nError := tp_rX1k(1,pcFile,"md_Progress")
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError






/* HYPERTEXT START
!short: md_sX1k()       Send file using Xmodem-1k protocol
md_sX1k()       Send file using Xmodem-1k protocol

^BDescription:^B

   md_sX1k() sends one file using the Xmodem-1k protocol.


^BSyntax:^B

   nError := md_sX1k( cFile )


^BPass:^B

   ^BcFile^B is a character expression that should contain the name of
   the file you want to send (including path).


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring the file.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_sX1k(pcFile)

   local nError := 0

   basSaveScrn()

   nError := tp_sX1k(1,pcFile,"md_Progress")
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError











/* HYPERTEXT START
!short: md_rXmodem()    Receive file using Xmodem-CRC protocol
md_rXmodem()    Receive file using Xmodem-CRC protocol

^BDescription:^B

   md_rXmodem() receives one file using the Xmodem-CRC protocol.


^BSyntax:^B

   nError := md_rXmodem( cFile )


^BPass:^B

   ^BcFile^B is a character expression that should contain the name of
   the file you want to receive (including path).


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring the file.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_rXmodem(pcFile)

   local nError := 0

   basSaveScrn()

   nError := tp_rXcrc(1,pcFile,"md_Progress")
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError






/* HYPERTEXT START
!short: md_sXmodem()    Send file using Xmodem (CRC or ChkSum) protocol
md_sXmodem()    Send file using Xmodem (CRC or ChkSum) protocol

^BDescription:^B

   md_sXmodem() sends one file using the Xmodem (CRC or ChkSum) protocol.


^BSyntax:^B

   nError := md_sXmodem( cFile )


^BPass:^B

   ^BcFile^B is a character expression that should contain the name of
   the file you want to send (including path).


^BReturns:^B

   ^BnError^B is a numeric expression that will contain the error number
   encountered while transferring the file.  See tp_ErrMsg() to convert
   ^BnError^B to English text.


^BNotes:^B

   Requires Telepathy from...

       Extrasensory Software
       4450 Murietta Ave., #8
       Sherman Oaks, CA  91423

       (818) 981-8367 Voice
       (818) 986-5411 Fax


^BSource:^B

   MODEM.PRG


HYPERTEXT END */
function md_sXmodem(pcFile)

   local nError := 0

   basSaveScrn()

   nError := tp_sXmodem(1,pcFile,"md_Progress")
   slBox := .f.

   basRestScrn()

   tp_ClearIn (1)
   tp_ClearOut(1)

   return nError





