/*===========================================================================
SOLAR v0.95.2 slrnews :: Module ..\slrnews\config.c
Original Author: Kevin Houle <kjhoule@iowegia.dsm.ia.us>

This software module has been placed in the public domain.
===========================================================================*/

/* Header Files */
#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <stdlib.h>
#include "..\common\yesno.h"
#include "config.h"

/*
 * Function: int load_static(char username[10])
 * Purpose : Load parameters from Waffle's static file.
 * Return  : 0 on success, non-zero on error and set _slrerr
*/

int load_static(char username[10])
{
  FILE *static_file = NULL;

  char buffer[128];
  char delimiter = ':';
  char *path = NULL;

  /* Get path to static file and open it */

  if ((path = getenv("WAFFLE")) == NULL)
  {
    strcpy(_slrerr,"load_static(): environment variable WAFFLE not set.");
    goto ErrorExit;
  }
  strcpy(static_path,path);
  if (verbose == YES)
    RPrintf("Waffle static file: %s\n",static_path);

  if ((static_file = fopen(static_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"load_static(): error opening %s for reading.",static_path);
    goto ErrorExit;
  }

  /* Read static file and pull out what we need. */

  while (fgets(buffer,128,static_file) != NULL)
  {
    /* Build path to user's directory */

    if (strnicmp(buffer,"USER",4) == 0)
    {
      strcpy(user_path,extract_parm(buffer,delimiter));
      strcat(user_path, "\\");
      strcat(user_path, username);
      if (verbose == YES)
        RPrintf("User Path: %s\n",user_path);
    }

    /* Set path to Solar's configuration file.
       Give 'solar95' priority. */

    if (strnicmp(buffer,"SOLAR95",7) == 0)
		{
			strcpy(config_path,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Solar config path: %s\n",config_path);
		}

    if ((strnicmp(buffer,"SOLAR",5) == 0) && (strcmp(config_path,"NONE") == 0))
		{
			strcpy(config_path,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Solar config path: %s\n",config_path);
		}

    /* Get path to temporary directory. This only gets used if
       'solarwork' is not set in Solar's configuration file. */

    if (strnicmp(buffer,"TEMPORARY",9) == 0)
    {
      if (strcmp(temp_path,"NONE") == 0)
      {
        strcpy(temp_path,extract_parm(buffer,delimiter));
        if (verbose == YES)
          RPrintf("Temp path: %s\n",temp_path);
      }
    }

    /* Store path to Waffle root directory */

    if (strnicmp(buffer,"WAFFLE",6) == 0)
    {
        strcpy(waffle_path,extract_parm(buffer,delimiter));

        /* Set default path to logfile */

        strcpy(log_path,waffle_path);
        strcat(log_path,"\\admin\\solar");
        if (verbose == YES)
          RPrintf("Log file: %s\n",log_path);
        strcpy(system_path,waffle_path);
        strcat(system_path,"\\system");
        if (verbose == YES)
          RPrintf("System path: %s\n",system_path);
    }

    /* Check the driver, use fossil only if selected on command line
       and in static file 'driver' parameter. */

    if (strnicmp(buffer,"DRIVER",6) == 0)
    {
      if (use_fossil == YES)
      {
        if (strnicmp(extract_parm(buffer,delimiter),"FOSSIL",6) == 0)
        {
          use_fossil = YES;
          if (verbose == YES) RPrintf("Fossil support enabled.\n");
        }
        else
        {
          use_fossil = NO;
          if (verbose == YES) RPrintf("Fossil support disabled.\n");
        }
      }
      else
      {
        if (verbose == YES) RPrintf("Fossil support disabled.\n");
      }
    }

    /* Load comm device, in case we are using a fossil driver. */

    if (strnicmp(buffer,"DEVICE",6) == 0)
    {
      port = atoi(extract_parm(buffer,delimiter)) - 1;
      if (verbose == YES)
        RPrintf("COM Port: %u\n",port);
    }
  }
  fclose(static_file);

  if (verbose == YES)
    RPrintf("Finished reading static, validating...\n");

  /* Check to make sure we got all that we need. */

  if (strcmp(temp_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): temporary: parameter not found in static file");
    goto ErrorExit;
  }
  if (strcmp(user_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): user: parameter not found in static file.");
    goto ErrorExit;
  }
  if (strcmp(waffle_path,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): waffle: parameter not found in static file.");
    goto ErrorExit;
  }
  if ((use_fossil == YES) && (port == 99))
  {
    strcpy(_slrerr,"load_static(): device: parameter not found in static file.");
    goto ErrorExit;
  }

  if (verbose == YES)
    RPrintf("Static file load OK\n");

GoodExit:
  return 0;
ErrorExit:
  if (static_file) fclose(static_file);
  return 1;
}

/*
 * Function: load_config()
 * Purpose : Load Solar's configuration file.
 * Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_config()
{
  FILE *config_file = NULL;
  char buf[128];
  char delimiter = '=';
  char *path = NULL;

  /* Get path to config file if not already set */

  if (strcmp(config_path,"NONE") == 0)
  {
    if ((path = getenv("SOLAR95")) == NULL)
    {
      if ((path = getenv("SOLAR")) == NULL)
      {
        strcpy(_slrerr,"load_config(): environment variable SOLAR not set.");
        goto ErrorExit;
      }
    }
    strcpy(config_path,path);
  }

  if (verbose == YES)
    RPrintf("Solar config file: %s\n",config_path);

  if ((config_file = fopen(config_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"load_config(): could not open %s for reading.",config_path);
    goto ErrorExit;
  }

  while (fgets(buf,128,config_file) != NULL)
  {
    /* Override the default temporary work directory */

    if (strnicmp(buf,"SOLARWORK",9) == 0)
    {
      strcpy(temp_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Temp path set to: %s\n",temp_path);
    }

    /* Override the default log file path */

    if (strnicmp(buf,"LOGFILE",7) == 0)
    {
      strcpy(log_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Log file set to: %s\n",log_path);
    }

    /* Override the default SOUP index type for news */

    if (strnicmp(buf,"NEWS-INDEX",10) == 0)
    {
      news_index = convert_to_number(extract_parm(buf,delimiter));
      if (verbose == YES)
      {
        if (news_index == n) RPrintf("News index: n (None)\n");
        if (news_index == C) RPrintf("News index: C (Partial C News)\n");
        if (news_index == c) RPrintf("News index: c (Full C News)\n");
      }
    }

    /* Override the default SOUP message type for news */

    if (strnicmp(buf,"NEWS-TYPE",9) == 0)
    {
      news_type = convert_to_number(extract_parm(buf,delimiter));
      if (verbose == YES)
      {
        if (news_type == u) RPrintf("news type: u (USENET Format)\n");
        if (news_type == B) RPrintf("news type: B (8-bit binary)\n");
        if (news_type == i) RPrintf("news type: i (Index only)\n");
      }
    }

    /* Set the path to the newsgroups description file */

    if (strnicmp(buf,"NEWSGROUPS",10) == 0)
    {
      strcpy(newsgroups_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Newsgroups file: %s\n",newsgroups_path);
    }

    /* Set the global limit of bytes per message area */

    if (strnicmp(buf,"NEWS-AREA-BYTES",15) == 0)
    {
      news_area_bytes = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Max. bytes per area: %lu\n",news_area_bytes);
    }

    /* Set the global limit of messages per message area */

    if (strnicmp(buf,"NEWS-AREA-MESSAGES",18) == 0)
    {
      news_area_messages = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Max. msgs per area: %lu\n",news_area_messages);
    }

    /* Set the global limit of total bytes */

    if (strnicmp(buf,"NEWS-TOTAL-BYTES",16) == 0)
    {
      news_total_bytes = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Max. bytes : %lu\n",news_total_bytes);
    }

    /* Set the global limit of total messages */

    if (strnicmp(buf,"NEWS-TOTAL-MESSAGES",18) == 0)
    {
      news_total_messages = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Max. msgs : %lu\n",news_total_messages);
    }

    /* Set flag to allow/disallow kills of summaries */

    if (strnicmp(buf,"KILL-SUMMARY",12) == 0)
    {
      kill_summary = convert_to_number(extract_parm(buf,delimiter));
      if (verbose == YES)
      {
        if (kill_summary == YES)
          RPrintf("Kill file enabled for summaries\n");
        else
          RPrintf("Kill file disabled for summaries\n");
      }
    }
  }
  fclose(config_file);

  if (verbose == YES)
    RPrintf("Solar config file loaded. Validating...\n");

  /* Make sure the SOUP index and message types for news are supported */

  switch (news_index) {
    case n  : break;
    case c  : break;
    case C  : break;
    default : sprintf(_slrerr,"load_config(): unsupported news-index in %s\n",config_path);
              goto ErrorExit;
  }
  switch (news_type) {
    case u  : break;
    case i  : break;
    case B  : break;
    default : sprintf(_slrerr,"load_config(): unsupported news-type in %s\n",config_path);
              goto ErrorExit;
  }

  if (verbose == YES)
    RPrintf("Solar config file load OK\n");

GoodExit:
  return 0;
ErrorExit:
  if (config_file) fclose(config_file);
  return 1;
}

/*
Function: user_defaults()
Purpose : Check for a user config file and load values if present.
Return  : zero on success, non-zero on error and set _slrerr.
*/

int user_defaults()
{
  FILE *user_config = NULL;

  char buf[128];
  char path[MAXPATH];
  char delimiter = '=';

  strcpy(path,user_path);
  strcat(path,"\\");
  strcat(path,OPTIONS_NAME);

  if (verbose == YES)
    RPrintf("User config file: %s\n",path);

  if ((user_config = fopen(path,"rt")) == NULL)
  {
    if (verbose == YES)
      RPrintf("No user config file. Using defaults.\n");
    goto GoodExit;    /* For some reason, user has no defaults */
  }

  while (fgets(buf,128,user_config) != NULL)
  {
    /* Override the SOUP index type for news */

    if (strnicmp(buf,"NEWS-INDEX",10) == 0)
    {
      news_index = convert_to_number(extract_parm(buf,delimiter));
      if (verbose == YES)
      {
        if (news_index == n) RPrintf("News index: n (None)\n");
        if (news_index == C) RPrintf("News index: C (Partial C News)\n");
        if (news_index == c) RPrintf("News index: c (Full C News)\n");
      }
    }

    /* Override the SOUP message type for news */

    if (strnicmp(buf,"NEWS-TYPE",9) == 0)
    {
      news_type = convert_to_number(extract_parm(buf,delimiter));
      if (verbose == YES)
      {
        if (news_type == u) RPrintf("news type: u (USENET Format)\n");
        if (news_type == B) RPrintf("news type: B (8-bit binary)\n");
        if (news_type == i) RPrintf("news type: i (Index only)\n");
      }
    }
  }
  fclose(user_config);

  if (verbose == YES)
    RPrintf("User config file loaded. Validating...\n");

  /* Make sure SOUP index and message types for news are supported */

  switch (news_index) {
    case n  : break;
    case c  : break;
    case C  : break;
    default : sprintf(_slrerr,"user_defaults(): unsupported news-index in %s\n",config_path);
              goto ErrorExit;
  }
  switch (news_type) {
    case u  : break;
    case i  : break;
    case B  : break;
    default : sprintf(_slrerr,"user_defaults(): unsupported news-type in %s\n",config_path);
              goto ErrorExit;
  }

  if (verbose == YES)
    RPrintf("User config file load OK\n");

GoodExit:
  return 0;
ErrorExit:
  if (user_config) fclose(user_config);
  return 1;
}

