/*===========================================================================
SOLAR v0.95 :: Common module newsrc.c
Original Author: Kevin Houle <kjhoule@iowegia.dsm.ia.us>

This software module has been placed in the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
03-29-94 KJH  Made a common module.
11-26-94 KJS  Changed fprintf() to RPrintf() for FOSSIL support.
===========================================================================*/

/* Include Files */
#include <stdio.h>
#include <dir.h>
#include <string.h>
#include <values.h>
#include <stdlib.h>
#include "..\common\yesno.h"

#define NEWSRC      "NEWSRC.SLR"
#define NEWSRC_TMP  "NEWSRC.TMP"

enum   SOUP_Types { u, M, m, n, C, c, i, b, B };  /* Possible SOUP types */

/* External Data */
extern int  verbose;              /* From config.c  */
extern char _slrerr[80];          /* From config.c  */
extern char user_path[MAXPATH];   /* From config.c  */
extern int  news_type;            /* From config.c  */

/* Global Data */
char   area_name[80];             /* Newsgroup name buffer          */
char   keyword;                   /* Newsgroup processing flag      */
long   index_high;                /* High index number processed    */
long   msg_high;                  /* High article number processed  */

#ifdef _SLRCMD
  extern char default_keyword;    /* Default new group keyword      */
#endif

/* External functions */
#ifdef _SLRCMD
  extern char *extract_parm(char string[128], char delimiter);
#endif

/* Global Functions */
FILE *open_newsrc(const char *newsrc_type, const char *open_mode);
int  write_newsrc(const char *newsrc_type, const char *open_mode);
int  kill_newsrc(const char *newsrc_type);
int  update_newsrc();

/*
 * Function: int read_newsrc(FILE *newsrc)
 * Purpose : Load a message area from newsrc into memory. This function
             assumes the newsrc file is already open.
 * Return  : Zero on success
 *           1 on EOF
 *          -1 and set _slrerr on error
*/

int read_newsrc(FILE *newsrc)
{
	char buf[1024];
	char *p = NULL;

	if (verbose == YES)
		RPrintf("Reading newsrc file...\n");

	/* Get the first line of multi-line newsrc record */

  if (fgets(buf,1024,newsrc))
	{
		/* Parse the first token; area name */

		if (!(p = strtok(buf,":")))
		{
      sprintf(_slrerr,"read_newsrc(): malformed newsrc entry: %s",buf);
			goto ErrorExit;
		}

    /* Copy newsgroup name to more permenant storage */

    strncpy(area_name,p,80);
		if (verbose == YES)
      RPrintf("Area found: %s\n",area_name);

    /* Parse the second token; keyword. Default to a value of 'a'
       meaning all. This should be a configurable value in solar.cfg */

		if ((p = strtok(NULL,":")) != NULL)
      keyword = p[0];
		else
      keyword = 'a';
    if (verbose == YES)
      RPrintf("Area keyword: %c\n",keyword);

    /* Parse highest index number processed, default to a long zero */

		if ((p = strtok(NULL,":")) != NULL)
      index_high = atol(p);
		else
      index_high = 0L;
    if (verbose == YES)
      RPrintf("Last index: %lu\n",index_high);

    /* Parse highest message number processed, default to a long zero */

		if ((p = strtok(NULL,":")) != NULL)
      msg_high = atol(p);
		else
      msg_high = 0L;
    if (verbose == YES)
      RPrintf("Last message: %lu\n",msg_high);
  }
	else
	{
    if (verbose == YES)
      RPrintf("End of newsrc file.\n");
    return 1; /* end of file */
	}

GoodExit:
  return 0;     /* Success */
ErrorExit:
  return -1;    /* Error   */
}

/*
 * Function: FILE *open_newsrc(const char *newsrc_type, const char *open_mode);
 * Purpose : Open newsrc file for whatever mode specified.
 * Return  : Pointer to newsrc file, NULL if not found.
*/

FILE *open_newsrc(const char *newsrc_type, const char *open_mode)
{
  FILE *temp_newsrc = NULL;
  char newsrc_path[MAXPATH];

	strcpy(newsrc_path,user_path);
  strcat(newsrc_path,"\\");
  strcat(newsrc_path,newsrc_type);

  if (verbose == YES)
    RPrintf("Trying to open newsrc file %s\n",newsrc_path);
  temp_newsrc = fopen(newsrc_path,open_mode);
  return temp_newsrc;
}

/*
 * Function: int write_newsrc(const char *newsrc_type, const char *open_mode)
 * Purpose : Write newsrc information from memory to file.
 * Return  : zero on success, non-zero and set _slrerr on error.
*/

int write_newsrc(const char *newsrc_type, const char *open_mode)
{
  FILE *temp_newsrc = NULL;

  temp_newsrc = open_newsrc(newsrc_type,open_mode);
  if (!temp_newsrc)
  {
    sprintf(_slrerr,"write_newsrc(): error opening %s in mode %s",newsrc_type,open_mode);
    goto ErrorExit;
  }

  fprintf(temp_newsrc,"%s:%c:",area_name,keyword);
  fprintf(temp_newsrc,"%lu:%lu\n",index_high,msg_high);
  fclose(temp_newsrc);
  if (verbose == YES)
  {
    RPrintf("Wrote newsrc: %s:%c:",area_name,keyword);
    RPrintf("%lu:%lu\n",index_high,msg_high);
  }

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
 * Function: int update_newsrc()
 * Purpose : Write NEWSRC_TMP as NEWSRC
 * Return  : Zero on success, 1 on error and set _slrerr.
*/

int update_newsrc()
{
  char old_newsrc[MAXPATH];
  char new_newsrc[MAXPATH];

  strcpy(old_newsrc,user_path);
  strcat(old_newsrc,"\\");
  strcat(old_newsrc,NEWSRC_TMP);
  strcpy(new_newsrc,user_path);
  strcat(new_newsrc,"\\");
  strcat(new_newsrc,NEWSRC);

  kill_newsrc(NEWSRC);
  if (rename(old_newsrc,new_newsrc) != 0)
  {
    sprintf(_slrerr,"update_newsrc(): error renaming %s to %s",old_newsrc,new_newsrc);
    goto ErrorExit;
  }
  kill_newsrc(NEWSRC_TMP);
  RPrintf("Updated message pointers in %s\n",NEWSRC);

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
 * Function: int kill_newsrc(const char *newsrc_type);
 * Purpose : Unlink a newsrc file
 * Return  : zero on success, -1 on error.
*/

int kill_newsrc(const char *newsrc_type)
{
  char path[MAXPATH];

  strcpy(path,user_path);
  strcat(path,"\\");
  strcat(path,newsrc_type);
  return unlink(path);
}

#ifdef _SLRCMD
/*
 * Function: int subscribe(char *area)
 * Purpose : Add a message area to newsrc file. There is no checking for
 *           a valid group or any security here because the user could
 *           always add a group to their newsrc file with an editor. The
 *           validity and security checks are left to slrnews.
 * Return  : Zero on success
 *           1 on security access denied
 *           -1 on error
*/

int subscribe(char *area)
{
  FILE *newsrc =  NULL;

  if (verbose == YES)
     RPrintf("Attempting to subscribe to %s\n",area);

  if ((newsrc = open_newsrc(NEWSRC,"rt")) != NULL)
  {
    while (read_newsrc(newsrc) != 1)
    {
      if (stricmp(area,area_name) == 0)
      {
        RPrintf("Already subscribed to %s\n",area);
        goto GoodExit;
      }
    }
  }
  else
  {
    if (verbose == YES)
       RPrintf("Creating %s file\n",NEWSRC);
  }
  fclose(newsrc); /* Finished reading */

  strcpy(area_name, area);
  keyword = default_keyword;
  msg_high    = 0L;
  index_high  = 0L;

  if (write_newsrc(NEWSRC,"at") != 0) goto ErrorExit;
  RPrintf("Subscription added to %s for %s\n",NEWSRC,area);

GoodExit:
  return 0;
ErrorExit:
  if (newsrc) fclose(newsrc);
  return -1;
}

/*
 * Function: int unsubscribe(char *area)
 * Purpose : Remove a message area from newsrc file.
 * Return  : Zero on success
 *           -1 on error
*/

int unsubscribe(char *area)
{
  FILE *newsrc  = NULL;
  int success   = NO;

  if (verbose == YES)
     RPrintf("Attempting to unsubscribe to %s\n",area);

  kill_newsrc(NEWSRC_TMP);
  if ((newsrc = open_newsrc(NEWSRC,"rt")) == NULL) goto ErrorExit;
  while (read_newsrc(newsrc) != 1)
  {
    if (stricmp(area,area_name) != 0)
    {
      if (write_newsrc(NEWSRC_TMP,"at") != 0)
        goto ErrorExit;
    }
    else
    {
      success = YES;
    }
  }
  fclose(newsrc);

  if (success == YES)
  {
    RPrintf("Subscription removed from %s for %s\n",NEWSRC,area);
    if (update_newsrc() != 0) goto ErrorExit;
  }
  else
  {
    RPrintf("Area %s not found in %s\n",area,NEWSRC);
  }

  kill_newsrc(NEWSRC_TMP);

GoodExit:
  return 0;
ErrorExit:
  if (newsrc) fclose(newsrc);
  return -1;
}

#endif

/*
Function: int join2newsrc()
Purpose : Convert Waffle's join file to Solar's newsrc format.
Return  : 0 on success, 1 on error, -1 on fatal. set _slrerr.
*/

int join2newsrc()
{
  FILE *join_file   = NULL;

  char join_path[MAXPATH];
  char joinbuf[80];
  char *p;

  /* Make path to JOIN file */

  strcpy(join_path, user_path);
  strcat(join_path, "\\JOIN");

  /* Open the JOIN file for reading. Assume an error means the user
     has no JOIN file, which is not fatal. */

  if ((join_file = fopen(join_path,"rt")) == NULL)
  {
    sprintf(_slrerr, "Waffle JOIN file not found.");
    goto ErrorExit;
  }

  /* OK, there is a JOIN file. Kill any existing NEWSRC.SLR file. */

  kill_newsrc(NEWSRC);

  /* Process each line in the JOIN file */

  while (fgets(joinbuf,80,join_file) != NULL)
  {
    /* The first token should be the area name. */

    if ((p = strtok(joinbuf," ")) == NULL)
    {
      sprintf(_slrerr,"join2newsrc(): invalid %s file format",join_path);
      goto FatalExit;
    }
    strcpy(area_name,p);

    /* The second token, if it exists, is the high message number read
       for that area. If non-existant, set the value to zero. */

    if ((p = strtok(NULL," ")) != NULL)
      msg_high = atol(p);
    else
      msg_high = 0L;

    /* Waffle's JOIN file does not allow for index tracking or an
       individual keyword for each newsgroup. Set index to match
       high message read, and set the keyword to match the default
       SOUP news type. */

    index_high = msg_high;
    if (news_type == u)
      keyword = 'a';
    else
      keyword = 'i';

    /* Write the newsrc record */

    if (write_newsrc(NEWSRC,"at") != 0)
      goto FatalExit;

  }
  fclose(join_file);

GoodExit:
  return 0;
ErrorExit:
  if (join_file) fclose(join_file);
  return 1;
FatalExit:
  if (join_file) fclose(join_file);
  return -1;
}

/*
Function: int newsrc2join()
Purpose : Convert newsrc.slr to Waffle's join format.
Return  : 0 on success, 1 on error, -1 on fatal. set _slrerr.
*/

int newsrc2join()
{
  FILE *join_file   = NULL;
  FILE *newsrc_file = NULL;

  char join_path[MAXPATH];
  char *p;

  /* Make path to JOIN file */

  strcpy(join_path, user_path);
  strcat(join_path, "\\JOIN");

  /* Open the newsrc file for reading. */

  if ((newsrc_file = open_newsrc(NEWSRC,"rt")) == NULL)
  {
    sprintf(_slrerr,"File %s not found.",NEWSRC);
    goto ErrorExit;
  }

  /* Open the JOIN file for writing. */

  if ((join_file = fopen(join_path,"wt")) == NULL)
  {
    sprintf(_slrerr, "newsrc2join(): Error opening %s for write.", join_path);
    goto FatalExit;
  }

  /* Process all records in newsrc and write records to JOIN. Note,
     the index tracking and keywords are lost in this process. */

  while (read_newsrc(newsrc_file) == 0)
  {
    fprintf(join_file,"%s %lu\n",area_name,msg_high);
	}
  fclose(join_file);

GoodExit:
  return 0;
ErrorExit:
  if (join_file) fclose(join_file);
  if (newsrc_file) fclose(newsrc_file);
  return 1;
FatalExit:
  if (join_file) fclose(join_file);
  if (newsrc_file) fclose(newsrc_file);
  return -1;
}


