{$G+,D-,E-,S-}
Unit XGraph;

{--------------------------------------------------------------------}

interface

procedure InitGraph;
procedure CloseGraph;
procedure Cls;
procedure ClrWin;
procedure SetColor(C:Byte);
function  GetColor:Byte;
procedure SetWin(X1, Y1, X2, Y2:Integer);
procedure GetWin(var X1, Y1, X2, Y2:Integer);
procedure PutPixel(X, Y:Integer);
procedure PutPixelC(X, Y:Integer; C:Byte);
function  GetPixel(X, Y:Integer):Byte;
procedure PutBigPixel(X, Y:Integer);
procedure PutBigPixelC(X, Y:Integer; C:Byte);
function  GetBigPixel(X, Y:Integer):Byte;
procedure Line(X1, Y1, X2, Y2:Integer);
procedure PutRow(X1, X2, Y:Integer);
procedure Rect(X1, Y1, X2, Y2:Integer);
procedure FRect(X1, Y1, X2, Y2:Integer);
procedure Circle(CX, CY, R:Integer);
procedure FCircle(CX, CY, R:Integer);
procedure WaitVRT;
procedure DelayVRT(N:Integer);
procedure SetVPage(Page:Integer);
procedure SetWPage(Page:Integer);
procedure SetPage(Page:Integer);
function  GetVPage:Integer;
function  GetWPage:Integer;
procedure CopyPage(PSrc, PDest:Integer);
procedure SetPal(PalNum, R, G, B:Byte);
procedure SetPals(S, N:Byte; var PalList);
procedure ResetPals;
procedure SetFont(var F);
procedure SetFont8x8;
procedure OutStr(X, Y:Integer; const S:String);
procedure OutSolidStr(X, Y:Integer; const S:String);
procedure OutBigStr(X, Y:Integer; const S:String; MagX, MagY, Skew:Integer);
procedure GetImage(var Buf; X1, Y1, X2, Y2:Integer);
procedure PutImage(var Buf; X1, Y1:Integer);
procedure PutSprite(var Buf; X1, Y1:Integer);
function  InitMouse:Boolean;
procedure CloseMouse;
procedure ShowMouse;
procedure HideMouse;
function  GetMouse(var X, Y:Integer):Integer;
procedure SetMouse(X, Y:Integer);
function  GetMousePress(Btn:Integer; var X, Y:Integer):Integer;
function  GetMouseRelease(Btn:Integer; var X, Y:Integer):Integer;
procedure SetMouseCursor(var Cur; hX, hY:Integer);

{--------------------------------------------------------------------}

implementation
uses Dos;

const
    XData : array[1..10] of Word = (
        $0011, $0B06, $3E07, $EA10,
        $8C11, $DF12, $0014, $E715,
        $0416, $E317);

    Gamma4  : array[0..3]  of Byte = (0, 29, 46, 63);
    Gamma16 : array[0..15] of Byte = (
        0, 14, 18, 22, 26, 30, 34, 38, 42, 45, 48, 51, 54, 57, 60, 63);

var
    XRegs:Registers;

    CurSeg  : Word;
    Color   : Byte;

    Font    : PChar;
    FontW   : Integer;
    FontH   : Integer;

    WinX1   : Integer;
    WinY1   : Integer;
    WinX2   : Integer;
    WinY2   : Integer;

{--------------------------------------------------------------------}

procedure Exch(var A, B:Integer);
begin
    A := A XOR B;
    B := B XOR A;
    A := A XOR B;
end;

{--------------------------------------------------------------------}

procedure Dummy;
begin
end;

{--------------------------------------------------------------------}

function Interp(A1, A2, B1, B2, B:Integer):Integer; assembler;
asm
    mov ax,B; sub ax,B1;    {       (B - B1) * (A2 - A1)  }
    mov bx,A2; sub bx,A1;   { A1 + ---------------------- }
    mov cx,B2; sub cx,B1;   {            (B2 - B1)        }
    imul bx;
    idiv cx;
    add ax,A1;
end;


{--------------------------------------------------------------------}

procedure InitGraph;
var
    N, I:Integer;
    B:Byte;
begin
    asm                         { Set mode 13h 320x200x256 }
        mov ax,13h; int 10h;
    end;

    B := Port[$03CC] OR $C0;    { Set ModeX 320x240x256 }
    Port[$03C2] := B;
    PortW[$03C4] := $0604;
    for I := 1 to 10 do begin
        PortW[$03D4] := XData[I];
    end;

    for N := 0 to 63 do begin   { Set up the palette }
        SetPal(N, Gamma4[(N shr 4) AND $03],
                  Gamma4[(N shr 2) AND $03],
                  Gamma4[(N shr 0) AND $03]);
    end;
    for N := 0 to 63 do SetPal(N + 192,  N, N, N);
    for N := 0 to 15 do begin
        I := Gamma16[N];
        SetPal(N + 64,  I, 0, 0); SetPal(N + 80,  I, I, 0);
        SetPal(N + 96,  0, I, 0); SetPal(N + 112, 0, I, I);
        SetPal(N + 128, 0, 0, I); SetPal(N + 144, I, 0, I);
        SetPal(N + 160, I, I div 2, 0);
        SetPal(N + 176, 0, I div 2, I);
    end;

    SetFont8x8;                 { Set the default font }

    CurSeg := $A000;            { Set initial variables }
    Color := $3F;
    WinX1 := 0;
    WinY1 := 0;
    WinX2 := 319;
    WinY2 := 239;

    asm                         { Clear all of video memory }
        mov es,CurSeg;
        xor di,di; xor ax,ax;
        mov cx,8000h; rep stosw;
    end;
end;

{--------------------------------------------------------------------}

procedure CloseGraph; assembler;
asm
    mov ax,03h;
    int 10h;
end;

{--------------------------------------------------------------------}

procedure Cls; assembler;
asm
    mov dx,03C4h;           { Turn on all planes }
    mov ax,0F02h;
    out dx,ax;
    mov es,CurSeg;          { ES:DI = video memory }
    xor di,di;
    xor ax,ax;              { Clear the current page }
    mov cx,2800h;
    rep stosw;
end;

{--------------------------------------------------------------------}

procedure ClrWin;
var
    oColor:Byte;
begin
    oColor := Color; Color := 0;
    FRect(WinX1, WinY1, WinX2, WinY2);
    Color := oColor;
end;

{--------------------------------------------------------------------}

procedure SetColor(C:Byte);
begin
    Color := C;
end;

{--------------------------------------------------------------------}

function GetColor:Byte;
begin
    GetColor := Color;
end;

{--------------------------------------------------------------------}

procedure SetWin(X1, Y1, X2, Y2:Integer);
begin
    WinX1 := X1;
    WinY1 := Y1;
    WinX2 := X2;
    WinY2 := Y2;
end;

{--------------------------------------------------------------------}

procedure GetWin(var X1, Y1, X2, Y2:Integer);
begin
    X1 := WinX1;
    Y1 := WinY1;
    X2 := WinX2;
    Y2 := WinY2;
end;

{--------------------------------------------------------------------}

procedure PutPixel(X, Y:Integer); assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    mov cx,X;               { CX = X, BX = Y }
    mov bx,Y;

    cmp cx,WinX1; jl @pp_done;  { Clip to window }
    cmp cx,WinX2; jg @pp_done;
    cmp bx,WinY1; jl @pp_done;
    cmp bx,WinY2; jg @pp_done;

    mov dx,03C4h;           { Set the memory plane }
    mov ax,1102h;
    rol ah,cl;
    out dx,ax;

    mov ax,bx;              { BX = offset in video memory }
    shl bx,2;
    add bx,ax;
    shl bx,4;
    shr cx,2;
    add bx,cx;
    mov cl,Color;           { Set the pixel }
    mov es:[bx],cl;
@pp_done:
end;

{--------------------------------------------------------------------}

procedure PutPixelC(X, Y:Integer; C:Byte); assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    mov cx,X;               { CX = X, BX = Y }
    mov bx,Y;

    cmp cx,WinX1; jl @pp_done;  { Clip to window }
    cmp cx,WinX2; jg @pp_done;
    cmp bx,WinY1; jl @pp_done;
    cmp bx,WinY2; jg @pp_done;

    mov dx,03C4h;           { Set the memory plane }
    mov ax,1102h;
    rol ah,cl;
    out dx,ax;

    mov ax,bx;              { BX = offset in video memory }
    shl bx,2;
    add bx,ax;
    shl bx,4;
    shr cx,2;
    add bx,cx;
    mov cl,C;               { Set the pixel }
    mov es:[bx],cl;
@pp_done:
end;

{--------------------------------------------------------------------}

procedure PutBigPixel(X, Y:Integer); assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    mov cx,X;               { CX = X, BX = Y }
    mov bx,Y;

    cmp cx,80; jae @pp_done;    { Clip to screen }
    cmp bx,60; jae @pp_done;

    imul bx,bx,320;         { BX = offset }
    add bx,cx;

    mov dx,03C4h;           { Turn on all planes }
    mov ax,0F02h; out dx,ax;
    mov al,Color;           { Draw the big pixel }
    mov es:[bx],al; mov es:[bx+80],al;
    mov es:[bx+160],al; mov es:[bx+240],al;
@pp_done:
end;

{--------------------------------------------------------------------}

procedure PutBigPixelC(X, Y:Integer; C:Byte); assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    mov cx,X;               { CX = X, BX = Y }
    mov bx,Y;

    cmp cx,80; jae @pp_done;    { Clip to screen }
    cmp bx,60; jae @pp_done;

    imul bx,bx,320;         { BX = offset }
    add bx,cx;

    mov dx,03C4h;           { Turn on all planes }
    mov ax,0F02h; out dx,ax;
    mov al,C;               { Draw the big pixel }
    mov es:[bx],al; mov es:[bx+80],al;
    mov es:[bx+160],al; mov es:[bx+240],al;
@pp_done:
end;

{--------------------------------------------------------------------}

function GetBigPixel(X, Y:Integer):Byte; assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    imul bx,Y,320;          { BX = offset }
    add bx,X;
    mov al,es:[bx];         { Read big pixel }
end;

{--------------------------------------------------------------------}

function GetPixel(X, Y:Integer):Byte; assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    mov cx,X;               { CX = X, BX = Y }
    mov bx,Y;

    mov dx,03CEh;           { Set the memory plane }
    mov al,4;
    mov ah,cl;
    and ah,3;
    out dx,ax;

    mov ax,bx;              { BX = offset in video memory }
    shl bx,2;
    add bx,ax;
    shl bx,4;
    shr cx,2;
    add bx,cx;
    mov al,es:[bx];         { Read the pixel }
    or al,es:[bx];
end;

{--------------------------------------------------------------------}

procedure _Line(X1, Y1, X2, Y2:Integer); assembler;
asm
    push bp;                { Save BP }
    mov es,CurSeg;          { ES = current segment }

    mov ax,X1;              { Get parameters }
    mov bx,Y1;
    mov cx,X2;
    mov dx,Y2;

    mov si,cx;              { Get X distance }
    sub si,ax;

    mov di,dx;              { Get Y distance }
    sub di,bx;
    jge @l_skip;
    neg di;

@l_skip:
    cmp si,di;              { Y-major? }
    jle @l_ymajor;

    sub cx,ax;              { CX = distance }
    sub dx,bx;              { DX = Y increment }
    sar dx,16;
    add dx,dx;
    inc dx;

    xchg cx,dx;             { Rotate registers }

    mov bp,si;              { BP = X distance }
    shr si,1;               { SI = error term }
    sub si,di;
    neg si;

    imul bx,bx,80;          { BX = offset in video memory }
    ror ax,2;
    add bl,al;
    adc bh,0;
    shr ax,14;
    push cx;                { Save CX }
    mov ah,11h;             { AH = plane }
    mov cl,al;
    rol ah,cl;
    pop cx;                 { Restore CX }

    imul cx,cx,80;          { CX = Y increment }
    mov al,ah;              { AL = plane value }

@l_xloop:
    or ah,al;               { OR in plane }
    cmp si,1;               { Check error value }
    jl @l_xstr;             { ZF clear if taken }

    push dx;                { Save registers }
    push ax;
    mov dx,03C4h;           { Set bit planes }
    mov al,2;
    out dx,ax;
    pop ax;                 { Restore registers }
    pop dx;

    mov ah,Color;           { AH = color }
    mov es:[bx],ah;         { Write pixels }

    add bx,cx;              { Next line }
    sub si,bp;
    xor ah,ah;              { Clear buffer, set ZF }

@l_xstr:
    rol al,1;               { Next plane }
    jnc @l_x2;              { New byte? }
    jz @l_x1;               { Buffer empty? }

    push dx;                { Save DX }
    mov dx,03C4h;           { Set bit planes }
    mov al,2;
    out dx,ax;
    pop dx;                 { Restore DX }

    mov al,Color;           { AL = color }
    mov es:[bx],al;         { Write pixels }
    mov ax,11h;             { Clear buffer }
@l_x1:
    inc bx;                 { Next byte }

@l_x2:
    add si,di;
    dec dx;                 { Loop back }
    jnl @l_xloop;

    mov dx,03C4h;           { Set bit planes }
    mov al,2;
    out dx,ax;
    mov al,Color;           { Write last pixels }
    mov es:[bx],al;
    jmp @l_done;            { Return }

@l_ymajor:
    xchg cx,dx;             { Switch X2, Y2 }
    mov dx,di;              { DX = distance }
    sub cx,bx;              { CX = Y increment }
    sar cx,16;
    add cx,cx;
    inc cx;

    mov bp,si;              { BP = X distance }
    mov si,di;              { SI = error term }
    shr si,1;
    sub si,bp;
    neg si;

    imul bx,bx,80;          { BX = offset in video memory }
    ror ax,2;
    add bl,al;
    adc bh,0;
    shr ax,14;
    push cx;                { Save CX }
    xchg cx,ax;             { AH = plane value }
    mov ah,11h;
    rol ah,cl;
    pop cx;                 { Restore CX }

    push dx;                { Save DX }
    mov al,2;               { Set first plane }
    mov dx,03C4h;
    out dx,ax;
    pop dx;                 { Restore DX }

    imul cx,cx,80;          { CX = Y increment }

@l_yloop:
    mov al,Color;           { AL = color }
    mov es:[bx],al;         { Set the pixel }
    test si,si;             { Check error value }
    jle @l_ystr;

    rol ah,1;               { Move in X direction }
    adc bx,0;

    push dx;                { Save DX }
    mov al,2;               { Set new plane }
    mov dx,03C4h;
    out dx,ax;
    pop dx;                 { Restore DX }
    sub si,di;              { Adjust error term }

@l_ystr:
    add bx,cx;              { Go straight }
    add si,bp;
    dec dx;                 { Loop back }
    jnl @l_yloop;

@l_done:
    pop bp;                 { Restore BP }
end;

{--------------------------------------------------------------------}

procedure Line(X1, Y1, X2, Y2:Integer);
var
    I:LongInt;
label 1;
begin
    if Y1 > Y2 then begin           { Put Y in order }
        Exch(X1, X2);
        Exch(Y1, Y2);
    end;
    if Y1 < WinY1 then begin        { Clip Y1 }
        if Y2 < WinY1 then goto 1;
        X1 := Interp(X1, X2, Y1, Y2, WinY1);
        Y1 := WinY1;
    end;
    if Y2 > WinY2 then begin        { Clip Y2 }
        if Y1 > WinY2 then goto 1;
        X2 := Interp(X1, X2, Y1, Y2, WinY2);
        Y2 := WinY2;
    end;

    if X1 > X2 then begin           { Put X in order }
        Exch(X1, X2);
        Exch(Y1, Y2);
    end;
    if X1 < WinX1 then begin        { Clip X1 }
        if X2 < WinX1 then goto 1;
        Y1 := Interp(Y1, Y2, X1, X2, WinX1);
        X1 := WinX1;
    end;
    if X2 > WinX2 then begin        { Clip X2 }
        if X1 > WinX2 then goto 1;
        Y2 := Interp(Y1, Y2, X1, X2, WinX2);
        X2 := WinX2;
    end;

    _Line(X1, Y1, X2, Y2);          { Draw the line}
1:
end;

{--------------------------------------------------------------------}

procedure _PutRow(X1, X2, Y:Integer); assembler;
asm
    mov es,CurSeg;          { ES = current segment }
    mov bx,X1;              { Get parameters }
    mov cx,X2;
    mov dx,Y;

    imul dx,dx,80;          { DX = row offset }
    mov di,bx;              { DI = start offset }
    shr di,2;
    add di,dx;
    mov si,cx;              { SI = end offset }
    shr si,2;
    add si,dx;

    mov ch,bl;              { CL = X2 plane }
    and cx,0303h;           { CH = X1 plane }

    mov bx,0201h;           { BH = mask for X2 }
    shl bh,cl;
    dec bh;
    mov cl,ch;              { BL = mask for X1 }
    shl bl,cl;
    neg bl;

    mov dx,03C4h;           { DX = SC port }
    mov al,2;               { AL = plane command }
    mov ch,Color;           { CH = color }

    cmp si,di;              { Check for tiny line }
    je @pr_tiny;

    mov ah,bl;              { Set left planes }
    out dx,ax;
    mov es:[di],ch;         { Write pixels }
    mov ah,bh;              { Set right planes }
    out dx,ax;
    mov es:[si],ch;         { Write pixels }
    inc di;                 { Advance pointer }

    mov ax,0F02h;           { Set all planes }
    out dx,ax;
    mov al,ch;              { AX = color }
    mov ah,al;
    mov cx,si;              { CX = byte count }
    sub cx,di;

    shr cx,1;               { Count in words }
    rep stosw;              { Draw solid line }
    adc cx,0;               { Possible odd byte }
    rep stosb;              { Write odd bytes }
    jmp @pr_done;

@pr_tiny:
    and bl,bh;              { BL = combined planes }
    mov ah,bl;              { Set planes }
    out dx,ax;
    mov es:[di],ch;         { Write pixels }

@pr_done:
end;

{--------------------------------------------------------------------}

procedure PutRow(X1, X2, Y:Integer);
begin
    if (Y < WinY1) or (Y > WinY2) then Exit;
    if X1 > X2 then Exch(X1, X2);
    if (X2 < WinX1) or (X1 > WinX2) then Exit;
    if X1 < WinX1 then X1 := WinX1;
    if X2 > WinX2 then X2 := WinX2;
    _PutRow(X1, X2, Y);
end;

{--------------------------------------------------------------------}

procedure Rect(X1, Y1, X2, Y2:Integer);
begin
    PutRow(X1, X2, Y1);
    PutRow(X1, X2, Y2);
    Line(X1, Y1, X1, Y2);
    Line(X2, Y1, X2, Y2);
end;

{--------------------------------------------------------------------}

procedure FRect(X1, Y1, X2, Y2:Integer);
var
    I:Integer;
begin
    if Y1 > Y2 then Exch(Y1, Y2);
    for I := Y1 to Y2 do begin
        PutRow(X1, X2, I);
    end;
end;

{--------------------------------------------------------------------}

procedure Circle(CX, CY, R:Integer);
var
    X, Y, A, B:Integer;
begin
    if R > 0 then begin
        X := 0;
        Y := R * 32;

        repeat begin
            A := (X + 16) shr 5; B := (Y + 16) shr 5;
            PutPixel(CX + A, CY + B); PutPixel(CX - A, CY + B);
            PutPixel(CX + A, CY - B); PutPixel(CX - A, CY - B);
            PutPixel(CX + B, CY + A); PutPixel(CX - B, CY + A);
            PutPixel(CX + B, CY - A); PutPixel(CX - B, CY - A);
            X := X + (Y div R);
            Y := Y - (X div R);
        end until B <= A;
    end else if R = 0 then PutPixel(CX, CY);
end;

{--------------------------------------------------------------------}

procedure FCircle(CX, CY, R:Integer);
var
    X, Y, oX, oY, A, B:Integer;
begin
    if R > 0 then begin
        X := 0;
        Y := R * 32;
        oY := -1;

        repeat begin
            oX := X; oY := Y;
            X := X + (Y div R);
            Y := Y - (X div R);
            if oY <> Y then begin
                A := (oX + 16) div 32; B := (oY + 16) div 32;
                PutRow(CX - A, CX + A, CY + B);
                PutRow(CX - A, CX + A, CY - B);
            end;
        end until B <= 0;
    end else if R = 0 then PutPixel(CX, CY);
end;

{--------------------------------------------------------------------}

procedure WaitVRT; assembler;
asm
    mov dx,03DAh;           { DX = IS1 port }
@wv_loop1:
    in al,dx;               { Wait for active period }
    test al,8;
    jnz @wv_loop1;
@wv_loop2:
    in al,dx;               { Wait for vertical retrace }
    test al,8;
    jz @wv_loop2;
end;

{--------------------------------------------------------------------}

procedure DelayVRT(N:Integer);
var
    I:Integer;
begin
    for I := 1 to N do WaitVRT;
end;

{--------------------------------------------------------------------}

procedure SetVPage(Page:Integer); assembler;
asm
    mov dx,03DAh;           { DX = IS1 port }
@vrt_loop:
    in al,dx;               { Wait for active period }
    test al,8;
    jnz @vrt_loop;

    mov ax,Page;            { AH = Page * $50 }
    cmp ax,3; jae @sv_done; { Page can't be >= 3 }
    imul ax,ax,5000h;
    mov dx,03D4h;           { Set new start address }
    mov al,0Ch; out dx,ax;

@sv_done:
end;

{--------------------------------------------------------------------}

procedure SetWPage(Page:Integer); assembler;
asm
    mov ax,Page;            { AX = Page * $500 }
    cmp ax,3; jae @sw_done; { Page can't be >= 3 }
    imul ax,ax,500h;
    add ax,0A000h;          { $A000, $A500, $AA00 }
    mov CurSeg,ax;          { Set CurSeg }
@sw_done:
end;

{--------------------------------------------------------------------}

procedure SetPage(Page:Integer);
begin
    SetVPage(Page);
    SetWPage(Page);
end;

{--------------------------------------------------------------------}

function GetVPage:Integer; assembler;
asm
    mov dx,03D4h;           { Select SAH register }
    mov al,0Ch; out dx,al;
    inc dx; in al,dx;       { Read page }
    shr al,6; and ax,3;     { Return high 2 bits (0-2) }
end;

{--------------------------------------------------------------------}

function GetWPage:Integer; assembler;
asm
    mov ax,CurSeg;          { Get segment, and convert to }
    shr ax,10; and ax,3;    { page number between 0 and 2 }
end;

{--------------------------------------------------------------------}

procedure CopyPage(PSrc, PDest:Integer); assembler;
asm
    push ds;                { Save DS }

    mov ax,PSrc;            { DS = source page }
    cmp ax,3; jae @cp_done;
    imul ax,ax,500h;
    add ax,0A000h; mov ds,ax;

    mov ax,PDest;           { ES = dest. page }
    cmp ax,3; jae @cp_done;
    imul ax,ax,500h;
    add ax,0A000h; mov es,ax;

    mov dx,03C4h;           { Activate all planes }
    mov ax,0F02h; out dx,ax;
    mov dl,0CEh;            { Clear bit mask (data from latches) }
    mov ax,0008h; out dx,ax;

    xor si,si; xor di,di;   { Copy the page }
    mov cx,5000h;
    rep movsb;

    mov ax,0FF08h;          { Reset bit mask (data from CPU) }
    out dx,ax;

@cp_done:
    pop ds;                 { Restore DS }
end;

{--------------------------------------------------------------------}

procedure SetPal(PalNum, R, G, B:Byte); assembler;
asm
    mov dx,03C8h;           { Select palette register }
    mov al,PalNum;
    out dx,al;
    inc dx;

    call Dummy;             { I/O delay }
    mov al,R; out dx,al;    { Set red value }
    call Dummy;             { I/O delay }
    mov al,G; out dx,al;    { Set green value }
    call Dummy;             { I/O delay }
    mov al,B; out dx,al;    { Set blue value }
end;

{--------------------------------------------------------------------}

procedure SetPals(S, N:Byte; var PalList); assembler;
asm
    push ds;                { Save DS }

    call WaitVRT;           { Wait for vertical retrace }
    mov dx,03C8h;           { DX = DACWA port }
    mov al,S;               { Select palette register }
    out dx,al;
    call Dummy;             { I/O delay }
    inc dx;

    mov cl,N; xor ch,ch;    { CX = num. of bytes }
    imul cx,cx,3;
    lds si,PalList;         { DS:SI = palette list }
    rep outsb;              { Send to DACD port }

    pop ds;                 { Restore DS }
end;

{--------------------------------------------------------------------}

procedure ResetPals;
var
    I, N:Integer;
begin
    for N := 0 to 63 do begin   { Set up the palette }
        SetPal(N, Gamma4[(N shr 4) AND $03],
                  Gamma4[(N shr 2) AND $03],
                  Gamma4[(N shr 0) AND $03]);
    end;
    for N := 0 to 63 do SetPal(N + 192,  N, N, N);
    for N := 0 to 15 do begin
        I := Gamma16[N];
        SetPal(N + 64,  I, 0, 0); SetPal(N + 80,  I, I, 0);
        SetPal(N + 96,  0, I, 0); SetPal(N + 112, 0, I, I);
        SetPal(N + 128, 0, 0, I); SetPal(N + 144, I, 0, I);
        SetPal(N + 160, I, I div 2, 0);
        SetPal(N + 176, 0, I div 2, I);
    end;
end;

{--------------------------------------------------------------------}

procedure SetFont(var F); assembler;
asm
    les si,F;               { Get the pointer }
    mov al,es:[si]; inc si; { Load width and height }
    mov bl,es:[si]; inc si;
    xor ah,ah; xor bh,bh;   { Set width and height }
    mov FontW,ax; mov FontH,bx;

    mov word ptr Font,si;   { Set font pointer }
    mov word ptr Font[2],es;
end;

{--------------------------------------------------------------------}

procedure SetFont8x8; assembler;
asm
    push bp;                { Save BP }
    mov ax,1130h; mov bh,3; { Get the 8x8 font pointer }
    int 10h;
    mov word ptr Font,bp;   { Store the pointer }
    mov word ptr Font+2,es;
    mov ax,8;               { Set width and height }
    mov FontW,ax;
    mov FontH,ax;
    pop bp;                 { Restore BP }
end;

{--------------------------------------------------------------------}

procedure OutStr(X, Y:Integer; const S:String);
var
    P:PChar;
    I, A, B, W:Integer;
    D, E:Byte;
begin
    P := @S;
    for I := 1 to Length(S) do begin
        for A := 0 to FontH - 1 do begin
            D := Ord(Font[Ord(P[I]) * FontH + A]);
            E := $80;
            for B := 0 to FontW - 1 do begin
                if (D AND E) <> 0 then
                    PutPixel(X + B, Y + A);
                E := E shr 1;
            end;
        end;

        X := X + FontW;
    end;
end;

{--------------------------------------------------------------------}

procedure OutSolidStr(X, Y:Integer; const S:String);
var
    I, A, B, W:Integer;
    D, E, C:Byte;
begin
    for I := 1 to Length(S) do begin
        for A := 0 to FontH - 1 do begin
            D := Ord(Font[Ord(S[I]) * FontH + A]);
            E := $80;
            for B := 0 to FontW - 1 do begin
                if (D AND E) <> 0 then C := Color else C := 0;
                PutPixelC(X + B, Y + A, C);
                E := E shr 1;
            end;
        end;

        X := X + FontW;
    end;
end;

{--------------------------------------------------------------------}

function IsFontPixel(F:Pointer; X, Y:Integer):Boolean; assembler;
asm
    xor ax,ax;                  { Start out FALSE }
    mov cx,X; mov bx,Y;         { Get parameters }

    les si,F;                   { AL = font byte }
    mov al,es:[si+bx];
    inc cx; rol al,cl;          { Rotate pixel into bit 0 }
    and ax,1;                   { AX = 1 if on, 0 if off }
@ip_done:
end;

{--------------------------------------------------------------------}

procedure OutBigStr(X, Y:Integer; const S:String; MagX, MagY, Skew:Integer);
var
    F:Pointer;
    I, A, B, X1, Y1, MX, MY:Integer;
label 1;
begin
    MX := MagX - 1; MY := MagY - 1;
    if (MX < 0) or (MY < 0) then goto 1;
    for I := 1 to Length(S) do begin
        F := @Font[Ord(S[I]) * FontH];
        X := X + Skew * FontH;
        for B := 0 to FontH - 1 do begin
            Y1 := Y + B * MagY; X := X - Skew;
            for A := 0 to FontW - 1 do begin
                X1 := X + A * MagX;
                if IsFontPixel(F, A, B) then
                    FRect(X1, Y1, X1 + MX, Y1 + MY);
            end;
        end;
        X := X + FontW * MagX;
    end;
1:
end;

{--------------------------------------------------------------------}

procedure GetImage(var Buf; X1, Y1, X2, Y2:Integer); assembler;
asm
    push ds;                { Save registers }
    push bp;

    les di,Buf;             { ES:DI = buffer }
    mov ax,X1; mov bx,Y1;   { Get parameters }
    mov cx,X2; mov dx,Y2;

    mov ds,CurSeg;          { DS = video memory }

    cmp ax,cx;              { Put X in order }
    jle @gi_xok;
    xchg ax,cx;
@gi_xok:
    cmp bx,dx;
    jle @gi_yok;
    xchg bx,dx;
@gi_yok:

    sub cx,ax;              { Get distances }
    mov bp,cx;              { CX = X distance }
    inc cx;                 { DX = Y distance }
    sub dx,bx;              { BP = X distance - 1 }
    inc dx;

    mov es:[di],cx;         { Store distances}
    mov es:[di+2],dx;
    add di,4;

    imul bx,bx,80;          { DI = offset }
    ror ax,2; add bl,al;    { AX = plane command }
    adc bh,0; shr ah,6;
    mov si,bx; mov al,04h;

    mov bx,dx;              { BX = Y distance }

@gi_oloop:
    mov dx,03CEh;           { Set the plane }
    out dx,ax;

    pusha;                  { Save all registers }

@gi_iloop:
    movsb;                  { Copy one pixel }
    add si,79;              { Next pixel }
    add di,bp;
    dec bx;                 { Loop back }
    jnz @gi_iloop;

    popa;                   { Restore registers }

    inc ah;                 { Next plane }
    and ah,3;
    jnz @gi_skip;           { Check for wrap }
    inc si;
@gi_skip:

    inc di;                 { Next column }
    dec cx;                 { Loop back }
    jnz @gi_oloop;

    pop bp;                 { Restore registers }
    pop ds;
end;

{--------------------------------------------------------------------}

procedure _PutImage(var Buf; X1, Y1:Integer); assembler;
asm
    push ds;                { Save registers }
    push bp;

    mov ax,X1; mov bx,Y1;   { Get parameters }
    mov es,CurSeg;          { ES = video memory }
    lds si,Buf;             { DS:SI = buffer }

    mov cx,[si];            { Get distances }
    mov dx,[si+2];          { CX = X distance }
    add si,4;               { DX = Y distance }
    mov bp,cx;              { BP = X distance - 1 }
    dec bp;

    imul bx,bx,80;          { DI = offset }
    ror ax,2; add bl,al;    { AX = plane number }
    adc bh,0; shr ax,14;
    mov di,bx;

    push cx; xchg ax,cx;    { AX = plane command }
    mov ax,1102h; rol ah,cl;
    pop cx;

    mov bx,dx;              { BX = Y distance }

@pi_oloop:
    mov dx,03C4h;           { Set the plane }
    out dx,ax;
    pusha;                  { Save all registers }

@pi_iloop:
    movsb;                  { Copy one pixel }
    add si,bp;              { Next pixel }
    add di,79;
    dec bx;                 { Loop back }
    jnz @pi_iloop;

    popa;                   { Restore registers }

    rol ah,1;               { Next plane }
    adc di,0;

    inc si;                 { Next column }
    dec cx;                 { Loop back }
    jnz @pi_oloop;

    pop bp;                 { Restore registers }
    pop ds;
end;

{--------------------------------------------------------------------}

procedure PutImage(var Buf; X1, Y1:Integer);
var
    X2, Y2:Integer;
begin
    asm                     { Get X2 and Y2 from image size }
        les bx,Buf;
        mov ax,es:[bx];   add ax,X1; mov X2,ax;
        mov ax,es:[bx+2]; add ax,Y1; mov Y2,ax;
    end;

    if (X1 < WinX1) or (X2 > WinX2) then Exit;
    if (Y1 < WinY1) or (Y2 > WinY2) then Exit;
    _PutImage(Buf, X1, Y1);
end;

{--------------------------------------------------------------------}

procedure _PutSprite(var Buf; X1, Y1:Integer); assembler;
asm
    push ds;                { Save registers }
    push bp;

    mov ax,X1; mov bx,Y1;   { Get parameters }
    mov es,CurSeg;          { ES = video memory }
    lds si,Buf;             { DS:SI = buffer }

    mov cx,[si];            { Get distances }
    mov dx,[si+2];          { CX = X distance }
    add si,4;               { DX = Y distance }
    mov bp,cx;              { BP = X distance }

    imul bx,bx,80;          { DI = offset }
    ror ax,2; add bl,al;    { AX = plane number }
    adc bh,0; shr ax,14;
    mov di,bx;

    push cx; xchg ax,cx;    { AX = plane command }
    mov ax,1102h; rol ah,cl;
    pop cx;

    mov bx,dx;              { BX = Y distance }

@ps_oloop:
    mov dx,03C4h;           { Set the plane }
    out dx,ax;
    pusha;                  { Save all registers }

@ps_iloop:
    mov dl,[si]             { Get this pixel }
    test dl,dl
    jz @ps_skip             { Write if not zero }
    mov es:[di],dl
@ps_skip:
    add si,bp;              { Next pixel }
    add di,80;
    dec bx;                 { Loop back }
    jnz @ps_iloop;

    popa;                   { Restore registers }

    rol ah,1;               { Next plane }
    adc di,0;

    inc si;                 { Next column }
    dec cx;                 { Loop back }
    jnz @ps_oloop;

    pop bp;                 { Restore registers }
    pop ds;
end;

{--------------------------------------------------------------------}

procedure PutSprite(var Buf; X1, Y1:Integer);
var
    X2, Y2:Integer;
begin
    asm                     { Get X2 and Y2 from image size }
        les bx,Buf;
        mov ax,es:[bx];   add ax,X1; mov X2,ax;
        mov ax,es:[bx+2]; add ax,Y1; mov Y2,ax;
    end;

    if (X1 < WinX1) or (X2 > WinX2) then Exit;
    if (Y1 < WinY1) or (Y2 > WinY2) then Exit;
    _PutSprite(Buf, X1, Y1);
end;

{--------------------------------------------------------------------}

const
    DefCursor : array[0..179] of Byte = (
        11,   0, 16,  0,
        $C0,$C0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
        $C0,$3F,$C0,  0,  0,  0,  0,  0,  0,  0,  0,
        $C0,$3F,$3F,$C0,  0,  0,  0,  0,  0,  0,  0,
        $C0,$3F,$3F,$3F,$C0,  0,  0,  0,  0,  0,  0,
        $C0,$3F,$3F,$3F,$3F,$C0,  0,  0,  0,  0,  0,
        $C0,$3F,$3F,$3F,$3F,$3F,$C0,  0,  0,  0,  0,
        $C0,$3F,$3F,$3F,$3F,$3F,$3F,$C0,  0,  0,  0,
        $C0,$3F,$3F,$3F,$3F,$3F,$3F,$3F,$C0,  0,  0,
        $C0,$3F,$3F,$3F,$3F,$3F,$3F,$3F,$3F,$C0,  0,
        $C0,$3F,$3F,$3F,$3F,$3F,$3F,$C0,$C0,$C0,$C0,
        $C0,$3F,$3F,$C0,$3F,$3F,$C0,  0,  0,  0,  0,
        $C0,$3F,$C0,  0,$C0,$3F,$3F,$C0,  0,  0,  0,
        $C0,$C0,  0,  0,$C0,$3F,$3F,$C0,  0,  0,  0,
        $C0,  0,  0,  0,  0,$C0,$3F,$3F,$C0,  0,  0,
          0,  0,  0,  0,  0,$C0,$3F,$3F,$C0,  0,  0,
          0,  0,  0,  0,  0,  0,$C0,$C0,  0,  0,  0
    );

var
    ScreenBuf : array[0..260] of Byte;

    MouseX, MouseY:Integer;
    MouseOn:Boolean;
    MouseCursor:Pointer;
    HotSpotX, HotSpotY:Integer;

{--------------------------------------------------------------------}

procedure PutCursor;
var
    X, Y, W, H, I, J:Integer;
    P:PChar;
    B:Byte;
begin
    P := MouseCursor; I := 4;
    W := Ord(P[0]); H := Ord(P[2]);
    for Y := 0 to H-1 do begin
        for X := 0 to W-1 do begin
            B := Ord(P[I]); J := MouseX + X; Inc(I);
            if (B > 0) and (J >= 0) and (J < 320) then
            asm
                mov es,CurSeg;  { ES = current segment }
                mov cx,J;       { CX = X, BX = Y }
                mov bx,MouseY;
                add bx,Y;
            
                mov dx,03C4h;   { Set the memory plane }
                mov ax,1102h;
                rol ah,cl;
                out dx,ax;

                mov ax,bx;      { BX = offset in video memory }
                shl bx,2;
                add bx,ax;
                shl bx,4;
                shr cx,2;
                add bx,cx;
                mov cl,B;       { Set the pixel }
                mov es:[bx],cl;
            end;
        end;
    end;
end;

{--------------------------------------------------------------------}

procedure GetScreen;
begin
    GetImage(ScreenBuf, MouseX, MouseY, MouseX + 15, MouseY + 15);
end;

{--------------------------------------------------------------------}

procedure PutScreen;
begin
    _PutImage(ScreenBuf, MouseX, MouseY);
end;

{--------------------------------------------------------------------}

procedure UpdateMouse(X, Y:Integer);
begin
    PutScreen;
    MouseX := X - HotSpotX; MouseY := Y - HotSpotY;
    GetScreen; PutCursor;
end;
    
{--------------------------------------------------------------------}

procedure MouseHandler; far; assembler;
asm
    pusha;                  { Save all registers }
    push ds; push es;
    mov ax,seg @Data;       { DS = data segment }
    mov ds,ax;

    cmp MouseOn,1;          { Quit if mouse is off }
    jne @mh_done;

    shr cx,1;               { CX = true X position }
    push cx; push dx;       { Update mouse cursor }
    call UpdateMouse;

@mh_done:
    pop es; pop ds;         { Restore registers and return }
    popa;
end;

{--------------------------------------------------------------------}

function InitMouse:Boolean; assembler;
asm
    xor ax,ax; int 33h;     { Check for mouse }
    test ax,ax; jnl @mi_nope

    mov ax,7; xor cx,cx;    { Set X/Y limits }
    mov dx,639; int 33h;
    mov ax,8; xor cx,cx;
    mov dx,239; int 33h;

    mov ax,4; xor cx,cx;    { Set position to (0, 0) }
    xor dx,dx; int 33h;

    xor ax,ax;              { Init mouse variables }
    mov MouseX,ax;
    mov MouseY,ax;
    mov HotSpotX,ax;
    mov HotSpotY,ax;
    mov MouseOn,0;          { Mouse starts hidden }

    mov dx,seg DefCursor;   { Set default cursor }
    mov ax,offset DefCursor;
    mov word ptr MouseCursor+2,dx;
    mov word ptr MouseCursor,ax;

    push cs; pop es;        { Set mouse handler }
    mov dx,offset MouseHandler;
    mov ax,0Ch; mov cx,1; int 33h;

    mov ax,1;               { Return 1 (TRUE) }
    jmp @mi_done;

@mi_nope:
    xor ax,ax;              { No mouse, return 0 (FALSE) }

@mi_done:
end;
    
{--------------------------------------------------------------------}

procedure CloseMouse;
begin
    if MouseOn then HideMouse;
    asm
        mov ax,21h;         { Reinitialize mouse }
        int 33h;
    end;
end;

{--------------------------------------------------------------------}

procedure ShowMouse;
begin
    if not MouseOn then begin
        GetScreen; PutCursor;
        MouseOn := True;
    end;
end;

{--------------------------------------------------------------------}

procedure HideMouse;
begin
    if MouseOn then begin
        MouseOn := False;
        PutScreen;
    end;
end;                     

{--------------------------------------------------------------------}

function GetMouse(var X, Y:Integer):Integer;
var
    B, M, N:Integer;
begin
    asm
        mov ax,3; int 33h;  { Get mouse info }
        mov B,bx; mov M,cx; mov N,dx;
    end;
    X := M shr 1; Y := N;
    GetMouse := B;
end;

{--------------------------------------------------------------------}

procedure SetMouse(X, Y:Integer); assembler;
asm
    mov ax,4; mov cx,X;     { Set mouse position }
    add cx,cx; mov dx,Y;
    int 33h;
end;

{--------------------------------------------------------------------}

function GetMousePress(Btn:Integer; var X, Y:Integer):Integer;
var
    B, M, N:Integer;
begin
    asm
        mov bx,Btn;         { Get mouse press info }
        mov ax,5; int 33h; mov B,bx;
        mov M,cx; mov N,dx;
    end;
    X := M shr 1; Y := N;
    GetMousePress := B;
end;

{--------------------------------------------------------------------}

function GetMouseRelease(Btn:Integer; var X, Y:Integer):Integer;
var
    B, M, N:Integer;
begin
    asm
        mov bx,Btn;         { Get mouse release info }
        mov ax,6; int 33h; mov B,bx;
        mov M,cx; mov N,dx;
    end;
    X := M shr 1; Y := N;
    GetMouseRelease := B;
end;

{--------------------------------------------------------------------}

procedure SetMouseCursor(var Cur; hX, hY:Integer);
begin
    MouseCursor := @Cur;
    HotSpotX := hX;
    HotSpotY := hY;
end;

{--------------------------------------------------------------------}

end.
