{$F+} {Compiler Directive: Generate far procedure calls: On }
{$O+} {Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Words
    version 2.0

  This unit holds a system which parses strings of words for justified
    printing.

  Purpose:
    Parsed sentences are useful for printing lines of text on paper in easy
      to read formats.

  How it works:
    The line first has multiple blank spaces removed and then the line is
      split up so that no line exceeds the given width.  Then blanks are
      added to enlarge the line to the given width if directed to do so.

  Features:
    A choice of justification styles are available.

  Limitations:
    Can only process strings up to the maximum string length.

  Copyright 1989, All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Words;

  Interface

    Uses
      CRT,
      DOS,
     {$IFNDEF OS2}
      String_Utilities;
     {$ELSE}
      String_U;
     {$ENDIF}

(***********************************************************

  PrintOut_Justify.
    This variable is used by the Word Parser PrintOut
    procedure to determine how the output will be presented
    to the PrintOut procedure.
      CENTERED - Centers the data by adding spaces to both
                 sides.
      RIGHT    - Makes the data flush against the right
                 side.
      LEFT     - Makes the data flush against the left side.
      BOTH     - Makes the data flush against both left and
                 right sides.
    The default value is both.

***********************************************************)

    Var
      PrintOut_Justify: ( Centered, Right, Left, Both );

(***********************************************************

  PrintOut.
    This variable procedure is supplied to allow the program
    to be able to intercept the output of the word parser
    PrintOut procedure.  It is used internally much like a
    writeln procedure.  The default procedure writes to the
    standard output.

***********************************************************)

     {$IFNDEF VER40}
      PrintOut: Procedure( Data: String );
     {$ENDIF}

(***********************************************************

  Procedure: Word parser print out.

    This procedure is designed to make it easier to create
    programs with word wrapping.
    Data contains a string of characters which will be
      broken down into several output lines.
    Width determines the size of the output lines to be
      returned in calls to the PrintOut function.
    Words are determined by the location of spaces or null
    characters; #0.  They are never broken unless the word
    length exceed the line width.  This procedure continues
    producing output until the entire data string is
    exhausted.

***********************************************************)

    Procedure Word_Parser_PrintOut( Data: String; Width: Byte );

(***********************************************************

  Procedure: Default print out.

    This procedure is the default PrintOut procedure used by
    the Word Parser PrintOut procedure.
    It is essentially a writeln of the data passed into it.

***********************************************************)

    Procedure Default_PrintOut( Data: String );

{----------------------------------------------------------------------------}

  Implementation

    Var
     { This variable holds the width in characters of the print out. }
      PrintOut_Width: Byte;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write out.
    This procedure prints out the given data
    string with the appropriate justification.

*************************************************)

    Procedure WriteOut( Data: String );
      Begin
        Case PrintOut_Justify of
          Left:
           {$IFNDEF VER40}
            PrintOut( Data );
           {$ELSE}
            Default_PrintOut( Data );
           {$ENDIF}
          Right:
            Begin
              While ( Length( Data ) < PrintOut_Width ) do
                Data := ' ' + Data;
             {$IFNDEF VER40}
              PrintOut( Data );
             {$ELSE}
              Default_PrintOut( Data );
             {$ENDIF}
            End;
          Centered:
            Begin
              While ( Length( Data ) < PrintOut_Width ) do
                Begin
                  Data := ' ' + Data;
                  If ( Length( Data ) < PrintOut_Width )
                    then
                      Data := Data + ' ';
                End;
             {$IFNDEF VER40}
              PrintOut( Data );
             {$ELSE}
              Default_PrintOut( Data );
             {$ENDIF}
            End;
          Both:
            Begin
              Spread_Out_String( Data, PrintOut_Width );
             {$IFNDEF VER40}
              PrintOut( Data );
             {$ELSE}
              Default_PrintOut( Data );
             {$ENDIF}
            End;
        End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Word parser print out.
    As previously defined.

*************************************************)

    Procedure Word_Parser_PrintOut( Data: String; Width: Byte );
      Var
        Count,
        The_Length: Byte;
        New_String,
        Previous_String: String;
      Begin
        PrintOut_Width := Width;
        Remove_Double_Blanks( Data );
        Count := 0;
        New_String := '';
        The_Length := Length( Data );
        While ( Count < The_Length ) do
          Begin
            Inc( Count );
            If ( Data[ Count ] = ' ' )
              then
                Begin
                  Previous_String := New_String;
                  New_String := New_String + ' ';
                End
              else
                Begin
                  New_String := New_String + Data[ Count ];
                End;
            If ( ( Length( New_String ) > Width ) and ( Previous_String <> '' ) )
              then
                Begin
                  If ( Length( Previous_String ) <= Width )
                    then
                      Begin
                        WriteOut( Previous_String );
                        Delete( New_String, 1, Length( Previous_String ) );
                        Previous_String := '';
                      End
                    else
                      Begin
                        WriteOut( Copy( Previous_String, 1, Width ) );
                        Delete( New_String, 1, Width );
                        Previous_String := '';
                      End;
                End;
            While ( ( Length( New_String ) > 0 ) and ( New_String[ 1 ] = ' ' ) ) do
              Delete( New_String, 1, 1 );
            While ( ( Length( Previous_String ) > 0 ) and ( Previous_String[ 1 ] = ' ' ) ) do
              Delete( Previous_String, 1, 1 );
          End;
        If ( Length( New_String ) <= Width )
          then
            WriteOut( New_String )
          else
            Begin
              WriteOut( Copy( New_String, 1, Width ) );
              WriteOut( Copy( New_String, Succ( Width ), ( Length( New_String ) - Width ) ) );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Default print out.
    As previously defined.

*************************************************)

    Procedure Default_PrintOut( Data: String );
      Begin
        WriteLn( Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Initialize the default variables.

*************************************************)

    Begin
     {$IFNDEF VER40}
      PrintOut := Default_PrintOut;
     {$ENDIF}
      PrintOut_Justify := Both;
    End.

