(*****************************************************************************

  WindExtras
    version 1.01 (Beta version)

  Purpose:
    This unit allows additional pointer activated features to be added to the
    current framed window.

  How it works:
    This unit intercepts the pointer codes before they reach the Keyboard
    alternative input and remaps them, depending on where about the current
    pointer location is.

  Features:
    Modifies the frame of the current window and adds new pointer buttons
    to it.
      Window close - Escape button.
      Page up, Page_Down, Home and End buttons.
      Up, Down, Left and Right buttons.
    Adds the [ Come Here ] command to the current window.  Simply click
      anywhere on the screen and the window will respond to the call.
    Allows the current window to be moved, simply by clicking on where you
      wish to move it and waiting for it to come.

  Limitations:
    Does not determine if the current window lacks a frame.
    In order to make this unit completely self contained, there are no
      direct links with any of the other units except through the KeyBoard
      unit.  This reduces the possibility of features, but greatly expands
      the flexibility of the programmer.

  Versions
    1.01 - Improved control timing constraints.

  Compilers:
    Turbo Pascal versions 5.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

{$IFNDEF OS2}
Unit WindExtra;
{$ELSE}
Unit WindExtr;
{$ENDIF}

  Interface

    Uses
      CRT,
      Core,
     {$IFNDEF OS2}
      Pointer,
     {$ELSE}
      Pointer_,
     {$ENDIF}
      KeyBoard;

    Const
     { The delays pause the command respond timing to allow better pointer control. }
      Wait_Time = 70;
      Wait_Time_Long = 200;

(***********************************************************

  Procedure: Add interface.
    This procedure is designed to be called after the window
    is defined on the screen and before the edit, menu or
    browsing routine is invokes.  What it does is sets the
    command intercept routine to operate and displays a
    simple window frame button routine so that when one of
    the units that use the KeyBoard unit is used and the
    pointer is activated over one of those buttons, the
    return code is intercepted and converted to the
    appropriate new code.

      Example...

        If Open_Window( ...
          then
            If VW_Create( ...
              then
                Begin
                  Add_Interface; { Call this routine before browse. }
                  VW_Browse;
                End;

***********************************************************)

    Procedure Add_Interface;

{----------------------------------------------------------------------------}

  Implementation

    Const
     { These values are used to define the current text attributes. }
      Black_Background = 0;
      Blue_Background = 16;
      Green_Background = 32;
      Cyan_Background = 48;
      Red_Background = 64;
      Magenta_Background = 80;
      Yellow_Background = 96;
      White_Background = 112;

      Black_Character = 0;
      Blue_Character = 1;
      Green_Character = 2;
      Cyan_Character = 3;
      Red_Character = 4;
      Magenta_Character = 5;
      Brown_Character = 6;
      Light_Gray_Character = 7;
      Dark_Gray_Character = 8;
      Light_Blue_Character = 9;
      Light_Green_Character = 10;
      Light_Cyan_Character = 11;
      Light_Red_Character = 12;
      Light_Magenta_Character = 13;
      Yellow_Character = 14;
      White_Character = 15;

      Flashing = 128;

    Var
     { Used to determine the current location of the current window. }
      Top,
      Left,
      Right,
      Bottom,
     { Used to create new text attributes out of the old one. }
      Old_Attribute,
      New_Attribute,
      Reverse_Attribute: Byte;
     { This holds the old alternative input routine for the new one. }
      Old_Interface: Function: Byte;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Near.
    This function returns true if the current two
    values are within one of each other or less.

*************************************************)

    Function Near( Value1, Value2: Byte ): Boolean;
      Begin
        Near := ( Abs( Value1 - Value2 ) < 2 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set locations.
    This procedure sets the global values to the
    current borders of the current window.

*************************************************)

    Procedure Set_Locations;
      Begin
        Top := Succ( Top_Of_Window^ );
        Left := Succ( Left_Of_Window^ );
        Right := Succ( Right_Of_Window^ );
        Bottom := Succ( Bottom_Of_Window^ );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Close location.
    This procedure determines the location of the
    close button.

*************************************************)

    Procedure Close_Location( Var Row, Column: Byte );
      Begin
        Row := Pred( Top );
        Column := Succ( Succ( Left ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Up location.
    This procedure determines the location of the
    up button.

*************************************************)

    Procedure Up_Location( Var Row, Column: Byte );
      Begin
        Row := Succ( Top );
        Column := Succ( Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Down location.
    This procedure determines the location of the
    down button.

*************************************************)

    Procedure Down_Location( Var Row, Column: Byte );
      Begin
        Row := Pred( Bottom );
        Column := Succ( Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page up location.
    This procedure determines the location of the
    page up button.

*************************************************)

    Procedure PgUp_Location( Var Row, Column: Byte );
      Begin
        Row := Top;
        Column := Succ( Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Page down location.
    This procedure determines the location of the
    page down button.

*************************************************)

    Procedure PgDown_Location( Var Row, Column: Byte );
      Begin
        Row := Bottom;
        Column := Succ( Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Left location.
    This procedure determines the location of the
    left button.

*************************************************)

    Procedure Left_Location( Var Row, Column: Byte );
      Begin
        Row := Succ( Bottom );
        Column := Succ( Succ( Left ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Right location.
    This procedure determines the location of the
    right button.

*************************************************)

    Procedure Right_Location( Var Row, Column: Byte );
      Begin
        Row := Succ( Bottom );
        Column := Pred( Pred( Right ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Home location.
    This procedure determines the location of the
    home button.

*************************************************)

    Procedure Home_Location( Var Row, Column: Byte );
      Begin
        Row := Succ( Bottom );
        Column := Succ( Left );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: End location.
    This procedure determines the location of the
    end button.

*************************************************)

    Procedure End_Location( Var Row, Column: Byte );
      Begin
        Row := Succ( Bottom );
        Column := Pred( Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw close.
    This procedure draws the close button.

*************************************************)

    Procedure Draw_Close;
      Var
        Row,
        Column: Byte;
      Begin
        Close_Location( Row, Column );
        Put_Character_On_Screen( Pred( Column ), Row, '[', Old_Attribute );
        Put_Character_On_Screen( Column, Row, #254, ( New_Attribute + 128 ) );
        Put_Character_On_Screen( Succ( Column ), Row, ']', Old_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw up and down.
    This procedure draws the up, down and long
    buttons on the right side of the window frame.

*************************************************)

    Procedure Draw_Up_And_Down;
      Var
        Row_1,
        Row_2,
        Count,
        Column_1,
        Column_2: Byte;
      Begin
        Up_Location( Row_1, Column_1 );
        Down_Location( Row_2, Column_2 );
        Put_Character_On_Screen( Column_1, Row_1, #24, Reverse_Attribute );
        Put_Character_On_Screen( Column_2, Row_2, #25, Reverse_Attribute );
        For Count := Succ( Row_1 ) to Pred( Row_2 ) do
          Put_Character_On_Screen( Column_1, Count, #176, Old_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw page up.
    This procedure draws the page up button.

*************************************************)

    Procedure Draw_PgUp;
      Var
        Row,
        Column: Byte;
      Begin
        PgUp_Location( Row, Column );
        Put_Character_On_Screen( Column, Row, #30, Reverse_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw page down.
    This procedure draws the page down button.

*************************************************)

    Procedure Draw_PgDown;
      Var
        Row,
        Column: Byte;
      Begin
        PgDown_Location( Row, Column );
        Put_Character_On_Screen( Column, Row, #31, Reverse_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure Draw left and right.
    This procedure draws the left, right and long
    buttons on the bottom of the window frame.

*************************************************)

    Procedure Draw_Left_And_Right;
      Var
        Row_1,
        Row_2,
        Count,
        Column_1,
        Column_2: Byte;
      Begin
        Left_Location( Row_1, Column_1 );
        Right_Location( Row_2, Column_2 );
        Put_Character_On_Screen( Column_1, Row_1, #27, Reverse_Attribute );
        Put_Character_On_Screen( Column_2, Row_2, #26, Reverse_Attribute );
        For Count := Succ( Column_1 ) to Pred( Column_2 ) do
          Put_Character_On_Screen( Count, Row_2, #176, Old_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw home.
    This procedure draws the home button.

*************************************************)

    Procedure Draw_Home;
      Var
        Row,
        Column: Byte;
      Begin
        Home_Location( Row, Column );
        Put_Character_On_Screen( Column, Row, #17, Reverse_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw end.
    This procedure draws the end button.

*************************************************)

    Procedure Draw_End;
      Var
        Row,
        Column: Byte;
      Begin
        End_Location( Row, Column );
        Put_Character_On_Screen( Column, Row, #16, Reverse_Attribute );
        Put_Character_On_Screen( Succ( Column ), Row, #205, Old_Attribute );
        Put_Character_On_Screen( Succ( Succ( Column ) ), Row, #188, Old_Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At close.
    This function returns true if the close button
    was clicked on.

*************************************************)

    Function At_Close: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        Close_Location( Row, Column );
        At_Close := ( Last_Row = Row ) and Near( Last_Column, Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At up.
    This function returns true if the up button
    was clicked on.

*************************************************)

    Function At_Up: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        Up_Location( Row, Column );
        At_Up := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At down
    This function returns true if the down button
    was clicked on.

*************************************************)

    Function At_Down: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        Down_Location( Row, Column );
        At_Down := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At page up.
    This function returns true if the page up
    button was clicked on.

*************************************************)

    Function At_PgUp: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        PgUp_Location( Row, Column );
        At_PgUp := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At page down.
    This function returns true if the page down
    button was clicked on.

*************************************************)

    Function At_PgDown: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        PgDown_Location( Row, Column );
        At_PgDown := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At left.
    This function returns true if the left button
    was clicked on.

*************************************************)

    Function At_Left: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        Left_Location( Row, Column );
        At_Left := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At right.
    This function returns true if the right button
    was clicked on.

*************************************************)

    Function At_Right: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        Right_Location( Row, Column );
        At_Right := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At home.
    This function returns true if the home button
    was clicked on.

*************************************************)

    Function At_Home: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        Home_Location( Row, Column );
        At_Home := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: At end.
    This function returns true if the end button
    was clicked on.

*************************************************)

    Function At_End: Boolean;
      Var
        Row,
        Column: Byte;
      Begin
        End_Location( Row, Column );
        At_End := ( Last_Row = Row ) and ( Last_Column = Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Update interface.
    This procedure is responsible for drawing the
    new pointer button interface on the screen.
    It has also been designed to get called after
    a window altering function or procedure has
    finished changing the screen so that it can
    redraw itself on the screen.

*************************************************)

   {$IFDEF VER50}
   {$F+}
    Procedure Update_Interface;
   {$ELSE}
    Procedure Update_Interface; Far;
   {$ENDIF}
      Begin
        Set_Locations;
        If Allow_Control and ( Top > 1 ) and ( Left > 1 ) and ( Bottom < Screen_Row_Limit ) and
           ( Right < Screen_Column_Limit ) and ( ( Bottom - Top ) > 2 ) and ( ( Right - Left ) > 4 )
          then
            Begin
              Old_Attribute := TextAttr;
              New_Attribute := Combine( TextAttr, Light_Gray_Character + Blue_BackGround, True, False );
              Reverse_Attribute := Reverse( New_Attribute );
              Draw_Close;
              Draw_PgUp;
              Draw_Up_And_Down;
              Draw_PgDown;
              Draw_Home;
              Draw_Left_And_Right;
              Draw_End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process up.
    This procedure is called to determine if any
    of the buttons above the window were pressed.

*************************************************)

    Procedure Process_Up( Var Data: Byte );
      Begin
        If At_Close
          then
            Data := Press_Escape;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process down.
    This procedure is called to determine if any
    of the buttons below the window were pressed.

*************************************************)

    Procedure Process_Down( Var Data: Byte );
      Begin
        If At_Home
          then
            Begin
              Data := Press_Home;
              Delay( Wait_Time );
            End
          else
            If At_End
              then
                Begin
                  Data := Press_End;
                  Delay( Wait_Time );
                End
              else
                If At_Right
                  then
                    Begin
                      Data := Press_Right_Arrow;
                      Delay( Wait_Time );
                    End
                  else
                    If At_Left
                      then
                        Begin
                          Data := Press_Left_Arrow;
                          Delay( Wait_Time );
                        End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process right.
    This procedure is called to determine if any
    of the buttons on the right side of the window
    was pressed.

*************************************************)

    Procedure Process_Right( Var Data: Byte );
      Begin
        If At_PgUp
          then
            Begin
              Data := Press_Page_Up;
              Delay( Wait_Time_Long );
            End
          else
            If At_PgDown
              then
                Begin
                  Data := Press_Page_Down;
                  Delay( Wait_Time_Long );
                End
              else
                If At_Up
                  then
                    Begin
                      Data := Press_Up_Arrow;
                      Delay( Wait_Time );
                    End
                  else
                    If At_Down
                      then
                        Begin
                          Data := Press_Down_Arrow;
                          Delay( Wait_Time );
                        End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: New interface.
    This function replaces the old KeyBoard
    alternative input routine so that the commands
    can be intercepted and converted to the new
    ones.
    First the old interface is called to get the
      pointer command.  If the pointer was clicked
      on the outside of the window, then the
      location of the pointer can be checked to
      determine if one of the new codes can be
      transmitted back to the KeyBoard main
      routine.

*************************************************)

    Function New_Interface: Byte; Far;
      Var
        Data: Byte;
      Begin
        Data := Old_Interface;
        If Allow_Control and ( Data in [ OutSide_Up, OutSide_Down, OutSide_Left, OutSide_Right,
                                         On_Frame_Top, On_Frame_Bottom, On_Frame_Right ] )
          then
            Begin
              Set_Locations;
              Case Data of
                Outside_Up: Data := Pressed_Move_Window_Up;
                Outside_Down: Data := Pressed_Move_Window_Down;
                Outside_Left: Data := Pressed_Move_Window_Left;
                Outside_Right: Data := Pressed_Move_Window_Right;
                On_Frame_Top: Process_Up( Data );
                On_Frame_Right: Process_Right( Data );
                On_Frame_Bottom: Process_Down( Data );
              End; { Case }
            End;
        New_Interface := Data;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add interface.
    As previously defined.

*************************************************)

   Procedure Add_Interface;
     Begin
       Allow_Control := True;
       Update_Interface;
     End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    This section sets up the keyboard alternative
    input routine to link to this unit.

*************************************************)

  Begin
    Old_Interface := Alternative_Input;
    Alternative_Input := New_Interface;
    UpDate_Control := Update_Interface;
  End.


