{----------------------------------------------------------------------------}

(*************************************************

  Function: Valid block operation.
    As previously defined.

*************************************************)

    Function Valid_Block_Operation( Start, Finish, Position: Point_Type ): Boolean;
      Begin
        Valid_Block_Operation := ( ( Start.Row < Finish.Row ) or ( ( Start.Row = Finish.Row ) and
                                                                   ( Start.Column < Finish.Column ) ) ) and
                                 ( ( Finish.Row < Position.Row ) or ( Start.Row > Position.Row ) ) or
                                 ( ( Finish.Row = Position.Row ) and ( Finish.Column <= Position.Column ) ) or
                                 ( ( Start.Row = Position.Row ) and ( Start.Column >= Position.Column ) )
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Copy structure.
    This function returns true only if it can
    generate a copy of the block marked off in
    Original by Start and Finish and put it in
    Copy.  Copy is expected to be uninitialized,
    or memory loss could occur.

*************************************************)

    Function Copy_Structure( Var Original, Copy: Text_Type; Start, Finish: Point_Type; Point, Limit: LongInt ): Boolean;
      Var
        Okay: Boolean;
        Count: LongInt;
      Begin
        Okay := Allocate_Text( Copy );
        If Okay
          then
            Begin
              Peek_Text_Line( Original, Start.Row, Work_Space^ );
              Delete_Line( Work_Space^, 1, Pred( Start.Column ) );
              For Count := Succ( Start.Row ) to Finish.Row do
                If Okay
                  then
                    Begin
                      Okay := Put_Text_Line( Copy, Point, Work_Space^, True );
                      If Okay
                        then
                          Begin
                            If Show_Status
                              then
                                Begin
                                  If Odd( Point )
                                    then
                                      Process_Status( Point, Limit, Okay );
                                  Inc( Point );
                                End;
                            Peek_Text_Line( Original, Count, Work_Space^ );
                          End;
                    End;
              If ( Work_Space^.Size > Finish.Column )
                then
                  Work_Space^.Size := Pred( Finish.Column );
              If Okay
                then
                  Okay := Put_Text_Line( Copy, Point, Work_Space^, True );
            End;
        Copy_Structure := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Clip structure.
    This function returns true only if it can
    remove the given block defined in Original by
    Start and Finish and put it in Clip.  Clip is
    expected to be uninitialized, or memory loss
    could occur.

*************************************************)

    Function Clip_Structure( Var Original, Clip: Text_Type; Var Buffer: Line_Type; Start, Finish: Point_Type; Point,
                             Limit: LongInt ): Boolean;
      Var
        Okay: Boolean;
        Count,
        Pointer: LongInt;
      Begin
        Okay := Allocate_Text( Clip );
        If Okay
          then
            Begin
              Get_Text_Line( Original, Finish.Row, Buffer );
              Append_Line( Buffer, ' ' );
              Okay := Split_Text( Original, Buffer, Finish, True );
              If Okay
                then
                  Begin
                    Get_Text_Line( Original, Start.Row, Buffer );
                    Append_Line( Buffer, ' ' );
                    Okay := Split_Text( Original, Buffer, Start, True );
                    If Okay
                      then
                        Begin
                          Pointer := Succ( Finish.Row - Start.Row );
                          For Count := Succ( Finish.Row ) downto Succ( Start.Row ) do
                            If Okay
                              then
                                Begin
                                  Peek_Text_Line( Original, Count, Buffer );
                                  Okay := Delete_Text_Line( Original, Count );
                                  If Okay
                                    then
                                      Begin
                                        Okay := Put_Text_Line( Clip, Pointer, Buffer, True );
                                        If Show_Status
                                          then
                                            Begin
                                              If Odd( Point )
                                                then
                                                  Process_Status( Point, Limit, Okay );
                                              Inc( Point );
                                            End;
                                        Dec( Pointer );
                                      End;
                                End;
                          If Okay
                            then
                              Begin
                                Get_Text_Line( Original, Start.Row, Buffer );
                                Append_Line( Buffer, ' ' );
                                Okay := Remove_Return( Original, Buffer, Start );
                              End;
                        End;
                  End;
            End;
        Clip_Structure := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Copy the block.
    As previously defined.

*************************************************)

    Function Copy_Block( Var Text, ClipBoard: Text_Type; Start, Finish: Point_Type ): Boolean;
      Begin
        Erase_Text( ClipBoard );
        Copy_Block := Copy_Structure( Text, ClipBoard, Start, Finish, 1, ( Finish.Row - Start.Row ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Cut the block.
    As previously defined.

*************************************************)

    Function Cut_Block( Var Text, ClipBoard: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                        Cursor: Point_Type ): Boolean;
      Begin
        Erase_Text( ClipBoard );
        Cut_Block := Clip_Structure( Text, ClipBoard, Buffer, Start, Finish, 0, ( Finish.Row - Start.Row ) );
        Finish := Start;
        Cursor := Start;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Paste the block.
    As previously defined.

*************************************************)

    Function Paste_Block( Var Text, ClipBoard: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type; Point,
                          Limit: LongInt ): Boolean;
      Var
        Okay: Boolean;
        Count: LongInt;
      Begin
        Okay := True;
        Start := Cursor;
        Finish := Cursor;
        Get_Text_Line( Text, Finish.Row, Buffer );
        Append_Line( Buffer, ' ' );
        For Count := 1 to ClipBoard.FileSize do
          If Okay
            then
              Begin
                Okay := Split_Text( Text, Buffer, Finish, True );
                If Okay
                  then
                    Begin
                      Peek_Text_Line( ClipBoard, Count, Work_Space^ );
                      Combine_Together( Buffer, Work_Space^, 1, Line_Limit );
                      Okay := Put_Text_Line( Text, Finish.Row, Buffer, True );
                      If Okay
                        then
                          Begin
                            Inc( Finish.Row );
                            Finish.Column := 1;
                            Get_Text_Line( Text, Finish.Row, Buffer );
                            Append_Line( Buffer, ' ' );
                          End;
                      If Show_Status
                        then
                          Begin
                            If Odd( Point )
                              then
                                Process_Status( Point, Limit, Okay );
                            Inc( Point );
                          End;
                    End
              End;
        If Okay
          then
            Begin
              Dec( Finish.Row );
              Get_Text_Line( Text, Finish.Row, Buffer );
              Append_Line( Buffer, ' ' );
              Finish.Column := Buffer.Size;
              Okay := Remove_Return( Text, Buffer, Finish );
            End;
        Paste_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: InPaste the block.

    This function attempts to paste the block in
    ClipBoard into the Text at the cursor point.
    It works like Paste the block, but disposes of
    the structure as it works.
    Buffer should not hold any necessary data.
    Start and Finish are automatically adjusted to
    the new locations of the block.  This function
    returns true only if it's successful.  In the
    event this function fails, part of the block
    may still have been inserted.

*************************************************)

    Function InPaste_Block( Var Text, ClipBoard: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type;
                            Point, Limit: LongInt ): Boolean;
      Var
        Okay: Boolean;
        Count: LongInt;
      Begin
        Okay := True;
        Start := Cursor;
        Finish := Cursor;
        Get_Text_Line( Text, Finish.Row, Buffer );
        Append_Line( Buffer, ' ' );
        For Count := 1 to ClipBoard.FileSize do
          If Okay
            then
              Begin
                Okay := Split_Text( Text, Buffer, Finish, True );
                If Okay
                  then
                    Begin
                      Peek_Text_Line( ClipBoard, Count, Work_Space^ );
                      Combine_Together( Buffer, Work_Space^, 1, Line_Limit );
                      Okay := Put_Text_Line( Text, Finish.Row, Buffer, True );
                      If Okay
                        then
                          Begin
                            Inc( Finish.Row );
                            Finish.Column := 1;
                            Get_Text_Line( Text, Finish.Row, Buffer );
                            Append_Line( Buffer, ' ' );
                            Work_Space^.Size := 0;
                            Okay := Put_Text_Line( ClipBoard, Count, Work_Space^, True );
                          End;
                      If Show_Status
                        then
                          Begin
                            If Odd( Point )
                              then
                                Process_Status( Point, Limit, Okay );
                            Inc( Point );
                          End;
                    End
              End;
        If Okay
          then
            Begin
              Dec( Finish.Row );
              Get_Text_Line( Text, Finish.Row, Buffer );
              Append_Line( Buffer, ' ' );
              Finish.Column := Buffer.Size;
              Okay := Remove_Return( Text, Buffer, Finish );
            End;
        InPaste_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Copy single line block.
    This function copies the block defined by
    Start and Finish, only if the block is
    entirely on one line.  Cursor specifies the
    location to copy it at.

*************************************************)

    Function Copy_Single_Line_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                                     Cursor: Point_Type ): Boolean;
      Var
        Hold: Point_Type;
        Count: Word;
        Character: Char;
      Begin
        { Get a copy of the block in Work_Space. }
        Peek_Text_Line( Text, Start.Row, Work_Space^ );
        Append_Line( Work_Space^, ' ' );
        If ( Work_Space^.Size > Finish.Column )
          then
            Work_Space^.Size := Pred( Finish.Column );
        Delete_Line( Work_Space^, 1, Pred( Start.Column ) );
        { Adjust hold for moving the block after the copy. }
        Hold.Column := ( Finish.Column - Start.Column );
        { Extend the cursor line and copy the block into it. }
        Append_Line( Buffer, ' ' );
        For Count := Work_Space^.Size downto 1 do
          Begin
            Character := Work_Space^.Data[ Count ];
            Insert_Line( Buffer, Cursor.Column, Character )
          End;
        Copy_Single_Line_Block := Put_Text_Line( Text, Cursor.Row, Buffer, True );
        { Move the block markers to the new location. }
        Start := Cursor;
        Finish.Row := Cursor.Row;
        Finish.Column := Cursor.Column + Hold.Column;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move single line block on same line.
    This Function attempts to move the single line
    block defined by Start and Finish only if it's
    on that same line, to a location specified by
    Cursor.

*************************************************)

    Function Move_Single_Line_Block_Same_Line( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                                               Cursor: Point_Type ): Boolean;
      Var
        Count,
        Adjustment: Word;
        Character: Char;
      Begin
        { Get a copy of the block in Work_Space. }
        Peek_Text_Line( Text, Start.Row, Work_Space^ );
        Append_Line( Work_Space^, ' ' );
        If ( Work_Space^.Size > Finish.Column )
          then
            Work_Space^.Size := Pred( Finish.Column );
        Delete_Line( Work_Space^, 1, Pred( Start.Column ) );
        { Adjust hold for moving the block after the move. }
        Adjustment := ( Finish.Column - Start.Column );
        If ( Cursor.Column > Finish.Column )
          then
            Cursor.Column := Cursor.Column - Adjustment;
        Append_Line( Buffer, ' ' );
        { Delete the block from the buffer. }
        Delete_Line( Buffer, Start.Column, ( Finish.Column - Start.Column ) );
        { Insert the copy into the buffer. }
        For Count := Work_Space^.Size downto 1 do
          Begin
            Character := Work_Space^.Data[ Count ];
            Insert_Line( Buffer, Cursor.Column, Character )
          End;
        { Store the result }
        Move_Single_Line_Block_Same_Line := Put_Text_Line( Text, Cursor.Row, Buffer, True );
        Start := Cursor;
        Finish.Row := Cursor.Row;
        Finish.Column := Cursor.Column + Adjustment;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Delete multiple line block.
    This function attempts to delete a block that
    is of multiple line length defined by
    Start and Finish.  Cursor is automatically
    updated.

*************************************************)

    Function Delete_Multiple_Line_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type;
                                         Point, Limit: LongInt ): Boolean;
      Var
        ClipBoard: Text_Type;
      Begin
        Initialize_Text( ClipBoard );
        Delete_Multiple_Line_Block := Clip_Structure( Text, ClipBoard, Buffer, Start, Finish, Point, Limit );
        Finish := Start;
        Cursor := Start;
        Dispose_Text( ClipBoard );
        Erase_Text( ClipBoard );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Delete the block.

    This function attempts to delete the block in
    Text that is marked off by Start and Finish.
    Cursor is automatically adjusted.  This
    function returns true only it it's successful.
    In the event this function fails, part of the
    block may still have been deleted.

*************************************************)

    Function Delete_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type ): Boolean;
      Begin
        If Valid_Block_Operation( Start, Finish, Start )
          then
            Delete_Block := Delete_Multiple_Line_Block( Text, Buffer, Start, Finish, Cursor, 0, ( Finish.Row - Start.Row ) )
          else
            Delete_Block := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move single line block.
    This function attempts to move the single line
    block defined by Start and Finish to another
    location specified by Cursor.

*************************************************)

    Function Move_Single_Line_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                                     Cursor: Point_Type ): Boolean;
      Var
        Okay: Boolean;
        Hold_Start,
        Hold_Finish,
        Hold_Cursor: Point_Type;
      Begin
        Okay := True;
        If ( Cursor.Row <> Start.Row )
          then
            Begin
              Hold_Start := Start;
              Hold_Finish := Finish;
              Okay := Copy_Single_Line_Block( Text, Buffer, Start, Finish, Cursor ) and
                      Delete_Block( Text, Buffer, Hold_Start, Hold_Finish, Hold_Cursor );
            End
          else
            Okay := Move_Single_Line_Block_Same_Line( Text, Buffer, Start, Finish, Cursor );
        Move_Single_Line_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Write the block.

    This function writes the block marked off in
    Text by Start and Finish into the given text
    file.  It returns false if it fails.

*************************************************)

    Function Write_Block( Var Text: Text_Type; Start, Finish: Point_Type; Var OutFile: Text ): Boolean;
      Var
        Okay: Boolean;
        Counter,
        Limit: LongInt;
      Begin
        Okay := True;
        Limit := ( Finish.Row - Start.Row );
        For Counter := Start.Row to Finish.Row do
          If Okay
            then
              Begin
                Peek_Text_Line( Text, Counter, Work_Space^ );
                If ( Counter = Finish.Row )
                  then
                    If ( Work_Space^.Size > Finish.Column )
                      then
                        Work_Space^.Size := Pred( Finish.Column );
                If ( Counter = Start.Row )
                  then
                    If ( Start.Column > 1 )
                      then
                        Delete_Line( Work_Space^, 1, Pred( Start.Column ) );
                Okay := Write_Line( OutFile, Work_Space^ );
                Write_Status( Counter, Limit );
              End;
        Write_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Copy multiple line block.
    This function attempts to copy a block of
    multiple line size specified by Start and
    Finish to the point specified by Cursor.

*************************************************)

    Function Copy_Multiple_Line_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                                       Cursor: Point_Type; Limit: LongInt ): Boolean;
      Var
        ClipBoard: Text_Type;
      Begin
        Initialize_Text( ClipBoard );
        Copy_Multiple_Line_Block := Copy_Structure( Text, ClipBoard, Start, Finish, 1, Limit ) and
                                    InPaste_Block( Text, ClipBoard, Buffer, Start, Finish, Cursor, ClipBoard.FileSize, Limit );
        Dispose_Text( ClipBoard );
        Erase_Text( ClipBoard );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Read the block.

    This function attempts to read data from the
    file into Text at the given cursor location.
    The read continues until the end of the file
    is reached, or there is no more room for the
    data.  Buffer should contain no necessary data
    and Start and Finish will be automatically
    adjusted.

*************************************************)

    Function Read_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type;
                         Var InFile: Text ): Boolean;
      Var
        Okay: Boolean;
        ClipBoard: Text_Type;
      Begin
        Initialize_Text( ClipBoard );
        Okay := Read_Text( InFile, ClipBoard );
        If Okay
          then
            Begin
              Okay := InPaste_Block( Text, ClipBoard, Buffer, Start, Finish, Cursor, 1, ClipBoard.FileSize );
              Dispose_Text( ClipBoard );
              Erase_Text( ClipBoard );
            End;
        Read_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move multiple line block.
    This function attempts to move the multiple
    line block, defined by Start and Finish to the
    location specified by Cursor.

*************************************************)

    Function Move_Multiple_Line_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                                       Cursor: Point_Type ): Boolean;
      Var
        Okay: Boolean;
        Hold_Start,
        Hold_Finish,
        Hold_Cursor: Point_Type;
        Limit,
        Adjustment: LongInt;

     { This procedure sets up for moving block. }
      Procedure Initialize;
        Begin
          Adjustment := ( Finish.Row - Start.Row );
          Limit := ( 3 * Adjustment );
          Hold_Start := Start;
          Hold_Finish := Finish;
          Hold_Cursor := Cursor;
        End;

      Begin
        If ( Cursor.Row < Start.Row )
          then
            Begin
              Initialize;
              Hold_Start.Row := Hold_Start.Row + Adjustment;
              Hold_Finish.Row := Hold_Finish.Row + Adjustment;
              Okay := Copy_Multiple_Line_Block( Text, Buffer, Start, Finish, Cursor, Limit ) and
                      Delete_Multiple_Line_Block( Text, Buffer, Hold_Start, Hold_Finish, Hold_Cursor, ( 2 * Adjustment ),
                                                  Limit );
            End
          else
            If ( Cursor.Row > Finish.Row )
              then
                Begin
                  Initialize;
                  Okay := Copy_Multiple_Line_Block( Text, Buffer, Start, Finish, Cursor, Limit ) and
                          Delete_Multiple_Line_Block( Text, Buffer, Hold_Start, Hold_Finish, Hold_Cursor, ( 2 * Adjustment ),
                                                      Limit );
                  If Okay
                    then
                      Begin
                        Start.Row := Start.Row - Adjustment;
                        Finish.Row := Finish.Row - Adjustment;
                        Cursor := Start;
                      End;
                End
              else
                Okay := False;
        Move_Multiple_Line_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: ICopy the block.

    This function attempts to copy the block in
    Text that is marked off by Start and Finish
    in another part of Text, determined by the
    Cursor.  Buffer should not hold any necessary
    data, and Start and Finish will automatically
    be adjusted.  This function returns true only
    if it's successful.  In the event this
    function fails, part of the block may still
    have been copied.

*************************************************)

    Function ICopy_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type ): Boolean;
      Var
        Limit: LongInt;
      Begin
        If ( Start.Row = Finish.Row )
          then
            ICopy_Block := Copy_Single_Line_Block( Text, Buffer, Start, Finish, Cursor )
          else
            If Valid_Block_Operation( Start, Finish, Cursor )
              then
                Begin
                  Limit := ( 2 * Succ( Finish.Row - Start.Row ) );
                  ICopy_Block := Copy_Multiple_Line_Block( Text, Buffer, Start, Finish, Cursor, Limit );
                End
              else
                ICopy_Block := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Move the block.

    This function attempts to move the given block
    in Text marked off by Start and Finish to the
    location referred to by Cursor.  Start and
    Finish are automatically adjusted.  Buffer
    should not hold any necessary data.  This
    function returns true only if it's successful.
    This function attempts to copy the block to
    the new location and then deletes the old part
    to prevent data loss.

*************************************************)

    Function Move_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type ): Boolean;
      Begin
        If ( Start.Row = Finish.Row )
          then
            Move_Block := Move_Single_Line_Block( Text, Buffer, Start, Finish, Cursor )
          else
            If Valid_Block_Operation( Start, Finish, Cursor )
              then
                Move_Block := Move_Multiple_Line_Block( Text, Buffer, Start, Finish, Cursor )
              else
                Move_Block := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Find the beginning.
    This procedure scans the given text line and
    tries to find the given character in that
    line.  If it is found, then Point will index
    it, otherwise, Point contains zero.

*************************************************)

    Procedure Find_Beginning( Var Line: Line_Type; Var Point: Word; Data: Char; Start: Word );
      Var
        Index: Word;
      Begin
        Point := 0;
        Index := Start;
        If ( Index < 1 )
          then
            Inc( Index );
        While ( Index <= Line.Size ) and ( Point = 0 ) do
          Begin
            If ( Line.Data[ Index ] = Data )
              then
                Point := Index;
            Inc( Index );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Find the beginning, but ignore case.
    This procedure scans the given text line and
    tries to find the given character in that
    line.  If it is found, then Point will index
    it, otherwise, Point contains zero.

*************************************************)

    Procedure Find_Beginning_Ignore_Case( Var Line: Line_Type; Var Pointer: Word; Data: Char; Start: Word );
      Var
        Index: Word;
      Begin
        Pointer := 0;
        Index := Start;
        If ( Index < 1 )
          then
            Inc( Index );
        While ( Index <= Line.Size ) and ( Pointer = 0 ) do
          Begin
            If ( UpCase( Line.Data[ Index ] ) = Data )
              then
                Pointer := Index;
            Inc( Index );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Check for match.
    This function checks the given part of Line,
    indexed by Start to the data string and
    returns true if they are found to match.

*************************************************)

    Function Check_Match( Var Line: Line_Type; Start: Word; Var Data: String ): Boolean;
      Var
        Okay: Boolean;
        Pointer: Byte;
      Begin
        If Search_Only_Whole_Words
          then
            Okay := ( ( ( Start = 1 ) or ( Line.Data[ Pred( Start ) ] in Word_Separators ) ) and
                      ( Line.Data[ Start + Length( Data ) ] in Word_Separators ) )
          else
            Okay := True;
        If Okay
          then
            Begin
              Pointer := 1;
              If Search_Case_Sensitive
                then
                  Repeat
                    If ( Data[ Pointer ] <> WildCard )
                      then
                        Okay := ( Data[ Pointer ] = Line.Data[ Start ] );
                    Inc( Start );
                    Inc( Pointer );
                  Until ( not Okay ) or ( Start > Line.Size ) or ( Pointer > Length( Data ) )
                else
                  Repeat
                    If ( Data[ Pointer ] <> WildCard )
                      then
                        Okay := ( UpCase( Data[ Pointer ] ) = UpCase( Line.Data[ Start ] ) );
                    Inc( Start );
                    Inc( Pointer );
                  Until ( not Okay ) or ( Start > Line.Size ) or ( Pointer > Length( Data ) );
              Okay := Okay and ( Pointer > Length( Data ) );
            End;
        Check_Match := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Scan the line.
    This function scans the given line of text for
    the given string and returns the location
    in Result.  If it doesn't find it, it returns
    false.  It it reaches the end of the line,
    then Finished_Row will be set to true.

*************************************************)

    Function Scan_Line( Var Line: Line_Type; Var Result: Point_Type; Var Data: String; Var Finished_Row: Boolean ): Boolean;
      Var
        Okay: Boolean;
        Pointer: Word;
      Begin
        Okay := False;
        If ( Length( Data ) > 0 )
          then
            Begin
              If Search_Case_Sensitive
                then
                  Find_Beginning( Line, Pointer, Data[ 1 ], Succ( Result.Column ) )
                else
                  Find_Beginning_Ignore_Case( Line, Pointer, UpCase( Data[ 1 ] ), Succ( Result.Column ) );
              If ( Pointer <> 0 )
                then
                  Begin
                    Finished_Row := ( Pointer = Line.Size );
                    Okay := Check_Match( Line, Pointer, Data );
                    Result.Column := Pointer;
                  End
                else
                  Finished_Row := True;
            End;
        Scan_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Scan row.
    This function gets the given row out of Text
    and Scans it for the given data string.

*************************************************)

    Function Scan_Row( Var Text: Text_Type; Row: LongInt; Var Result: Point_Type; Var Search_Data: String;
                       Var Finished_Row: Boolean ): Boolean;
      Begin
        Get_Text_Line( Text, Row, Work_Space^ );
        If ( Result.Row <> Row )
          then
            Begin
              Result.Row := Row;
              Result.Column := 0;
            End;
        Scan_Row := Scan_Line( Work_Space^, Result, Search_Data, Finished_Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function:  Display the status.
    This function displays the status of the
    current operation on the screen and also looks
    for the user to press escape.

*************************************************)

    Function Display_Status( Current_Row: LongInt ): Boolean;
      Var
        Command: Byte;
        Value: Char;
      Begin
        If Show_Status
          then
            Display_Where_Status( Current_Row );
        If Data_Ready
          then
            Begin
              Get_Command( Value, Command );
              Display_Status := ( Command = Press_Escape );
            End
          else
            Display_Status := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Scan text.
    This function looks in the text file for the
    specified search string starting at the given
    value in result.  If the search string is
    found, the function returns true and the
    location is returned in Result.

*************************************************)

    Function Scan_Text( Var Text: Text_Type; Var Result: Point_Type; Var Search_Data: String ): Boolean;
      Var
        Okay,
        Exit,
        Finished_Row: Boolean;
        Current_Row: LongInt;
      Begin
        Exit := False;
        Current_Row := Result.Row;
        Repeat
          Okay := Scan_Row( Text, Current_Row, Result, Search_Data, Finished_Row );
          If Finished_Row
            then
              Begin
                Inc( Current_Row );
                If Odd( Current_Row )
                  then
                    Exit := Display_Status( Current_Row );
              End;
        Until Okay or ( Current_Row > Text.FileSize ) or Exit;
        Scan_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Look through text.
    This function works similar to scan text, with
    the exception that it only scans the given row
    of the text.  With subsequent calls to this
    function, the entire file can be searched, but
    care should be taken to check that the row
    isn't looking past the end of the text.

*************************************************)

    Function Look_Text( Var Text: Text_Type; Var Result: Point_Type; Var Search_Data: String; Var Row: LongInt ): Boolean;
      Var
        Finished_Row: Boolean;
      Begin
        Look_Text := Scan_Row( Text, Row, Result, Search_Data, Finished_Row );
        If Finished_Row
          then
            Inc( Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display.
    This procedure is the main routine for trans-
    furing the editing data to the screen.  Row
    points to the screen row where Data is
    intended to appear.  Start and Finish position
    where the HighLighting takes place.

*************************************************)

    Procedure Display( Row: Byte; Data: String; Start, Finish: LongInt; HighLight: Byte );
      Var
        Size: LongInt;
      Begin
        GotoXY( 1, Row );
        If ( ( Start <> Finish ) and ( Finish > Start ) and Display_Block )
          then
            Begin
              If ( Start > Length( Data ) )
                then
                  Start := Length( Data )
                else
                  If ( Start < 1 )
                    then
                      Start := 0;
              If ( Finish > Length( Data ) )
                then
                  Finish := Length( Data )
                else
                  If ( Finish < 0 )
                    then
                      Finish := 0;
              If ( Start > 0 )
                then
                  Begin
                    TextAttr := Normal;
                    Write( Screen, Copy( Data, 1, Start ) );
                  End;
              Size := ( Finish - Start );
              If ( Size > 0 )
                then
                  Begin
                    TextAttr := HighLight;
                    Write( Screen, Copy( Data, Succ( Start ), Size ) );
                  End;
              TextAttr := Normal;
              Size := ( Length( Data ) - Finish );
              If ( Size > 0 )
                then
                  Write( Screen, Copy( Data, Succ( Finish ), Size ) );
            End
          else
            Begin
              TextAttr := Normal;
              Write( Screen, Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display truncated.
    This procedure takes the given data line,
    performs the calculations and puts what is
    necessary on the screen.

*************************************************)

    Procedure Display_Truncated( New_Row: Byte; Var Buffer: Line_Type; Start, Finish: LongInt );
      Var
        Row,
        Column: Byte;
      Begin
        Row := WhereY;
        Column := WhereX;
        Start := ( Start - Wave.Start );
        Finish := ( Finish - Wave.Start );
        Display( New_Row, Copy_Line( Buffer, Wave.Start, Succ( Wave.Finish - Wave.Start ) ), Start, Finish, HighLight );
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write truncated static.
    This procedure is designed to take the given
    line row and adjust it so that the cursor will
    appear in the right location on the screen.
    If the cursor is normally off the screen, the
    values are automatically adjusted to
    compensate for it.

*************************************************)

    Procedure Write_Truncated_Static( Row: Byte; Var Buffer: Line_Type; Var Data: Cluster_Type; Start, Finish: LongInt );
      Var
        Column: Byte;
        Temporary_Finish: LongInt;
      Begin
        Temporary_Finish := Data.Finish;
        If ( Data.Where.Column <= Buffer.Size )
          then
            While ( Data.Where.Column > Data.Finish ) or ( Finish > Temporary_Finish ) do
              Begin
                Inc( Data.Start );
                Inc( Data.Finish );
                Dec( Start );
                Dec( Finish );
              End;
        Dec( Start );
        Dec( Finish );
        Display( Row, Copy_Line( Buffer, Data.Start, Succ( Data.Finish - Data.Start ) ), Start, Finish, Search_HighLight );
        Column := Succ( Data.Where.Column - Data.Start );
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw screen.
    As previously defined.

*************************************************)

    Procedure Draw_Screen( Var Text: Text_Type; Cursor: Point_Type; Row, Adjustment, Max_Window_Row: Byte );
      Var
        Count,
        Save_Column: Byte;
        Working_Row: LongInt;
        Data: Cluster_Type;
      Begin
        Data.Start := 1;
        Data.Finish := ( Right_Of_Window^ - Left_Of_Window^ );
        Data.Where := Cursor;
        Working_Row := Succ( Data.Where.Row - Row );
        Get_Text_Line( Text, Data.Where.Row, Working_Buffer^ );
        Data.String_Length := Working_Buffer^.Size;
        If ( Data.Where.Column > Data.String_Length )
          then
            Data.Where.Column := Data.String_Length;
        Write_Truncated_Static( Row, Working_Buffer^, Data, Data.Where.Column, Data.Where.Column + Adjustment );
        Save_Column := WhereX;
        For Count := 1 to Max_Window_Row do
          Begin
            If ( Count <> Row )
              then
                Begin
                  Get_Text_Line( Text, Working_Row, Working_Buffer^ );
                  Data.String_Length := Working_Buffer^.Size;
                  Write_Truncated_Static( Count, Working_Buffer^, Data, 0, 0 );
                End;
            Inc( Working_Row );
          End;
        GotoXY( Save_Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write truncated.
    This procedure is designed to take the given
    line row and adjust it so that the cursor will
    appear in the right location on the screen.
    If the cursor is normally off the screen, the
    values are automatically adjusted to
    compensate for it.  When the values have to
    be automatically adjusted, the wave cluster
    must be informed to allow the ripple effect to
    start.

*************************************************)

    Procedure Write_Truncated( Row: Byte; Var Buffer: Line_Type; Var Data: Cluster_Type; Start, Finish: LongInt );
     { This procedure sets up the system for the rippling effect. }
      Procedure Set_Up;
        Begin
          Wave.Screen_Up := Pred( Row );
          Wave.Screen_Down := Succ( Row );
          Wave.Ripple_Up := Pred( Wave.Where );
          Wave.Ripple_Down := Succ( Wave.Where );
          Rippling := True;
        End;
      Begin
        If ( Data.Where.Column <= Buffer.Size )
          then
            While ( Data.Where.Column > Data.Finish ) do
              Begin
                Inc( Data.Start );
                Inc( Data.Finish );
                Set_Up;
              End;
        While ( Data.Where.Column < Data.Start ) do
          Begin
            Dec( Data.Start );
            Dec( Data.Finish );
            Set_Up;
          End;
        Start := ( Start - Data.Start );
        Finish := ( Finish - Data.Start );
        Display( Row, Copy_Line( Buffer, Data.Start, Succ( Data.Finish - Data.Start ) ), Start, Finish, HighLight );
        Cursor_Column_Start := Succ( Data.Where.Column - Data.Start );
        GotoXY( Cursor_Column_Start, Row );
        Cursor_Column_Finish := Cursor_Column_Start;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Alter block.
    This function returns true if Cursor point is
    on the same line as the Block point and if the
    Cursor column is less than the Block column.

*************************************************)

    Function Alter_Block( Var Block, Cursor: Point_Type ): Boolean;
      Begin
        Alter_Block := ( Block.Row = Cursor.Row ) and ( Block.Column > Cursor.Column );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Increment block.
    This procedure adjusts the Start and Finish
    values depending on where the cursor is on
    the screen.

*************************************************)

    Procedure Increment_Block( Cursor: Point_Type; Var Start, Finish: LongInt );
      Begin
        If Alter_Block( Wave.Block_Start, Cursor )
          then
            Begin
              Inc( Wave.Block_Start.Column );
              Inc( Start );
            End;
        If Alter_Block( Wave.Block_Finish, Cursor )
          then
            Begin
              Inc( Wave.Block_Finish.Column );
              Inc( Finish );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

   Procedure: Decrement block.
     This procedure adjusts the Start and Finish
     values depending on where the Cursor is on
     the screen.

*************************************************)

    Procedure Decrement_Block( Cursor: Point_Type; Var Start, Finish: LongInt );
      Begin
        If Alter_Block( Wave.Block_Start, Cursor )
          then
            Begin
              Dec( Wave.Block_Start.Column );
              Dec( Start );
            End;
        If Alter_Block( Wave.Block_Finish, Cursor )
          then
            Begin
              Dec( Wave.Block_Finish.Column );
              Dec( Finish );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Truncate block.
    This procedure adjusts the values of the wave
    columns depending on where the cursor is on
    the screen.

*************************************************)

    Procedure Truncate_Block( Cursor: Point_Type; Var Start, Finish: LongInt );
      Begin
        If Alter_Block( Wave.Block_Start, Cursor )
          then
            Begin
              Wave.Block_Start.Column := Cursor.Column;
              Start := Cursor.Column;
            End;
        If Alter_Block( Wave.Block_Finish, Cursor )
          then
            Begin
              Wave.Block_Start.Column := Cursor.Column;
              Finish := Cursor.Column;
            End;
      End;

