{$F+} { Compiler Directive: Generate far procedure calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  TextCorrect
    Version 1.34 (Beta version)

  Purpose:
    Generation of a complete word correction system to compliment the
    TextEdit text editor engine.

  Features:
    Using Soundex code system for word key look up allows phonetic word
      matchings.
    Multiple dictionaries allow quick word lookup.
    Automatically capitalizes replaced words.
    Uses the B_Tree data management system unit to store the dictionary for
      extremely quick word access.
    Ability to recognize and flag words that should always be capitalized.
    Ability to capitalize automatically or on demand.
    Idle time background checking increases processing speed.
    Easily allows adding of words to main dictionaries.
    AutoCorrect facility allows permanent addictions.
    Word checking of user entered corrections.
    Displays current status while checking.
    Automatically counts the word while checking them.
    Automatically links into TextEdit;
      Activation keystrokes are
        ^QL - to start spell checking at current cursor location.
        ^KL - to start spell checking at beginning of block.
        ^QW - to spell check the current text line.
    Uses Search's Super_Find to look across the system for it's dictionaries.

  Possibilities:
    Ability to find and flag repeated words.
    Dictionary Support for grammar checking.

  Limitations:
    Does not generate compressed dictionary structures.
      Compressed dictionary structures are common among professional systems
        because they save disk storage space.
    Does not separate words for user edited word search.

  Versions:
    1.1 - Corrected words skipped on current line in background.
    1.2 - Corrected alignment of highlight during flagging.
    1.3 - Corrected miss counting of words during background search.
    1.31 - Corrected dropped word count during background correction.
    1.32 - Repaired line count status error.
    1.33 - Included support for dictionary searches across system.
    1.34 - Included support for unique temporary file name.

  CopyRight 1994, 1995 All rights reserved.
    By Paul Renaud.

  Compiler:
    Turbo Pascal versions 5.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit TextCorrect;

  Interface

    Uses
      CRT,
      Core,
      BTree,
      Search,
      KeyBoard,
      TextLine,
      TextEdit,
      TextLink,
      String_Utilities,
      Generate_FileName;

(***********************************************************

  This alters the operation of the system to automatically
    perform any capitalization it would normally flag.

***********************************************************)

    Const
      AutoCapitalize: Boolean = False;

(***********************************************************

  This alters the operation of the system to automatically
    perform the text checking in the background.

***********************************************************)

      AllowBackGroundChecking = True;

(***********************************************************

  This flag produces alternate code that uses a unique name
    for the temporary file instead of the default name.

***********************************************************)

      Use_Unique_FileName = True;

(***********************************************************

   Definition of the word information storage structure.

***********************************************************)

    Type
      Info_Type = Set of ( Capital );

(***********************************************************

  These variable procedures are available for replacement by
  the main program.  They are all initialized to the default
  procedures which allow maximum versatility, but can still
  be altered to make a more consistent and elegant user
  interface.  Study the code thoroughly before attempting to
  replace it, because it may also perform a operation that
  isn't so obvious at first.

    To replace one of these routines, define a replacement
    routine that mimics the default code in functions.

      For example...

        Procedure Replacement( parameters ); Far;
          Begin
            do something here.
          End;

    Then, somewhere in the initialization section of the
    main program, substitute your new routine for the old
    one.

      For example...

        Old_Routine := Replacement;

   That's all there is to it.  Now the new routine will
   be called in place of the old one.

***********************************************************)

    Var

     { This procedure is called to set up the screen for the word. }
      Prepare_Screen: Procedure;

     { This procedure is called to display the suspected word. }
      Display_Word,

     { This procedure is called to get a suggested word from the user. }
      Get_Suggestion: Procedure( Var Word: String );

     { This procedure is called to display the suggested word. }
      Display_Suggestion: Procedure( Word: String );

     { This procedure is called to show and receive the options when there is a suggestion. }
      Offer_Options,

     { This procedure is called to show and receive the options when there isn't a suggestion. }
      Offer_Options2: Procedure( Var Result: Char );

     { This procedure flags when there are no more suggestions. }
      Out_Of_Suggestions: Procedure;

     { This procedure prompts to capitalize the word and performs the operation. }
      Prompt_To_Capitalize: Procedure( Var Data: String; Var Result: Info_Type; Var Escape: Boolean );

     { This function is called for correct words that aren't capitalized. }
      Prompt_For_Capitalize: Function( Var Escape: Boolean ): Char;

     { This function asks if the suggestion is to be used globally. }
      Ask_Global_Replacement: Function( Var Escape: Boolean ): Boolean;

     { This procedure clears the options from the screen. }
      Clear_Options: Procedure;

     { This procedure is called to display the text checking results. }
      Display_Status: Procedure( Total_Words, Replaced_Words, Total_Lines, Capitalized_Words, Altered_Words: LongInt );

     { This procedure is called to display the filename during dictionary search. }
      Write_Search: Procedure( FileName: String );

(***********************************************************

  Procedure: Read dictionary.

    This procedure reads in the words from the given text
    file and adds them into the main dictionaries without
    prompting for permission.  It's main purpose is to allow
    easy building of the dictionary in the file system.

***********************************************************)

    Procedure Read_Dictionary( FileName: String );

(***********************************************************

  Procedure: Write dictionary.

    This procedure writes all the words in the main
    dictionaries into a text file of the given name.

***********************************************************)

    Procedure Write_Dictionary( FileName: String );

(***********************************************************

  Error codes returned to Write_Error are

    1000 - Error in opening read dictionary name.
    1001 - Error in reading the dictionary.
    1002 - Error in opening write dictionary name.
    1003 - Error in writing the dictionary.

***********************************************************)

{----------------------------------------------------------------------------}

  Implementation

    Const
     { Tells if the system is ready. }
      Correct_Ready: Boolean = False;
     { Maximum allowable word size. }
      Word_Data_Length = 30;
     { Maximum allowable words in the suggestion list. }
      Maximum_Word_List_Amount = 40;
     { Maximum allowable word sizes for the various dictionaries. }
      Word1_Maximum = 6;
      Word2_Maximum = 8;
      Word3_Maximum = 10;
      Word4_Maximum = 12;
     { Used for stripping a word out of the text. The extended characters are also included. }
      Valid_Word_Character = [ 'A' .. 'Z', 'a' .. 'z', #131, #132, #133, #160, #134, #142, #143, #136, #137, #138, #130, #144,
                               #140, #139, #141, #161, #147, #148, #149, #162, #153, #150, #129, #151, #163, #154, #152, #145,
                               #146, #166, #167, #164, #165, #39 ];

    Type
     { These are the possible results during a single word check. }
      Result_Type = ( A_Fine, A_Not_Found, A_Alter, A_Capitalize, A_Error );
     { Definition of the key for the soundex system. }
      Look_Type = packed array [ 1 .. 6 ] of Char;
     { Definition of the word storage structure. }
      Word_String_Type = packed array [ 1 .. Word_Data_Length ] of Char;
     { Definition of the word storage record and key. }
      Word_Data_Type = Record
                         Look: Look_Type;
                         Info: Info_Type;
                         Word: Word_String_Type;
                       End;
     { Definition of the word storage record for automatic corrections and ignores. }
      Change_Data_Type = Record
                           Look: Look_Type;
                           Info: Info_Type;
                           Word,
                           Word2: Word_String_Type;
                         End;
     { Definition of the various words for the different dictionaries. }
      Word1_String_Type = packed array [ 1 .. Word1_Maximum ] of Char;
      Word2_String_Type = packed array [ 1 .. Word2_Maximum ] of Char;
      Word3_String_Type = packed array [ 1 .. Word3_Maximum ] of Char;
      Word4_String_Type = packed array [ 1 .. Word4_Maximum ] of Char;
     { Definition of the various word types for the different dictionaries. }
      Word1_Type = Record
                     Look: Look_Type;
                     Info: Info_Type;
                     Word: Word1_String_Type;
                   End;
      Word2_Type = Record
                     Look: Look_Type;
                     Info: Info_Type;
                     Word: Word2_String_Type;
                   End;
      Word3_Type = Record
                     Look: Look_Type;
                     Info: Info_Type;
                     Word: Word3_String_Type;
                   End;
      Word4_Type = Record
                     Look: Look_Type;
                     Info: Info_Type;
                     Word: Word4_String_Type;
                   End;
     { Definition of the storage array for the suggestion list. }
      Word_Array_Type = packed array [ 1 .. Maximum_Word_List_Amount ] of Word_String_Type;
     { Structure to hold the suggestion list. }
      Word_List_Type = Record
                         Amount,
                         Look_Point: 0 .. Maximum_Word_List_Amount;
                         Word: Word_Array_Type;
                       End;
      Dictionary_Array_Type = array [ 1 .. 5 ] of Tree_Type;

    Var
     { Holds the old exit routine - used to make sure the dictionary will be
       saved in the case of an emergency exit. }
      Old_Exit_Procedure: Pointer;
     { Holds the main word dictionaries. }
      Auto_Dictionary,
      Correct_Dictionary,
      Personal_Dictionary: Tree_Type;
      Main_Dictionaries: Dictionary_Array_Type;
     { Holds the word alternative lists. }
      Word_List: ^Word_List_Type;
     { Restricts access to Replace_The_Word. }
      Allow_Go,
     { This flag is used to advance the background checking feature. }
      BackGroundCheck: Boolean;
     { Used to save the old window. }
      Save_Bottom: Byte;
     { These pointers are used during background checking to hold the current
       location. }
      BackGroundLook,
      BackGroundStart: Point_Type;
     { This variable holds the amount of words automatically altered. }
      Altered_Count,
     { This value is used during background checking to limit the search. }
      Checking_Limit,
     { This is used to count the words during background checking. }
      BackGroundCount: LongInt;
     { This variable points to the current text data during background
       operations. }
      Checking_Text: ^All_Type;
     { Hold the old help routine. }
      Old_Help: Procedure;
     { This variable holds the old link. }
      Old_K_Link,
      Old_Q_Link: Procedure( Var All: All_Type; Data: Char; Var Start, Finish: Point_Type; Var Done: Boolean );
     { Used to hold the temporary file name. }
      Temporary_FileName: String[ 80 ];

{----------------------------------------------------------------------------}

  {$I TextCor2.Pas }

{----------------------------------------------------------------------------}

(*************************************************

  Function: Replace the word.
    This function should be called after Analyze
    the word, when the result is A_Not_Found.  It
    attempts to find a suitable replacement and
    will return true if it does.  If the escape
    key is pressed then this function will change
    Escape to true.  This function is not allowed
    to be called in the background;

*************************************************)

    Function Replace_The_Word( Word: String; Var New_Word: String; Var Escape: Boolean ): Boolean;
      Var
        Size: Byte;
        Change_Word,
        Capitalized: Boolean;
        Result: Char;
        Replacement: Word_String_Type;
        Working_Word: Change_Data_Type;
      Begin
        Escape := False;
        Change_Word := False;
        If Correct_Ready and Allow_Go
          then
            Begin
              Allow_Go := False;
              Size := Length( Word );
              Find_Look_Value( Word, Working_Word.Look );
              Make_Word_LowerCase( Word, Working_Word.Word, Capitalized );
              Working_Word.Info := [];
              Present_Options( Word_List^, Word, Result, Replacement );
              Case Result of
                'A': Begin
                       If Capitalized
                         then
                           Prompt_To_Capitalize( Word, Working_Word.Info, Escape );
                       If ( not Escape )
                         then
                           Add_To_Dictionary( Personal_Dictionary, Working_Word );
                     End;
                'C': Change_Word := True;
                'D': Begin
                       If Capitalized
                         then
                           Prompt_To_Capitalize( Word, Working_Word.Info, Escape );
                       If ( not Escape )
                         then
                           Add_To_Dictionaries( Working_Word, Size );
                     End;
                'E': Begin
                       Change_Word := True;
                       Working_Word.Word2 := Replacement;
                       Add_To_Dictionary( Auto_Dictionary, Working_Word );
                     End;
                'G': Begin
                       Change_Word := True;
                       Working_Word.Word2 := Replacement;
                       Add_To_Dictionary( Auto_Dictionary, Working_Word );
                     End;
                'I': Begin
                       Working_Word.Word2 := Working_Word.Word;
                       Add_To_Dictionary( Auto_Dictionary, Working_Word );
                     End;
                'L': Begin
                       Change_Word := True;
                       Working_Word.Word2 := Replacement;
                       Add_To_Dictionary( Correct_Dictionary, Working_Word );
                     End;
                'Z': Escape := True;
              End; { Case }
              Clear_Options;
              If Change_Word
                then
                  Convert_To_String( Replacement, New_Word, Capitalized );
              Allow_Go := True;
            End;
        Replace_The_Word := Change_Word;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Capitalize the word.
    This function should be called after Analyze
    the word, when the result is A_Capitalize.  It
    returns true if the word should be
    capitalized.  If the escape key is pressed
    then this function will change Escape to true.
    This function is not allowed to be called in
    the background;

*************************************************)

    Function Capitalize_The_Word( Var Word: String; Var Escape: Boolean ): Boolean;
      Var
        Capitalize: Boolean;
        Working_Word: Change_Data_Type;
      Begin
        If AutoCapitalize
          then
            Begin
              Capitalize := True;
              Word[ 1 ] := UpCase( Word[ 1 ] );
            End
          else
            Begin
              Escape := False;
              Capitalize := False;
              If Correct_Ready and Allow_Go
                then
                  Begin
                    Allow_Go := False;
                    Prepare_Screen;
                    Display_Word( Word );
                    Case Prompt_For_Capitalize( Escape ) of
                      'A': If ( not Escape )
                             then
                               Begin
                                 Find_Look_Value( Word, Working_Word.Look );
                                 Make_Word_LowerCase( Word, Working_Word.Word, Capitalize );
                                 Working_Word.Info := [ Capital ];
                                 Working_Word.Word2 := Working_Word.Word;
                                 Working_Word.Word2[ 1 ] := UpCase( Working_Word.Word2[ 1 ] );
                                 Add_To_Dictionary( Auto_Dictionary, Working_Word );
                                 Word[ 1 ] := UpCase( Word[ 1 ] );
                                 Capitalize := True;
                               End;
                      'N': Capitalize := False;
                      'Y': If ( not Escape )
                             then
                               Begin
                                 Capitalize := True;
                                 Word[ 1 ] := UpCase( Word[ 1 ] );
                               End
                    End; { Case }
                    Clear_Options;
                    Allow_Go := True;
                  End;
            End;
        Capitalize_The_Word := Capitalize;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Emergency exit.
    This procedure is called when the program
    halts to close the dictionary file.

*************************************************)

    Procedure Emergency_Exit; Far;
      Var
        The_File: Text;
      Begin
        Close_Tree_File( Main_Dictionaries[ 1 ] );
        Close_Tree_File( Main_Dictionaries[ 2 ] );
        Close_Tree_File( Main_Dictionaries[ 3 ] );
        Close_Tree_File( Main_Dictionaries[ 4 ] );
        Close_Tree_File( Main_Dictionaries[ 5 ] );
        Close_Tree_File( Auto_Dictionary );
        Close_Tree_File( Personal_Dictionary );
        Close_Tree_File( Correct_Dictionary );
        Assign( The_File, Temporary_FileName );
        Erase( The_File );
        ExitProc := Old_Exit_Procedure;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Search and set.
    This procedure initializes the dictionary and
    opens it up for use.

*************************************************)

    Procedure Search_And_Set( File_Name: String; Var Tree: Tree_Type; Size: Word );
      Var
        Path: String;
      Begin
        Write_Search( File_Name );
        Path := Super_Find( File_Name, False );
        If ( Path = '' )
          then
            Open_Tree_File( Tree, File_Name, Size, 1, SizeOf( Look_Type ) )
          else
            Open_Tree_File( Tree, Path, Size, 1, SizeOf( Look_Type ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set things up for correction system.
    This procedure initializes the dictionaries
    and opens them up for use.

*************************************************)

    Procedure Set_Up_For_Correct;
      Begin
        If ( not Correct_Ready )
          then
            Begin
              Write_Wait;
              Search_And_Set( 'Words1.Tre', Main_Dictionaries[ 1 ], SizeOf( Word1_Type ) );
              Search_And_Set( 'Words2.Tre', Main_Dictionaries[ 2 ], SizeOf( Word2_Type ) );
              Search_And_Set( 'Words3.Tre', Main_Dictionaries[ 3 ], SizeOf( Word3_Type ) );
              Search_And_Set( 'Words4.Tre', Main_Dictionaries[ 4 ], SizeOf( Word4_Type ) );
              Search_And_Set( 'Words5.Tre', Main_Dictionaries[ 5 ], SizeOf( Word_Data_Type ) );
              Search_And_Set( 'Correct.Tre', Correct_Dictionary, SizeOf( Change_Data_Type ) );
              Search_And_Set( 'Person.Tre', Personal_Dictionary, SizeOf( Word_Data_Type ) );
              If Use_Unique_FileName
                then
                  Temporary_FileName := Generate_Unique_FileName
                else
                  Temporary_FileName := 'Temp.Tre';
              Open_Tree_File( Auto_Dictionary, Temporary_FileName, SizeOf( Change_Data_Type ), 1, SizeOf( Look_Type ) );
              Old_Exit_Procedure := ExitProc;
              ExitProc := @Emergency_Exit;
              Correct_Ready := True;
              Allow_Go := True;
              Write_Complete;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get word internal.
    This procedure separates the word internally
    from the surrounding line.

*************************************************)

    Procedure Get_Word_Internal( Var Line: Line_Type; Var Data: String; Var Index, Start: Word );
      Var
        Pointer: Byte;
      Begin
       { Examine the characters until we get a character that isn't a blank. }
        While ( Index < Line.Size ) and ( not ( Line.Data[ Index ] in Valid_Word_Character ) ) do
          Inc( Index );
       { Separate the word. }
        Data := '';
        Start := Index;
        While ( Index <= Line.Size ) and ( Line.Data[ Index ] in Valid_Word_Character ) do
          Begin
            Data := Data + Line.Data[ Index ];
            Inc( Index );
          End;
       { Eliminate trailing apostrophes. }
        Pointer := Length( Data );
        While ( ( Pointer > 0 ) and ( Data[ Pointer ] = '''' ) ) do
          Begin
            Delete( Data, Pointer, 1 );
            Dec( Pointer );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Build line.
    This procedure takes the given line and
    separates any of the internal words.  Then it
    puts the words into the main dictionary.

*************************************************)

    Procedure Build_Line( Var Line: Line_Type );
      Var
        Index,
        Start: Word;
        Data_String: String;
      Begin
        If ( Line.Size > 0 )
          then
            Begin
              Index := 1;
              Repeat
                Get_Word_Internal( Line, Data_String, Index, Start );
                If ( ( Data_String <> '' ) and ( Data_String[ 1 ] <> #39 ) )
                  then
                    Begin
                      Write( Screen, Data_String, ' ' );
                      Add_Word_To_Dictionaries( Data_String );
                    End;
              Until ( Index >= Line.Size );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Build dictionary.
    This function reads the data from the given
    text file and puts all the words into the
    main dictionary.  It returns false if it
    fails.

*************************************************)

    Function Build_Dictionary( Var InFile: Text; Var Data: Line_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := True;
        While Okay and ( not EOF( InFile ) ) do
          Begin
            If Read_Line( InFile, Data )
              then
                Build_Line( Data )
              else
                Okay := False;
            WriteLn( Screen );
          End;
        Build_Dictionary := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Change the line.
    This procedure takes the given line and makes
    the appropriate changes to it to correct it.

*************************************************)

    Procedure Change_The_Line( Var Line: Line_Type; Var Start, Index: Word; Var Replacement: String );
      Var
        Size: Word;
      Begin
        Size := Length( Replacement );
        Change_Line( Line, Start, ( Index - Start ), Replacement );
        While ( Index - Start ) < Size do
          Inc( Index );
        While ( Index - Start ) > Size do
          Dec( Index );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read dictionary.
    As previously defined.

*************************************************)

    Procedure Read_Dictionary( FileName: String );
      Var
        InFile: Text;
      Begin
        If ( not Correct_Ready )
          then
            Set_Up_For_Correct;
        Assign( InFile, FileName );
       {$I-}
        Reset( InFile );
       {$I+}
        If ( IoResult = 0 )
          then
            If Build_Dictionary( InFile, Working_Buffer^ )
              then
                Close( InFile )
              else
                Write_Error( 1001 )
          else
            Write_Error( 1000 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write dictionary.
    As previously defined.

*************************************************)

    Procedure Write_Dictionary( FileName: String );
      Var
        OutFile: Text;
      Begin
        If ( not Correct_Ready )
          then
            Set_Up_For_Correct;
        Assign( OutFile, FileName );
       {$I-}
        Rewrite( OutFile );
       {$I+}
        If ( IoResult = 0 )
          then
            If Dump_The_Dictionaries( OutFile )
              then
                Close( OutFile )
              else
                Write_Error( 1002 )
          else
            Write_Error( 1003 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get text word.
    This function gets a single word from the text
    line.  If there are no more words on that
    line, either it will read in the next line
    from text, or it will return false in which
    case there is no more text to use.

*************************************************)

    Function Get_Text_Word( Var All: All_Type; Var Line: Line_Type; Var Where, Start: Point_Type; Var Data: String ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Repeat
          If ( Line.Size > 0 ) and ( Where.Column < Line.Size )
            then
              Get_Word_Internal( Line, Data, Where.Column, Start.Column )
            else
              Begin
                Inc( Where.Row );
                Where.Column := 1;
                Get_Line( All.Text, Where.Row, Line );
                Start := Where;
                Data := '';
              End;
          Okay := ( Where.Row <= All.Text.FileSize );
        Until ( ( Data <> '' ) and ( Data[ 1 ] <> '''' ) ) or ( not Okay );
        Get_Text_Word := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Alter text.
    This procedure makes the alteration in the
    data line.

*************************************************)

    Procedure Alter_Text( Var All: All_Type; Var Line: Line_Type; Var Look, Word: Point_Type; Var Okay: Boolean;
                          Var Data: String );
      Begin
        If ( Look.Row <= All.Text.FileSize )
          then
            Begin
              Get_Line( All.Text, Look.Row, Line );
              Change_The_Line( Line, Word.Column, Look.Column, Data );
              Okay := Put_Line( All.Text, Look.Row, Line, True );
              If Okay
                then
                  Inc( Altered_Count );
            End
          else
            Okay := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Capitalize text.
    This procedure capitalizes the word.

*************************************************)

    Procedure Capitalize_Text( Var All: All_Type; Var Line: Line_Type; Var Look, Word: Point_Type; Var Okay, Escape: Boolean;
                               Var Data: String; Var Count: LongInt );
      Var
        Middle: Byte;
      Begin
        Middle := ( Bottom_Of_Window^ - ( Bottom_Of_Window^ div 5 ) );
        Draw_Screen( All.Text, Word, Middle, Length( Data ), Succ( Bottom_Of_Window^ ) );
        If Capitalize_The_Word( Data, Escape )
          then
            Begin
              Alter_Text( All, Line, Look, Word, Okay, Data );
              If Okay
                then
                  Inc( Count );
            End
          else
            Begin
              Get_Line( All.Text, Look.Row, Line );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Replace text.
    This procedure replaces the word in the text
    with the new word.

*************************************************)

    Procedure Replace_Text( Var All: All_Type; Var Line: Line_Type; Var Look, Word: Point_Type; Var Okay, Escape: Boolean;
                            Var Data, Replacement: String; Var Count: LongInt );
      Var
        Middle: Byte;
      Begin
        Middle := ( Bottom_Of_Window^ - ( Bottom_Of_Window^ div 5 ) );
        Draw_Screen( All.Text, Word, Middle, Length( Data ), Succ( Bottom_Of_Window^ ) );
        If Replace_The_Word( Data, Replacement, Escape )
          then
            Begin
              Alter_Text( All, Line, Look, Word, Okay, Replacement );
              If Okay
                then
                  Inc( Count );
            End
          else
            Begin
              Get_Line( All.Text, Look.Row, Line );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Background checking.
    During a normal spell checking, the system may
    sometimes have to stop and wait for the user
    to decide which action to take.  In order to
    speed up the procedure, this routine takes
    control during idle time and continues the
    checking in the background.

*************************************************)

    Procedure Background_Checking; Far;
      Var
        Okay: Boolean;
        Data,
        Replacement: String;
      Begin
        If BackGroundCheck
          then
            Begin
              Okay := ( BackGroundLook.Row <= Checking_Limit );
              If Okay
                then
                  Begin
                    Okay := Get_Text_Word( Checking_Text^, Other_Buffer^, BackGroundLook, BackGroundStart, Data );
                    If Okay
                      then
                        Begin
                          Case Analyze_The_Word_Single( Data, Replacement ) of
                            A_Alter:
                              Alter_Text( Checking_Text^, Other_Buffer^, BackGroundLook, BackGroundStart, Okay,
                                          Replacement );
                            A_Capitalize:
                              Okay := False;
                            A_Not_Found:
                              Okay := False;
                            A_Error:
                              Okay := False;
                          End; { Case }
                          Inc( BackGroundCount );
                        End;
                  End;
              BackGroundCheck := Okay;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up background check.
    This procedure sets up the global variables
    for the background checking routine.

*************************************************)

    Procedure Set_Up_BackGround_Check( Var Line: Line_Type; Look, Start: Point_Type );
      Begin
        BackGroundCheck := True;
        BackGroundLook := Look;
        BackGroundStart := Start;
        Other_Buffer^ := Line;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process background check.
    This routine checks the background global
    variables to see what happened.  It the
    parameters allow it, the variables are updated
    to where the background check left off.

*************************************************)

    Procedure Process_BackGround_Check( Var Line: Line_Type; Var Look, Start: Point_Type; Var Word_Count: LongInt );
      Begin
        If BackGroundCheck
          then
            Begin  { Didn't find anything. }
              BackGroundCheck := False;
              If ( BackGroundLook.Row <> Look.Row )
                then
                  Begin
                    Look := BackGroundLook;
                    Start := BackGroundStart;
                    Line := Other_Buffer^;
                    Word_Count := Word_Count + BackGroundCount;
                  End;
              BackGroundCount := 0;
            End
          else { Found something. }
            Begin
              If ( BackGroundLook.Row <> Look.Row )
                then
                  Begin
                    Look := BackGroundStart;
                    Start := BackGroundStart;
                    Start.Column := Start.Column + ( BackGroundStart.Column - BackGroundLook.Column );
                    Line := Other_Buffer^;
                    Word_Count := Word_Count + Pred( BackGroundCount );
                  End;
              BackGroundCount := 0;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process capitalize.
    This procedure is called when the word should
    be capitalized.

*************************************************)

    Procedure Process_Capitalize( Var All: All_Type; Var Line: Line_Type; Var Look, Start: Point_Type; Var Okay,
                                  Escape: Boolean; Var Data: String; Var Count, Word_Count: LongInt );
      Begin
        If AllowBackGroundChecking
          then
            Set_Up_BackGround_Check( Line, Look, Start );
        Write_Complete;
        Capitalize_Text( All, Line, Look, Start, Okay, Escape, Data, Count );
        Write_Wait;
        If AllowBackGroundChecking
          then
            Process_BackGround_Check( Line, Look, Start, Word_Count );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process not found.
    This procedure is called when the word isn't
    found so that an alternative could be
    attained.

*************************************************)

    Procedure Process_Not_Found( Var All: All_Type; Var Line: Line_Type; Var Look, Start: Point_Type; Var Okay,
                                 Escape: Boolean; Var Data, Replacement: String; Var Count, Word_Count: LongInt );
      Begin
        If AllowBackGroundChecking
          then
            Set_Up_BackGround_Check( Line, Look, Start );
        Write_Complete;
        Replace_Text( All, Working_Buffer^, Look, Start, Okay, Escape, Data, Replacement, Count );
        Write_Wait;
        If AllowBackGroundChecking
          then
            Process_BackGround_Check( Line, Look, Start, Word_Count );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Check test.
    This procedure performs a spelling check on
    the given text structure and makes corrections
    to it.  It works with background checking to
    increase performance levels.

*************************************************)

    Procedure Check_Text( Var All: All_Type; Start_Row, Finish_Row: LongInt; Start_Column: Word );
      Var
        Okay,
        Escape: Boolean;
        Look,
        Start: Point_Type;
        Result: Result_Type;
        Data,
        Replacement: String;
        Limit,
        Point,
        Word_Count,
        Replaced_Count,
        Capitalized_Count: LongInt;
        Hold_KeyBoard_Procedure: Procedure;
      Begin
        If ( not Correct_Ready )
          then
            Set_Up_For_Correct;
        Working_Buffer^.Size := 0;
        If ( Start_Row > 1 )
          then
            Look.Row := Start_Row
          else
            Look.Row := 1;
        If ( Finish_Row < 1 )
          then
            Finish_Row := All.Text.FileSize;
        Look.Column := Start_Column;
        If ( Look.Column < 1 )
          then
            Look.Column := 1;
        Okay := True;
        Escape := False;
        If AllowBackGroundChecking
          then
            Begin
              Hold_KeyBoard_Procedure := KeyBoard.Wait_Routine;
              Checking_Text := @All;
              KeyBoard.Wait_Routine := BackGround_Checking;
              Checking_Limit := Finish_Row;
              BackGroundCount := 0;
            End;
        If Show_Status
          then
            Begin
              Limit := ( Finish_Row - Start_Row );
              If ( Limit < 1 )
                then
                  Limit := 1;
            End;
        Point := 0;
        Word_Count := 0;
        Altered_Count := 0;
        Replaced_Count := 0;
        Capitalized_Count := 0;
        Start := Look;
        Write_Wait;
        Get_Line( All.Text, Look.Row, Working_Buffer^ );
        While ( Okay and Get_Text_Word( All, Working_Buffer^, Look, Start, Data ) ) do
          Begin
            Result := Analyze_The_Word( Data, Replacement );
            Case Result of
              A_Capitalize:
                Process_Capitalize( All, Working_Buffer^, Look, Start, Okay, Escape, Data, Capitalized_Count, Word_Count );
              A_Not_Found:
                Process_Not_Found( All, Working_Buffer^, Look, Start, Okay, Escape, Data, Replacement, Replaced_Count,
                                   Word_Count );
              A_Alter:
                Alter_Text( All, Working_Buffer^, Look, Start, Okay, Replacement );
            End; { Case }
            Okay := ( Look.Row <= Finish_Row ) and ( not Escape );
            If ( Okay and Show_Status )
              then
                Begin
                  Point := Succ( Look.Row - Start_Row );
                  If Odd( Point )
                    then
                      Process_Status( Point, Limit, Okay );
                End;
            Inc( Word_Count );
          End;
        Write_Complete;
        If AllowBackGroundChecking
          then
            KeyBoard.Wait_Routine := Hold_KeyBoard_Procedure;
        If Show_Status
          then
            Display_Status( Word_Count, Replaced_Count, Point, Capitalized_Count, Altered_Count );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Prepare Screen default.
    This routine is called at the start of the
    word correction command routine.

*************************************************)

    Procedure Prepare_Screen_Default; Far;
      Begin
        Save_Bottom := Succ( Bottom_Of_Window^ );
        Window( Succ( Left_Of_Window^ ), Succ( Top_Of_Window^ ), Succ( Right_Of_Window^ ), 14 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display word default.
    This routine is called to display the
    suspected word.

*************************************************)

    Procedure Display_Word_Default( Var Word: String ); Far;
      Begin
        ClrScr;
        GotoXY( 1, 1 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Suspect word : ' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Word );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display suggestion default.
    This routine is called each time a new
    suggestion is supposed to be displayed on the
    screen.

*************************************************)

    Procedure Display_Suggestion_Default( Word: String ); Far;
      Begin
        GotoXY( 1, 2 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Suggested word : ' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Word: Word_Data_Length );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Offer options default.
    This procedure offers the options to the user
    when a potential replacement can be found.

*************************************************)

    Procedure Offer_Options_Default( Var Result: Char ); Far;
      Var
        Command: Byte;
      Begin
        TextAttr := Message_HighLight;
        WriteLn( Screen, ' Please press the appropriate letter.' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Pred( Right_Of_Window^ - Left_Of_Window^ ) ) );
        TextAttr := Message_HighLight;
        Write( Screen, '   A' );
        TextAttr := Message_Normal;
        Write( Screen, ' to add to personal dictionary.' );
        TextAttr := Message_HighLight;
        Write( Screen, '  B' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to bypass word this time.' );
        TextAttr := Message_HighLight;
        Write( Screen, '   C' );
        TextAttr := Message_Normal;
        Write( Screen, ' to correct as suggested.      ' );
        TextAttr := Message_HighLight;
        Write( Screen, '  E' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to enter correction from keyboard.' );
        TextAttr := Message_HighLight;
        Write( Screen, '   G' );
        TextAttr := Message_Normal;
        Write( Screen, ' to correct globally.          ' );
        TextAttr := Message_HighLight;
        Write( Screen, '  I' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to ignore word throughout document.' );
        TextAttr := Message_HighLight;
        Write( Screen, '   P' );
        TextAttr := Message_Normal;
        Write( Screen, ' to show previous suggestion.  ' );
        TextAttr := Message_HighLight;
        Write( Screen, '  N' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to show next suggestion.' );
        WriteLn( Screen, Expand( '-', Pred( Right_Of_Window^ - Left_Of_Window^ ) ) );
        TextAttr := Message_HighLight;
        Write( Screen, '   D' );
        TextAttr := Message_Normal;
        Write( Screen, ' to add to main dictionary.   ' );
        TextAttr := Message_HighLight;
        Write( Screen, '   L' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to add to auto correct dictionary.' );
        WriteLn( Screen, Expand( '-', Pred( Right_Of_Window^ - Left_Of_Window^ ) ) );
        Write( Screen, '  Press' );
        TextAttr := Message_HighLight;
        Write( Screen, ' Escape' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to cancel' );
        Repeat
          Repeat
            Get_Command( Result, Command );
          Until ( Command in [ Press_Lower_Letters, Press_Capital_Letters, Press_Escape, Press_Up_Arrow, Press_Down_Arrow,
                               Press_Left_Arrow, Press_Right_Arrow ] );
          Case Command of
            Press_Escape: Result := 'Z';
            Press_Up_Arrow,
            Press_Left_Arrow: Result := 'P';
            Press_Down_Arrow,
            Press_Right_Arrow: Result := 'N';
            else Result := UpCase( Result );
          End; { Case }
        Until ( Result in [ 'A' .. 'E', 'G', 'I', 'L', 'P', 'N', 'Z' ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Offer options two default.
    This procedure offers the options to the user
    when a potential replacement can't be found.

*************************************************)

    Procedure Offer_Options2_Default( Var Result: Char ); Far;
      Var
        Command: Byte;
      Begin
        TextAttr := Message_HighLight;
        WriteLn( Screen, ' Please press the appropriate letter.' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Pred( Right_Of_Window^ - Left_Of_Window^ ) ) );
        TextAttr := Message_HighLight;
        Write( Screen, '   A' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to add to personal dictionary.' );
        TextAttr := Message_HighLight;
        Write( Screen, '   B' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to bypass word this time.' );
        TextAttr := Message_HighLight;
        Write( Screen, '   E' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to enter correction from keyboard.' );
        TextAttr := Message_HighLight;
        Write( Screen, '   I' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to ignore word throughout document.' );
        WriteLn( Screen, Expand( '-', Pred( Right_Of_Window^ - Left_Of_Window^ ) ) );
        TextAttr := Message_HighLight;
        Write( Screen, '   D' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to add to main dictionary.   ' );
        WriteLn( Screen, Expand( '-', Pred( Right_Of_Window^ - Left_Of_Window^ ) ) );
        Write( Screen, '  Press' );
        TextAttr := Message_HighLight;
        Write( Screen, ' Escape' );
        TextAttr := Message_Normal;
        WriteLn( Screen, ' to cancel' );
        Repeat
          Repeat
            Get_Command( Result, Command )
          Until ( Command in [ Press_Lower_Letters, Press_Capital_Letters, Press_Escape ] );
          If ( Command = Press_Escape )
            then
              Result := 'Z'
            else
              Result := UpCase( Result );
        Until ( Result in [ 'A', 'B', 'D', 'E', 'I', 'Z' ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Out of suggestions default.
    This routine is called only when there are no
    more suggestions.

*************************************************)

    Procedure Out_Of_Suggestions_Default; Far;
      Begin
        TextAttr := Message_Normal;
        WriteLn( Screen, 'No more suggestions. '#7 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Clear options default.
    This routine is called after the present
    options to clear away the clutter.

*************************************************)

    Procedure Clear_Options_Default; Far;
      Begin
        ClrScr;
        TextAttr := Message_HighLight;
        GotoXY( 1, 2 );
        Write( Screen, 'Checking text' );
        Window( Succ( Left_Of_Window^ ), Succ( Top_Of_Window^ ), Succ( Right_Of_Window^ ), Save_Bottom );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get suggestion default.
    This routine is called to get a suggestion
    from the keyboard.

*************************************************)

    Procedure Get_Suggestion_Default( Var Word: String ); Far;
      Begin
        GotoXY( 1, 2 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Enter correction: ', ' ':Word_Data_Length );
        TextAttr := Message_Normal;
        GotoXY( 19, 2 );
        ReadLn( Keys, Word );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Ask global replacement default.
    This function is called to determine if the
    replacement is supposed to be used globally.

*************************************************)

    Function Ask_Global_Replacement_Default( Var Escape: Boolean ): Boolean; Far;
      Var
        Result: Char;
      Begin
        ClrScr;
        TextAttr := Message_HighLight;
        Write( Screen, 'Global replacement? ' );
        Result := Get_Answer;
        Ask_Global_Replacement_Default := ( Result = 'Y' );
        Escape := ( Result = 'E' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Prompt to capitalize default.
    This procedure is called when the word in
    question is capitalized and is to be included
    in one of the dictionaries.

*************************************************)

    Procedure Prompt_To_Capitalize_Default( Var Data: String; Var Result: Info_Type; Var Escape: Boolean ); Far;
      Var
        Answer: Char;
      Begin
        ClrScr;
        TextAttr := Message_HighLight;
        Write( Screen, 'Should ' );
        TextAttr := Message_Normal;
        Write( Screen, Data );
        TextAttr := Message_HighLight;
        Write( Screen, ' always be capitalized? ' );
        Answer := Get_Answer;
        If ( Answer = 'Y' )
          then
            Result := Result + [ Capital ]
          else
            Escape := ( Answer = 'E' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Prompt for capitalize default.
    This function prompts to capitalize the word.

*************************************************)

    Function Prompt_For_Capitalize_Default( Var Escape: Boolean ): Char; Far;
      Var
        Command: Byte;
        Result: Char;
      Begin
        TextAttr := Message_HighLight;
        Write( Screen, 'Capitalize this word? ' );
        TextAttr := Message_Normal;
        Write( Screen, '( ' );
        TextAttr := Message_HighLight;
        Write( Screen, 'Y' );
        TextAttr := Message_Normal;
        Write( Screen, ', ' );
        TextAttr := Message_HighLight;
        Write( Screen, 'N ' );
        TextAttr := Message_Normal;
        Write( Screen, 'or ' );
        TextAttr := Message_HighLight;
        Write( Screen, 'A ' );
        TextAttr := Message_Normal;
        Write( Screen, ')' );
        Repeat
          Repeat
            Get_Command( Result, Command )
          Until ( Command in [ Press_Lower_Letters, Press_Capital_Letters, Press_Escape ] );
          Result := UpCase( Result );
          Escape := ( Command = Press_Escape );
        Until ( ( Result in [ 'Y', 'N', 'A' ] ) or Escape );
        Prompt_For_Capitalize_Default := Result;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write search default.
    This procedure is called to inform the user
    that the system is attempting to find the file
    of the given name.

*************************************************)

    Procedure Write_Search_Default( FileName: String ); Far;
      Begin
        TextAttr := Message_HighLight;
        Write( Screen, ' Searching for ' );
        TextAttr := Message_Normal;
        Write( Screen, FileName );
        TextAttr := Message_HighLight;
        WriteLn( Screen, '.' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display status default.
    This routine is called after the text is
    checked to display the results.

*************************************************)

    Procedure Display_Status_Default( Total_Words, Replaced_Words, Total_Lines, Capitalized_Words, Altered_Words: LongInt );
              Far;
      Begin
        ClrScr;
        TextAttr := Message_HighLight;
        Write( Screen, 'Total lines processed:' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Total_Lines:14 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Total words processed:' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Total_Words:14 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Total words flaged:' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Replaced_Words:17 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Total words capitalized:' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Capitalized_Words:12 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Words automatically altered:' );
        TextAttr := Message_Normal;
        WriteLn( Screen, Altered_Words:8 );
      End;


{----------------------------------------------------------------------------}

(*************************************************

  Procedure: K link.
    This procedure links into the control K
    interface to add the new functions to the
    TextEdit editing routine.

*************************************************)

  Procedure K_Link( Var All: All_Type; Character: Char; Var Block_Start, Block_Finish: Point_Type; Var Done: Boolean ); Far;
    Begin
      If ( Character = 'L' )
        then
          Begin
            Check_Text( All, Block_Start.Row, Block_Finish.Row, Block_Start.Column );
            TextAttr := Message_Normal;
            WriteLn( Screen );
            Pause_For_Key;
          End
        else
          Old_K_Link( All, Character, Block_Start, Block_Finish, Done );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Q link.
    This procedure links into the control Q
    interface to add the new functions to the
    TextEdit editing routine.

*************************************************)

  Procedure Q_Link( Var All: All_Type; Character: Char; Var Block_Start, Block_Finish: Point_Type; Var Done: Boolean ); Far;
    Begin
      Case Character of
        'L':
          Begin
            Check_Text( All, All.Data.Cursor.Row, 0, All.Data.Cursor.Column );
            TextAttr := Message_Normal;
            WriteLn( Screen );
            Pause_For_Key;
          End;
        'W':
          Begin
            Check_Text( All, All.Data.Cursor.Row, All.Data.Cursor.Row, 0 );
            TextAttr := Message_Normal;
            WriteLn( Screen );
            Pause_For_Key;
          End;
        else Old_Q_Link( All, Character, Block_Start, Block_Finish, Done );
      End; { Case }
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: New help.
    This function adds a new page to the help
    routine explaining the new functions.

*************************************************)

  Procedure New_Help; Far;
    Var
      Width: Byte;
    Begin
      Old_Help;
      Width := Pred( Right_Of_Window^ - Left_Of_Window^ );
      ClrScr;
      TextAttr := Message_HighLight;
      WriteLn( Screen, Center( 'Help Screen', Width, ' ' ) );
      TextAttr := Message_Normal;
      WriteLn( Screen, Expand( '-', Width ) );
      TextAttr := Message_HighLight;
      WriteLn( Screen, Center( 'Spell checking keys', Width, ' ' ) );
      TextAttr := Message_Normal;
      WriteLn( Screen, Expand( '-', Width ) );
      TextAttr := Message_Normal;
      Write( Screen, Push_To( 'Check text spelling', '^QL', 37, '.' ) );
      TextAttr := Message_HighLight;
      WriteLn( Screen, '^QL' );
      TextAttr := Message_Normal;
      Write( Screen, Push_To( 'Check block spelling', '^KL', 37, '.' ) );
      TextAttr := Message_HighLight;
      WriteLn( Screen, '^KL' );
      TextAttr := Message_Normal;
      Write( Screen, Push_To( 'Check line spelling', '^QW', 37, '.' ) );
      TextAttr := Message_HighLight;
      WriteLn( Screen, '^QW' );
      WriteLn( Screen );
      Pause_For_Key;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    This section initializes the variable
    procedures and functions.

*************************************************)

    Begin
      New( Word_List );
      If ( Word_List = Nil )
        then
          RunError( 203 );
      Prepare_Screen := Prepare_Screen_Default;
      Get_Suggestion := Get_Suggestion_Default;
      Display_Suggestion := Display_Suggestion_Default;
      Offer_Options := Offer_Options_Default;
      Offer_Options2 := Offer_Options2_Default;
      Ask_Global_Replacement := Ask_Global_Replacement_Default;
      Out_Of_Suggestions := Out_Of_Suggestions_Default;
      Prompt_To_Capitalize := Prompt_To_Capitalize_Default;
      Prompt_For_Capitalize := Prompt_For_Capitalize_Default;
      Write_Search := Write_Search_Default;
      Clear_Options := Clear_Options_Default;
      Display_Word := Display_Word_Default;
      Display_Status := Display_Status_Default;
      Old_Q_Link := Control_Q_Link;
      Control_Q_Link := Q_Link;
      Old_K_Link := Control_K_Link;
      Control_K_Link := K_Link;
      Old_Help := Help;
      Help := New_Help;
    End.

