(*****************************************************************************

  Pointer
    version 1.20

  This unit contains an interface for the use of the pointer device.  The
    standard pointer device is accessed through the operating system and the
    device driver.

  Purpose:
    To make adding the pointer to any program simple.

  How it works:
    The system handles the pointer operations, or the program can call the
      procedures directly.
    This unit works through the keyboard in a somewhat indirect manner.
      Note that nothing is generated from the pointer when neither buttons
        are active. (Down)  All action takes place when the button is held
        down or the last button condition had changed.
      The following occurs when either of the buttons are pressed.
        First, the position of the pointer is calculated relative to the
          position of the keyboard supplied cursor.
        Next, the interface generates cursor movement commands as it waits
          for the keyboard supplied cursor to move to the pointer's location.
        When or if the cursor and pointer are in alignment, a command is
          generated that indicates the button is in the down position.
        When the button is released, a command is generated that indicates
          the button was released.
        If the pointer is moved while the button is held down, the
          appropriate movement commands are again generated between the up and
          down signals until the pointer and keyboard cursor are again in
          alignment. (For dragging operations)
      Double-clicks are generated separately and may override the up or down
        signal.  Double clicks are recognized in a better manner than before.

  Features:
    This pointer interface is designed to allow mice, trackballs and light
      pens to be used within the program.
    When a program uses the KeyBoard unit, Pointer will link up with it
      automatically and include commands as keystrokes.
    Supports screen manipulation through the KeyBoard unit.
    New version improves the interface for the keyboard by increased the
      possible keyboard alterable commands and providing better movement
      using the secondary button.

  Limitations:
    The system service is assumed to interlink with the standard basic
      input output system pointer device link.
    While both Pointer and JoyStick can work nicely together with Keyboard,
      it isn't recommended.
    To avoid messing the screen up, the pointer should first be turned off
      while writing to the screen, then should be turned back on.
    This unit does not compile under Turbo Pascal version 4.0
    The keyboard interface routines are rather complex and should not be
      altered unless they are completely understood.

  Versions
    1.10 - Separated the pointer double-click from the enter commands.
    1.11 - Added support for selecting text with the pointer.
    1.20 - Revamped the pointer system to add more flexibility and work in a
           more consistent and easy working manner.

  Compilers:
    Turbo Pascal versions 5.0 to 6.0

  Systems:
    MS-DOS, MDOS;

*****************************************************************************)

Unit Pointer;

  Interface

    Uses
      DOS,
      CRT,
      Core,
      KeyBoard;

    Const
     { These variable constants are alterable by the program to alter the commands generated. }
      Button1_Up: Byte = Pointer_Button1_Up;
      Button2_Up: Byte = Pointer_Button2_Up;
      Button1_Down: Byte = Pointer_Button1_Down;
      Button2_Down: Byte = Pointer_Button2_Down;
      Button1_Double: Byte = Pointer_Button1_Double;
      Button2_Double: Byte = Pointer_Button2_Double;
     { Amount of hundredths of a second in which to register a double click. }
      Time_Delay: Word = 75;
     { Used to simulate the delay time for a keystroke from the keyboard. }
      Wait_Time: Word = 150;

    Type
     { This is the basic data return record. }
      Pointer_Data_Type = Record
                            X,
                            Y: Word;
                            Row,
                            Column: Byte;
                            Button1,
                            Button2: Boolean;
                          End;

    Var
     { These values determine the last location of the cursor. }
      Last_Row,
      Last_Column: Byte;

(***********************************************************

  Flag: Find pointer.

    This flag is set to true if the pointer was detected,
    otherwise it returns false.

***********************************************************)

      Find_Pointer: Boolean;

(*************************************************

  Procedure: Read position and buttons.

    This procedure reads the values of the
    pointer's X and Y screen coordinates and the
    buttons using the pointer device driver
    interface.

*************************************************)

    Procedure Read_Position_And_Buttons( Var X, Y: Word; Var Button1, Button2: Boolean );

(*************************************************

  Procedure: Display pointer.

    This procedure puts the pointer on the screen.

*************************************************)

    Procedure Display_Pointer;

(*************************************************

  Procedure: Hide pointer.

    This procedure removes the graphical pointer
    from the screen.

*************************************************)

    Procedure Hide_Pointer;

(***********************************************************

  Procedure: Set for pointer.

    This procedure sets up the system for the pointer.

***********************************************************)

    Procedure Set_For_Pointer;

(***********************************************************

  Function: Reset pointer.

    This function performs a pointer reset.  It returns true
    if everything is fine, false if it isn't.

***********************************************************)

    Function Reset_Pointer: Boolean;

(***********************************************************

  Function: Check for pointer.

    This function returns true if the pointer is detected.
    It works much quicker than using Reset_Pointer.

***********************************************************)

    Function Check_For_Pointer: Boolean;

(***********************************************************

  Procedure: Read the pointer.

    This procedure reads the status of the pointer and
    returns the results in the supplied data structures.
    X and Y return the value of the pointer in screen pixels
    while Row and Column returns the location in terms of
    cursor coordinates.

***********************************************************)

    Procedure Read_Pointer( Var Data: Pointer_Data_Type );

{-----------------------------------------------------------------------------}

(*****************************************************************************

    Useful Information:
      The pointer device driver is accessed through the interrupt 33h.
        Initialize the pointer - set AX to 0, returns AX <> 0 if successful.
        Turn the pointer on - set AX to 1
        Turn the pointer off - set AX to 2
        Get the pointer position - set AX to 3, returns CX with horizontal
                                   position, DX with the vertical position,
                                   and the buttons in BX
        Move the pointer - set AX to 4, CX to horizontal position, DX to
                           vertical position
        Redefine the cursor - set Ax to 9, BX,CX define the hotspot, ES:DX
                              points to bitmapped data of 64 bytes in length.
                              The first 32 bytes define the mask  ( the cursor
                              is a 16x16 bit map )

*****************************************************************************)

  Implementation

    Type
      Pointer_Procedure_Type = Procedure( Var Pointer_Data: Pointer_Data_Type );

    Var
     { Used to watch the status of the two pointer buttons. }
      Down_Button1,
      Down_Button2: Boolean;
     { Used to store the last two events to watch for double clicking. }
      Previous_Event,
      Previous_Event2: Byte;
     { Used to time a double pointer device click. }
      Old_Time_Last_Event: LongInt;
     { Used for keeping track of the on screen pointer. }
      Last_Attribute,
      Last_New_Attribute: Byte;
      Last_Character: Char;
      Old_Interface: Function: Byte;
      Internal_Read_Pointer: Pointer_procedure_Type;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read buttons.
    This procedure reads the button values using
    the standard pointer interface.

*************************************************)

    Procedure Read_Buttons( Var Button1, Button2: Boolean );
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AX := 3;
        Intr( $33, The_Registers );
        Button1 := ( ( The_Registers.BX and 1 ) <> 0 );
        Button2 := ( ( The_Registers.BX and 2 ) <> 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read position and buttons.
    As previously defined.

*************************************************)

    Procedure Read_Position_And_Buttons( Var X, Y: Word; Var Button1, Button2: Boolean );
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AX := 3;
        Intr( $33, The_Registers );
        X := The_Registers.CX;
        Y := The_Registers.DX;
        Button1 := ( ( The_Registers.BX and 1 ) <> 0 );
        Button2 := ( ( The_Registers.BX and 2 ) <> 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display_Pointer.
    As previously defined.

*************************************************)

    Procedure Display_Pointer;
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AX := 1;
        Intr( $33, The_Registers );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Hide_Pointer.
    As previously defined.

*************************************************)

    Procedure Hide_Pointer;
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AX := 2;
        Intr( $33, The_Registers );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read pointer short
    This procedure reads the pointer from the
    system using the short method.  In certain
    video modes, the values for Y are returned
    in a byte instead of a word.

*************************************************)

   {$IFDEF VER50}
    {$F+}
   {$ENDIF}

    Procedure Read_Pointer_Short( Var Pointer_Data: Pointer_Data_Type );
     {$IFNDEF VER50}
      Far;
     {$ENDIF}
      Var
        The_Registers: Registers;
      Begin
        With The_Registers, Pointer_Data do
          Begin
            Ah := $04;                  { Use the BIOS to read the position. }
            Intr( $10, The_Registers ); { Call the BIOS. }
            Row := Dh;
            Column := Dl;
            Y := Ch;
            X := Bx;
            Read_Buttons( Button1, Button2 );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read pointer long.
    This procedure reads the pointer from the
    system using the long method.  In certain
    video modes, the values for Y are returned
    in a word instead of a byte.

*************************************************)

    Procedure Read_Pointer_Long( Var Pointer_Data: Pointer_Data_Type );
     {$IFNDEF VER50}
      Far;
     {$ENDIF}
      Var
        The_Registers: Registers;
      Begin
        With The_Registers, Pointer_Data do
          Begin
            Ah := $04;                  { Use the BIOS to read the position. }
            Intr( $10, The_Registers ); { Call the BIOS. }
            Row := Dh;
            Column := Dl;
            Y := Cx;
            X := Bx;
            Read_Buttons( Button1, Button2 );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set for pointer.
    As previously defined.

*************************************************)

    Procedure Set_for_Pointer;
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ah := $0F;    { Use the BIOS to get the Video Mode. }
        Intr( $10, The_Registers ); { Call the BIOS. }
        Case The_Registers.Al of
          0..8,
          13..19: Internal_Read_Pointer := Read_Pointer_short;
          else Internal_Read_Pointer := Read_Pointer_long;
        End; { Case }
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Reset pointer.
    As previously defined.

*************************************************)

    Function Reset_Pointer: Boolean;
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AX := 0;       { Use the device interface to reset the pointer. }
        Intr( $33, The_Registers );  { Call the device driver. }
        Reset_Pointer := ( The_Registers.AX <> 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Check for pointer.
    This function uses the pointer interrupt to
    check and see if the pointer driver is
    installed and working.

*************************************************)

    Function Check_For_Pointer: Boolean;
      Var
        The_Registers: Registers;
      Begin
        The_Registers.AX := 3; { Use the device interface to read the position. }
        The_Registers.CX := $FFFF;
        The_Registers.DX := $FFFF;
        Intr( $33, The_Registers );  { Call the device driver. }
        Check_For_Pointer := ( The_Registers.CX <> $FFFF ) or ( The_Registers.DX <> $FFFF );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read pointer.
    As previously defined.

*************************************************)

    Procedure Read_Pointer( Var Data: Pointer_Data_Type );
      Begin
        Internal_Read_Pointer( Data );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Move.
    This function sets the Data byte to the
    appropriate value depending on the location
    of the selector cursor.  It returns false if
    the nothing is moved.

*************************************************)

    Function Move( Point: Pointer_Data_Type; Var Data: Byte ): Boolean;
      Var
        Adjust_Row,
        Adjust_Column: Byte;
      Begin
        Inc( Point.Row );
        Inc( Point.Column );
        Adjust_Row := ( Point.Row - Top_Of_Window^ );
        Adjust_Column := ( Point.Column - Left_Of_Window^ );
        If ( Adjust_Row > Cursor_Row )
          then
            Begin
              Data := Pointer_Down;
              Adjust_Amount := ( Adjust_Row - Cursor_Row );
              Move := True;
            End
          else
            If ( Adjust_Row < Cursor_Row )
              then
                Begin
                  Data := Pointer_Up;
                  Adjust_Amount := ( Cursor_Row - Adjust_Row );
                  Move := True;
                End
              else
                If ( Adjust_Column > Cursor_Column_Finish )
                  then
                    Begin
                      Data := Pointer_Right;
                      Adjust_Amount := ( Adjust_Column - Cursor_Column_Finish );
                      Move := True;
                    End
                  else
                    If ( Adjust_Column < Cursor_Column_Start )
                      then
                        Begin
                          Data := Pointer_Left;
                          Adjust_Amount := ( Cursor_Column_Start - Adjust_Column );
                          Move := True;
                        End
                      else
                        Move := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Outside.
    This procedure sets the Data byte to the
    appropriate value depending on the location
    outside the current window.

*************************************************)

    Procedure OutSide( Point: Pointer_Data_Type; Var Data: Byte );
      Begin
        If ( Point.Row > Bottom_Of_Window^ )
          then
            Begin
              If ( Point.Row > Succ( Bottom_Of_Window^ ) )
                then
                  Data := Outside_Down
                else
                  Data := On_Frame_Bottom;
            End
          else
            If ( Point.Row < Top_Of_Window^ )
              then
                Begin
                  If ( Point.Row < Pred( Top_Of_Window^ ) )
                    then
                      Data := Outside_Up
                    else
                      Data := On_Frame_Top;
                End
              else
                If ( Point.Column > Right_Of_Window^ )
                  then
                    Begin
                      If ( Point.Column > Succ( Right_Of_Window^ ) )
                        then
                          Data := Outside_Right
                        else
                          Data := On_Frame_Right;
                    End
                  else
                    If ( Point.Column < Left_Of_Window^ )
                      then
                        Begin
                          If ( Point.Column < Pred( Left_Of_Window^ ) )
                            then
                              Data := Outside_Left
                            else
                              Data := On_Frame_Left;
                        End
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display text pointer.
    While in text mode, this procedure puts a
    block on the screen to simulate the display
    pointer.  It works better than the default
    system because it allows data to be written
    to the screen without turning off the cursor
    first.

*************************************************)

    Procedure Display_Text_Pointer( Row, Column: Byte );
      Var
        Character: Char;
        Attribute: Byte;
      Begin
        Inc( Row );
        Inc( Column );
        Get_Character_From_Screen( Last_Column, Last_Row, Character, Attribute );
        If ( Character = Last_Character ) and ( Attribute = Last_New_Attribute )
          then
            Begin
              If ( Row = Last_Row ) and ( Column = Last_Column )
                then
                  Exit;
              Put_Character_On_Screen( Last_Column, Last_Row, Character, Last_Attribute );
            End;
        Last_Row := Row;
        Last_Column := Column;
        Get_Character_From_Screen( Last_Column, Last_Row, Last_Character, Last_Attribute );
        Last_New_Attribute := Reverse( Last_Attribute );
        Put_Character_On_Screen( Last_Column, Last_Row, Last_Character, Last_New_Attribute );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove text pointer.
    This procedure removes the text mode pointer
    from the screen.

*************************************************)

    Procedure Remove_Text_Pointer;
     {$IFNDEF VER50}
      Far;
     {$ENDIF}
      Var
        Character: Char;
        Attribute: Byte;
      Begin
        Get_Character_From_Screen( Last_Column, Last_Row, Character, Attribute );
        If ( Character = Last_Character )
          then
            Put_Character_On_Screen( Last_Column, Last_Row, Character, Last_Attribute );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Evaluate pointer.
    This procedure determines if the pointer is
    within the current screen window or outside
    of it.  In either case, it takes the
    appropriate measures.

*************************************************)

    Procedure Evaluate_Pointer( Var Point: Pointer_Data_Type; Var Data: Byte );
      Begin
        Display_Text_Pointer( Point.Row, Point.Column );
        If ( Point.Row >= Top_Of_Window^ ) and ( Point.Row <= Bottom_Of_Window^ ) and
           ( Point.Column >= Left_Of_Window^ ) and ( Point.Column <= Right_Of_Window^ )
          then
            Begin
              If ( Point.Button1 or Point.Button2 )
                then
                  Begin
                    If not Move( Point, Data )
                      then
                        Begin
                          If Point.Button1 and ( not Down_Button1 )
                            then
                              Begin
                                Data := Button1_Down;
                                Down_Button1 := True;
                              End;
                          If Point.Button2 and ( not Down_Button2 )
                            then
                              Begin
                                Data := Button2_Down;
                                Down_Button2 := True;
                              End;
                        End;
                  End
                else
                  Begin
                    If ( Down_Button1 )
                      then
                        Data := Button1_Up;
                    Down_Button1 := False;
                    If ( Down_Button2 )
                      then
                        Data := Button2_Up;
                    Down_Button2 := False;
                  End;
            End
          else
            Begin
              Down_Button1 := False;
              Down_Button2 := False;
              If ( Point.Button1 or Point.Button2 )
                then
                  OutSide( Point, Data );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Look for clicks.
    This procedure checks for when double clicks

*************************************************)

    Procedure Look_For_Clicks( Var Data: Byte );
      Var
        Hours,
        Minutes,
        Seconds,
        Hundredths: Word;
        New_Time: LongInt;
      Begin
        If ( Data in [ Button1_Up, Button2_Up ] )
          then
            Begin
              GetTime( Hours, Minutes, Seconds, Hundredths );
              New_Time := ( ( Minutes * 6000 ) + ( Seconds * 100 ) + Hundredths );
              If ( Data = Previous_Event2 ) and ( Previous_Event in [ Button1_Down, Button2_Down ] )
                then
                  If ( ( New_Time - Old_Time_Last_Event ) < Time_Delay )
                    then
                      If ( Data = Button1_Up )
                        then
                          Data := Button1_Double
                        else
                          Data := Button2_Double;
              Old_Time_Last_Event := New_Time;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: New interface.
    This function is the routine that allows the
    pointer to generate keystrokes for the
    Keyboard unit.

*************************************************)

    Function New_Interface: Byte;
     {$IFNDEF VER50}
      Far;
     {$ENDIF}
      Var
        Data: Byte;
        Point: Pointer_Data_Type;
      Begin
        Data := Old_Interface;
        If ( Data = 0 )
          then
            Begin
              Internal_Read_Pointer( Point );
              Evaluate_Pointer( Point, Data );
              If ( Data <> 0 )
                then
                  Begin
                    Look_For_Clicks( Data );
                    Previous_Event2 := Previous_Event;
                    Previous_Event := Data;
                  End;
            End;
        New_Interface := Data;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Set the pointer for default settings.
    Hook up the new interface.
    Initialize the click parameters.

*************************************************)

  Begin
    Set_for_Pointer;
    Find_Pointer := Check_For_Pointer;
    If Find_Pointer
      then
        Begin
         {$IFNDEF VER40}
          Old_Interface := Alternative_Input;
          Alternative_Input := New_Interface;
          Erase_Pointer := Remove_Text_Pointer;
         {$ENDIF}
          Previous_Event := 0;
          Old_Time_Last_Event := 0;
          Down_Button1 := False;
          Down_Button2 := False;
        End;
  End.

