{$O-} { Compiler Directive: Generate overlay code: Off } { DO NOT CHANGE }

(*****************************************************************************

  Music
    version 1.1

  This unit contains an easy to use background music playing system.

  Purpose:
    This unit allows the calling program to play music on the computer's
    built in speaker without having to worry about the overhead.

  How it works:
    The music is programmed into a circular queue so that it will continue
    playing until directed to stop.
    DOS:  The music playing is done in the background by utilizing the
          computer's timing interrupt.  This allows the music procedure to
          gain control 18.2 times per second.
    OS/2: The music is handled entirely by another thread.  This allows the
          system to handle the music as an entirely seperate case.

  Features:
    The procedure does not effect the computer's timing mechanism and thus
    is fully compatible with all pop-up utilities and multitasking
    operating systems.

  Versions:
    1.0  -  The original music system.
    1.1  -  Extended the code to run under OS/2.

  Copyright 1990, 1996, All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0.
    Speed Pascal/2 version Beta 4.

  Systems:
    MS-DOS, MDOS, OS/2.

*****************************************************************************)

Unit Music;

  Interface

    Uses
      DOS,
      CRT;

(***********************************************************

  Notes.
    The notes are defined here to make programming of music
    easier.  Each note will generate the corresponding
    frequency on the computer's internal speaker.

***********************************************************)

    Type
      Notes = ( C0, CN0, D0, DN0, E0, F0, FN0, G0, GN0, A0, AN0, B0,
                C1, CN1, D1, DN1, E1, F1, FN1, G1, GN1, A1, AN1, B1,
                C2, CN2, D2, DN2, E2, F2, FN2, G2, GN2, A2, AN2, B2,
                C3, CN3, D3, DN3, E3, F3, FN3, G3, GN3, A3, AN3, B3,
                C4, CN4, D4, DN4, E4, F4, FN4, G4, GN4, A4, AN4, B4,
                C5, CN5, D5, DN5, E5, F5, FN5, G5, GN5, A5, AN5, B5,
                C6, CN6, D6, DN6, E6, F6, FN6, G6, GN6, A6, AN6, B6,
                C7, CN7, D7, DN7, E7, F7, FN7, G7, GN7, A7, AN7, B7,
                C8 );

(***********************************************************

  Procedure: Add a note to the music queue.

    This procedure is one of the procedures for adding notes
    to the music system.  Notes must be one of the
    predefined values while Length is in seconds.

***********************************************************)

    Procedure Add_Note_To_Music( New_Note: Notes; Length: Real );

(***********************************************************

  Procedure: Add sound to the music queue.

    This procedure will add any frequency to the music
    queue.  It is the primary procedure for the music system
    and is slightly more versatile than the note adding
    procedure.  Length is in seconds.

***********************************************************)

    Procedure Add_Sound_To_Music( Frequency: Word; Length: Real );

(***********************************************************

  Procedure: Clear the music.

    This procedure will stop any music playing and clear the
    music queue for new music to be added, or for memory
    reclaiming.

***********************************************************)

    Procedure Clear_Music;

(***********************************************************

  Procedure: Start the music.

    This procedure begins the playing of the music in the
    music queue. It must be called to begin playing of music
    after the music tones is placed in the music queue.
    Music begins where it was left off.

***********************************************************)

    Procedure Start_Music;

(***********************************************************

  Procedure: Stop the music.

    This procedure stops the playing of the notes in the
    music queue.  The music will halt until Start_Music is
    executed.  The music queue will remain intact.

***********************************************************)

    Procedure Stop_Music;

{----------------------------------------------------------------------------}

  Implementation

    Const
     {$IFNDEF OS2}
      Adjustment = 18.2;
     {$ELSE}
      Adjustment = 1000;
     {$ENDIF}
      { This is an array holding all the default frequencies for the notes. }
      Frequencies: array[ C0 .. C8 ] of Word =
        (   16,   17,   18,   19,   21,   22,   23,   25,   26,   28,   29,   31,
            33,   35,   37,   39,   41,   44,   46,   49,   52,   55,   58,   62,
            65,   69,   73,   78,   82,   87,   93,   98,  104,  110,  117,  123,
           131,  139,  147,  156,  165,  175,  185,  196,  208,  220,  233,  247,
           262,  277,  294,  311,  330,  349,  370,  392,  415,  440,  466,  494,
           523,  554,  587,  622,  659,  698,  740,  784,  831,  880,  932,  988,
          1047, 1109, 1175, 1245, 1329, 1397, 1480, 1568, 1661, 1760, 1865, 1976,
          2093, 2217, 2349, 2489, 2637, 2794, 2960, 3136, 3322, 3520, 3729, 3951,
          4186 );

    Type
     {$IFNDEF OS2}
      Hold_Type = Word;
     {$ELSE}
      Hold_Type = LongWord;
     {$ENDIF}
      { This is the music queue structure. }
      Note_Node_Pointer = ^Note_Node_Record;
      Note_Node_Record =  Record
                            Note,
                            Length: Hold_Type;
                            Next,
                            Previous: Note_Node_Pointer;
                          End;

    Var
      { This holds the location of the music queue. }
      Chain_Pointer,
      { This points to the current note on the music queue. }
      Music_Pointer: Note_Node_Pointer;
      { This holds the current count of the music timer. }
      Music_Count: Word;
      { This holds the location of the old timer interrupt procedure. }
      Old_Interrupt,
      { This holds the location of the old exit procedure. }
      Exit_Interrupt: Pointer;
     {$IFDEF OS2}
      Thread_Id: LongWord;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Start music.
    As previously defined.

*************************************************)

    Procedure Start_Music;
      Begin
        Music_Count := 0;
        Music_Pointer := Chain_Pointer;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Stop music.
    As previously defined.

*************************************************)

    Procedure Stop_Music;
      Begin
        Music_Pointer := Nil;
       {$IFNDEF OS2}
        NoSound;
       {$ELSE}
        Beep( 100000, 1 );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add to chain.
    This procedure adds the note to a node on the
    circular music queue at the end of the music
    queue.

*************************************************)

    Procedure Add_To_Chain( New_Note, New_Length: Hold_Type );
      Var
        Temporary_Pointer: Note_Node_Pointer;
      Begin
        If ( Chain_Pointer = Nil )
          then
            Begin
              New( Chain_Pointer );
              With Chain_Pointer^ Do
                Begin
                  Next := Chain_Pointer;
                  Previous := Chain_Pointer;
                  Note := New_Note;
                  Length := New_Length;
                End;
            End
          Else
            Begin
              New( Temporary_Pointer );
              With Temporary_Pointer^ Do
                Begin
                  Note := New_Note;
                  Length := New_Length;
                  Next := Chain_Pointer^.Next;
                  Previous := Chain_Pointer^.Next^.Previous;
                End;
              Chain_Pointer^.Next^.Previous := Temporary_Pointer;
              Chain_Pointer^.Next := Temporary_Pointer;
              Chain_Pointer := Temporary_Pointer;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove from chain.
    This procedure removes the current music note,
    the one on the top of the music queue, from
    the music note queue.

*************************************************)

    Procedure Remove_From_Chain;
      Var
        Temporary_Pointer: Note_Node_Pointer;
      Begin
        If ( Chain_Pointer^.Next <> Chain_Pointer )
          then
            Begin
              Temporary_Pointer := Chain_Pointer;
              Chain_Pointer := Chain_Pointer^.Previous;
              Temporary_Pointer^.Next^.Previous := Chain_Pointer;
              Chain_Pointer^.Next := Temporary_Pointer^.Next;
              Dispose( Temporary_Pointer );
            End
          Else
            Begin
              Dispose( Chain_Pointer );
              Chain_Pointer := Nil;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Clear music.
    As previously defined.

*************************************************)

    Procedure Clear_Music;
      Var
        Temporary_Pointer: Note_Node_Pointer;
      Begin
        Stop_Music;
        While ( Chain_Pointer <> Nil ) Do
          Remove_From_Chain;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add note to music.
    As previously defined.

*************************************************)

    Procedure Add_Note_To_Music( New_Note: Notes; Length: Real );
      Var
        The_Length,
        The_Frequency: Hold_Type;
        Music_On: Boolean;
      Begin
        Music_On := ( Music_Pointer <> Nil );
        If Music_On
          then
            Stop_Music;
        The_Length := Round( Length * Adjustment );
        The_Frequency := Frequencies[ New_Note ];
        Add_To_Chain( The_Frequency, The_Length );
        If Music_On
          then
            Start_Music;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add sound to music.
    As previously defined.

*************************************************)

    Procedure Add_Sound_To_Music( Frequency: Word; Length: Real );
      Var
        The_Length: Hold_Type;
        Music_On: Boolean;
      Begin
        Music_On := ( Music_Pointer <> Nil );
        If Music_On
          then
            Stop_Music;
        The_Length := Round( Length * Adjustment );
        Add_To_Chain( Frequency, The_Length );
        If Music_On
          then
            Start_Music;
      End;

{----------------------------------------------------------------------------}

{$IFNDEF OS2}

(*************************************************

  Procedure: New interrupt.
    This interrupt replaces the old timer
    interrupt with the new one.  Then it plays the
    particular note which it's pointing to on the
    music queue if the time is up for the last
    note.

*************************************************)

    Procedure New_Interrupt; Interrupt;
      Begin
        If ( Music_Pointer <> Nil )
          then
            Begin
              Inc( Music_Count );
              If ( Music_Count > Music_Pointer^.Length )
                then
                  Begin
                    Music_Pointer := Music_Pointer^.Next;
                    Music_Count := 0;
                    Sound( Music_Pointer^.Note );
                  End;
            End;
      End;

{$ELSE}

(*************************************************

  Procedure: Play_Music.
    This procedure is a thread that plays the note
    queue for OS/2.  Since I can't implement an
    interrupt with OS/2, I rewrote the interrupt
    routine to operate as this thread.

*************************************************)

    Procedure Play_Music; Far;
      Begin
        Repeat
          If ( Music_Pointer <> Nil )
            then
              Begin
                Music_Pointer := Music_Pointer^.Next;
                Beep( Music_Pointer^.Note, Music_Pointer^.Length );
              End;
        Until False;
      End;

{$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: New exit procedure.
    This procedure is placed in the exit procedure
    queue to replace the old interrupts or stop 
    the thread if something goes wrong.  That way, 
    the interrupt will be kept current and the 
    system will not crash or hang.

*************************************************)

   {$F+}  { Generate far procedure calls: On } { Do not change! }
    Procedure New_Exit_Procedure;
      Begin
        ExitProc := Exit_Interrupt;     { Restore previous exit procedure }
       {$IFNDEF OS2}
        SetIntVec( $1C, Old_Interrupt ); { Restore old interrupt vector }
        NoSound;
       {$ELSE}
        KillThread( Thread_Id );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Initialize the music queue.
    Replace the interrupt or start the thread.
    Replace the exit procedure.

*************************************************)

  Begin
    Chain_Pointer := Nil;
    Music_Pointer := Nil;
   {$IFNDEF OS2}
    GetIntVec( $1C, Old_Interrupt );  { Save "Old" interrupt }
    SetIntVec( $1C, @New_Interrupt ); { Install new interrupt }
   {$ELSE}
    StartThread( @Play_Music, 3000, Nil, Thread_Id );
   {$ENDIF}
    Exit_Interrupt := ExitProc;      { Save old exit procedure }
    ExitProc := @New_Exit_Procedure; { Install new exit procedure }
  End.

