{$F+} { Generate far procedure calls: On } { Do not Change! }
{$B-} { Generate complete boolean evaluation: Off } { Do not Change! }

(*****************************************************************************

  FileMem
    version 2.4

    This Unit is designed to simulate memory ( Virtual memory ) using a file.
      It uses the fastest possible resources to access the simulated memory
      in the file, but due to file management overhead, accessing the file may
      be much slower than using actual memory.
    To reduce the access time, this version which succeeds Disk, introduces
      disk buffering in conventional memory.  Processing time is greatly
      reduced by using the heap area for holding the last accessed allocation.
      Using a last served policy, only the most recently accessed areas are
      stored in memory.

    Purpose:
      To allow maximum use of file buffering with minimum amount of overhead
      and worry.

    How it works:
      First, space is allocated in the file with the New_Pointer function.
        The amount of space is determined by the value passed in to the
        allocate procedure.  The allocate procedure will then return a
        pointer that points to the allocated file space.
      Next, the allocated space is initialized by the Put_Data procedure.
      Next, the allocated space can be accessed by the Get_Data procedure.
      Lastly, when the space is no longer used, the pointer can be
        deallocated using the dispose procedure.

    Features:
      With a long Integer pointer type, the size of available virtual memory
        is almost unlimited.  The maximum amount is only limited by available
        disk space or the maximum value of pointer. ( roughly 2 gigabytes )
      Disposing of allocated memory is easy because the unit keeps track of
        the size of allocated units.  And all deallocated memory is tracked
        for later possible use.
      New file space is only reserved when none of available file space is
        able to handle the new allocated size.
      Chunks of data are allocated in sizes ranging from the minimum size
        { 6 - bytes } to the maximum size { maximum value of a word - 2. }
      Allocation and deallocation are handled by maximum performance code.
        Unfortunately, due to the nature of file accessing, performance can
        be slowed with large deallocation lists.
      The temporary file is created before execution of the program begins
        using the default file name.
      Memory is automatically released to the system through the HeapError
        function.  This makes the use of memory almost invisible to the
        application program except for the slight time delay when the memory
        is being deallocated.

    Limitations:
      Manipulation of the specified pointers is definitely not recommended
        since pointers are verified before use.

    Versions:
      2.0 - Added the memory buffering system to the fileMem file system.
      2.1 - Added the Put_New_Pointer function.
      2.2 - Added the Peek_Data and Peek_Buffer_Data functions.
      2.3 - Added the Generate_Unique_File switch.
      2.31 - Corrected file list bug that caused location confusion.
      2.32 - Added code to check correct drive for free space.
      2.33 - Added optional code to test for data integrity.
      2.4 - Added support for Speed Pascal/2 beta 4.
      2.41 - Altered code to compiler under Speed Pascal/2 version 1.5.

    Copyright 1992, 1994, 1996, all rights reserved.
      Paul R. Renaud.

    Compilers:
      Turbo Pascal versions 4.0 to 6.0
      Speed Pascal/2 version 1.5 Beta 4 - Field Test 1.

    Systems:
      MS-DOS, MDOS, OS/2.

*****************************************************************************)

Unit FileMem;

  Interface

    Uses
      DOS,
     {$IFNDEF OS2}
      Generate_FileName;
     {$ELSE}
      Generate;
     {$ENDIF}

(***********************************************************

  This switch turns the file buffering feature on when set
    to true.  Otherwise the code does not include support
    for the buffering feature.

***********************************************************)

    Const
      Allow_Buffering = True;

(***********************************************************

  This switch turns the file integrity testing code on when
    set to true.  Otherwise the code does not include the
    extra code to test file integrity.

***********************************************************)

      Integrity_Check = False;

(***********************************************************

  This switch turns the immediate write feature on.  When
    set to true all updates take place immediately,
    otherwise they only occur when the memory copy is to be
    flushed.

***********************************************************)

      Immediate_Write = False;

(***********************************************************

  This switch alters the method used to name the temporary
    file.  When set to true the system finds a unique file
    name for the temporary file, when false is uses a
    standard file name.

***********************************************************)

      Generate_Unique_File = True;

(***********************************************************

  This is the default file name for the temporary file.
    This value is used only when the Generate_Unique_File
    switch is set to false.

***********************************************************)

      File_Name = 'FileVMem.$$$';

(***********************************************************

  Definition of the file memory address pointer used to
    uniquely identify a part of the memory map.

***********************************************************)

    Type
      Pointer_Type = LongInt;

(***********************************************************

  This type is used to define the size of the memory blocks
    allowed.

***********************************************************)

     {$IFNDEF OS2}
      Word_Type = Word;
     {$ELSE}
      Word_Type = LongWord;
     {$ENDIF}

(***********************************************************

  Function:  New pointer. (Allocate data block)

    This function attempts to allocate the specified amount
    of space on disk and returns a pointer to reference
    that space.  If there is available space already
    available, this function attempts to find it before new
    disk space is reserved.  If there is no disk space, a
    nul pointer is returned.

***********************************************************)

    Function New_Pointer( Size: Word_Type ): Pointer_Type;

(***********************************************************

  Function:  Put new pointer.  (Allocate and initialize data
                                block)

    This function attempts to allocate the specified amount
    of space on disk and returns a pointer to reference that
    space.  If there is available space already available,
    this function attempts to find it before new disk space
    is reserved.  If there is no disk space, a nul pointer
    is returned.

***********************************************************)

    Function Put_New_Pointer( Size: Word_Type; Var Data ): Pointer_Type;

(***********************************************************

  Procedure: Put data on disk.

    This procedure transfers the specified data to the disk
    file.  The specified size must be smaller or equal to
    the size of the allocated data block.

***********************************************************)

    Procedure Put_Data( Address: Pointer_Type; Var Data; Size: Word_Type );

(***********************************************************

  Procedure: Get data from disk.

    This procedure transfers the specified data from the disk
    file into the variable.  The specified size must be
    smaller or equal to the size of the allocated data
    block.

***********************************************************)

    Procedure Get_Data( Address: Pointer_Type; Var Data; Size: Word_Type );

(***********************************************************

  Procedure: Get buffer data from disk.

    This procedure transfers up to the specified data from
    the disk file into the variable.  The specified size
    must be greater or equal to the size of the allocated
    data block.

***********************************************************)

    Procedure Get_Buffer_Data( Address: Pointer_Type; Var Data; Size: Word_Type );

(***********************************************************

  Function: Despose the pointer. (Deallocate data block)

    This function takes a pointer featuring an allocated
    data block and puts it on the unneeded data block list
    for later possible reuse.  After this function is
    called, that data block must be consider invalid
    although it may be possible to read the data from that
    block before any more blocks are reallocated.

***********************************************************)

    Function Dispose_Pointer( Var Pointer: Pointer_Type ): Boolean;

(***********************************************************

  Procedure:  Dump deallocated listings.

    This procedure outputs to the specified file, a listing
    of all the unused data blocks that have been disposed of
    and their respective sizes.  This procedure is intended
    only for debugging purposes only.

***********************************************************)

    Procedure Dump_Deallocated_Listings( Var Out_File: Text );

(***********************************************************

  Procedure:  Flush the buffers.

    This procedure flushes out the memory buffers regard-
    less of the contents.  It comes in very handy when all
    the memory space used by this unit must be reclaimed.
    Succeeding calls to the routines, will begin allocating
    memory when it's available.

***********************************************************)

    Procedure Flush_Buffers;

(***********************************************************

  Procedure: Peek data from disk.

    This procedure transfers the specified data from the disk
    file into the variable with the exception that the data
    isn't stored in the memory buffer.  The specified size
    must be smaller or equal to the size of the allocated
    data block.

***********************************************************)

    Procedure Peek_Data( Address: Pointer_Type; Var Data; Size: Word_Type );

(***********************************************************

  Procedure: Peek buffer data from disk.

    This procedure transfers up to the specified data from
    the disk file into the variable with the exception that
    the data isn't stored in the memory buffer.  The
    specified size must be greater or equal to the size of
    the allocated data block.

***********************************************************)

    Procedure Peek_Buffer_Data( Address: Pointer_Type; Var Data; Size: Word_Type );

{----------------------------------------------------------------------------}

  Implementation

   {$DEFINE NoDebug}  { Used only for debugging this unit. }
   {$DEFINE Quick}  { Allows alternate use of code to speed processing. }

    Const
     { Signifies end of the pointer list. }
      Nul = -1;
     { Limit to the size of the memory buffer.  Should range from 2 to 5000. }
      List_Limit = 3000;
     { Limit to the amount of records per list.  Should range from 2 to 10000. }
      File_List_Limit = 300;
     {$IFNDEF Ver40 }
      Minimum_Size = SizeOf( Pointer_Type );
     {$ELSE}
      Minimum_Size = 4;
     {$ENDIF}
      Integrity_Word: Word = $12CD;

    Type
     { Signifies disk block size.  Warning for OS/2, this number should
       be significantly less than the total amount of memory used by the
       program since the operating system will not allow references to
       memory out of your allotment.   Also, using a value less than 66000
       with OS/2 will cause the program to crash. }
     {$IFNDEF OS2}
      Size_Type = 0 .. 65534;
     {$ELSE}
      Size_Type = 0 .. 100000; { This works best for OS/2. }
     {$ENDIF}
     { Signifies the current buffer state. }
      Status_Type = ( Status_Read, Status_Write );
     { Signifies memory list range. }
     {$IFNDEF Ver40 }
      List_Range = 0 .. Succ( File_List_Limit );
     {$ELSE}
      List_Range = Word;
     {$ENDIF}
     { Signifies the data variable. }
      Data_Type = Packed array[ Size_Type ] of Char;
     { Disk link type used for block management. }
      Link_Type = Packed Record
                           Size: Size_Type;
                           Pointer: Pointer_Type;
                         End;
     { List type for memory management. }
      File_List_Type = Packed Record
                                Amount: List_Range;
                                Data: Packed array[ 1 .. File_List_Limit ] of Link_Type;
                                Next: Pointer_Type;
                              End;
     { Pointer to the free list storage area. }
      File_Pointer_Type = ^File_List_Type;
     { Defines the limits of the internal buffer list. }
      Amount_Range = 0 .. List_Limit;
     { Defines the data necessary to manage an internal buffer. }
      Memory_Pointer_Type = Packed Record
                                     Address: Pointer_Type;
                                     Memory_Pointer: Pointer;
                                     Memory_Size: Word_Type;
                                     Status: Status_Type;
                                   End;
     { Defines the internal buffer list structure. }
      Memory_List = Packed array[ 1 .. List_Limit ] of Memory_Pointer_Type;
     { Defines the internal buffer structure. }
      Memory_List_Type = Packed Record
                                  Amount: Amount_Range;
                                  List: Memory_List;
                                End;

    Var
     { Holds the old heap routine. }
      Old_Heap_Pointer,
     { Holds the old exit routine. }
      Old_Exit_Procedure: Pointer;
     { File for disk management. }
      Memory_File: File;
     { Used to speed processing. }
      Free_Amount: LongInt;
     { Points to the memory block list. }
      Free_Pointer,
     { Points to the current list block. }
      Current_Pointer: Pointer_Type;
     { Holds the memory block list. }
      Free_List: File_Pointer_Type;
     { Holds the memory buffer. }
      Memory: ^Memory_List_Type;
     { Holds the default drive number }
      Default_Drive: Byte;

    {$IFNDEF VER40 }
     { This variable function allows for deallocating buffer memory on demand. }
      Old_Heap_Function: Function( Size: Word_Type ): Integer absolute Old_Heap_Pointer;
    {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Check for error.
    This procedure checks for any error type.  If
    an error occurs, it displays the procedure in
    which the error occurred and what type of error
    it most likely was.  Then the program is
    halted.

*************************************************)

    Procedure Check_Error( IOResult: Integer; Sentence: String );
      Begin
        If ( IOResult <> 0 )
          then
            Begin
              WriteLn( 'Error in ', Sentence, '.' );
             {$IFDEF OS2}
              WriteLn( 'Press enter to continue.' );
              ReadLn;
             {$ENDIF}
              Halt( IOResult );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate list.
    This function allocates a list block.  If
    there isn't any more disk space, this function
    returns a nul pointer.  It writes in stuff to
    initialize the disk space.

*************************************************)

    Function Allocate_List: Pointer_Type;
      Var
        New_Size: LongInt;
        Fresh_Pointer: Pointer_Type;
        Data: File_List_Type;
      Begin
        New_Size := SizeOf( File_List_Type );
        If ( DiskFree( Default_Drive ) >= New_Size )
          then
            Begin
              Fresh_Pointer := FileSize( Memory_File );
             {$IFDEF Debug}
              FillChar( Data, New_Size, 0 );
             {$ENDIF}
             {$I-}
              Seek( Memory_File, Fresh_Pointer );
              Check_Error( IOResult, 'Allocate: Address out of range' );
              If Integrity_Check
                then
                  Begin
                    BlockWrite( Memory_File, Integrity_Word, SizeOf( Integrity_Word ) );
                    Check_Error( IOResult, 'Allocate: Integrity 1 write error' );
                  End;
              BlockWrite( Memory_File, Data, New_Size );
              Check_Error( IOResult, 'Allocate: File write error' );
              If Integrity_Check
                then
                  Begin
                    BlockWrite( Memory_File, Integrity_Word, SizeOf( Integrity_Word ) );
                    Check_Error( IOResult, 'Allocate: Integrity 2 write error' );
                  End;
             {$I+}
            End
          else
            Fresh_Pointer := Nul;
        Allocate_List := Fresh_Pointer;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate and initialize.
    This function allocates a block of the given
    size.  If there isn't any more disk space,
    this function returns a nul pointer.  It will
    write the size in the file and also writes in
    the data to initialize the disk space.

*************************************************)

    Function Allocate_And_Initialize( New_Size: Size_Type; Var Data: Data_Type ): Pointer_Type;
      Var
        Fresh_Pointer: Pointer_Type;
      Begin
        If ( DiskFree( Default_Drive ) >= New_Size )
          then
            Begin
              Fresh_Pointer := FileSize( Memory_File );
             {$I-}
              Seek( Memory_File, Fresh_Pointer );
              Check_Error( IOResult, 'Allocate_And_Initialize: Address out of range' );
              BlockWrite( Memory_File, New_Size, SizeOf( Size_Type ) );
              Check_Error( IOResult, 'Allocate_And_Initialize: File write error' );
              BlockWrite( Memory_File, Data, ( New_Size - SizeOf( Size_Type ) ) );
              Check_Error( IOResult, 'Allocate_And_Initialize: File write error' );
             {$I+}
            End
          else
            Fresh_Pointer := Nul;
        Allocate_And_Initialize := Fresh_Pointer;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Perform Read.
    This procedure is a core procedure to read in
    data.  First it performs a seek to the memory
    location.  If the address is invalid, the
    program will halt.  Next it reads in the size
    from the file and compares it with the
    specified size.  If the specified size is
    greater than the allocated size, the procedure
    will fail and halt the program.  Otherwise,
    the data reading is attempted.

*************************************************)

    Procedure Perform_Read( Address: Pointer_Type; Var Buffer: Data_Type; Amount: Word_Type );
      Var
        Safe_Amount: Size_Type;
      Begin
       {$I-}
        Seek( Memory_File, Address );
        Check_Error( IOResult, 'Perform_Read: Address out of range' );
        BlockRead( Memory_File, Safe_Amount, SizeOf( Size_Type ) );
        Check_Error( IOResult, 'Perform_Read: Address uninitialized' );
        Safe_Amount := Safe_Amount - SizeOf( Size_Type );
        If ( Safe_Amount < Amount )
          then
            BlockRead( Memory_File, Buffer, Safe_Amount )
          else
            BlockRead( Memory_File, Buffer, Amount );
        Check_Error( IOResult, 'Perform_Read: File read error' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Read_Disk_Size.
    This function returns only the size of the
    allocated block at that address.

*************************************************)

    Function Read_Disk_Size( Address: Pointer_Type ): Word_Type;
      Var
        Data: Size_Type;
      Begin
       {$I-}
        Seek( Memory_File, Address );
        Check_Error( IOResult, 'Read_Disk_Size: Address out of range' );
        BlockRead( Memory_File, Data, SizeOf( Size_Type ) );
        Check_Error( IOResult, 'Read_Disk_Size: Address uninitialized' );
       {$I+}
        Read_Disk_Size := ( Data - SizeOf( Size_Type ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Perform Write.
    This procedure is the core procedure to write
    out data.  First it performs a seek to the
    memory location.  If the address is invalid,
    the program will halt.  Next it reads in the
    size from the file and compares it with the
    specified size.  If the specified size is
    greater than the allocated size, the procedure
    will fail and halt the program.  Otherwise,
    the data writing is attempted.

*************************************************)

    Procedure Perform_Write( Address: Pointer_Type; Var Buffer: Data_Type; Amount: Word_Type );
      Var
        Safe_Amount: Size_Type;
      Begin
       {$I-}
        Seek( Memory_File, Address );
        Check_Error( IOResult, 'Perform_Write: Address out of range' );
        BlockRead( Memory_File, Safe_Amount, SizeOf( Size_Type ) );
        Check_Error( IOResult, 'Perform_Write: Address uninitialized' );
        If ( Safe_Amount < Amount )
          then
            Begin
             {$IFDEF Debug}
              WriteLn;
              WriteLn( 'Error: Safe_Amount = ', Safe_Amount, ':', SizeOf( Safe_Amount ) );
              WriteLn( 'Error: Amount = ', Amount, ':', SizeOf( Amount ) );
             {$ENDIF}
              Check_Error( 201, 'Perform_Write: Amount is incompatible with memory block size' );
            End;
        BlockWrite( Memory_File, Buffer, Amount );
        Check_Error( IOResult, 'Perform_Write: File write error' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read Link.
    This procedure is a core procedure used to
    read in the data block size.

*************************************************)

    Procedure Read_Link( Address: Pointer_Type; Var Link: Link_Type );
      Begin
       {$I-}
        Seek( Memory_File, Address );
        Check_Error( IOResult, 'Read_Link: Address out of range' );
        BlockRead( Memory_File, Link, SizeOf( Link_Type ) );
        Check_Error( IOResult, 'Read_Link: Address uninitialized' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read List.
    This procedure is a core procedure used to
    read in the block managing list.

*************************************************)

    Procedure Read_List( Where: Pointer_Type; Var List: File_List_Type );
      Var
        Test_Word: Word;
      Begin
        Seek( Memory_File, Where );
        Check_Error( IOResult, 'Read_List: Address out of range' );
        If Integrity_Check
          then
            Begin
              BlockRead( Memory_File, Test_Word, SizeOf( Test_Word ) );
              Check_Error( IOResult, 'Read_List: Integrity 1 read error' );
              If ( Test_Word <> Integrity_Word )
                then
                  Begin
                    WriteLn;
                    WriteLn( 'Location = ', Where );
                    WriteLn( 'Test_Word = ', Test_Word );
                    Check_Error( 1, 'Read_List: First integrity word found corrupt' );
                  End;
            End;
        BlockRead( Memory_File, List, SizeOf( File_List_Type ) );
        Check_Error( IOResult, 'Read_List: Address uninitialized' );
        If Integrity_Check
          then
            Begin
              BlockRead( Memory_File, Test_Word, SizeOf( Test_Word ) );
              Check_Error( IOResult, 'Read_List: Integrity 2 read error' );
              If ( Test_Word <> Integrity_Word )
                then
                  Begin
                    WriteLn;
                    WriteLn( 'Location = ', Where );
                    WriteLn( 'Test_Word = ', Test_Word );
                    Check_Error( 1, 'Read_List: Last integrity word found corrupt' );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write List.
    This procedure is a core procedure used to
    write out the block managing list.

*************************************************)

    Procedure Write_List( Where: Pointer_Type; Var List: File_List_Type );
      Var
        Test_Word: Word;
      Begin
        Seek( Memory_File, Where );
        Check_Error( IOResult, 'Write_List: Address out of range' );
        If Integrity_Check
          then
            Begin
              BlockRead( Memory_File, Test_Word, SizeOf( Test_Word ) );
              Check_Error( IOResult, 'Write_List: Integrity 1 read error' );
              If ( Test_Word <> Integrity_Word )
                then
                  Begin
                    WriteLn;
                    WriteLn( 'Location = ', Where );
                    WriteLn( 'Test_Word = ', Test_Word );
                    Check_Error( 1, 'Write_List: First integrity word found corrupt' );
                  End;
            End;
        BlockWrite( Memory_File, List, SizeOf( File_List_Type ) );
        Check_Error( IOResult, 'Write_List: Address uninitialized' );
        If Integrity_Check
          then
            Begin
              BlockRead( Memory_File, Test_Word, SizeOf( Test_Word ) );
              Check_Error( IOResult, 'Write_List: Integrity 2 read error' );
              If ( Test_Word <> Integrity_Word )
                then
                  Begin
                    WriteLn;
                    WriteLn( 'Location = ', Where );
                    WriteLn( 'Test_Word = ', Test_Word );
                    Check_Error( 1, 'Write_List: Last integrity word found corrupt' );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Switch list.
    This procedure takes care of switching the
    block list.

*************************************************)

    Procedure Switch_List( Where: Pointer_Type; Var List: File_List_Type );
      Begin
        If ( Current_Pointer <> Where )
          then
            Begin
              Write_List( Current_Pointer, List );
              Current_Pointer := Where;
              Read_List( Current_Pointer, List );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate and update the list.
    This function will allocate space on the disk
    for the list, then performs all the necessary
    initialization.  It returns false if it fails.

*************************************************)

    Function Allocate_And_Update_List( Var List: File_List_Type; Where: Pointer_Type; Hold: Link_Type ): Boolean;
      Begin
        List.Next := Allocate_List;
        If ( List.Next <> Nul )
          then
            Begin
              Write_List( Where, List );
              Where := List.Next;
             {$IFDEF Debug}
              FillChar( List, SizeOf( File_List_Type ), 0 );
             {$ENDIF}
              List.Next := Nul;
              List.Amount := 1;
              List.Data[ 1 ] := Hold;
              Allocate_And_Update_List := True;
              Current_Pointer := Where;
            End
          else
            Allocate_And_Update_List := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal get data.
    This function gets the data from the allocated
    buffer address.

*************************************************)

    Procedure Internal_Get_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Var
        Buffer: Data_Type absolute Data;
      Begin
        Perform_Read( Address, Buffer, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal put data.
    This procedure transfers the given data to the
    allocated file space.

*************************************************)

    Procedure Internal_Put_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Var
        Buffer: Data_Type absolute Data;
      Begin
        Perform_Write( Address, Buffer, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Inner scan.
    This procedure looks in the current memory
    block list for a memory block larger or equal
    to the specified size.  It will return the
    location in Look if it is found and will
    output it's success status in Found_It.  This
    procedure uses a binary search for maximum
    speed.  The list is expected to be sorted in
    order of size.

*************************************************)

    Procedure Inner_Scan( Var List: File_List_Type; Var Look: List_Range; Var Found_It: Boolean; New_Size: Size_Type );
      Var
        Top,
        Bottom: List_Range;
      Begin
        Top := 1;
        Bottom := List.Amount;
        Repeat
          Look := ( ( Top + Bottom ) div 2 );
          If ( List.Data[ Look ].Size < New_Size )
            then
              Top := Look
            else
              Bottom := Look
        Until ( Succ( Top ) >= Bottom );
        Look := Top;
        Found_It := ( List.Data[ Look ].Size >= New_Size );
        If ( not Found_It )
          then
            Begin
              Inc( Look );
              Found_It := ( Look <= List.Amount ) and ( List.Data[ Look ].Size >= New_Size );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function Find It.
    This function is designed to look for the
    memory block of the specified size in the list
    of memory block lists.  It uses a specially
    designed advanced look feature to speed up the
    process.  If the last block in the list is too
    small, the entire block is skipped.  The lists
    are expected to be sorted in size order.

*************************************************)

    Function Find_It( Var List: File_List_Type; Var Where: Pointer_Type; Var Look: List_Range; New_Size: Size_Type ):
                    Boolean;
      Var
        Found_It: Boolean;
        Search: Pointer_Type;
      Begin
        Found_It := False;
        Search := Free_Pointer;
        Repeat
          Switch_List( Search, List );
          If ( List.Amount > 0 ) and ( List.Data[ List.Amount ].Size >= New_Size )
            then
              Inner_Scan( List, Look, Found_It, New_Size )
            else
              Look := Succ( List.Amount );
          Where := Search;
          Search := List.Next;
        Until ( Search = Nul ) or Found_It;
        Find_It := Found_It;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get from list.
    This function is designed to remove the
    requested block of new_size in the memory
    block list if it finds one of the proper size.
    Otherwise, it will return the value, false.

*************************************************)

    Function Get_From_List( Var List: File_List_Type; Var Pointer: Pointer_Type; New_Size: Size_Type ): Boolean;
      Var
        Where: Pointer_Type;
        Look,
        Counter: List_Range;
      Begin
        Get_From_List := False;
        If ( Free_Amount >= 1 )
          then
            If Find_It( List, Where, Look, New_Size )
              then
                Begin
                  Pointer := List.Data[ Look ].Pointer;
                 {$IFDEF Quick}
                  If ( Look < File_List_Limit )
                    then
                      Move( List.Data[ Succ( Look ) ], List.Data[ Look ],
                            ( ( List.Amount - Look ) * SizeOf( Link_Type ) ) );
                 {$ELSE}
                  For Counter := Look to Pred( List.Amount ) do
                    List.Data[ Counter ] := List.Data[ Succ( Counter ) ];
                 {$ENDIF}
                 {$IFDEF Debug}
                  FillChar( List.Data[ List.Amount ], SizeOf( Link_Type ), 0 );
                 {$ENDIF}
                  Dec( List.Amount );
                  Get_From_List := True;
                  Dec( Free_Amount );
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Initialize new pointer.
    This function tries to get a pointer of the
    specified size off the free list.  If it
    can't, it attempts to allocate it.

*************************************************)

    Function Initialize_New_Pointer( Size: Word_Type; Var Data ): Pointer_Type;
      Var
        New_Size: Size_Type;
        Fresh_Pointer: Pointer_Type;
        New_Data: Data_Type absolute Data;
      Begin
        If ( Size < Minimum_Size )
          then
            Size := Minimum_Size;
        New_Size := ( Size + SizeOf( Size_Type ) );
        If Get_From_List( Free_List^, Fresh_Pointer, New_Size )
          then
            Begin
              Initialize_New_Pointer := Fresh_Pointer;
              Internal_Put_Data( Fresh_Pointer, Data, Size );
            End
          else
            Initialize_New_Pointer := Allocate_And_Initialize( New_Size, New_Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Push back and add.
    When a pointer is about to be added to the
    list, in order to keep them in assorted order,
    the list has to be reshaped.  In order to do
    that, the entire list is pushed backwards.  If
    the list is unfilled, then it is merely a
    question of pushing back the list and
    inserting the new block.  On the other hand,
    if the list is full, then the end of the
    current list has to be pushed back to the next
    one and the process has to be repeated until
    the very end.  This is done recursively.

*************************************************)

    Function Push_Back_Add( Var List: File_List_Type; Where: Pointer_Type; Look: List_Range; Link: Link_Type ): Boolean;

(***************************************

  Procedure: Slide Back.
    This procedure pushes the current
    list backwards and updates the
    opened space.

***************************************)

      Procedure Slide_Back( Var List: File_List_Type; Look, Bottom: List_Range; Link: Link_Type );
        Var
          Count: List_Range;
        Begin
         {$IFDEF Quick}
          If ( Look < File_List_Limit )
            then
              Move( List.Data[ Look ], List.Data[ Succ( Look ) ],
                    ( Succ( Bottom - Look ) * SizeOf( Link_Type ) ) );
         {$ELSE}
          For Count := Bottom downto Look do
            List.Data[ Succ( Count ) ] := List.Data[ Count ];
         {$ENDIF}
          List.Data[ Look ] := Link;
        End;

{------------------------------------------------}

      Var
        Hold: Link_Type;
      Begin
        If ( List.Amount < File_List_Limit )
          then
            Begin
              Slide_Back( List, Look, List.Amount, Link );
              Inc( List.Amount );
              Push_Back_Add := True;
            End
          else
            Begin
              Hold := List.Data[ File_List_Limit ];
              Slide_Back( List, Look, Pred( File_List_Limit ), Link );
              If ( List.Next = Nul )
                then
                  Push_Back_Add := Allocate_And_Update_List( List, Where, Hold )
                else
                  Begin
                    Where := List.Next;
                    Switch_List( Where, List );
                    Push_Back_Add := Push_Back_Add( List, Where, 1, Hold );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Put in list.
    This Function adds a memory block to the
    memory block list.  If the list is too small,
    it will allocate a new link to the list list
    and initialize it.  It returns false is some-
    thing goes wrong.

*************************************************)

    Function Put_In_List( Var List: File_List_Type; Var Pointer: Pointer_Type; Size: Size_Type ): Boolean;
      Var
        Look: List_Range;
        Search: Pointer_Type;
        Link: Link_Type;
      Begin
        If Find_It( List, Search, Look, Size )
          then
            Begin
              Read_Link( Pointer, Link );
              Link.Pointer := Pointer;
              Put_In_List := Push_Back_Add( List, Search, Look, Link );
            End
          else
            Begin
              Read_Link( Pointer, Link );
              Link.Pointer := Pointer;
              If ( Look <= File_List_Limit )
                then
                  Begin
                    List.Data[ Look ] := Link;
                    Inc( List.Amount );
                    Put_In_List := True;
                  End
                else
                  Put_In_List := Allocate_And_Update_List( List, Search, Link );
            End;
        Pointer := Nul;
        Inc( Free_Amount );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal dispose pointer.
    This function gets the address and deallocates
    it, putting the unused space in the free list
    for later possible use.

*************************************************)

    Function Internal_Dispose_Pointer( Var Pointer: Pointer_Type ): Boolean;
      Var
        Link: Link_Type;
      Begin
        Read_Link( Pointer, Link );
        Internal_Dispose_Pointer := Put_In_List( Free_List^, Pointer, Link.Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write out the list.
    This procedure writes out the list of memory
    blocks in an easy to read format.  It assumes
    to use the standard Input/Output file.

*************************************************)

    Procedure Write_Out_List( Var Out_File: Text; Var List: File_List_Type );
      Var
        Search: Pointer_Type;
        Link: Link_Type;
        Page: Byte;
        Count: List_Range;
      Begin
        Page := 0;
        Search := Free_Pointer;
        Repeat
          Switch_List( Search, List );
          For Count := 1 to List.Amount do
            With List.Data[ Count ] do
              Begin
                If ( Page = 0 )
                  then
                    WriteLn( Out_File, ' Size           Address' );
                Write( Out_File, Size:5, '       ' );
                If ( Pointer = Nul )
                  then
                    WriteLn( Out_File, '       Nul' )
                  else
                    WriteLn( Out_File, Pointer:10 );
                Inc( Page );
                If ( Page > 21 )
                  then
                    Begin
                      Page := 0;
                      Write( '{ More data to follow, press enter }' );
                      ReadLn;
                    End;
              End;
          Search := List.Next;
        Until ( Search = Nul );
        Write( '{ End of data, press enter }' );
        ReadLn;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dump deallocated listings.
    As previously defined.

*************************************************)

    Procedure Dump_deallocated_Listings( Var Out_File: Text );
      Begin
        WriteLn( 'Start of deallocated listings dump' );
        Write_Out_List( Out_File, Free_List^ );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: New exit procedure.
    This procedure is automatically called at the
    termination of the program to close and delete
    the temporary file we used for the disk
    memory.

*************************************************)

    Procedure New_Exit_Procedure;
      Begin
        ExitProc := Old_Exit_Procedure;
        Close( Memory_File );
        Erase( Memory_File );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure Dispose memory.
    This procedure disposes of the memory that
    the given memory buffer occupies.  If the
    delayed buffer write feature is enabled,
    the data is written to the disk before the
    memory is released.

*************************************************)

    Procedure Dispose_Memory( Var Memory: Memory_Pointer_Type; Save: Boolean );
      Begin
        If ( Memory.Memory_Pointer <> Nil )
          then
            Begin
              If ( ( not Immediate_Write ) and Save and ( Memory.Status = Status_Write ) )
                then
                  Internal_Put_Data( Memory.Address, Memory.Memory_Pointer^, Memory.Memory_Size );
              FreeMem( Memory.Memory_Pointer, Memory.Memory_Size );
              Memory.Address := 0;
              Memory.Memory_Pointer := Nil;
              Memory.Memory_Size := 0;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add memory.
    This procedure makes room, if necessary on the
    top of the buffer list for the new buffer.
    The other buffer pointers are pushed down and
    the last one on the list is removed if the
    list can't hold it.

*************************************************)

    Procedure Add_Memory( Var Memory: Memory_List_Type; Memory_Node: Memory_Pointer_Type );
      Var
        Count: Amount_Range;
      Begin
        If ( Memory.Amount = List_Limit )
          then
            Begin
              Dispose_Memory( Memory.List[ List_Limit ], True );
              Dec( Memory.Amount );
            End;
       {$IFDEF Quick}
        If ( Memory.Amount > 0 )
          then
            Move( Memory.List[ 1 ], Memory.List[ 2 ], ( Memory.Amount * SizeOf( Memory_Pointer_Type ) ) );
       {$ELSE}
        For Count := Memory.Amount downto 1 do
          Memory.List[ Succ( Count ) ] := Memory.List[ Count ];
       {$ENDIF}
        Inc( Memory.Amount );
        Memory.List[ 1 ] := Memory_Node;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Subtract memory.
    This procedure removes the specified buffer
    pointer from the list and pushes the sub-
    ceeding pointers upward.

*************************************************)

    Procedure Subtract_Memory( Var Memory: Memory_List_Type; Where: Amount_Range );
      Var
        Count: Word;
      Begin
        If ( Where <= Memory.Amount )
          then
            Begin
             Dispose_Memory( Memory.List[ Where ], False );
            {$IFDEF Quick}
             Count := ( Memory.Amount - Where );
             If ( Count > 0 )
               then
                 Move( Memory.List[ Succ( Where ) ], Memory.List[ Where ], ( Count * SizeOf( Memory_Pointer_Type ) ) );
            {$ELSE}
             For Count := Where to Pred( Memory.Amount ) do
               Memory.List[ Count ] := Memory.List[ Succ( Count ) ];
            {$ENDIF}
             Dec( Memory.Amount );
           End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Scan memory.
    This function searches the memory buffer list
    for the address matching the one supplied.
    If it finds it, it's location is returned,
    otherwise zero is returned.

*************************************************)

    Function Scan_Memory( Var Memory: Memory_List_Type; Address: Pointer_Type ): Amount_Range;
      Var
        Count: Amount_Range;
      Begin
        Scan_Memory := 0;
        For Count := 1 to Memory.Amount do
          If ( Memory.List[ Count ].Address = Address )
            then
              Begin
                Scan_Memory := Count;
                Exit;
              End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Rearrange memory.
    This procedure moves the given buffer pointer
    to the top of the buffer list.

*************************************************)

    Procedure Rearrange_Memory( Var Memory: Memory_List_Type; Where: Amount_Range );
      Var
        Count: Amount_Range;
        Hold: Memory_Pointer_Type;
      Begin
        If ( Where <> 0 )
          then
            Begin
              Hold := Memory.List[ Where ];
             {$IFDEF Quick}
              Count := Pred( Where );
              If ( Count > 0 )
                then
                  Move( Memory.List[ 1 ], Memory.List[ 2 ], ( Count * SizeOf( Memory_Pointer_Type ) ) );
             {$ELSE}
              For Count := Where downto 2 do
                Memory.List[ Count ] := Memory.List[ Pred( Count ) ];
             {$ENDIF}
              Memory.List[ 1 ] := Hold;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add to memory.
    This procedure adds the data to the memory
    buffer.  If there is no more room for it,
    either an error occurs or nothing is added.

*************************************************)

    Procedure Add_To_Memory( Var Memory: Memory_List_Type; Address: Pointer_Type; Var Data; Size: Word_Type;
                             Status: Status_Type );
      Var
        New_Node: Memory_Pointer_Type;
      Begin
        New_Node.Status := Status;
        New_Node.Memory_Size := Size;
        GetMem( New_Node.Memory_Pointer, New_Node.Memory_Size );
        If ( New_Node.Memory_Pointer <> Nil )
          then
            Begin
              Move( Data, New_Node.Memory_Pointer^, Size );
              New_Node.Address := Address;
              Add_Memory( Memory, New_Node );
            End
          else
            If ( Status = Status_Write )
              then
                Internal_Put_Data( Address, Data, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get from memory.
    This procedure looks in the buffer list for
    the specified address.  If it finds it, then
    the buffer data is returned to the calling
    routine, or the data is retrieved from the
    file and stored in memory for the next access.

*************************************************)

    Procedure Get_From_Memory( Var Memory: Memory_List_Type; Address: Pointer_Type; Var Data; Size: Word_Type; Keep: Boolean );
      Var
        Where: Amount_Range;
        Memory_Size: Word_Type;
      Begin
        Where := Scan_Memory( Memory, Address );
        If ( Where <> 0 )
          then
            Begin
              Rearrange_Memory( Memory, Where );
              Memory_Size := Memory.List[ 1 ].Memory_Size;
              If ( Memory_Size >= Size )
                then
                  Move( Memory.List[ 1 ].Memory_Pointer^, Data, Size )
                else
                  Move( Memory.List[ 1 ].Memory_Pointer^, Data, Memory_Size );
            End
          else
            Begin
              Internal_Get_Data( Address, Data, Size );
              If Keep
                then
                  Add_To_Memory( Memory, Address, Data, Size, Status_Read );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure Get buffer from memory.
    Like get from memory, this procedure gets the
    data from the file.  With the exception that
    this function allows a buffer size larger than
    the allocated size.

*************************************************)

    Procedure Get_Buffer_From_Memory( Var Memory: Memory_List_Type; Address: Pointer_Type; Var Data; Size: Word_Type;
                                      Keep: Boolean );
      Var
        Where: Amount_Range;
        Memory_Size: Word_Type;
      Begin
        Where := Scan_Memory( Memory, Address );
        If ( Where <> 0 )
          then
            Begin
              Rearrange_Memory( Memory, Where );
              Memory_Size := Memory.List[ 1 ].Memory_Size;
              If ( Memory_Size >= Size )
                then
                  Move( Memory.List[ 1 ].Memory_Pointer^, Data, Size )
                else
                  Move( Memory.List[ 1 ].Memory_Pointer^, Data, Memory_Size );
            End
          else
            Begin
              Internal_Get_Data( Address, Data, Size );
              If Keep
                then
                  Begin
                    Memory_Size := Read_Disk_Size( Address );
                    Add_To_Memory( Memory, Address, Data, Memory_Size, Status_Read );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put to disk.
    This procedure transfers the data to the
    allocated file space, or to the memory buffer,
    depending on if the immediate write option is
    activated.  In either case, an attempt is made
    to store the data in the memory buffer list.

*************************************************)

    Procedure Put_To_Disk( Var Memory: Memory_List_Type; Address: Pointer_Type; Var Data; Size: Word_Type );
      Var
        Where: Amount_Range;
        New_Node: Memory_Pointer_Type;
        Memory_Size: Word_Type;
      Begin
        If Immediate_Write
          then
            Internal_Put_Data( Address, Data, Size );
        Where := Scan_Memory( Memory, Address );
        If ( Where <> 0 )
          then
            Begin
              Rearrange_Memory( Memory, Where );
              Memory_Size := Memory.List[ 1 ].Memory_Size;
              If ( Memory_Size >= Size )
                then
                  Move( Data, Memory.List[ 1 ].Memory_Pointer^, Size )
                else
                  Move( Data, Memory.List[ 1 ].Memory_Pointer^, Memory_Size );
              Memory.List[ 1 ].Status := Status_Write;
            End
          else
            Add_To_Memory( Memory, Address, Data, Size, Status_Write );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Free memory.
    This function is called when file space is
    being deallocated to deallocate the memory
    buffer along with it.

*************************************************)

    Function Free_Memory( Var Memory: Memory_List_Type; Var Address: Pointer_Type ): Boolean;
      Var
        Where: Amount_Range;
      Begin
        Where := Scan_Memory( Memory, Address );
        If ( Where <> 0 )
          then
            Subtract_Memory( Memory, Where );
        Free_Memory := Internal_Dispose_Pointer( Address );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Deallocate memory.
    This procedure is called to deallocate buffer
    pointers from the bottom of the list upwards
    until there is enough free space to accommodate
    a block of the given size.

*************************************************)

    Procedure Deallocate_Memory( Var Memory: Memory_List_Type; Size: Word_Type );
      Begin
        While ( ( Memory.Amount > 0 ) and ( MaxAvail < Size ) ) do
          Begin
            Dispose_Memory( Memory.List[ Memory.Amount ], True );
            Dec( Memory.Amount );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: New heap function.
    This function replaces the old heap function
    and is called when an attempt to allocate
    heap space fails.  The purpose of this
    function is to dispose of buffer memory until
    the heap request can be satisfied.

*************************************************)

    Function New_Heap_Function( Size: Word_Type ): Integer;
      Begin
        If ( Size > 0 )
          then
            Begin
              New_Heap_Function := 0;
              Deallocate_Memory( Memory^, Size );
              If ( MaxAvail >= Size )
                then
                  New_Heap_Function := 2
                else
                  Begin
                    New_Heap_Function := 1;
                   {$IFDEF Ver60}
                    New_Heap_Function := Old_Heap_Function( Size );
                   {$ENDIF}
                   {$IFDEF OS2}
                    New_Heap_Function := Old_Heap_Function( Size );
                   {$ENDIF}
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set heap function.
    This function is called to install the new heap
    function.  It must be called somewhere in the
    program, because sometimes, a succeeding unit
    could override it.  The heap function is
    chained to the previous one.

*************************************************)

    Procedure Set_Heap_Function;
     {$IFDEF OS2}
      Var
        Pointer1,
        Pointer2: Pointer;
     {$ENDIF}
      Begin
       {$IFNDEF OS2}
        If ( HeapError <> @New_Heap_Function )
       {$ELSE}
        Pointer1 := @HeapError;
        Pointer2 := @New_Heap_Function;
        If ( Pointer1 <> Pointer2 )
       {$ENDIF}
          then
            Begin
              Old_Heap_Pointer := @HeapError;
              HeapError := @New_Heap_Function;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal flush buffers.
    This procedure flushes the entire file buffer
    structure.

*************************************************)

    Procedure Internal_Flush_Buffer( Var Memory: Memory_List_Type );
      Begin
        While ( Memory.Amount > 0 ) do
          Begin
            Dispose_Memory( Memory.List[ Memory.Amount ], True );
            Dec( Memory.Amount );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function:  Allocate data block;
    As previously defined.

*************************************************)

    Function New_Pointer( Size: Word_Type ): Pointer_Type;
      Begin
        If Allow_Buffering
          then
            Set_Heap_Function;
        New_Pointer := Initialize_New_Pointer( Size, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function:  Allocate and initialize data block;
    As previously defined.

*************************************************)

    Function Put_New_Pointer( Size: Word_Type; Var Data ): Pointer_Type;
      Begin
        If Allow_Buffering
          then
            Set_Heap_Function;
        Put_New_Pointer := Initialize_New_Pointer( Size, Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put data on disk.
    As previously defined.

*************************************************)

    Procedure Put_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Begin
        If ( Address = Nul )
          then
            Check_Error( 204, 'Put_Data: Invalid address for Address' );
        If Allow_Buffering
          then
            Put_To_Disk( Memory^, Address, Data, Size )
          else
            Internal_Put_Data( Address, Data, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get data from disk.
    As previously defined.

*************************************************)

    Procedure Get_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Begin
        If ( Address = Nul )
          then
            Check_Error( 204, 'Get_Data: Invalid address for Address' );
        If Allow_Buffering
          then
            Get_From_Memory( Memory^, Address, Data, Size, True )
          else
            Internal_Get_Data( Address, Data, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Peek data from disk.
    As previously defined.

*************************************************)

    Procedure Peek_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Begin
        If ( Address = Nul )
          then
            Check_Error( 204, 'Peek_Data: Invalid address for Address' );
        If Allow_Buffering
          then
            Get_From_Memory( Memory^, Address, Data, Size, False )
          else
            Internal_Get_Data( Address, Data, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Deallocate the data block.
    As previously defined.

*************************************************)

    Function Dispose_Pointer( Var Pointer: Pointer_Type ): Boolean;
      Begin
        If Allow_Buffering
          then
            Dispose_Pointer := Free_Memory( Memory^, Pointer )
          else
            Dispose_Pointer := Internal_Dispose_Pointer( Pointer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get buffer data from disk.
    As previously defined.

*************************************************)

    Procedure Get_Buffer_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Begin
        If ( Address = Nul )
          then
            Check_Error( 204, 'Get_Buffer_Data: Invalid address for Address' );
        If Allow_Buffering
          then
            Get_Buffer_From_Memory( Memory^, Address, Data, Size, True )
          else
            Internal_Get_Data( Address, Data, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Peek buffer data from disk.
    As previously defined.

*************************************************)

    Procedure Peek_Buffer_Data( Address: Pointer_Type; Var Data; Size: Word_Type );
      Begin
        If ( Address = Nul )
          then
            Check_Error( 204, 'Peek_Buffer_Data: Invalid address for Address' );
        If Allow_Buffering
          then
            Get_Buffer_From_Memory( Memory^, Address, Data, Size, False )
          else
            Internal_Get_Data( Address, Data, Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure:  Flush the buffers.
    As previously defined.

*************************************************)

    Procedure Flush_Buffers;
      Begin
        Internal_Flush_Buffer( Memory^ );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Initialization section.
    First, we initialize the buffer list.
    Next we link in the exit procedure.
    Then we open up our file for the data
      management.
    Then we allocate and initialize our memory
      block list management data.

*************************************************)

    Begin
      New( Memory );
      If ( Memory <> Nil )
        then
          Memory^.Amount := 0
        else
         {$IFDEF Ver40 }
          Halt( 203 );
         {$ELSE}
          RunError( 203 );
         {$ENDIF}
      New( Free_List );
      If ( Free_List <> Nil )
        then
          Begin
            { Initializing exit procedure. }
            Old_Exit_Procedure := ExitProc;
            ExitProc := @New_Exit_Procedure;
            { Initializing free lists. }
            Free_Amount := 0;
           {$IFDEF Debug}
            FillChar( Free_List^, SizeOf( File_List_Type ), 0 );
           {$ENDIF}
            Free_List^.Amount := 0;
            Free_List^.Next := Nul;
            { Initializing disk file. }
            If Generate_Unique_File
              then
                Begin
                  Assign( Memory_File, Generate_Unique_FileName );
                  Default_Drive := Unique_File_Drive;
                End
              else
                Begin
                  Assign( Memory_File, File_Name );
                  Default_Drive := 0;
                End;
            Rewrite( Memory_File, 1 );
            BlockWrite( Memory_File, Free_List, 1 );
            Free_Pointer := Allocate_List;
            If ( Free_Pointer = Nul )
              then
               {$IFDEF Ver40}
                Halt( 101 );
               {$ELSE}
                RunError( 101 );
               {$ENDIF}
            Current_Pointer := Free_Pointer;
            Write_List( Free_Pointer, Free_List^ );
          End
        else
         {$IFDEF Ver40 }
          Halt( 203 );
         {$ELSE}
          RunError( 203 );
         {$ENDIF}
    End.

