%title "VIDL86 - IBM/PC video display routine library (Intel 8088/8086)"
%pagesize 78,132

;title		: VIDL86
;version	: 1.0
;date		: Dec 15,1995
;author		: J R Ferguson
;language	: Intel 8068/8088 assembler
;O.S.		: MS-DOS v3.2
;assembler	: Borland Turbo Assembler v3.2, Ideal mode
;memory model	: compatible with TPASCAL
;function	: routine library
;usage		: link with assembler program
;
;Supports video memory access in 80 x 25 monochrome or color text mode for
;- MDA : Monochrome Video Adapter
;- HGA : Hercules Graphics Adapter
;- CGA : Color Graphics Adapter
;- EGA : Enhanced Graphics Adapter
;All display routines write to video page 0.
;
;Remarks:
;  All routines in this library expect DS to point to the common DATA
;  segment.

		ideal
		locals	@@

VidBufSegMono	=	0b000h		;video display buffer, monochrome
VidBufSegColor	=	0b800h		;video display buffer, color

EquipDatSeg	=	0040h		;BIOS equipment data area
 EquipDatByte1	=	0010h		; equipment status byte 1
  VidModeMask	=	00110000b	;  bit 4-5: initial video mode
  VidModeC40x25	=	00010000b	;   01 : color 40 x 25
  VidModeC80x25	=	00100000b	;   10 : color 80 x 25
  VidModeM80x25	=	00110000b	;   11 : monochrome 80 x 25
 EquipDatABP	=	0063h		; adapter base port
   ABP_MDA	=	03b4h		;  adapter base port value for MDA
   ABP_CGA	=	03d4h		;  adapter base port value for CGA
   ABP_HGA	=	03b4h		;  adapter base port value for HGA
   ABP_EGA	=	03d4h		;  adapter base port value for EGA

DflAttr		=	07h		;default display attribute

BS		=	08h
HT		=	09h
LF		=	0ah
CR		=	0dh

segment		DATA	byte public
		assume	ds:DATA

%newpage
;-------------------------------------------------------------------------
;Static parameters.
;-------------------------------------------------------------------------
;
;These parameters are not declared public. The _VidInit routine initializes
;them dynamically, and MUST therefore be called before any other routine in
;this library is used. Allthough an attempt is made to define a reasonable
;hard-coded initial value for them that prevents crashing in most cases, it
;is unwise to rely on that since the parameters concerning the video
;display adapter in use and the current display mode (monochrome or color)
;are hardware-dependant and can only be chosen right at run-time.
;
;Note that the hard-coded values are simply ignored when this library is
;included in a Turbo Pascal program or unit by means of the {$Iname} direc-
;tive. (This is true for Turbo Pascal release 4.0, 5.0 and 5.5 at least).
;The best way to ensure that the fields are initialized when used with
;Turbo Pascal is to include the library in a unit that calls _VidInit in
;its Initialization part.

VidStaPort	dw	ABP_CGA + 6	;video status I/O port
VidBufSeg	dw	VidBufSegColor	;video display buffer segment
VidBufOfs	dw	0000h		;current video display buffer
					;offset
SnowCheck	db	1		;avoid "snow" if <> 0
DispAttr	db	DflAttr		;current display attribute

ends		DATA

segment		CODE	byte public
		assume	cs:CODE

%newpage
;-------------------------------------------------------------------------
public _VidInit		;Initialize video parameters
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	-
;alters	AX,Flags
;
;This routine MUST be called once before any other routine in this
;library is used. It performs the following initialization:
; - Determine the type of video adapter currently in use.
; - Determine the current video mode (monochrome or color).
; - Set the current display position to row 0 and column 0.
; - Set the snow check mode on.
; - Set the current display attribute to an initial value.
;It will not alter the current screen contents and does not interfere with
;screen I/O not going through this library (for instance MS-DOS or BIOS
;calls. 
;
;The routine may be called repeatedly to re-initialize the state of the
;library. The current display position, snow check mode and current
;display attribute can be reset afterwards by means of the routines
;_VidSetPos, _VidSnow and _VidSetAttr.

proc		_VidInit
		push	es

		mov	ax,EquipDatSeg		;BIOS video data area
		mov	es,ax
		mov	ax,[es:EquipDatABP]	;video adapter base port
		add	ax,6			;video status I/O port
		mov	[VidStaPort],ax

		mov	al,[es:EquipDatByte1]	;video equipment byte 1
		and	al,VidModeMask		;initial video mode
		cmp	al,VidModeM80x25	;test for monochrome mode
		mov	ax,VidBufSegColor
		jne	@@setseg
		mov	ax,VidBufSegMono
@@setseg:	mov	[VidBufSeg],ax		;save video buffer segment

		mov	[VidBufOfs],0		;set row=0, column=0
		mov	[SnowCheck],1		;avoid "snow"
		mov	[DispAttr],DflAttr	;set default display attrib

		pop	es
		ret

endp		_VidInit

%newpage
;-------------------------------------------------------------------------
public _VidSnow		;Set snow check on/off
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	0    : snow check off (do not avoid "snow")
;		other: snow check on (avoid "snow" when writing to video)
;out	-
;alters	-
;
;Avoiding "snow" means the video write routines will wait for the start of
;a horizontal video beam retraction period before writing to the video
;buffer. This slows down the display speed a little, but avoids glitches
;("snow") on some monitors.
; 
;Note that _VidInit will set snow check on. If you want the initial snow
;check mode off, you should specify so by calling this routine after the
;call to _VidInit.
 
proc		_VidSnow

		mov	[SnowCheck],al
		ret

endp		_VidSnow

%newpage
;-------------------------------------------------------------------------
public _VidSetPos	;Set current display position
;-------------------------------------------------------------------------
;
;Register usage:
;in	BH	row    (0..24)
; 	BL	column (0..79)
;out	Flags	cond C = ok
;		cond NC= error (row or column out of screen boundaries)
;alters	Flags
;
;The row range is 0..24, the column range 0..79. The upper-left corner of
;the screen corresponds to row 0, column 0. When the row or column are out
;of screen boundaries the carry flag is set on return and the current
;position is left unchanged.
;
;Note that _VidInit sets the current display position to row 0, column 0.
;If you want another initial value, specify it by calling this routine
;after the call to _VidInit.

proc		_VidSetPos

		cmp	bh,25		;check row bound
		jnc	@@return
		cmp	bl,80		;check column bound
		jnc	@@return

		push	ax		;compute video buffer offset
		mov	al,80		;ax = ( 80 * row
		mul	bh
		add	al,bl		;	+ column
		adc	ah,0
		shl	ax,1		;     ) * 2
		mov	[VidBufOfs],ax	;save as current offset
		pop	ax

		stc			;normal exit: set carry flag
@@return:	ret

endp		_VidSetPos

%newpage
;-------------------------------------------------------------------------
public _VidGetPos	;Get current display position
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	BH	row    (0..24)
;	BL	column (0..79)
;alters	BX,Flags
;
;The row range is 0..24, the column range 0..79. The upper-left corner of
;the screen corresponds to row 0, column 0.

proc		_VidGetPos

		push	ax
		mov	ax,[VidBufOfs]		;current video buffer offset
		shr	ax,1			;2 bytes per character
		mov	bl,80			;80 characters per line
		div	bl			;al=result, ah=remainder
		mov	bh,al
		mov	bl,ah
		pop	ax
		ret

endp		_VidGetPos

%newpage
;-------------------------------------------------------------------------
public _VidSetAttr	;Set current display attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AH	attribute
;out	-
;alters	-
;
;Note that _VidInit sets the current display attribute to a default value.
;If you want another initial value, specify it by calling this routine
;after the call to _VidInit.

proc		_VidSetAttr

		mov	[DispAttr],ah
		ret

endp		_VidSetAttr

%newpage
;-------------------------------------------------------------------------
public _VidGetAttr	;Get current display attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	AH	attribute
;alters	AH
;
proc		_VidGetAttr

		mov	ah,[DispAttr]
		ret

endp		_VidGetAttr

%newpage
;-------------------------------------------------------------------------
public _VidDspChr	;Display ASCII character with current attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character code (0..255)
;out	-
;alters	AX,Flags
;
;Display a character using the current display attribute at the current
;row and column position. The current display position is incremented,
;wrapping around to the next line if needed. When the last screen position
;is written, the current position wraps around to the upperleft corner of
;the screen.
;
;The full character code value range of 0..255 is accepted. The parity bit
;is stripped before display, and the following control characters are
;recognized:
; 08h,88h: Backspace	Decrement current display position
; 09h,89h: Tab		Increment current column to next 8-position stop
; 0Ah,8Ah: Line feed	Increment current row
; 0Dh,8Dh: Return	Return to start of current row
;Control characters may cause the current display position to wrap around.

proc		_VidDspChr

		and	al,7Fh			;strip parity bit
		cmp	al,CR			;check if control character
		je	@@cr
		cmp	al,LF
		je	@@lf
		cmp	al,HT
		je	@@ht
		cmp	al,BS
		je	@@bs
		call	_VidPutChr		;display characters
		ret

		;Return
@@cr:		push	bx
		mov	ax,[VidBufOfs]		;get video buffer offset
		mov	bl,80*2			;get nr of bytes per line
		div	bl			;compute current line nr
		mul	bl			;rescale to offset (column 0)
		mov	[VidBufOfs],ax		;store new offset
		pop	bx
		ret

		;Line feed
@@lf:		add	[VidBufOfs],80*2	;skip to next line
		cmp	[VidBufOfs],25*80*2	;wrap if needed
		jb	@@lf1
		sub	[VidBufOfs],25*80*2
@@lf1:		ret

		;Tab
@@ht:		push	bx
		mov	al,' '
		mov	ah,[DispAttr]
		call	_VidGetPos
@@ht1:		call	_VidPutChrAttr		;put space
		inc	bl			;until column at tabstop
		and	bl,07h
		jnz	@@ht1
		pop	bx
		ret

		;Backspace
@@bs:		sub	[VidBufOfs],2		;decrement current position
		jnc	@@bs1			;wrap around if needed
		add	[VidBufOfs],25*80*2
@@bs1:		ret

endp		_VidDspChr

%newpage
;-------------------------------------------------------------------------
public _VidPutChr	;Display literal character with current attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character code (0..255)
;out	-
;alters	AH,Flags
;
;Display a character using the current display attribute at the current
;row and column position. The current display position is incremented,
;wrapping around to the next line if needed. When the last screen position
;is written, the current position wraps around to the upperleft corner of
;the screen.
;
;The full character code value range of 0..255 is accepted. Special
;characters like Return, Linefeed, Backspace, Horizontal Tab and Bell are
;not recognized as such but are displayed as found in the computer's
;character font table.

label		_VidPutChr	proc

		mov	ah,[DispAttr]
;proceed with	jmp	_VidPutChrAttr

%newpage
;-------------------------------------------------------------------------
public _VidPutChrAttr  ;Display literal character with specified attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character code (0..255)
;	AH	display attribute
;out	-
;alters	Flags
;
;Display a character using the specified display attribute at the current
;row and column position. The current display position is incremented,
;wrapping around to the next line if needed. When the last screen position
;is written, the current position wraps around to the upper-left corner of
;the screen. The current display attribute is left unchanged.
;
;The full character code value range of 0..255 is accepted. Special
;characters like Return, Linefeed, Backspace, Horizontal Tab and Bell are
;not recognized as such but are displayed as found in the computer's
;character font table.

proc		_VidPutChrAttr

		push	es
		push	di

		mov	es,[VidBufSeg]
		mov	di,[VidBufOfs]
		cmp	[SnowCheck],0		;avoid snow?
		je	@@put			;no: skip synchronizing
		push	dx
		mov	dx,[VidStaPort]
		cli				;disable interrupts
		push	ax			;save character + attribute
@@wait1:	in	al,dx			;if currently in retrace mode
		test	al,1			;then wait until it ends
		jnz	@@wait1
@@wait2:	in	al,dx			;wait for start of new retrace
		test	al,1
		jz	@@wait2
		pop	ax			;restore character+attribute
		stosw				;store them in video buffer
		sti				;enable interrupts
		pop	dx
		jmp	short @@wrap

@@put:		stosw				;store char + attribute
@@wrap:		cmp	di,25*80*2		;end of screen?
		jb	@@pos
		mov	di,0			;yes: set to start
@@pos:		mov	[VidBufOfs],di		;update screen position

		pop	di
		pop	es
		ret

endp		_VidPutChrAttr

%newpage
;-------------------------------------------------------------------------
public _VidChgAttr	;Change display attribute at current position
;-------------------------------------------------------------------------
;
;Register usage:
;in	AH	attribute
;out	-
;alters	Flags
;
;Change the display attribute of the character displayed at the current
;row and column position without altering its contents nor the current
;(default) display attribute used with the _VidDsp... and _VidPut...
;routines. The current display position is incremented, wrapping around
;if needed.

proc		_VidChgAttr

		push	es
		push	di

		mov	es,[VidBufSeg]
		mov	di,[VidBufOfs]
		inc	di			;point to attribute byte
		mov	al,ah			;new attribute in al
		cmp	[SnowCheck],0		;avoid snow?
		je	@@put			;no: skip synchronizing
		push	dx
		mov	dx,[VidStaPort]
		cli				;disable interrupts
		push	ax			;save character + attribute
@@wait1:	in	al,dx			;if currently in retrace mode
		test	al,1			;then wait until it ends
		jnz	@@wait1
@@wait2:	in	al,dx			;wait for start of new retrace
		test	al,1
		jz	@@wait2
		pop	ax			;restore attribute
		stosb				;store it in video buffer
		sti				;enable interrupts
		pop	dx
		jmp	short @@wrap

@@put:		stosb				;store attribute
@@wrap:		cmp	di,25*80*2		;end of screen?
		jb	@@pos
		mov	di,0			;yes: set to start
@@pos:		mov	[VidBufOfs],di		;update screen position

		pop	di
		pop	es
		ret

endp		_VidChgAttr

%newpage
;-------------------------------------------------------------------------
public _VidChgChr	;Change character at current position
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character
;out	-
;alters	Flags
;
;Change the character displayed at the current row and column position
;without altering its attribute. The current display position is
;incremented, wrapping around if needed.

proc		_VidChgChr

		push	es
		push	di

		mov	es,[VidBufSeg]
		mov	di,[VidBufOfs]
		cmp	[SnowCheck],0		;avoid snow?
		je	@@put			;no: skip synchronizing
		push	dx
		mov	dx,[VidStaPort]
		cli				;disable interrupts
		push	ax			;save character + attribute
@@wait1:	in	al,dx			;if currently in retrace mode
		test	al,1			;then wait until it ends
		jnz	@@wait1
@@wait2:	in	al,dx			;wait for start of new retrace
		test	al,1
		jz	@@wait2
		pop	ax			;restore attribute
		stosb				;store it in video buffer
		sti				;enable interrupts
		pop	dx
		jmp	short @@wrap

@@put:		stosb				;store attribute
@@wrap:		inc	di			;skip attribute position
		cmp	di,25*80*2		;end of screen?
		jb	@@pos
		mov	di,0			;yes: set to start
@@pos:		mov	[VidBufOfs],di		;update screen position

		pop	di
		pop	es
		ret

endp		_VidChgChr

%newpage
;-------------------------------------------------------------------------
public _VidRetChrAttr	;Retrieve character,attribute at current position
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	AL	character
;	AH	attribute
;alters	AX, Flags
;
;Get the contents and the display attribute of the character displayed
;at the current row and column position. The current display position is
;incremented, wrapping around if needed.

proc		_VidRetChrAttr

		push	es
		push	di

		mov	es,[VidBufSeg]
		mov	di,[VidBufOfs]
		mov	ax,[es:di]		;get character + attribute
		inc	di			;point to next position
		inc	di
		cmp	di,25*80*2		;end of screen?
		jb	@@pos
		mov	di,0			;yes: set to start
@@pos:		mov	[VidBufOfs],di		;update screen position

		pop	di
		pop	es
		ret

endp		_VidRetChrAttr

%newpage
;-------------------------------------------------------------------------
public _VidPutNChr	;Display literal character repeatedly
;			;using the current display attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character code (0..255)
;	CX	repeat count
;out	-
;alters	AH, CX, Flags
;
;Display character AL using the current display attribute CX times in a
;row, starting at the current display position. The display position is
;adjusted with each character and may wrap aroud to a following line or to
;the top of the page.
;
;The full character code value range of 0..255 is accepted. Special
;characters like Return, Linefeed, Backspace, Horizontal Tab and Bell are
;not recognized as such but are displayed as found in the computer's
;character font table.

label		_VidPutNChr	proc

		mov	ah,[DispAttr]
;proceed with	jmp	_VidPutNChrAttr

%newpage
;-------------------------------------------------------------------------
public _VidPutNChrAttr	;Display literal character repeatedly
;			;using a specified display attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character code (0..255)
;	AH	display attribute
;	CX	repeat count
;out	-
;alters	CX, Flags
;
;Display character AL using attribute AH for CX times in a row, starting
;at the current display position. The display position is adjusted with
;each character and may wrap aroud to a following line or to the top of
;the page. The current display attribute is left unchanged.
;
;The full character code value range of 0..255 is accepted. Special
;characters like Return, Linefeed, Backspace, Horizontal Tab and Bell are
;not recognized as such but are displayed as found in the computer's
;character font table.

proc		_VidPutNChrAttr

		jcxz	@@exit
@@put:		call	_VidPutChrAttr
		loop	@@put
@@exit:		ret

endp		_VidPutNChrAttr

%newpage
;-------------------------------------------------------------------------
public _VidChgNAttr	;Change attribute repeatedly
;-------------------------------------------------------------------------
;
;Register usage:
;in	AH	attribute
;in	CX	repeat count
;out	-
;alters	CX, Flags
;
;Change the display attribute for CX consecutive characters starting at
;the current display position, without altering their contents or the
;current display attribute.
;The current display position is incremented by CX, wrapping around if
;needed.

proc		_VidChgNAttr

		jcxz	@@exit
@@put:		call	_VidChgAttr
		loop	@@put
@@exit:		ret

endp		_VidChgNAttr

%newpage
;-------------------------------------------------------------------------
public _VidChgNChr	;Change character repeatedly
;-------------------------------------------------------------------------
;
;Register usage:
;in	AL	character
;	CX	repeat count
;out	-
;alters	CX, Flags
;
;Change the contents of CX consecutive characters to AL, starting at
;the current display position, without altering their display attribute.
;The current display position is incremented by CX, wrapping around if
;needed.

proc		_VidChgNChr

		jcxz	@@exit
@@put:		call	_VidChgChr
		loop	@@put
@@exit:		ret

endp		_VidChgNChr

%newpage
;-------------------------------------------------------------------------
public _VidDspStz	;Display 0-ended ASCII string
;-------------------------------------------------------------------------
;
;Register usage:
;in	ES:DI	string address
;out	DI	position following concluding 0-byte
;alters	AX,DI,Flags
;
;Display a 0-ended character string using the current display attribute at
;the current row and column position. The current display position is
;incremented with each character, wrapping around to the next line if
;needed. When the last screen position is written, the current position
;wraps around to the upperleft corner of the screen.
;
;The full character code value range of 0..255 is accepted. The parity bit
;is stripped before display, and the following control characters are
;recognized:
;
; 08h,88h: Backspace	Decrement current display position
; 09h,89h: Tab		Increment current column to next 8-position stop
; 0Ah,8Ah: Line feed	Increment current row
; 0Dh,8Dh: Return	Return to start of current row
;
;Control characters may cause the current display position to wrap around.
;
;Remarks:
; The string may not exceed its segment.

proc		_VidDspStz

@@put:		mov	al,[es:di]
		inc	di
		and	al,al
		jz	@@endstr
		call	_VidDspChr
		jmp	short @@put
@@endstr:	ret

endp		_VidDspStz

%newpage
;-------------------------------------------------------------------------
public _VidPutStz	;Display literal 0-ended string
;-------------------------------------------------------------------------
;
;Register usage:
;in	ES:DI	string address
;out	DI	position following concluding 0-byte
;alters	AX,DI,Flags
;
;Display a 0-ended character string using the current display attribute at
;the current row and column position. The current display position is
;incremented with each character, wrapping around to the next line if
;needed. When the last screen position is written, the current position
;wraps around to the upperleft corner of the screen.
;
;The full character code value range of 0..255 is accepted. Special
;characters like Return, Linefeed, Backspace, Horizontal Tab and Bell are
;not recognized as such but are displayed as found in the computer's
;character font table.
;
;Remarks:
; The string may not exceed its segment.

proc		_VidPutStz

		mov	ah,[DispAttr]
@@put:		mov	al,[es:di]
		inc	di
		and	al,al
		jz	@@endstr
		call	_VidPutChrAttr
		jmp	short @@put
@@endstr:	ret

endp		_VidPutStz

%newpage
;-------------------------------------------------------------------------
public _VidPutStp	;Display Turbo Pascal type ASCII string
;-------------------------------------------------------------------------
;
;Register usage:
;in	ES:DI	string address (first byte = string length)
;out	DI	position following the last valid character of the string
;alters	AX,DI,Flags
;
;Display a Turbo Pascal type string using the current display attribute at
;the current row and column position. The current display position is
;incremented with each character, wrapping around to the next line if
;needed. When the last screen position is written, the current position
;wraps around to the upperleft corner of the screen.
;
;The full character code value range of 0..255 is accepted. Special
;characters like Return, Linefeed, Backspace, Horizontal Tab and Bell are
;not recognized as such but are displayed as found in the computer's
;character font table.
;
;The string consists of:
; 1 byte : string length n (0..255)
; n bytes: string characters
;
;Remarks:
; The string may not exceed its segment.

proc		_VidPutStp

		push	cx
		mov	ch,0
		mov	cl,[es:di]
		inc	di
		jcxz	@@exit
		mov	ah,[DispAttr]
@@put:		mov	al,[es:di]
		inc	di
		call	_VidPutChrAttr
		loop	@@put

@@exit:		pop	cx
		ret

endp		_VidPutStp

%newpage
;-------------------------------------------------------------------------
public _VidDspStp	;Display Turbo Pascal type literal string
;-------------------------------------------------------------------------
;
;Register usage:
;in	ES:DI	string address (first byte = string length)
;out	DI	position following the last valid character of the string
;alters	AX,DI,Flags
;
;Display a Turbo Pascal type string using the current display attribute at
;the current row and column position. The current display position is
;incremented with each character, wrapping around to the next line if
;needed. When the last screen position is written, the current position
;wraps around to the upperleft corner of the screen.
;
;The full character code value range of 0..255 is accepted. The parity bit
;is stripped before display, and the following control characters are
;recognized:
;
; 08h,88h: Backspace	Decrement current display position
; 09h,89h: Tab		Increment current column to next 8-position stop
; 0Ah,8Ah: Line feed	Increment current row
; 0Dh,8Dh: Return	Return to start of current row
;
;Control characters may cause the current display position to wrap around.
;
;The string consists of:
; 1 byte : string length n (0..255)
; n bytes: string characters
;
;Remarks:
; The string may not exceed its segment.

proc		_VidDspStp

		push	cx
		mov	ch,0
		mov	cl,[es:di]
		inc	di
		jcxz	@@exit
@@put:		mov	al,[es:di]
		inc	di
		call	_VidDspChr
		loop	@@put

@@exit:		pop	cx
		ret

endp		_VidDspStp

%newpage
;-------------------------------------------------------------------------
public _VidClrEol	;Clear to end of line using current attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	-
;alters	AH, Flags
;
;Clears from current position to the end of the line by writing spaces
;with the current attribute. The current display position is not changed.

label		_VidClrEol	proc

		mov	ah,[DispAttr]
;proceed with	jmp	_VidClrEolAttr

%newpage
;-------------------------------------------------------------------------
public _VidClrEolAttr	;Clear to end of line using specified attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AH	display attribute
;out	-
;alters	Flags
;
;Clears from current position to the end of the line by writing spaces
;with the specified attribute. The current display position is not
;changed.

proc		_VidClrEolAttr

		push	ax
		push	bx
		push	cx
		push	[VidBufOfs]		;save current position
		call	_VidGetPos		;get current column
		mov	cx,80			;compute number of chars
		sub	cl,bl
		mov	al,' '			;fill with spaces
		call	_VidPutNChrAttr
		pop	[VidBufOfs]		;restore original position
		pop	cx
		pop	bx
		pop	ax

		ret

endp		_VidClrEolAttr

%newpage
;-------------------------------------------------------------------------
public _VidClrEos	;Clear to end of screen using current attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	-
;alters	AH, Flags
;
;Clears from current position to the end of the screen by writing spaces
;with the current attribute. The current display position is not changed.

label		_VidClrEos	proc

		mov	ah,[DispAttr]
;proceed with	jmp	_VidClrEosAttr

%newpage
;-------------------------------------------------------------------------
public _VidClrEosAttr	;Clear Screen using the specified attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AH	display attribute
;out	-
;alters	Flags
;
;Clears from current position to the end of the screen by writing spaces
;with the specified attribute. The current display position is not
;changed.

proc		_VidClrEosAttr

		push	ax
		push	bx
		push	cx
		mov	bx,[VidBufOfs]		;get current position
		mov	cx,25*80*2		;compute number of chars
		sub	cx,bx
		shr	cx,1
		mov	al,' '			;fill with spaces
		call	_VidPutNChrAttr
		mov	[VidBufOfs],bx		;reposition
		pop	cx
		pop	bx
		pop	ax

		ret

endp		_VidClrEosAttr

%newpage
;-------------------------------------------------------------------------
public _VidClrScr	;Clear Screen using current attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	-
;out	-
;alters	AH, Flags
;
;Clears the entire screen, writing spaces with the current attribute to
;all rows and columns. Repositions the current display position to row 0,
;column 0 afterwards.

label		_VidClrScr	proc

		mov	ah,[DispAttr]
;proceed with	jmp	_VidClrScrAttr

%newpage
;-------------------------------------------------------------------------
public _VidClrScrAttr	;Clear Screen using the specified attribute
;-------------------------------------------------------------------------
;
;Register usage:
;in	AH	display attribute
;out	-
;alters	Flags
;
;Clears the entire screen, writing spaces with the specified attribute to
;all rows and columns. Repositions the current display position to row 0,
;column 0 afterwards. The current display attribute is left unchanged.

proc		_VidClrScrAttr

		push	ax
		push	cx
		mov	[VidBufOfs],0		;position to row 0, col 0
		mov	al,' '			;fill screen with spaces
		mov	cx,25*80		;25 lines, 80 cols
		call	_VidPutNChrAttr		;(wraps to row 0,col 0)
		pop	cx
		pop	ax

		ret

endp		_VidClrScrAttr

ends		CODE

		end
