{ DTMLIB.PAS : Date manipulation routine library

  Title   : DTMLIB
  Version : 2.0
  Date    : Nov 10,1996
  Language: Borland Pascal 4.0 through 7.0 (all targets)
            Borland Delphi 1.0
  Author  : J R Ferguson
  Usage   : Unit

  Supports dates from Jan 01,1901 through Dec 31,2099.
}

UNIT DtmLib;


INTERFACE
Uses
{$IFDEF WINDOWS}
  {$IFDEF VER80}
    SysUtils;
  {$ELSE}
    WinDos;
  {$ENDIF}
{$ELSE}
  Dos;
{$ENDIF}

type

  DtmFnTyp      { function code :                                         }
  = ( DtmFnCur, {   Current date                                          }
      DtmFnIdf, {   Internal Date Format                                  }
      DtmFnJul, {   Julian Date                                           }
      DtmFnYmd, {   Year, Month, Day of month                             }
      DtmFnCal  {   Office-Calendar Year, Week number, Day of Week        }
    );

  DtmRcTyp      { return code :                                           }
  = ( DtmRcOk , {   ok     : success                        date valid    }
      DtmRcWrn, {   warning: out of scale                   date rescaled }
      DtmRcRng, {   error  : out of range                   date invalid  }
      DtmRcFun  {   error  : unknown function code          date invalid  }
    );

  DtmIdfTyp    = longint;      { number of days since a fixed reference   }
                               { date used throughout this unit.          }

  DtmJulTyp    = record        { Julian date :                            }
                   Year : word;{   year                      (1901..2099) }
                   Day  : word {   day of the year               (1..366) }
                 end;

  DtmYmdTyp    = record        { Normal calendar date :                   }
                   Year : word;{   year                      (1901..2099) }
                   Month: byte;{   month                          (1..12) }
                   Day  : byte {   day of the month               (1..31) }
                 end;

  DtmCalTyp    = record        { Office calendar date :                   }
                   Year: word; {   office calendar year      (1901..2099) }
                   Week: byte; {   week number                    (1..53) }
                   Day : byte  {   Day of the week (1=monday .. 7=sunday  }
                 end;
                 { Week 1 contains the first thursday of the Ymd-year     }

  DtmDateRec   = record
                   Fn  : DtmFnTyp;   { in     : function code             }
                   Rc  : DtmRcTyp;   { out    : return code               }
                   Idf : DtmIdfTyp;  { in/out : internal date format      }
                   Jul : DtmJulTyp;  { in/out : julians date              }
                   Ymd : DtmYmdTyp;  { in/out : normal calendar date      }
                   Cal : DtmCalTyp;  { in/out : office calendar date      }
                 end;

function DtmConvert  (var Date: DtmDateRec): boolean;
{ Convert date format, get current date, or check date validity.
  Usage:
  1. Get current date
     a. Set Date.Fn to DtmFnCur.
     b. Execute function DtmConvert(Date).
     c. Read the current date information you need from Date.Idf,
        Date.Jul, Date.Ymd and/or Date.Cal.
  2. Check date validity.
     a. Fill Date.Idf, Date.Jul, Date.Ymd or Date.Cal.
     b. Set Date.Fn to the corresponding function code.
     c. Execute function DtmConvert(Date). Check the functon result
        to see if the date is valid.
     d. If the function result is false, optionally check Date.Rc for
        more information on the type of error.
  3. Convert date format.
     a. through d. : same as 2a. through 2d.
     e. If the conversion was successfull (function result TRUE, Date.Rc
        = DtmRcOk) the fields Date.Idf, Date.Jul, Date.Ymd and/or Date.Cal
        contain the date information in each date format.

        If the conversion results in a warning return code (function result
        is FALSE, Date.Rc = DtmRcWarning), this means the date was invalid
        but could be rescaled, and all date fields are now filled with
        valid date information. Examples: 'March 3' will be rescaled to
        'April 2', and 'January 0' will become 'Decenber 31' of the 
        previous year.
}

function DtmAdd      (var DateBeg : DtmDateRec;  { in : starting date    }
                          Days    : longint;     { in : number of days   }
                      var DateEnd : DtmDateRec   { out: ending date      }
                     )            : boolean;     { out: function result  }
{ Add/subtract days to/from starting date, get ending date.
  Usage:
  a. Fill DateBeg.Fn and - unless it is DtmFnCur - the corresponding date
     information. This is the starting date.
  b. Execute the function with Days set to the number of days you want to
     add to this starting date, and DateEnd as an output parameter.
  c. Check the function result. If it is TRUE, DateEnd contains the ending
     date in all formats. If it is FALSE, DateBeg.Rc and DateEnd.Rc can be
     checked - in that order - to obtain information on the type of error.
  To subtract days from a date, add a negative number of days to DateBeg.
  You will then get a DateEnd value that is before DateBeg.
}

function DtmSubtract (var DateEnd : DtmDateRec; { in : ending date      }
                      var DateBeg : DtmDateRec; { in : starting date    }
                      var Days    : longint     { out: number of days   }
                     )            : boolean;    { out: function result  }
{ Subtract to dates, get number of days in-between.
  Usage:
  a. Fill the functioncode and - if it is not DtmFnCur - the corresponding
     date fields of the input parameters DateEnd and DateBeg.
  b. Execute the function and check the function result.
  c. If the function result is TRUE, output parameter Days is set to the
     number of days between DateBeg and DateEnd. The value of Days will be
     positive if DateEnd comes later then DateBeg, negative if DateEnd
     comes sooner, and zero if both dates are the same.
     If the function result is FALSE, DateBeg.Rc and DateEnd.Rc can be
     checked - in that order - to obtain information on the type of error.
}



IMPLEMENTATION

const
  IdfMin       = 694325;
  IdfMax       = 767008;
  JulOfsIdf    = 694325;
  JulOfsYear   = 1901;
  JulOfsDay    = 001;
  YmdOfsIdf    = 694325;
  YmdOfsYear   = 1901;
  YmdOfsMonth  = 01;
  YmdOfsDay    = 01;
  CalOfsIdf    = 691769;
  CalOfsYear   = 1894;
  CalOfsWeek   = 01;
  CalOfsDay    = 01;

const
  MonthLen     : array[{leap_year: }boolean, {month:} 1..12] of integer
               = ((000,031,059,090,120,151,181,212,243,273,304,334),
                  (000,031,060,091,121,152,182,213,244,274,305,335));

  WeekTable    : array[1..28] of word
               = (   0,  52, 104, 156, 209, 261, 313, 365, 417, 469,
                   522, 574, 626, 678, 730, 783, 835, 887, 939, 991,
                  1043,1096,1148,1200,1252,1304,1356,1409);
  WeekTbLen    = 28;
  WeekMultiple = 1461;
  DaysMultiple = 7 * WeekMultiple;


{--- local routines ---}

function LeapYear(Year: word): boolean;
begin LeapYear:= ((Year and 3) = 0) and (Year <> 1900); end;

procedure JulToIdf(var dt: DtmDateRec);
begin with dt,Jul do begin
  Idf:= JulOfsIdf
        + trunc((longint(Year) - longint(JulOfsYear)) * 365.25)
        + Day - JulOfsDay;
  Rc := DtmRcOk;
end end;

procedure YmdToIdf(var dt: DtmDateRec);
var years,months: longint;
begin with dt,Ymd do begin
  years:= Year; months:= Month;
  while months < 01 do begin Inc(months,12); Dec(years) end;
  while months > 12 do begin Dec(months,12); Inc(years) end;
  Idf:= YmdOfsIdf
        + trunc( (longint(years) - longint(YmdOfsYear)) * 365.25 )
        + MonthLen[LeapYear(year),months]
        + Day - YmdOfsDay;
  Rc:= DtmRcOk;
end end;

procedure CalToIdf(var dt: DtmDateRec);
var years,weeks,days,MultipleCount: longint;
begin with dt do begin
  years:= longint(Cal.Year) - longint(CalOfsYear);
  weeks:= Cal.Week - CalOfsWeek;
  days := Cal.Day  - CalOfsDay;
  Idf  := CalOfsIdf;
  if years < 0 then begin  {non-negative value required for div and mod}
    MultipleCount:= (-years) div WeekTbLen + 1;
    Inc(years,MultipleCount*WeekTbLen);
    Dec(Idf,MultipleCount*DaysMultiple);
  end;
  Inc(Idf,( (years div WeekTbLen) * WeekMultiple
            + WeekTable[1 + years mod WeekTbLen]
            + weeks
          ) * 7
          + days);
  Rc:= DtmRcOk;
end end;

procedure CurToIdf(var dt: DtmDateRec);
var CurYear, CurMonth, CurDay, CurDayOfWeek: word;
begin
{$IFDEF VER80}
  DecodeDate(Date,CurYear,CurMonth,CurDay);
{$ELSE}
  GetDate(CurYear,CurMonth,CurDay,CurDayOfWeek);
{$ENDIF}
  with dt.Ymd do begin Year:= CurYear; month:= CurMonth; Day:= CurDay end;
  YmdToIdf(dt);
end;

procedure IdfToJul(var dt: DtmDateRec);
var years, days, quotient: longint;
begin with dt do begin
  Rc:= DtmRcOk;
  days   := Idf - JulOfsIdf;
  quotient:= days div 1461; days:= days mod 1461; years:= 4 * quotient;
  quotient:= days div  365; days:= days mod  365;
  if quotient=4 then begin dec(quotient); inc(days,365) end;
  years:= years + quotient + JulOfsYear;
  Inc(days,JulOfsDay);
  if Fn=DtmFnJul then
     if (years <> Jul.Year) or (days <> Jul.Day) then Rc:= DtmRcWrn;
  Jul.Year:= years; Jul.Day:= days;
end end;

procedure IdfToYmd(var dt: DtmDateRec);
  var years, months, days, quotient: longint;
      leap: boolean;
begin with dt do begin
  Rc:= DtmRcOk;
  days    := Idf - YmdOfsIdf;
  quotient:= days div 1461; days:= days mod 1461;
  years   := YmdOfsYear + 4 * quotient;
  quotient:= days div 365;  days:= days mod 365;
  if quotient=4 then begin dec(quotient); Inc(days,365) end;
  Inc(years,quotient);
  months := 12; leap:= LeapYear(years);
  while MonthLen[leap,months] > days do dec(months);
  days:= days - MonthLen[leap,months] + YmdOfsDay;
  if Fn=DtmFnYmd then
     if (years <> Ymd.Year) or (months <> Ymd.Month) or (days <> Ymd.Day)
     then Rc:= DtmRcWrn;
  Ymd.Year := years;
  Ymd.Month:= months;
  Ymd.Day  := days;
end end;

procedure IdfToCal(var dt: DtmDateRec);
var years, weeks, days, quotient: longint;
    i: 0..WeekTbLen;
begin with dt do begin
  Rc:= DtmRcOk;
  days  := Idf - CalOfsIdf;
  weeks := days div 7; days:= days mod 7;
  years := 0;
  while weeks < 0 do begin
    Inc(weeks,WeekMultiple); Dec(years,WeekTbLen);
  end;
  while weeks >= WeekMultiple do begin
    dec(weeks,WeekMultiple); inc(years,WeekTbLen);
  end;
  i:= WeekTbLen; years:= years + WeekTbLen - 1;
  while WeekTable[i] > weeks do begin dec(i); dec(years) end;
  Dec(weeks,WeekTable[i]);
  Inc(years,CalOfsYear);
  Inc(weeks,CalOfsWeek);
  Inc(days ,CalOfsDay ); if days>7 then Dec(days,7);
  if Fn=DtmFnCal then
     if (years <> Cal.Year) or (weeks <> Cal.Week) or (days <> Cal.Day)
     then Rc:= DtmRcWrn;
  Cal.Year:= years;
  Cal.Week:= weeks;
  Cal.Day := days;
end end;



{--- interface routines ---}

function DtmConvert  (var Date: DtmDateRec): boolean;
var dt: DtmDateRec; MaxRc: DtmRcTyp;
begin with dt do begin
  dt:= Date;
  MaxRc:= DtmRcOk;
  case Fn of
    DtmFnCur : CurToIdf(dt);
    DtmFnIdf : Rc:= DtmRcOk;
    DtmFnJul : JulToIdf(dt);
    DtmFnYmd : YmdToIdf(dt);
    DtmFnCal : CalToIdf(dt);
    else       Rc:= DtmRcFun;
  end;
  if Rc > MaxRc then MaxRc:= Rc;
  if MaxRc <= DtmRcWrn then begin
    if (Idf < IdfMin) or (Idf > IdfMax) then MaxRc:= DtmRcRng
    else begin
      IdfToJul(dt); if Rc > MaxRc then MaxRc:= Rc;
      if MaxRc <= DtmRcWrn then begin
        IdfToYmd(dt); if Rc > MaxRc then MaxRc:= Rc;
        if MaxRc <= DtmRcWrn then begin
          IdfToCal(dt); if Rc > MaxRc then MaxRc:= Rc;
        end;
      end;
    end;
  end;
  Rc:= MaxRc;
  if (Rc <= DtmRcWrn) then Date:= dt else Date.Rc:= Rc;
  DtmConvert:= Rc=DtmRcOk;
end end;


function DtmAdd      (var DateBeg : DtmDateRec;
                          Days    : longint;
                      var DateEnd : DtmDateRec ): boolean;
var ok1, ok2: boolean;
begin
  ok1:= DtmConvert(DateBeg);
  if DateBeg.Rc > DtmRcWrn then begin
    DateEnd.Rc:= DateBeg.Rc;
    DtmAdd:= false;
  end
  else with DateEnd do begin
    Idf:= DateBeg.Idf + days;
    Fn := DtmFnIdf;
    ok2:= DtmConvert(DateEnd);
    DtmAdd:= ok1 and ok2;
  end;
end;


function DtmSubtract (var DateEnd : DtmDateRec;
                      var DateBeg : DtmDateRec;
                      var Days    : longint    ): boolean;
var ok1, ok2: boolean;
begin
  ok1:= DtmConvert(DateEnd);
  if DateEnd.Rc > DtmRcWrn then begin
    DateBeg.Rc := DateEnd.Rc;
    DtmSubtract:= false;
  end
  else begin
    ok2:= DtmConvert(DateBeg);
    if DateBeg.Rc > DtmRcWrn then DtmSubtract:= false
    else begin
      days:= DateEnd.Idf - DateBeg.Idf;
      DtmSubtract:= ok1 and ok2;
    end;
  end;
end;


END.
